"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.Operator = void 0;
const node_util_1 = require("node:util");
const sqlstring_1 = require("./sqlstring");
const literals_1 = __importDefault(require("./literals"));
const channels_1 = __importDefault(require("./channels"));
const debug = (0, node_util_1.debuglog)('ali-rds:operator');
/**
 * Operator Interface
 */
class Operator {
    #connection;
    constructor(connection) {
        if (connection) {
            this.#connection = connection;
        }
    }
    beforeQueryHandlers = [];
    afterQueryHandlers = [];
    get literals() { return literals_1.default; }
    get threadId() {
        return this.#connection?.threadId;
    }
    beforeQuery(beforeQueryHandler) {
        this.beforeQueryHandlers.push(beforeQueryHandler);
    }
    afterQuery(afterQueryHandler) {
        this.afterQueryHandlers.push(afterQueryHandler);
    }
    escape(value, stringifyObjects, timeZone) {
        return sqlstring_1.SqlString.escape(value, stringifyObjects, timeZone);
    }
    escapeId(value, forbidQualified) {
        return sqlstring_1.SqlString.escapeId(value, forbidQualified);
    }
    format(sql, values, stringifyObjects, timeZone) {
        // if values is object, not null, not Array;
        if (!Array.isArray(values) && typeof values === 'object' && values !== null) {
            // object not support replace column like ??;
            return sql.replace(/\:(\w+)/g, (text, key) => {
                if (values.hasOwnProperty(key)) {
                    return sqlstring_1.SqlString.escape(values[key]);
                }
                // if values don't hasOwnProperty, return origin text;
                return text;
            });
        }
        return sqlstring_1.SqlString.format(sql, values, stringifyObjects, timeZone);
    }
    async query(sql, values) {
        // query(sql, values)
        if (values) {
            sql = this.format(sql, values);
        }
        if (this.beforeQueryHandlers.length > 0) {
            for (const beforeQueryHandler of this.beforeQueryHandlers) {
                const newSql = beforeQueryHandler(sql);
                if (newSql) {
                    sql = newSql;
                }
            }
        }
        debug('[connection#%s] query %o', this.threadId, sql);
        const queryStart = performance.now();
        let rows;
        let lastError;
        channels_1.default.queryStart.publish({
            sql,
            connection: this.#connection,
        });
        try {
            rows = await this._query(sql);
            if (Array.isArray(rows)) {
                debug('[connection#%s] query get %o rows', this.threadId, rows.length);
            }
            else {
                debug('[connection#%s] query result: %o', this.threadId, rows);
            }
            return rows;
        }
        catch (err) {
            lastError = err;
            err.stack = `${err.stack}\n    sql: ${sql}`;
            debug('[connection#%s] query error: %o', this.threadId, err);
            throw err;
        }
        finally {
            const duration = Math.floor((performance.now() - queryStart) * 1000) / 1000;
            channels_1.default.queryEnd.publish({
                sql,
                connection: this.#connection,
                duration,
                error: lastError,
            });
            if (this.afterQueryHandlers.length > 0) {
                for (const afterQueryHandler of this.afterQueryHandlers) {
                    afterQueryHandler(sql, rows, duration, lastError);
                }
            }
        }
    }
    async queryOne(sql, values) {
        const rows = await this.query(sql, values);
        return rows && rows[0] || null;
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    async _query(_sql) {
        throw new Error('SubClass must impl this');
    }
    async count(table, where) {
        const sql = this.format('SELECT COUNT(*) as count FROM ??', [table]) +
            this._where(where);
        debug('count(%j, %j) \n=> %j', table, where, sql);
        const rows = await this.query(sql);
        return rows[0].count;
    }
    /**
     * Select rows from a table
     *
     * @param  {String} table     table name
     * @param  {Object} [option] optional params
     *  - {Object} where          query condition object
     *  - {Array|String} columns  select columns, default is `'*'`
     *  - {Array|String} orders   result rows sort condition
     *  - {Number} limit          result limit count, default is no limit
     *  - {Number} offset         result offset, default is `0`
     * @return {Array} result rows
     */
    async select(table, option) {
        option = option || {};
        const sql = this._selectColumns(table, option.columns) +
            this._where(option.where) +
            this._orders(option.orders) +
            this._limit(option.limit, option.offset);
        debug('select(%o, %o) \n=> %o', table, option, sql);
        return await this.query(sql);
    }
    async get(table, where, option) {
        option = option || {};
        option.where = where;
        option.limit = 1;
        option.offset = 0;
        const rows = await this.select(table, option);
        return rows && rows[0] || null;
    }
    async insert(table, rows, option) {
        option = option || {};
        let insertRows;
        let firstObj;
        // insert(table, rows)
        if (Array.isArray(rows)) {
            firstObj = rows[0];
            insertRows = rows;
        }
        else {
            // insert(table, row)
            firstObj = rows;
            insertRows = [rows];
        }
        if (!option.columns) {
            option.columns = Object.keys(firstObj);
        }
        const params = [table, option.columns];
        const strs = [];
        for (const row of insertRows) {
            const values = [];
            for (const column of option.columns) {
                values.push(row[column]);
            }
            strs.push('(?)');
            params.push(values);
        }
        const sql = this.format('INSERT INTO ??(??) VALUES' + strs.join(', '), params);
        debug('insert(%o, %o, %o) \n=> %o', table, rows, option, sql);
        return await this.query(sql);
    }
    async update(table, row, option) {
        option = option || {};
        if (!option.columns) {
            option.columns = Object.keys(row);
        }
        if (!option.where) {
            if (!('id' in row)) {
                throw new Error('Can not auto detect update condition, please set option.where, or make sure obj.id exists');
            }
            option.where = {
                id: row.id,
            };
        }
        const sets = [];
        const values = [];
        for (const column of option.columns) {
            sets.push('?? = ?');
            values.push(column);
            values.push(row[column]);
        }
        const sql = this.format('UPDATE ?? SET ', [table]) +
            this.format(sets.join(', '), values) +
            this._where(option.where);
        debug('update(%o, %o, %o) \n=> %o', table, row, option, sql);
        return await this.query(sql);
    }
    /**
     * Update multiple rows from a table
     *
     * UPDATE `table_name` SET
     *  `column1` CASE
     *     WHEN  condition1 THEN 'value11'
     *     WHEN  condition2 THEN 'value12'
     *     WHEN  condition3 THEN 'value13'
     *     ELSE `column1` END,
     *  `column2` CASE
     *     WHEN  condition1 THEN 'value21'
     *     WHEN  condition2 THEN 'value22'
     *     WHEN  condition3 THEN 'value23'
     *     ELSE `column2` END
     * WHERE condition
     *
     * See MySQL Case Syntax: https://dev.mysql.com/doc/refman/5.7/en/case.html
     *
     * @param {String} table table name
     * @param {Array<Object>} updateRows Object Arrays
     *    each Object needs a primary key `id`, or each Object has `row` and `where` properties
     *    e.g.
     *      [{ id: 1, name: 'fengmk21' }]
     *      or [{ row: { name: 'fengmk21' }, where: { id: 1 } }]
     * @return {object} update result
     */
    async updateRows(table, updateRows) {
        if (!Array.isArray(updateRows)) {
            throw new Error('updateRows should be array');
        }
        /**
         * {
         *  column: {
         *    when: [ 'WHEN condition1 THEN ?', 'WHEN condition12 THEN ?' ],
         *    then: [ value1, value1 ]
         *  }
         * }
         */
        const SQL_CASE = {};
        // e.g. { id: [], column: [] }
        const WHERE = {};
        for (const updateRow of updateRows) {
            const row = updateRow.row ?? updateRow;
            let where = updateRow.where;
            const hasId = 'id' in row;
            if (!hasId && !where) {
                throw new Error('Can not auto detect updateRows condition, please set updateRow.where, or make sure updateRow.id exists');
            }
            // convert { id, column } to { row: { column }, where: { id } }
            if (hasId) {
                where = { id: updateRow.id };
            }
            let whereString = this._where(where);
            whereString = !whereString.includes('WHERE') ? whereString : whereString.substring(whereString.indexOf('WHERE') + 5);
            for (const key in row) {
                if (key === 'id')
                    continue;
                if (!SQL_CASE[key]) {
                    SQL_CASE[key] = { when: [], then: [] };
                }
                SQL_CASE[key].when.push(' WHEN ' + whereString + ' THEN ? ');
                SQL_CASE[key].then.push(row[key]);
            }
            for (const key in where) {
                if (!WHERE[key]) {
                    WHERE[key] = [];
                }
                if (!WHERE[key].includes(where[key])) {
                    WHERE[key].push(where[key]);
                }
            }
        }
        let SQL = 'UPDATE ?? SET ';
        let VALUES = [table];
        const TEMPLATE = [];
        for (const key in SQL_CASE) {
            let templateSql = ' ?? = CASE ';
            VALUES.push(key);
            templateSql += SQL_CASE[key].when.join(' ');
            VALUES = VALUES.concat(SQL_CASE[key].then);
            templateSql += ' ELSE ?? END ';
            TEMPLATE.push(templateSql);
            VALUES.push(key);
        }
        SQL += TEMPLATE.join(' , ');
        SQL += this._where(WHERE);
        /**
         * e.g.
         *
         * updateRows(table, [
         *  {id: 1, name: 'fengmk21', email: 'm@fengmk21.com'},
         *  {id: 2, name: 'fengmk22', email: 'm@fengmk22.com'},
         *  {id: 3, name: 'fengmk23', email: 'm@fengmk23.com'},
         * ])
         *
         * UPDATE `ali-sdk-test-user` SET
         *  `name` =
         *    CASE
         *      WHEN  `id` = 1 THEN 'fengmk21'
         *      WHEN  `id` = 2 THEN 'fengmk22'
         *      WHEN  `id` = 3 THEN 'fengmk23'
         *      ELSE `name` END,
         *  `email` =
         *    CASE
         *      WHEN  `id` = 1 THEN 'm@fengmk21.com'
         *      WHEN  `id` = 2 THEN 'm@fengmk22.com'
         *      WHEN  `id` = 3 THEN 'm@fengmk23.com'
         *      ELSE `email` END
         *  WHERE `id` IN (1, 2, 3)
         */
        const sql = this.format(SQL, VALUES);
        debug('updateRows(%o, %o) \n=> %o', table, updateRows, sql);
        return await this.query(sql);
    }
    async delete(table, where) {
        const sql = this.format('DELETE FROM ??', [table]) +
            this._where(where);
        debug('delete(%j, %j) \n=> %j', table, where, sql);
        return await this.query(sql);
    }
    _where(where) {
        if (!where) {
            return '';
        }
        const wheres = [];
        const values = [];
        for (const key in where) {
            const value = where[key];
            if (Array.isArray(value)) {
                wheres.push('?? IN (?)');
            }
            else {
                if (value === null || value === undefined) {
                    wheres.push('?? IS ?');
                }
                else {
                    wheres.push('?? = ?');
                }
            }
            values.push(key);
            values.push(value);
        }
        if (wheres.length > 0) {
            return this.format(' WHERE ' + wheres.join(' AND '), values);
        }
        return '';
    }
    _selectColumns(table, columns) {
        if (!columns || columns.length === 0) {
            columns = '*';
        }
        if (columns === '*') {
            return this.format('SELECT * FROM ??', [table]);
        }
        return this.format('SELECT ?? FROM ??', [columns, table]);
    }
    _orders(orders) {
        if (!orders) {
            return '';
        }
        if (typeof orders === 'string') {
            orders = [orders];
        }
        const values = [];
        for (const value of orders) {
            if (typeof value === 'string') {
                values.push(this.escapeId(value));
            }
            else if (Array.isArray(value)) {
                // value format: ['name', 'desc'], ['name'], ['name', 'asc']
                let sort = String(value[1]).toUpperCase();
                if (sort !== 'ASC' && sort !== 'DESC') {
                    sort = '';
                }
                if (sort) {
                    values.push(this.escapeId(value[0]) + ' ' + sort);
                }
                else {
                    values.push(this.escapeId(value[0]));
                }
            }
        }
        return ' ORDER BY ' + values.join(', ');
    }
    _limit(limit, offset) {
        if (!limit || typeof limit !== 'number') {
            return '';
        }
        if (typeof offset !== 'number') {
            offset = 0;
        }
        return ' LIMIT ' + offset + ', ' + limit;
    }
    /**
     * Lock tables.
     * @param {object[]} lockTableOptions table lock descriptions.
     * @description
     * LOCK TABLES
     *   tbl_name [[AS] alias] lock_type
     *   [, tbl_name [[AS] alias] lock_type] ...
     * lock_type: {
     *   READ [LOCAL]
     *   | [LOW_PRIORITY] WRITE
     * }
     * For more details:
     * https://dev.mysql.com/doc/refman/8.0/en/lock-tables.html
     * @example
     * await locks([{ tableName: 'posts', lockType: 'READ', tableAlias: 't' }]);
     */
    async locks(lockTableOptions) {
        const sql = this.#locks(lockTableOptions);
        debug('lock tables \n=> %o', sql);
        return await this.query(sql);
    }
    /**
     * Lock a single table.
     * @param {string} tableName table name
     * @param {string} lockType lock type
     * @param {string} tableAlias table alias
     * @description
     * LOCK TABLES
     *   tbl_name [[AS] alias] lock_type
     *   [, tbl_name [[AS] alias] lock_type] ...
     * lock_type: {
     *   READ [LOCAL]
     *   | [LOW_PRIORITY] WRITE
     * }
     * For more details:
     * https://dev.mysql.com/doc/refman/8.0/en/lock-tables.html
     * @example
     * await lockOne('posts_table', 'READ', 't'); // LOCK TABLS 'posts_table' AS t READ
     */
    async lockOne(tableName, lockType, tableAlias) {
        const sql = this.#locks([{ tableName, lockType, tableAlias }]);
        debug('lock one table \n=> %o', sql);
        return await this.query(sql);
    }
    #locks(lockTableOptions) {
        if (lockTableOptions.length === 0) {
            throw new Error('Cannot lock empty tables.');
        }
        let sql = 'LOCK TABLES ';
        for (const [index, lockTableOption] of lockTableOptions.entries()) {
            const { tableName, lockType, tableAlias } = lockTableOption;
            if (!tableName) {
                throw new Error('No table_name provided while trying to lock table');
            }
            if (!lockType) {
                throw new Error('No lock_type provided while trying to lock table `' + tableName + '`');
            }
            if (!['READ', 'WRITE', 'READ LOCAL', 'LOW_PRIORITY WRITE'].includes(lockType.toUpperCase())) {
                throw new Error('lock_type provided while trying to lock table `' + tableName +
                    '` must be one of the following(CASE INSENSITIVE):\n`READ` | `WRITE` | `READ LOCAL` | `LOW_PRIORITY WRITE`');
            }
            if (index > 0) {
                sql += ', ';
            }
            sql += ' ' + this.escapeId(tableName) + ' ';
            if (tableAlias) {
                sql += ' AS ' + this.escapeId(tableAlias) + ' ';
            }
            sql += ' ' + lockType;
        }
        return sql + ';';
    }
    /**
     * To unlock all tables locked in current session.
     * For more details:
     * https://dev.mysql.com/doc/refman/8.0/en/lock-tables.html
     * @example
     * await unlock(); // unlock all tables.
     */
    async unlock() {
        debug('unlock tables');
        return await this.query('UNLOCK TABLES;');
    }
}
exports.Operator = Operator;
//# sourceMappingURL=data:application/json;base64,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