"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RDSClient = void 0;
const node_async_hooks_1 = require("node:async_hooks");
const node_util_1 = require("node:util");
const mysql_1 = __importDefault(require("mysql"));
const operator_1 = require("./operator");
const connection_1 = require("./connection");
const transaction_1 = require("./transaction");
const PoolConfig_1 = require("./PoolConfig");
const literals_1 = __importDefault(require("./literals"));
const channels_1 = __importDefault(require("./channels"));
class RDSClient extends operator_1.Operator {
    static get literals() { return literals_1.default; }
    static get escape() { return mysql_1.default.escape; }
    static get escapeId() { return mysql_1.default.escapeId; }
    static get format() { return mysql_1.default.format; }
    static get raw() { return mysql_1.default.raw; }
    static #DEFAULT_STORAGE_KEY = Symbol('RDSClient#storage#default');
    static #TRANSACTION_NEST_COUNT = Symbol('RDSClient#transaction#nestCount');
    #pool;
    #connectionStorage;
    #connectionStorageKey;
    constructor(options) {
        super();
        const { connectionStorage, connectionStorageKey, ...mysqlOptions } = options;
        // get connection options from getConnectionConfig method every time
        if (mysqlOptions.getConnectionConfig) {
            // eslint-disable-next-line @typescript-eslint/no-var-requires
            const MySQLPool = require('mysql/lib/Pool');
            this.#pool = new MySQLPool({
                config: new PoolConfig_1.RDSPoolConfig(mysqlOptions, mysqlOptions.getConnectionConfig),
            });
            // override _needsChangeUser to return false
            this.#pool._needsChangeUser = () => false;
        }
        else {
            this.#pool = mysql_1.default.createPool(mysqlOptions);
        }
        [
            'query',
            'getConnection',
            'end',
        ].forEach(method => {
            this.#pool[method] = (0, node_util_1.promisify)(this.#pool[method]);
        });
        this.#connectionStorage = connectionStorage || new node_async_hooks_1.AsyncLocalStorage();
        this.#connectionStorageKey = connectionStorageKey || RDSClient.#DEFAULT_STORAGE_KEY;
        // https://github.com/mysqljs/mysql#pool-events
        this.#pool.on('connection', (connection) => {
            channels_1.default.connectionNew.publish({
                client: this,
                connection,
            });
        });
        this.#pool.on('enqueue', () => {
            channels_1.default.connectionEnqueue.publish({
                client: this,
            });
        });
        this.#pool.on('acquire', (connection) => {
            channels_1.default.connectionAcquire.publish({
                client: this,
                connection,
            });
        });
        this.#pool.on('release', (connection) => {
            channels_1.default.connectionRelease.publish({
                client: this,
                connection,
            });
        });
    }
    async query(sql, values) {
        const conn = await this.getConnection();
        try {
            return await conn.query(sql, values);
        }
        finally {
            conn.release();
        }
    }
    get pool() {
        return this.#pool;
    }
    get stats() {
        const acquiringConnections = this.#pool._acquiringConnections.length;
        const allConnections = this.#pool._allConnections.length;
        const freeConnections = this.#pool._freeConnections.length;
        const connectionQueue = this.#pool._connectionQueue.length;
        const busyConnections = allConnections - freeConnections - acquiringConnections;
        return {
            acquiringConnections,
            allConnections,
            freeConnections,
            connectionQueue,
            busyConnections,
        };
    }
    async getConnection() {
        try {
            const _conn = await this.#pool.getConnection();
            const conn = new connection_1.RDSConnection(_conn);
            if (this.beforeQueryHandlers.length > 0) {
                for (const handler of this.beforeQueryHandlers) {
                    conn.beforeQuery(handler);
                }
            }
            if (this.afterQueryHandlers.length > 0) {
                for (const handler of this.afterQueryHandlers) {
                    conn.afterQuery(handler);
                }
            }
            return conn;
        }
        catch (err) {
            if (err.name === 'Error') {
                err.name = 'RDSClientGetConnectionError';
            }
            throw err;
        }
    }
    /**
     * Begin a transaction
     *
     * @return {RDSTransaction} transaction instance
     */
    async beginTransaction() {
        const conn = await this.getConnection();
        try {
            await conn.beginTransaction();
        }
        catch (err) {
            conn.release();
            throw err;
        }
        const tran = new transaction_1.RDSTransaction(conn);
        tran[RDSClient.#TRANSACTION_NEST_COUNT] = 1;
        if (this.beforeQueryHandlers.length > 0) {
            for (const handler of this.beforeQueryHandlers) {
                tran.beforeQuery(handler);
            }
        }
        if (this.afterQueryHandlers.length > 0) {
            for (const handler of this.afterQueryHandlers) {
                tran.afterQuery(handler);
            }
        }
        return tran;
    }
    /**
     * Auto commit or rollback on a transaction scope
     *
     * @param {Function} scope - scope with code
     * @param {Object} [ctx] - transaction context
     * @return {Object} - scope return result
     */
    async #beginTransactionScope(scope, ctx) {
        let tran;
        let shouldRelease = false;
        if (!ctx[this.#connectionStorageKey]) {
            // there is no transaction in ctx, create a new one
            tran = await this.beginTransaction();
            ctx[this.#connectionStorageKey] = tran;
            shouldRelease = true;
        }
        else {
            // use transaction in ctx
            tran = ctx[this.#connectionStorageKey];
            tran[RDSClient.#TRANSACTION_NEST_COUNT]++;
        }
        let result;
        let scopeError;
        let internalError;
        try {
            result = await scope(tran);
        }
        catch (err) {
            scopeError = err;
        }
        tran[RDSClient.#TRANSACTION_NEST_COUNT]--;
        // null connection means the nested scope has been rollback, we can do nothing here
        if (tran.conn) {
            try {
                // execution error, should rollback
                if (scopeError) {
                    await tran.rollback();
                }
                else if (tran[RDSClient.#TRANSACTION_NEST_COUNT] < 1) {
                    // nestedCount smaller than 1 means all the nested scopes have executed successfully
                    await tran.commit();
                }
            }
            catch (err) {
                internalError = err;
            }
        }
        // remove transaction in ctx
        if (shouldRelease && tran[RDSClient.#TRANSACTION_NEST_COUNT] < 1) {
            ctx[this.#connectionStorageKey] = null;
        }
        if (internalError) {
            if (scopeError) {
                internalError.cause = scopeError;
            }
            throw internalError;
        }
        if (scopeError) {
            throw scopeError;
        }
        return result;
    }
    /**
     * Auto commit or rollback on a transaction scope
     *
     * @param scope - scope with code
     * @return {Object} - scope return result
     */
    async beginTransactionScope(scope) {
        let ctx = this.#connectionStorage.getStore();
        if (ctx) {
            return await this.#beginTransactionScope(scope, ctx);
        }
        ctx = {};
        return await this.#connectionStorage.run(ctx, async () => {
            return await this.#beginTransactionScope(scope, ctx);
        });
    }
    /**
     * doomed to be rollbacked after transaction scope
     * useful on writing tests which are related with database
     *
     * @param scope - scope with code
     * @param ctx - transaction context
     * @return {Object} - scope return result
     */
    async #beginDoomedTransactionScope(scope, ctx) {
        let tran;
        if (!ctx[this.#connectionStorageKey]) {
            // there is no transaction in ctx, create a new one
            tran = await this.beginTransaction();
            ctx[this.#connectionStorageKey] = tran;
        }
        else {
            // use transaction in ctx
            tran = ctx[this.#connectionStorageKey];
            tran[RDSClient.#TRANSACTION_NEST_COUNT]++;
        }
        try {
            const result = await scope(tran);
            tran[RDSClient.#TRANSACTION_NEST_COUNT]--;
            if (tran[RDSClient.#TRANSACTION_NEST_COUNT] === 0) {
                ctx[this.#connectionStorageKey] = null;
                await tran.rollback();
            }
            return result;
        }
        catch (err) {
            if (ctx[this.#connectionStorageKey]) {
                ctx[this.#connectionStorageKey] = null;
                await tran.rollback();
            }
            throw err;
        }
    }
    /**
     * doomed to be rollbacked after transaction scope
     * useful on writing tests which are related with database
     *
     * @param scope - scope with code
     * @return {Object} - scope return result
     */
    async beginDoomedTransactionScope(scope) {
        let ctx = this.#connectionStorage.getStore();
        if (ctx) {
            return await this.#beginDoomedTransactionScope(scope, ctx);
        }
        ctx = {};
        return await this.#connectionStorage.run(ctx, async () => {
            return await this.#beginDoomedTransactionScope(scope, ctx);
        });
    }
    async end() {
        await this.#pool.end();
    }
}
exports.RDSClient = RDSClient;
//# sourceMappingURL=data:application/json;base64,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