/**
 * Help users access and manage their external accounts from within your app, without requiring them to enter login credentials.
 */
declare module Accounts {
    export class ACAccountStore extends NSObject {
        /**
         * The accounts managed by this account store.
         */
        accounts: NSArray!;
        /**
         * Renews account credentials when the credentials are no longer valid.
         */
        renewCredentials(account: ACAccount!, completionHandler: ACAccountStoreCredentialRenewalHandler!):void;
        /**
         * Removes an account from the account store.
         */
        removeAccount(account: ACAccount!, completionHandler: ACAccountStoreRemoveCompletionHandler!):void;
        /**
         * Obtains permission to access protected user properties.
         */
        requestAccessToAccounts(accountType: ACAccountType!, options: Record<AnyHashable, any>, completion: ACAccountStoreRequestAccessCompletionHandler!):void;
        /**
         * Returns the account with the specified identifier.
         */
        account(identifier: String!):ACAccount!;
        /**
         * Returns an account type that matches the specified identifier.
         */
        accountType(typeIdentifier: String!):ACAccountType!;
        /**
         * Saves an account to the Accounts database.
         */
        saveAccount(account: ACAccount!, completionHandler: ACAccountStoreSaveCompletionHandler!):void;
    }
    export enum ACAccountCredentialRenewResult {
        renewed = 0,
        rejected = 1,
        failed = 2,
    }
    export class ACAccount extends NSObject {
        /**
         * A human-readable description of the account.
         */
        accountDescription: String!;
        /**
         * The type of service account.
         */
        accountType: ACAccountType!;
        /**
         * The credential used to authenticate the user of this account.
         */
        credential: ACAccountCredential!;
        /**
         * A unique identifier for this account.
         */
        identifier: NSString!;
        /**
         * The username for this account.
         */
        username: String!;
        /**
         * The full name associated with the user account.
         */
        userFullName: String!;
        /**
         * Initializes a new account of the specified type.
         */
        constructor (type: ACAccountType!);
    }
    export class ACAccountCredential extends NSObject {
        /**
         * The token used for the credential.
         */
        oauthToken: String!;
        /**
         * Initializes an account credential using OAuth.
         */
        constructor (token: String!, secret: String!);
        /**
         * Initializes an account credential using OAuth 2.
         */
        constructor (token: String!, refreshToken: String!, expiryDate: Date!);
    }
    export class ACAccountType extends NSObject {
        /**
         * A Boolean value indicating whether the user granted the application access to accounts of this type.
         */
        accessGranted: boolean;
        /**
         * A human-readable description of the account type.
         */
        accountTypeDescription: String!;
        /**
         * The unique identifier for the account type.
         */
        identifier: String!;
    }
    export class ACErrorCode {
        /**
         * The raw integer value of the error code.
         */
        rawValue: UInt32;
        /**
         * Initializes a new error code structure.
         */
        constructor (rawValue: UInt32);
        /**
         * Initializes a new error code structure.
         */
        constructor (rawValue: UInt32);
    }
}

/**
 * Access essential data types, collections, and operating-system services to define the base layer of functionality for your app.
 */
declare module Foundation {
    export class NSNotification extends NSObject {}
    /**
     * the same name namespace of class NSNotification
     */
    export namespace NSNotification {
        export class Name {
            /**
             * Posted when the accounts managed by this account store changed in the database.
             */
            static ACAccountStoreDidChange: NSNotification.Name;
            /**
             * Sent when the contents of the assets library have changed from under the app that is using the data.
             */
            static ALAssetsLibraryChanged: NSNotification.Name;
            /**
             * A notification the framework posts when the audio engine configuration changes.
             */
            static AVAudioEngineConfigurationChange: NSNotification.Name;
            /**
             * A notification that indicates when component tags change.
             */
            static AVAudioUnitComponentTagsDidChange: NSNotification.Name;
            /**
             * A notification the system posts when a player item plays to its end time.
             */
            static AVPlayerItemDidPlayToEndTime: NSNotification.Name;
            /**
             * A notification that the system posts when a player item fails to play to its end time.
             */
            static AVPlayerItemFailedToPlayToEndTime: NSNotification.Name;
            /**
             * A notification the system posts when a player item media doesn’t arrive in time to continue playback.
             */
            static AVPlayerItemPlaybackStalled: NSNotification.Name;
            /**
             * A notification the system posts when a player item adds a new entry to its access log.
             */
            static AVPlayerItemNewAccessLogEntry: NSNotification.Name;
            /**
             * A notification the system posts when a player item adds a new entry to its error log.
             */
            static AVPlayerItemNewErrorLogEntry: NSNotification.Name;
            /**
             * A notification the system posts when changes occur to its detected routes.
             */
            static AVRouteDetectorMultipleRoutesDetectedDidChange: NSNotification.Name;
            /**
             * A notification the system posts when a sample buffer display layer fails to decode.
             */
            static AVSampleBufferDisplayLayerFailedToDecode: NSNotification.Name;
            /**
             * A notification the system posts when a capture session starts.
             */
            static AVCaptureSessionDidStartRunning: NSNotification.Name;
            /**
             * A notification the system posts when a capture session stops.
             */
            static AVCaptureSessionDidStopRunning: NSNotification.Name;
            /**
             * A notification the system posts when it interrupts a capture session.
             */
            static AVCaptureSessionWasInterrupted: NSNotification.Name;
            /**
             * A notification the system posts when an interruption to a capture session finishes.
             */
            static AVCaptureSessionInterruptionEnded: NSNotification.Name;
            /**
             * A notification the system posts when an error occurs during a capture session.
             */
            static AVCaptureSessionRuntimeError: NSNotification.Name;
            /**
             * A notification the system posts when the capture input port’s format description changes.
             */
            static AVCaptureInputPortFormatDescriptionDidChange: NSNotification.Name;
            /**
             * A notification the system posts when a new capture device becomes available.
             */
            static AVCaptureDeviceWasConnected: NSNotification.Name;
            /**
             * A notification the system posts when an existing device becomes unavailable.
             */
            static AVCaptureDeviceWasDisconnected: NSNotification.Name;
            /**
             * A notification the system posts when a capture device detects a substantial change to the video subject area.
             */
            static AVCaptureDeviceSubjectAreaDidChange: NSNotification.Name;
            /**
             * Posted when the set of active complications changes.
             */
            static CLKComplicationServerActiveComplicationsDidChange: NSNotification.Name;
            /**
             * A notification that a container posts when the status of an iCloud account changes.
             */
            static CKAccountChanged: NSNotification.Name;
            /**
             * Posted when changes occur to the contact store.
             */
            static CNContactStoreDidChange: NSNotification.Name;
            /**
             * Posted after records are imported from the ubiquitous content store.
             */
            static NSPersistentStoreDidImportUbiquitousContentChanges: NSNotification.Name;
            /**
             * A notification of changes made to managed objects associated with this context.
             */
            static NSManagedObjectContextObjectsDidChange: NSNotification.Name;
            /**
             * A notification that the context completed a save.
             */
            static NSManagedObjectContextDidSave: NSNotification.Name;
            /**
             * A notification that the context is about to save.
             */
            static NSManagedObjectContextWillSave: NSNotification.Name;
            /**
             * A notification that posts when a coordinator adds, removes, or updates a persistent store.
             */
            static NSPersistentStoreCoordinatorStoresDidChange: NSNotification.Name;
            /**
             * A notification that posts immediately before the coordinator updates its collection of stores.
             */
            static NSPersistentStoreCoordinatorStoresWillChange: NSNotification.Name;
            /**
             * A notification that posts immediately before a coordinator removes a persistent store.
             */
            static NSPersistentStoreCoordinatorWillRemoveStore: NSNotification.Name;
            /**
             * The name of the notification indicating that the radio access technology changed for one of the services.
             */
            static CTRadioAccessTechnologyDidChange: NSNotification.Name;
            /**
             * A notification that the system sends when an accessory becomes connected and available for your application to use.
             */
            static EAAccessoryDidConnect: NSNotification.Name;
            /**
             * A notification that is posted when an accessory is disconnected and no longer available for your application to use.
             */
            static EAAccessoryDidDisconnect: NSNotification.Name;
            /**
             * A notification posted whenever the system clock is changed. 
             */
            static NSSystemClockDidChange: NSNotification.Name;
            /**
             * A notification that is posted whenever the calendar day of the system changes, as determined by the system calendar, locale, and time zone.
             */
            static NSCalendarDayChanged: NSNotification.Name;
            /**
             * A notification posted when the time zone changes.
             */
            static NSSystemTimeZoneDidChange: NSNotification.Name;
            /**
             * Posted whenever an 
             *  object opens or closes an undo group (except when it opens a top-level group) and when checking the redo stack in 
             * . 
             */
            static NSUndoManagerCheckpoint: NSNotification.Name;
            /**
             * Posted whenever an 
             *  object opens an undo group, which occurs in the implementation of the 
             *  method. 
             */
            static NSUndoManagerDidOpenUndoGroup: NSNotification.Name;
            /**
             * Posted just after an 
             *  object performs a redo operation (
             * ).
             */
            static NSUndoManagerDidRedoChange: NSNotification.Name;
            /**
             * Posted just after an 
             *  object performs an undo operation.
             */
            static NSUndoManagerDidUndoChange: NSNotification.Name;
            /**
             * Posted before an 
             *  object closes an undo group, which occurs in the implementation of the 
             *  method.
             */
            static NSUndoManagerWillCloseUndoGroup: NSNotification.Name;
            /**
             * Posted after an 
             *  object closes an undo group, which occurs in the implementation of the 
             *  method.
             */
            static NSUndoManagerDidCloseUndoGroup: NSNotification.Name;
            /**
             * Posted just before an 
             *  object performs a redo operation (
             * ).
             */
            static NSUndoManagerWillRedoChange: NSNotification.Name;
            /**
             * Posted just before an 
             *  object performs an undo operation.
             */
            static NSUndoManagerWillUndoChange: NSNotification.Name;
            /**
             * Posts when the power state of a device changes.
             */
            static NSProcessInfoPowerStateDidChange: NSNotification.Name;
            /**
             * Posted after the system detects that the amount of available disk space is getting low. The notification is posted to the default notification center.
             */
            static NSBundleResourceRequestLowDiskSpace: NSNotification.Name;
        }
    }
    export class NSUserActivity extends NSObject {
        /**
         * A Boolean value that indicates whether a user activity represents a ClassKit context.
         */
        isClassKitDeepLink: boolean;
        /**
         * The identifier path associated with a user activity generated by an app that adopts ClassKit.
         */
        contextIdentifierPath: string[] | null;
    }
    export class Locale {
        /**
         * A list of the user’s preferred languages.
         */
        static preferredLanguages: string[];
        /**
         * The language of a locale.
         */
        language: Locale.Language;
        /**
         * The first day of the week as represented by this locale.
         */
        firstDayOfWeek: Locale.Weekday;
        /**
         * The hour cycle used by the locale, like one-to-twelve or zero-to-twenty-three.
         */
        hourCycle: Locale.HourCycle;
        /**
         * The time zone associated with the locale, if any.
         */
        timeZone: TimeZone | null;
        /**
         * The currency used by the locale.
         */
        currency: Locale.Currency | null;
        /**
         * The measurement system used by the locale, like metric or the US system.
         */
        measurementSystem: Locale.MeasurementSystem;
        /**
         * The numbering system used by the locale.
         */
        numberingSystem: Locale.NumberingSystem;
        /**
         * An array containing all the valid numbering systems for the locale.
         */
        availableNumberingSystems: Locale.NumberingSystem[];
        /**
         * The region used by the locale.
         */
        region: Locale.Region | null;
        /**
         * The optional subdivision of the region used by this locale.
         */
        subdivision: Locale.Subdivision | null;
        /**
         * An optional variant used by the locale.
         */
        variant: Locale.Variant | null;
        /**
         * The string sort order of the locale.
         */
        collation: Locale.Collation;
        /**
         * The identifier of the locale.
         */
        identifier: string;
        /**
         * The calendar for the locale, or the Gregorian calendar as a fallback.
         */
        calendar: Calendar;
        /**
         * The region code of the locale, or 
         *  if it has none.
         */
        regionCode: string;
        /**
         * The language code of the locale, or 
         *  if has none.
         */
        languageCode: string;
        /**
         * The script code of the locale, or 
         *  if has none.
         */
        scriptCode: string;
        /**
         * The variant code for the locale, or 
         *  if it has none.
         */
        variantCode: string;
        /**
         * The exemplar character set for the locale, or 
         *  if has none.
         */
        exemplarCharacterSet: CharacterSet | null;
        /**
         * The collation identifier for the locale, or 
         *  if it has none.
         */
        collationIdentifier: string;
        /**
         * The collator identifier of the locale.
         */
        collatorIdentifier: string;
        /**
         * A Boolean that is true if the locale uses the metric system.
         */
        usesMetricSystem: boolean;
        /**
         * The decimal separator of the locale.
         */
        decimalSeparator: string;
        /**
         * The grouping separator of the locale.
         */
        groupingSeparator: string;
        /**
         * The currency code of the locale.
         */
        currencyCode: string;
        /**
         * The currency symbol of the locale.
         */
        currencySymbol: string;
        /**
         * The quotation begin delimiter of the locale.
         */
        quotationBeginDelimiter: string;
        /**
         * The quotation end delimiter of the locale.
         */
        quotationEndDelimiter: string;
        /**
         * The alternate quotation begin delimiter of the locale.
         */
        alternateQuotationBeginDelimiter: string;
        /**
         * The alternate quotation end delimiter of the locale.
         */
        alternateQuotationEndDelimiter: string;
        /**
         * A textual description of the locale.
         */
        description: string;
        /**
         * A textual description of the locale suitable for debugging.
         */
        debugDescription: string;
        /**
         * A mirror that reflects the locale.
         */
        customMirror: Mirror;
        /**
         * The computed hash value for the locale.
         */
        hashValue: number;
        /**
         * Creates a new instance by decoding from the given decoder.
         */
        constructor (decoder: Decoder);
        /**
         * Returns the character direction for a specified language code.
         */
        static characterDirection(isoLangCode: string):Locale.LanguageDirection;
        /**
         * Returns the line direction for a specified language code.
         */
        static lineDirection(isoLangCode: string):Locale.LanguageDirection;
        /**
         * Returns a localized string for a specified calendar.
         */
        localizedString(calendarIdentifier: Calendar.Identifier):string;
        /**
         * Encodes this value into the given encoder.
         */
        encode(encoder: Encoder):void;
        /**
         * Hashes the essential components of this value by feeding them into the given hasher.
         */
        hash(hasher: Hasher):void;
    }
    /**
     * the same name namespace of class Locale
     */
    export namespace Locale {
        export class Region {
            static turksCaicosIslands: Locale.Region;
            static tuvalu: Locale.Region;
            static uganda: Locale.Region;
            static ukraine: Locale.Region;
            static unitedArabEmirates: Locale.Region;
            static unitedKingdom: Locale.Region;
            static unitedStates: Locale.Region;
            static unitedStatesOutlyingIslands: Locale.Region;
            static unitedStatesVirginIslands: Locale.Region;
            static uruguay: Locale.Region;
            static uzbekistan: Locale.Region;
            static vanuatu: Locale.Region;
            static vaticanCity: Locale.Region;
            static venezuela: Locale.Region;
            static vietnam: Locale.Region;
            static wallisFutuna: Locale.Region;
            static westernSahara: Locale.Region;
            static world: Locale.Region;
            static yemen: Locale.Region;
            static zambia: Locale.Region;
            static zimbabwe: Locale.Region;
            static ålandIslands: Locale.Region;
        }
        export class Language {}
        /**
         * the same name namespace of class Language
         */
        export namespace Language {
            export class Components {
                /**
                 * The written script used by this language.
                 */
                script: Locale.Script | null;
                /**
                 * The hash value.
                 */
                hashValue: number;
                /**
                 * Creates a new instance by decoding from the given decoder.
                 */
                constructor (decoder: Decoder);
                /**
                 * Encodes this value into the given encoder.
                 */
                encode(encoder: Encoder):void;
                /**
                 * Hashes the essential components of this value by feeding them into the given hasher.
                 */
                hash(hasher: Hasher):void;
            }
        }
        export class Script {
            static unknown: Locale.Script;
            static adlam: Locale.Script;
            static arabic: Locale.Script;
            static arabicNastaliq: Locale.Script;
            static armenian: Locale.Script;
            static bangla: Locale.Script;
            static cherokee: Locale.Script;
            static cyrillic: Locale.Script;
            static devanagari: Locale.Script;
            static ethiopic: Locale.Script;
            static georgian: Locale.Script;
            static greek: Locale.Script;
            static gujarati: Locale.Script;
            static gurmukhi: Locale.Script;
            static hanSimplified: Locale.Script;
            static hanTraditional: Locale.Script;
            static hanifiRohingya: Locale.Script;
            static hebrew: Locale.Script;
            static hiragana: Locale.Script;
            static japanese: Locale.Script;
            static kannada: Locale.Script;
            static katakana: Locale.Script;
            static khmer: Locale.Script;
            static korean: Locale.Script;
            static lao: Locale.Script;
            static latin: Locale.Script;
            static malayalam: Locale.Script;
            static meiteiMayek: Locale.Script;
            static myanmar: Locale.Script;
            static odia: Locale.Script;
            static olChiki: Locale.Script;
            static sinhala: Locale.Script;
            static syriac: Locale.Script;
            static tamil: Locale.Script;
            static telugu: Locale.Script;
            static thaana: Locale.Script;
            static thai: Locale.Script;
            static tibetan: Locale.Script;
            identifier: string;
            /**
             * The hash value.
             */
            hashValue: number;
            isISOScript: boolean;
            /**
             * Creates a script from a BCP 47 identifier.
             */
            constructor (identifier: string);
            /**
             * Creates a script from a BCP 47 identifier as a string literal.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
        export class Components {
            /**
             * The calendar used by the locale.
             */
            calendar: Calendar.Identifier | null;
            /**
             * The first day of the week as represented by this locale.
             */
            firstDayOfWeek: Locale.Weekday | null;
            /**
             * The hour cycle used by the locale, like one-to-twelve or zero-to-twenty-three.
             */
            hourCycle: Locale.HourCycle | null;
            /**
             * The time zone used by the locale.
             */
            timeZone: TimeZone | null;
            /**
             * The currency used by the locale.
             */
            currency: Locale.Currency | null;
            /**
             * The measurement system used by the locale, like metric or the US system.
             */
            measurementSystem: Locale.MeasurementSystem | null;
            /**
             * The numbering system used by the locale.
             */
            numberingSystem: Locale.NumberingSystem | null;
            /**
             * The region used by the locale.
             */
            region: Locale.Region | null;
            /**
             * The optional subdivision of the region used by this locale.
             */
            subdivision: Locale.Subdivision | null;
            /**
             * An optional variant used by the locale.
             */
            variant: Locale.Variant | null;
            /**
             * The string sort order of the locale.
             */
            collation: Locale.Collation | null;
            /**
             * The hash value.
             */
            hashValue: number;
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
        export enum Weekday {
            monday,
            tuesday,
            wednesday,
            thursday,
            friday,
            saturday,
            sunday,
        }
        export enum HourCycle {
            zeroToEleven,
            oneToTwelve,
            zeroToTwentyThree,
            oneToTwentyFour,
        }
        export class Currency {
            /**
             * An array containing currencies defined by the currency codes in ISO-4217.
             */
            static isoCurrencies: Locale.Currency[];
            /**
             * A representation of an “unknown” currency, for use with transactions that don’t involve any currency.
             */
            static unknown: Locale.Currency;
            /**
             * The currency’s identifier.
             */
            identifier: string;
            /**
             * A Boolean value that indicates whether the currency is in the list of ISO-defined currencies.
             */
            isISOCurrency: boolean;
            /**
             * The hash value.
             */
            hashValue: number;
            /**
             * Creates a currency instance from a BCP 47 identifier.
             */
            constructor (identifier: string);
            /**
             * Creates a currency instance from a BCP 47 identifier as a string literal.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
        export class MeasurementSystem {
            /**
             * An array of the measurement systems defined by the Unicode Common Locale Data Repository (CLDR).
             */
            static measurementSystems: Locale.MeasurementSystem[];
            /**
             * The metric measurement system.
             */
            static metric: Locale.MeasurementSystem;
            /**
             * The United Kingdom measurement system.
             */
            static uk: Locale.MeasurementSystem;
            /**
             * The United States measurement system.
             */
            static us: Locale.MeasurementSystem;
            /**
             * The measurement system’s BCP 47 identifier.
             */
            identifier: string;
            /**
             * The hash value.
             */
            hashValue: number;
            /**
             * Creates a measurement system instance from a BCP 47 identifier.
             */
            constructor (identifier: string);
            /**
             * Creates a measurement system instance from a BCP 47 identifier as a string literal.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
        export class NumberingSystem {
            static availableNumberingSystems: Locale.NumberingSystem[];
            /**
             * The numbering system’s BCP 47 identifier.
             */
            identifier: string;
            /**
             * The hash value.
             */
            hashValue: number;
            /**
             * Creates a numbering system instance from a BCP 47 identifier.
             */
            constructor (identifier: string);
            /**
             * Creates a numbering system instance from a BCP 47 identifier as a string literal.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
        export class Subdivision {
            /**
             * The subdivision’s Unicode identifier.
             */
            identifier: string;
            /**
             * The hash value.
             */
            hashValue: number;
            /**
             * Creates a sudivision from a Unicode identifier.
             */
            constructor (identifier: string);
            /**
             * Creates a sudivision from a Unicode identifier as a string literal.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            /**
             * Returns the subdivision representing the given region as a whole.
             */
            static subdivision(region: Locale.Region):Locale.Subdivision;
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
        export class Variant {
            static posix: Locale.Variant;
            /**
             * The variant’s BCP 47 identifier.
             */
            identifier: string;
            /**
             * The hash value.
             */
            hashValue: number;
            /**
             * Creates a variant from a BCP 47 identifier.
             */
            constructor (identifier: string);
            /**
             * Creates a variant from a BCP 47 identifier as a string literal.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
        export class Collation {
            /**
             * A collation that provides the default ordering for each language.
             */
            static standard: Locale.Collation;
            /**
             * A collation used for string search.
             */
            static searchRules: Locale.Collation;
            static availableCollations: Locale.Collation[];
            /**
             * The collation’s BCP 47 identifier.
             */
            identifier: string;
            /**
             * The hash value.
             */
            hashValue: number;
            /**
             * Creates a collation from a BCP 47 identifier.
             */
            constructor (identifier: string);
            /**
             * Creates a collation from a BCP 47 identifier as a string literal.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
        export class LanguageCode {
            /**
             * Returns an array of ISO-defined language codes.
             */
            static isoLanguageCodes: Locale.LanguageCode[];
            static ainu: Locale.LanguageCode;
            static albanian: Locale.LanguageCode;
            static amharic: Locale.LanguageCode;
            static apacheWestern: Locale.LanguageCode;
            static arabic: Locale.LanguageCode;
            static armenian: Locale.LanguageCode;
            static assamese: Locale.LanguageCode;
            static assyrian: Locale.LanguageCode;
            static azerbaijani: Locale.LanguageCode;
            static bangla: Locale.LanguageCode;
            static belarusian: Locale.LanguageCode;
            static bodo: Locale.LanguageCode;
            static bulgarian: Locale.LanguageCode;
            static burmese: Locale.LanguageCode;
            static cantonese: Locale.LanguageCode;
            static catalan: Locale.LanguageCode;
            static cherokee: Locale.LanguageCode;
            static chinese: Locale.LanguageCode;
            static croatian: Locale.LanguageCode;
            static czech: Locale.LanguageCode;
            static danish: Locale.LanguageCode;
            static dhivehi: Locale.LanguageCode;
            static dogri: Locale.LanguageCode;
            static dutch: Locale.LanguageCode;
            static dzongkha: Locale.LanguageCode;
            static english: Locale.LanguageCode;
            static estonian: Locale.LanguageCode;
            static faroese: Locale.LanguageCode;
            static finnish: Locale.LanguageCode;
            static french: Locale.LanguageCode;
            static fula: Locale.LanguageCode;
            static georgian: Locale.LanguageCode;
            static german: Locale.LanguageCode;
            static greek: Locale.LanguageCode;
            static gujarati: Locale.LanguageCode;
            static hawaiian: Locale.LanguageCode;
            static hebrew: Locale.LanguageCode;
            static hindi: Locale.LanguageCode;
            static hungarian: Locale.LanguageCode;
            static icelandic: Locale.LanguageCode;
            static igbo: Locale.LanguageCode;
            static indonesian: Locale.LanguageCode;
            static irish: Locale.LanguageCode;
            static italian: Locale.LanguageCode;
            static japanese: Locale.LanguageCode;
            static kannada: Locale.LanguageCode;
            static kashmiri: Locale.LanguageCode;
            static kazakh: Locale.LanguageCode;
            static khmer: Locale.LanguageCode;
            static konkani: Locale.LanguageCode;
            static korean: Locale.LanguageCode;
            static kurdish: Locale.LanguageCode;
            static kurdishSorani: Locale.LanguageCode;
            static kyrgyz: Locale.LanguageCode;
            static lao: Locale.LanguageCode;
            static latvian: Locale.LanguageCode;
            static lithuanian: Locale.LanguageCode;
            static macedonian: Locale.LanguageCode;
            static maithili: Locale.LanguageCode;
            static malay: Locale.LanguageCode;
            static malayalam: Locale.LanguageCode;
            static maltese: Locale.LanguageCode;
            static manipuri: Locale.LanguageCode;
            static marathi: Locale.LanguageCode;
            static mongolian: Locale.LanguageCode;
            static multiple: Locale.LanguageCode;
            static māori: Locale.LanguageCode;
            static navajo: Locale.LanguageCode;
            static nepali: Locale.LanguageCode;
            static norwegian: Locale.LanguageCode;
            static norwegianBokmål: Locale.LanguageCode;
            static norwegianNynorsk: Locale.LanguageCode;
            static odia: Locale.LanguageCode;
            static pashto: Locale.LanguageCode;
            static persian: Locale.LanguageCode;
            static polish: Locale.LanguageCode;
            static portuguese: Locale.LanguageCode;
            static punjabi: Locale.LanguageCode;
            static rohingya: Locale.LanguageCode;
            static romanian: Locale.LanguageCode;
            static russian: Locale.LanguageCode;
            static samoan: Locale.LanguageCode;
            static sanskrit: Locale.LanguageCode;
            static santali: Locale.LanguageCode;
            static serbian: Locale.LanguageCode;
            static sindhi: Locale.LanguageCode;
            static sinhala: Locale.LanguageCode;
            static slovak: Locale.LanguageCode;
            static slovenian: Locale.LanguageCode;
            static spanish: Locale.LanguageCode;
            static swahili: Locale.LanguageCode;
            static swedish: Locale.LanguageCode;
            static tagalog: Locale.LanguageCode;
            static tajik: Locale.LanguageCode;
            static tamil: Locale.LanguageCode;
            static telugu: Locale.LanguageCode;
            static thai: Locale.LanguageCode;
            static tibetan: Locale.LanguageCode;
            static tongan: Locale.LanguageCode;
            static turkish: Locale.LanguageCode;
            static turkmen: Locale.LanguageCode;
            static ukrainian: Locale.LanguageCode;
            static unavailable: Locale.LanguageCode;
            static uncoded: Locale.LanguageCode;
            static unidentified: Locale.LanguageCode;
            static urdu: Locale.LanguageCode;
            static uyghur: Locale.LanguageCode;
            static uzbek: Locale.LanguageCode;
            static vietnamese: Locale.LanguageCode;
            static welsh: Locale.LanguageCode;
            static yiddish: Locale.LanguageCode;
            /**
             * The identifier used to create the language code.
             */
            identifier: string;
            /**
             * A Boolean value that indicates whether this language code is in the list of ISO-defined languages.
             */
            isISOLanguage: boolean;
            /**
             * The hash value.
             */
            hashValue: number;
            /**
             * Creates a language code from an identifier.
             */
            constructor (identifier: string);
            /**
             * Creates a language code from an identifier as a string literal.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
        /**
         * the same name namespace of class LanguageCode
         */
        export namespace LanguageCode {
            export enum IdentifierType {
                alpha2,
                alpha3,
            }
        }
        export enum IdentifierType {
            icu,
            cldr,
            bcp47,
        }
    }
    export class Decimal {
        primitivePlottable: number;
        constructor (primitivePlottable: number);
        constructor (_string: string, locale: Locale | null);
        formatted():string;
        hash(hasher: Hasher):void;
    }
    /**
     * the same name namespace of class Decimal
     */
    export namespace Decimal {
        export class FormatStyle {
            static number: Decimal.FormatStyle;
            attributed: Decimal.FormatStyle.Attributed;
            hashValue: number;
            locale: Locale;
            parseStrategy: Decimal.ParseStrategy<Decimal.FormatStyle>;
            regex: Regex<Decimal>;
            constructor (decoder: Decoder);
            constructor (locale: Locale);
            static localizedDecimal(locale: Locale):Decimal.FormatStyle;
            static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
            consuming(input: string, index: String.Index, bounds: Range<String.Index>):any;
            decimalSeparator(strategy: Decimal.FormatStyle.Configuration.DecimalSeparatorDisplayStrategy):Decimal.FormatStyle;
            encode(encoder: Encoder):void;
            format(value: Decimal):string;
            grouping(group: Decimal.FormatStyle.Configuration.Grouping):Decimal.FormatStyle;
            hash(hasher: Hasher):void;
            notation(notation: Decimal.FormatStyle.Configuration.Notation):Decimal.FormatStyle;
            precision(p: Decimal.FormatStyle.Configuration.Precision):Decimal.FormatStyle;
            rounded(rule: Decimal.FormatStyle.Configuration.RoundingRule, increment: number):Decimal.FormatStyle;
            scale(multiplicand: number):Decimal.FormatStyle;
            sign(strategy: Decimal.FormatStyle.Configuration.SignDisplayStrategy):Decimal.FormatStyle;
        }
        /**
         * the same name namespace of class FormatStyle
         */
        export namespace FormatStyle {
            export class Attributed {
                hashValue: number;
                constructor (decoder: Decoder);
                static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
                encode(encoder: Encoder):void;
                format(value: Decimal):AttributedString;
                hash(hasher: Hasher):void;
                locale(locale: Locale):Decimal.FormatStyle.Attributed;
            }
            export class Currency {
                attributed: Decimal.FormatStyle.Attributed;
                currencyCode: string;
                hashValue: number;
                locale: Locale;
                parseStrategy: Decimal.ParseStrategy<Decimal.FormatStyle.Currency>;
                regex: Regex<Decimal>;
                constructor (code: string, locale: Locale);
                constructor (decoder: Decoder);
                static currency(code: string):Decimal.FormatStyle.Currency;
                static localizedCurrency(code: Locale.Currency, locale: Locale):Decimal.FormatStyle.Currency;
                static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
                consuming(input: string, index: String.Index, bounds: Range<String.Index>):any;
                decimalSeparator(strategy: Decimal.FormatStyle.Currency.Configuration.DecimalSeparatorDisplayStrategy):Decimal.FormatStyle.Currency;
                encode(encoder: Encoder):void;
                format(value: Decimal):string;
                grouping(group: Decimal.FormatStyle.Currency.Configuration.Grouping):Decimal.FormatStyle.Currency;
                hash(hasher: Hasher):void;
                precision(p: Decimal.FormatStyle.Currency.Configuration.Precision):Decimal.FormatStyle.Currency;
                presentation(p: Decimal.FormatStyle.Currency.Configuration.Presentation):Decimal.FormatStyle.Currency;
                rounded(rule: Decimal.FormatStyle.Currency.Configuration.RoundingRule, increment: number):Decimal.FormatStyle.Currency;
                scale(multiplicand: number):Decimal.FormatStyle.Currency;
                sign(strategy: Decimal.FormatStyle.Currency.Configuration.SignDisplayStrategy):Decimal.FormatStyle.Currency;
            }
            export class Percent {
                static percent: Decimal.FormatStyle.Percent;
                attributed: Decimal.FormatStyle.Attributed;
                hashValue: number;
                locale: Locale;
                parseStrategy: Decimal.ParseStrategy<Decimal.FormatStyle.Percent>;
                regex: Regex<Decimal>;
                constructor (decoder: Decoder);
                constructor (locale: Locale);
                static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
                consuming(input: string, index: String.Index, bounds: Range<String.Index>):any;
                decimalSeparator(strategy: Decimal.FormatStyle.Percent.Configuration.DecimalSeparatorDisplayStrategy):Decimal.FormatStyle.Percent;
                encode(encoder: Encoder):void;
                format(value: Decimal):string;
                grouping(group: Decimal.FormatStyle.Percent.Configuration.Grouping):Decimal.FormatStyle.Percent;
                hash(hasher: Hasher):void;
                notation(notation: Decimal.FormatStyle.Percent.Configuration.Notation):Decimal.FormatStyle.Percent;
                precision(p: Decimal.FormatStyle.Percent.Configuration.Precision):Decimal.FormatStyle.Percent;
                rounded(rule: Decimal.FormatStyle.Percent.Configuration.RoundingRule, increment: number):Decimal.FormatStyle.Percent;
                scale(multiplicand: number):Decimal.FormatStyle.Percent;
                sign(strategy: Decimal.FormatStyle.Percent.Configuration.SignDisplayStrategy):Decimal.FormatStyle.Percent;
            }
        }
    }
    export class Calendar {}
    /**
     * the same name namespace of class Calendar
     */
    export namespace Calendar {
        export enum Identifier {
            buddhist,
            chinese,
            coptic,
            ethiopicAmeteAlem,
            ethiopicAmeteMihret,
            gregorian,
            hebrew,
            indian,
            islamic,
            islamicCivil,
            islamicTabular,
            islamicUmmAlQura,
            iso8601,
            japanese,
            persian,
            republicOfChina,
        }
        export enum Component {
            era,
            year,
            yearForWeekOfYear,
            quarter,
            month,
            weekOfYear,
            weekOfMonth,
            weekday,
            weekdayOrdinal,
            day,
            hour,
            minute,
            second,
            nanosecond,
            calendar,
            timeZone,
        }
    }
    export class NumberFormatter extends Formatter {
        /**
         * The formatter behavior of the receiver.
         */
        formatterBehavior: NumberFormatter.Behavior;
        /**
         * The number style used by the receiver.
         */
        numberStyle: NumberFormatter.Style;
        /**
         * Determines whether the receiver creates instances of 
         *  when it converts strings to number objects.
         */
        generatesDecimalNumbers: boolean;
        /**
         * Determines whether the dollar sign character (
         * ), decimal separator character (
         * ), and thousand separator character (
         * ) are converted to appropriately localized characters as specified by the user’s localization preference.
         */
        localizesFormat: boolean;
        /**
         * The locale of the receiver.
         */
        locale: Locale!;
        /**
         * The rounding behavior used by the receiver.
         */
        roundingBehavior: NSDecimalNumberHandler;
        /**
         * The rounding increment used by the receiver.
         */
        roundingIncrement: NSNumber!;
        /**
         * The rounding mode used by the receiver.
         */
        roundingMode: NumberFormatter.RoundingMode;
        /**
         * The minimum number of digits before the decimal separator.
         */
        minimumIntegerDigits: number;
        /**
         * The maximum number of digits before the decimal separator.
         */
        maximumIntegerDigits: number;
        /**
         * The minimum number of digits after the decimal separator.
         */
        minimumFractionDigits: number;
        /**
         * The maximum number of digits after the decimal separator.
         */
        maximumFractionDigits: number;
        /**
         * A Boolean value indicating whether the formatter uses minimum and maximum significant digits when formatting numbers.
         */
        usesSignificantDigits: boolean;
        /**
         * The minimum number of significant digits for the number formatter.
         */
        minimumSignificantDigits: number;
        /**
         * The maximum number of significant digits for the number formatter.
         */
        maximumSignificantDigits: number;
        /**
         * The receiver’s format.
         */
        format: string;
        /**
         * The capitalization formatting context used when formatting a number.
         */
        formattingContext: Formatter.Context;
        /**
         * The format width used by the receiver.
         */
        formatWidth: number;
        /**
         * The format the receiver uses to display negative values.
         */
        negativeFormat: String!;
        /**
         * The format the receiver uses to display positive values.
         */
        positiveFormat: String!;
        /**
         * The multiplier of the receiver.
         */
        multiplier: NSNumber | null;
        /**
         * The string used to represent a percent symbol.
         */
        percentSymbol: String!;
        /**
         * The string used to represent a per-mill (per-thousand) symbol.
         */
        perMillSymbol: String!;
        /**
         * The string used to represent a minus sign.
         */
        minusSign: String!;
        /**
         * The string used to represent a plus sign.
         */
        plusSign: String!;
        /**
         * The string used to represent an exponent symbol.
         */
        exponentSymbol: String!;
        /**
         * The string used to represent a zero value.
         */
        zeroSymbol: string;
        /**
         * The string used to represent a 
         *  value.
         */
        nilSymbol: string;
        /**
         * The string used to represent a NaN (“not a number”) value.
         */
        notANumberSymbol: String!;
        /**
         * The string used to represent a negative infinity symbol.
         */
        negativeInfinitySymbol: string;
        /**
         * The string used to represent a positive infinity symbol. 
         */
        positiveInfinitySymbol: string;
        /**
         * The string used by the receiver as a local currency symbol.
         */
        currencySymbol: String!;
        /**
         * The receiver’s currency code.
         */
        currencyCode: String!;
        /**
         * The international currency symbol used by the receiver.
         */
        internationalCurrencySymbol: String!;
        /**
         * The currency grouping separator for the receiver.
         */
        currencyGroupingSeparator: String!;
        /**
         * The string the receiver uses as the prefix for positive values.
         */
        positivePrefix: String!;
        /**
         * The string the receiver uses as the suffix for positive values.
         */
        positiveSuffix: String!;
        /**
         * The string the receiver uses as a prefix for negative values.
         */
        negativePrefix: String!;
        /**
         * The string the receiver uses as a suffix for negative values.
         */
        negativeSuffix: String!;
        /**
         * The text attributes to be used in displaying negative values.
         */
        textAttributesForNegativeValues: Record<string, any> | null;
        /**
         * The text attributes to be used in displaying positive values.
         */
        textAttributesForPositiveValues: Record<string, any> | null;
        /**
         * The attributed string that the receiver uses to display zero values.
         */
        attributedStringForZero: NSAttributedString;
        /**
         * The text attributes used to display a zero value.
         */
        textAttributesForZero: Record<string, any> | null;
        /**
         * The attributed string the receiver uses to display 
         *  values.
         */
        attributedStringForNil: NSAttributedString;
        /**
         * The text attributes used to display the 
         *  symbol.
         */
        textAttributesForNil: Record<string, any> | null;
        /**
         * The attributed string the receiver uses to display “not a number” values.
         */
        attributedStringForNotANumber: NSAttributedString;
        /**
         * The text attributes used to display the NaN (“not a number”) string.
         */
        textAttributesForNotANumber: Record<string, any> | null;
        /**
         * The text attributes used to display the positive infinity symbol.
         */
        textAttributesForPositiveInfinity: Record<string, any> | null;
        /**
         * The text attributes used to display the negative infinity symbol.
         */
        textAttributesForNegativeInfinity: Record<string, any> | null;
        /**
         * The string used by the receiver for a grouping separator.
         */
        groupingSeparator: String!;
        /**
         * Determines whether the receiver displays the group separator.
         */
        usesGroupingSeparator: boolean;
        /**
         * The character the receiver uses as a thousand separator.
         */
        thousandSeparator: String!;
        /**
         * Determines whether the receiver uses thousand separators.
         */
        hasThousandSeparators: boolean;
        /**
         * The character the receiver uses as a decimal separator.
         */
        decimalSeparator: String!;
        /**
         * Determines whether the receiver always shows the decimal separator, even for integer numbers.
         */
        alwaysShowsDecimalSeparator: boolean;
        /**
         * The string used by the receiver as a currency decimal separator.
         */
        currencyDecimalSeparator: String!;
        /**
         * The grouping size of the receiver.
         */
        groupingSize: number;
        /**
         * The secondary grouping size of the receiver.
         */
        secondaryGroupingSize: number;
        /**
         * The string that the receiver uses to pad numbers in the formatted string representation.
         */
        paddingCharacter: String!;
        /**
         * The padding position used by the receiver.
         */
        paddingPosition: NumberFormatter.PadPosition;
        /**
         * Determines whether the receiver allows as input floating-point values (that is, values that include the period character [
         * ]).
         */
        allowsFloats: boolean;
        /**
         * The lowest number allowed as input by the receiver.
         */
        minimum: NSNumber | null;
        /**
         * The highest number allowed as input by the receiver.
         */
        maximum: NSNumber | null;
        /**
         * Determines whether the receiver will use heuristics to guess at the number which is intended by a string.
         */
        isLenient: boolean;
        /**
         * Determines whether partial string validation is enabled for the receiver.
         */
        isPartialStringValidationEnabled: boolean;
        /**
         * Sets the default formatter behavior for new instances of 
         *  .
         */
        static setDefaultFormatterBehavior(behavior: NumberFormatter.Behavior):void;
        /**
         * Returns an 
         *  constant that indicates default formatter behavior for new instances of 
         * .
         */
        static defaultFormatterBehavior():NumberFormatter.Behavior;
        /**
         * Returns a localized number string with the specified style.
         */
        static localizedString(num: NSNumber, nstyle: NumberFormatter.Style):string;
        /**
         * Returns by reference a cell-content object after creating it from a range of characters in a given string.
         */
        getObjectValue(obj: AutoreleasingUnsafeMutablePointer<AnyObject?> | null, _string: string, rangep: UnsafeMutablePointer<NSRange> | null):void;
        /**
         * Returns an 
         *  object created by parsing a given string.
         */
        number(_string: string):NSNumber | null;
        /**
         * Returns a string containing the formatted value of the provided number object.
         */
        string(_number: NSNumber):string;
    }
    /**
     * the same name namespace of class NumberFormatter
     */
    export namespace NumberFormatter {
        export enum Style {
            none = 0,
            decimal = 1,
            percent = 3,
            scientific = 4,
            spellOut = 5,
            ordinal = 6,
            currency = 2,
            currencyAccounting = 10,
            currencyISOCode = 8,
            currencyPlural = 9,
        }
        export enum Behavior {
            default = 0,
            behavior10_0 = 1000,
            behavior10_4 = 1040,
        }
        export enum PadPosition {
            beforePrefix = 0,
            afterPrefix = 1,
            beforeSuffix = 2,
            afterSuffix = 3,
        }
        export enum RoundingMode {
            ceiling = 0,
            floor = 1,
            down = 2,
            up = 3,
            halfEven = 4,
            halfDown = 5,
            halfUp = 6,
        }
    }
    export class NSDecimalNumberHandler extends NSObject {
        /**
         * Returns the default instance of 
         * .
         */
        static default: NSDecimalNumberHandler;
        /**
         * Returns an 
         *  object initialized so it behaves as specified by the method’s arguments.
         */
        constructor (roundingMode: NSDecimalNumber.RoundingMode, scale: Int16, exact: boolean, overflow: boolean, underflow: boolean, divideByZero: boolean);
    }
    export class NSLocale extends NSObject {}
    /**
     * the same name namespace of class NSLocale
     */
    export namespace NSLocale {
        export enum LanguageDirection {
            unknown = 0,
            leftToRight = 1,
            rightToLeft = 2,
            topToBottom = 3,
            bottomToTop = 4,
        }
    }
    export class NSOrthography extends NSObject {
        /**
         * A dictionary that maps script tags to arrays of language tags.
         */
        languageMap: Record<string, string[]>;
        /**
         * The first language in the list of languages for the dominant script.
         */
        dominantLanguage: string;
        /**
         * The dominant script for the text.
         */
        dominantScript: string;
        /**
         * The scripts appearing as keys in the language map.
         */
        allScripts: string[];
        /**
         * The languages appearing in values of the language map.
         */
        allLanguages: string[];
        /**
         * Creates an orthography object with the specified dominant script and language map.
         */
        constructor (script: string, map: Record<string, string[]>);
        constructor (coder: NSCoder);
        /**
         * Creates and returns an orthography object with the default language map for the specified language.
         */
        static defaultOrthography(language: string):Self;
        /**
         * Returns the list of languages for the specified script.
         */
        languages(script: string):string[] | null;
    }
    export class LocalizedStringResource {
        /**
         * The key to use to look up a localized string.
         */
        key: string;
        /**
         * The resource’s default value.
         */
        defaultValue: String.LocalizationValue;
        /**
         * The name of the table containing the key-value pairs.
         */
        table: string;
        /**
         * The bundle containing the table’s strings file.
         */
        bundle: LocalizedStringResource.BundleDescription;
        /**
         * The locale to use to look up the localized string from the string resource.
         */
        locale: Locale;
        /**
         * A resource that helps provide a description of the instance.
         */
        localizedStringResource: LocalizedStringResource;
        /**
         * Creates a localized string resource from a localization key and its bundle properties.
         */
        constructor (keyAndValue: String.LocalizationValue, table: string, locale: Locale, bundle: LocalizedStringResource.BundleDescription, comment: StaticString | null);
        /**
         * Creates a localized string resource from a static string and its bundle properties.
         */
        constructor (key: StaticString, defaultValue: String.LocalizationValue, table: string, locale: Locale, bundle: LocalizedStringResource.BundleDescription, comment: StaticString | null);
        /**
         * Creates a localized string resource from the specified string literal.
         */
        constructor (value: string);
        /**
         * Creates an instance initialized to the given value.
         */
        constructor (value: string);
        /**
         * Creates an instance initialized to the given value.
         */
        constructor (value: string);
        /**
         * Creates a localized string resource from the given string interpolation.
         */
        constructor (stringInterpolation: String.LocalizationValue.StringInterpolation);
        /**
         * Creates a new instance by decoding from the given decoder.
         */
        constructor (decoder: Decoder);
        /**
         * Encodes this value into the given encoder.
         */
        encode(encoder: Encoder):void;
    }
    /**
     * the same name namespace of class LocalizedStringResource
     */
    export namespace LocalizedStringResource {
        export enum BundleDescription {
            main,
            atURL,
            forClass,
        }
    }
    export interface CustomLocalizedStringResourceConvertible {
        /**
         * A resource that helps provide a description of this instance.
         */
        localizedStringResource: LocalizedStringResource;
    }
    export class URLResource {
        /**
         * The bundle containing the resource.
         */
        bundle: Bundle;
        /**
         * The name of the resource in the bundle.
         */
        name: string;
        /**
         * The subdirectory, if any, of the resource.
         */
        subdirectory: string;
        /**
         * The bundle containing the resource.
         */
        locale: Locale;
        /**
         * The hash value.
         */
        hashValue: number;
        /**
         * Creates a URL resource from the given bundle, name, and subdirectory, optionally specifying a locale.
         */
        constructor (name: string, subdirectory: string, locale: Locale, bundle: Bundle);
        /**
         * Creates a new instance by decoding from the given decoder.
         */
        constructor (decoder: Decoder);
        /**
         * Hashes the essential components of the value by feeding them into the given hasher.
         */
        hash(hasher: Hasher):void;
        /**
         * Encodes the value into the given encoder.
         */
        encode(encoder: Encoder):void;
    }
    export class NSLinguisticTagger extends NSObject {
        /**
         * The string being analyzed by the linguistic tagger.
         */
        string: string;
        /**
         * Returns the tag schemes configured for this linguistic tagger. For possible values, see 
         * .
         */
        tagSchemes: NSLinguisticTagScheme[];
        /**
         * Creates a linguistic tagger instance using the specified tag schemes and options.
         */
        constructor (tagSchemes: NSLinguisticTagScheme[], opts: number);
        /**
         * Returns the tag schemes available for a particular unit and language on the current device. 
         */
        static availableTagSchemes(unit: NSLinguisticTaggerUnit, language: string):NSLinguisticTagScheme[];
        /**
         * Returns the dominant language for the specified string.
         */
        static dominantLanguage(_string: string):string;
        /**
         * Returns the orthography at the index and also returns the effective range.
         */
        orthography(charIndex: number, effectiveRange: NSRangePointer | null):NSOrthography | null;
        /**
         * Sets the orthography for the specified range.
         */
        setOrthography(orthography: NSOrthography | null, range: NSRange):void;
        /**
         * Enumerates over a given range of the string for a particular unit and calls the specified block for each tag.
         */
        enumerateTags(range: NSRange, unit: NSLinguisticTaggerUnit, scheme: NSLinguisticTagScheme, options: NSLinguisticTagger.Options, block: ()=>any):void;
        /**
         * Returns a tag for a single scheme, for a given linguistic unit, at the specified character position.
         */
        tag(charIndex: number, unit: NSLinguisticTaggerUnit, scheme: NSLinguisticTagScheme, tokenRange: NSRangePointer | null):NSLinguisticTag | null;
        /**
         * Returns an array of linguistic tags and token ranges for a given string range and linguistic unit.
         */
        tags(range: NSRange, unit: NSLinguisticTaggerUnit, scheme: NSLinguisticTagScheme, options: NSLinguisticTagger.Options, tokenRanges: AutoreleasingUnsafeMutablePointer<NSArray?> | null):NSLinguisticTag[];
        /**
         * Returns the range of the linguistic unit containing the specified character index. 
         */
        tokenRange(charIndex: number, unit: NSLinguisticTaggerUnit):NSRange;
        /**
         * Returns the range of a sentence containing the specified range.
         */
        sentenceRange(range: NSRange):NSRange;
        /**
         * Returns an array of possible tags for the given scheme at the specified range, supplying matching scores.
         */
        possibleTags(charIndex: number, tagScheme: string, tokenRange: NSRangePointer | null, sentenceRange: NSRangePointer | null, scores: AutoreleasingUnsafeMutablePointer<NSArray?> | null):string[] | null;
        /**
         * Notifies the linguistic tagger that the string (if mutable) has changed as specified by the parameters.
         */
        stringEdited(newRange: NSRange, delta: number):void;
    }
    /**
     * the same name namespace of class NSLinguisticTagger
     */
    export namespace NSLinguisticTagger {
        export class Options {
            /**
             * Omit tokens of type 
             *  (items considered to be words).
             */
            static omitWords: NSLinguisticTagger.Options;
            /**
             * Omit tokens of type 
             *  (all punctuation).
             */
            static omitPunctuation: NSLinguisticTagger.Options;
            /**
             * Omit tokens of type 
             *  (whitespace of all sorts).
             */
            static omitWhitespace: NSLinguisticTagger.Options;
            /**
             * Omit tokens of type 
             *  (non-linguistic items, such as symbols).
             */
            static omitOther: NSLinguisticTagger.Options;
            /**
             * Typically, multiple-word names will be returned as multiple tokens, following the standard tokenization practice of the tagger.  If this option is set, then multiple-word names will be joined together and returned as a single token.
             */
            static joinNames: NSLinguisticTagger.Options;
            constructor (rawValue: number);
        }
    }
    export class NSLinguisticTagScheme {
        /**
         * Classifies tokens according to their broad type:  word, punctuation, or whitespace.
         */
        static tokenType: NSLinguisticTagScheme;
        /**
         * Classifies tokens according to class:  part of speech, type of punctuation, or whitespace.
         */
        static lexicalClass: NSLinguisticTagScheme;
        /**
         * Classifies tokens according to whether they are part of a named entity.
         */
        static nameType: NSLinguisticTagScheme;
        /**
         * Classifies tokens corresponding to names according to 
         * , and classifies all other tokens according to 
         * .
         */
        static nameTypeOrLexicalClass: NSLinguisticTagScheme;
        /**
         * Supplies a stem form of a word token, if known.
         */
        static lemma: NSLinguisticTagScheme;
        /**
         * Supplies the language for a token, if one can be determined.
         */
        static language: NSLinguisticTagScheme;
        /**
         * Supplies the script for a token, if one can be determined.
         */
        static script: NSLinguisticTagScheme;
        constructor (rawValue: string);
        constructor (rawValue: string);
    }
    export enum NSLinguisticTaggerUnit {
        document = 3,
        paragraph = 2,
        sentence = 1,
        word = 0,
    }
    export class NSLinguisticTag {
        /**
         * The token indicates a word.
         */
        static word: NSLinguisticTag;
        /**
         * The token indicates punctuation.
         */
        static punctuation: NSLinguisticTag;
        /**
         * The token indicates white space of any sort.
         */
        static whitespace: NSLinguisticTag;
        /**
         * The token indicates a non-linguistic item, such as a symbol.
         */
        static other: NSLinguisticTag;
        /**
         * The token is a noun.
         */
        static noun: NSLinguisticTag;
        /**
         * This token is a verb.
         */
        static verb: NSLinguisticTag;
        /**
         * This token is an adjective
         */
        static adjective: NSLinguisticTag;
        /**
         * This token is an adverb.
         */
        static adverb: NSLinguisticTag;
        /**
         * This token is a pronoun.
         */
        static pronoun: NSLinguisticTag;
        /**
         * This token is a determiner.
         */
        static determiner: NSLinguisticTag;
        /**
         * This token is a particle.
         */
        static particle: NSLinguisticTag;
        /**
         * This token is a preposition.
         */
        static preposition: NSLinguisticTag;
        /**
         * This token is a number.
         */
        static number: NSLinguisticTag;
        /**
         * This token is a conjunction.
         */
        static conjunction: NSLinguisticTag;
        /**
         * This token is an interjection.
         */
        static interjection: NSLinguisticTag;
        /**
         * This token is a classifier.
         */
        static classifier: NSLinguisticTag;
        /**
         * This token is an idiom.
         */
        static idiom: NSLinguisticTag;
        /**
         * This token is a word other than a kind described by other lexical classes (noun, verb, adjective, adverb, pronoun, determiner, particle, preposition, number, conjunction, interjection, classifier, and idiom).
         */
        static otherWord: NSLinguisticTag;
        /**
         * This token is a sentence terminator.
         */
        static sentenceTerminator: NSLinguisticTag;
        /**
         * This token is an open quote.
         */
        static openQuote: NSLinguisticTag;
        /**
         * This token is a close quote.
         */
        static closeQuote: NSLinguisticTag;
        /**
         * This token is an open parenthesis.
         */
        static openParenthesis: NSLinguisticTag;
        /**
         * This token is a close parenthesis.
         */
        static closeParenthesis: NSLinguisticTag;
        /**
         * This token is a word joiner.
         */
        static wordJoiner: NSLinguisticTag;
        /**
         * This token is a dash.
         */
        static dash: NSLinguisticTag;
        /**
         * This token is punctuation other than a kind described by other lexical classes (sentence terminator, open or close quote, open or close parenthesis, word joiner, and dash).
         */
        static otherPunctuation: NSLinguisticTag;
        /**
         * This token is a paragraph break.
         */
        static paragraphBreak: NSLinguisticTag;
        /**
         * This token is whitespace other than a kind described by other lexical classes (paragraph break).
         */
        static otherWhitespace: NSLinguisticTag;
        /**
         * This token is a personal name.
         */
        static personalName: NSLinguisticTag;
        /**
         * This token is an organization name.
         */
        static organizationName: NSLinguisticTag;
        /**
         * This token is a place name.
         */
        static placeName: NSLinguisticTag;
        constructor (rawValue: string);
        constructor (rawValue: string);
    }
    export class Data {
        static transferRepresentation: TransferRepresentation;
        /**
         * A Boolean value indicating whether the collection is empty.
         */
        isEmpty: boolean;
        /**
         * A sequence containing the same elements as this sequence, but on which some operations, such as 
         *  and 
         * , are implemented lazily.
         */
        lazy: LazySequence<Data>;
        /**
         * The beginning index into the data.
         */
        startIndex: Data.Index;
        /**
         * The end index into the data.
         */
        endIndex: Data.Index;
        /**
         * A textual description of the data.
         */
        description: string;
        /**
         * A textual description the data suitable for debugging.
         */
        debugDescription: string;
        /**
         * A mirror that reflects the data.
         */
        customMirror: Mirror;
        /**
         * The hash value for the data.
         */
        hashValue: number;
        count: number;
        indices: Range<Int>;
        /**
         * The last element of the collection.
         */
        last: UInt8 | null;
        publisher: any;
        regions: CollectionOfOne<Data>;
        /**
         * A value less than or equal to the number of elements in the collection.
         */
        underestimatedCount: number;
        /**
         * Creates an empty data buffer.
         */
        constructor ();
        /**
         * Creates an empty data buffer of a specified size.
         */
        constructor (capacity: number);
        /**
         * Creates a new data buffer with the specified count of zeroed bytes.
         */
        constructor (count: number);
        /**
         * Creates data with copied memory content.
         */
        constructor (bytes: UnsafeRawPointer, count: number);
        /**
         * Creates a data buffer with memory content without copying the bytes.
         */
        constructor (bytes: UnsafeMutableRawPointer, count: number, deallocator: Data.Deallocator);
        constructor (base64Data: Data, options: Data.Base64DecodingOptions);
        constructor (url: URL, options: Data.ReadingOptions);
        constructor (decoder: Decoder);
        constructor (reference: NSData);
        constructor (repeatedValue: UInt8, count: number);
        /**
         * Sets a region of the data buffer to 
         * .
         */
        resetBytes(range: Range<Data.Index>):void;
        /**
         * Writes the contents of the data buffer to a location.
         */
        write(url: URL, options: Data.WritingOptions):void;
        /**
         * Returns Base-64 encoded data.
         */
        base64EncodedData(options: Data.Base64EncodingOptions):Data;
        /**
         * Returns a Base-64 encoded string.
         */
        base64EncodedString(options: Data.Base64EncodingOptions):string;
        /**
         * Copies the contents of the data to memory.
         */
        copyBytes(pointer: UnsafeMutablePointer<UInt8>, count: number):void;
        /**
         * Appends the specified data to the end of this data.
         */
        append(other: Data):void;
        /**
         * Prepares the collection to store the specified number of elements, when doing so is appropriate for the underlying type.
         */
        reserveCapacity(minimumCapacity: number):void;
        /**
         * Removes and returns the element at the specified position.
         */
        remove(position: number):UInt8;
        /**
         * Removes all elements from the collection.
         */
        removeAll(keepCapacity: boolean):void;
        /**
         * Removes the elements in the specified subrange from the collection.
         */
        removeSubrange(bounds: Range<Int>):void;
        /**
         * Replaces a region of bytes in the data with new data.
         */
        replaceSubrange(subrange: Range<Data.Index>, data: Data):void;
        /**
         * Returns the first element of the sequence that satisfies the given predicate.
         */
        first(predicate: Bool):UInt8 | null;
        /**
         * Returns the maximum element in the sequence.
         */
        max():UInt8 | null;
        /**
         * Returns the minimum element in the sequence.
         */
        min():UInt8 | null;
        /**
         * Finds the range of the specified data as a subsequence of this data, if it exists.
         */
        range(dataToFind: Data, options: Data.SearchOptions, range: Range<Data.Index> | null):Range<Data.Index> | null;
        /**
         * Returns a new collection of the same type containing, in order, the elements of the original collection that satisfy the given predicate.
         */
        filter(isIncluded: Bool):Data;
        /**
         * Returns a subsequence, up to the specified maximum length, containing the initial elements of the collection.
         */
        prefix(maxLength: number):Data;
        /**
         * Returns a subsequence, up to the given maximum length, containing the final elements of the collection.
         */
        suffix(maxLength: number):Data;
        /**
         * Returns a subsequence containing all but the specified number of final elements.
         */
        dropLast(k: number):Data;
        /**
         * Returns a subsequence containing all but the given number of initial elements.
         */
        dropFirst(k: number):Data;
        /**
         * Returns a subsequence by skipping elements while 
         *  returns 
         *  and returning the remaining elements.
         */
        drop(predicate: Bool):Data;
        /**
         * Returns a new data buffer created by removing the given number of bytes from the front of the original buffer.
         */
        advanced(amount: number):Data;
        /**
         * Calls the given closure on each element in the sequence in the same order as a 
         * -
         *  loop.
         */
        forEach(body: Void):void;
        /**
         * Returns a sequence of pairs (
         * , 
         * ), where 
         *  represents a consecutive integer starting at zero and 
         *  represents an element of the sequence.
         */
        enumerated():EnumeratedSequence<Data>;
        /**
         * Returns an iterator over the contents of the data.
         */
        makeIterator():Data.Iterator;
        /**
         * Enumerates the contents of the data's buffer.
         */
        enumerateBytes(block: ()=>any):void;
        /**
         * Sorts the collection in place, using the given predicate as the comparison between elements.
         */
        sort(areInIncreasingOrder: Bool):void;
        /**
         * Returns the elements of the sequence, sorted.
         */
        sorted():UInt8[];
        /**
         * Returns a view presenting the elements of the collection in reverse order.
         */
        reversed():ReversedCollection<Data>;
        /**
         * Returns a new copy of the data in a specified range.
         */
        subdata(range: Range<Data.Index>):Data;
        /**
         * Returns the longest possible subsequences of the collection, in order, that don’t contain elements satisfying the given predicate.
         */
        split(maxSplits: number, omittingEmptySubsequences: boolean, isSeparator: Bool):Data[];
        /**
         * Returns the index that immediately follows the specified index.
         */
        index(i: Data.Index):Data.Index;
        /**
         * Returns a Boolean value indicating whether every element of a sequence satisfies a given predicate.
         */
        allSatisfy(predicate: Bool):boolean;
        /**
         * Applies the given difference to this collection.
         */
        applying(difference: CollectionDifference<UInt8>):Data | null;
        /**
         * Returns a Boolean value indicating whether the sequence contains the given element.
         */
        contains(element: UInt8):boolean;
        encode(encoder: Encoder):void;
        /**
         * Returns the first index where the specified value appears in the collection.
         */
        firstIndex(element: UInt8):number;
        /**
         * Offsets the given index by the specified distance.
         */
        formIndex(i: Int, distance: number):void;
        hash(hasher: Hasher):void;
        /**
         * Inserts a new element into the collection at the specified position.
         */
        insert(newElement: UInt8, i: number):void;
        /**
         * Returns the last index where the specified value appears in the collection.
         */
        lastIndex(element: UInt8):number;
        /**
         * Reorders the elements of the collection such that all the elements that match the given predicate are after all the elements that don’t match.
         */
        partition(belongsInSecondPartition: Bool):number;
        /**
         * Removes and returns the first element of the collection.
         */
        popFirst():UInt8 | null;
        /**
         * Removes and returns the last element of the collection.
         */
        popLast():UInt8 | null;
        /**
         * Returns a random element of the collection.
         */
        randomElement():UInt8 | null;
        /**
         * Removes and returns the first element of the collection.
         */
        removeFirst():UInt8;
        /**
         * Removes and returns the last element of the collection.
         */
        removeLast():UInt8;
        /**
         * Reverses the elements of the collection in place.
         */
        reverse():void;
        /**
         * Shuffles the collection in place.
         */
        shuffle():void;
        /**
         * Returns the elements of the sequence, shuffled.
         */
        shuffled():UInt8[];
        /**
         * Exchanges the values at the specified indices of the collection.
         */
        swapAt(i: number, j: number):void;
        trimPrefix(predicate: Bool):void;
        trimmingPrefix(predicate: Bool):Data;
    }
    /**
     * the same name namespace of class Data
     */
    export namespace Data {
        export enum Deallocator {
            custom,
            free,
            none,
            unmap,
            virtualMemory,
        }
        export class Iterator {
            next():UInt8 | null;
        }
    }
    export class NSArray extends NSObject {
        /**
         * The number of objects in the array.
         */
        count: number;
        /**
         * The first object in the array.
         */
        firstObject: any;
        /**
         * The last object in the array.
         */
        lastObject: any;
        /**
         * Analyzes the array and returns a “hint” that speeds the sorting of the array when the hint is supplied to 
         * .
         */
        sortedArrayHint: Data;
        /**
         * A string that represents the contents of the array, formatted as a property list.
         */
        description: string;
        customMirror: Mirror;
        underestimatedCount: number;
        /**
         * Creates and returns an array containing a given object.
         */
        constructor (anObject: any);
        /**
         * Creates and returns an array that includes a given number of objects from a given C array.
         */
        constructor (objects: UnsafePointer<AnyObject>, cnt: number);
        /**
         * Initializes a newly allocated array.
         */
        constructor ();
        /**
         * Initializes a newly allocated array by placing in it the objects contained in a given array.
         */
        constructor (array: any[]);
        /**
         * Initializes a newly allocated array using 
         *  as the source of data objects for the array.  
         */
        constructor (array: any[], flag: boolean);
        /**
         * Initializes a newly allocated array with the contents of the file specified by a given path.
         */
        constructor (path: string);
        /**
         * Initializes a newly allocated array with the contents of the location specified by a given URL.
         */
        constructor (url: URL);
        constructor (coder: NSCoder);
        /**
         * Create an instance initialized with 
         * .
         */
        constructor (...elements: any);
        constructor (url: URL, error: any);
        constructor (...elements: any);
        /**
         * Returns a Boolean value that indicates whether a given object is present in the array.
         */
        contains(anObject: any):boolean;
        /**
         * Returns the object located at the specified index.
         */
        object(index: number):any;
        /**
         * Returns an array containing the objects in the array at the indexes specified by a given index set.
         */
        objects(indexes: IndexSet):any[];
        /**
         * Returns an enumerator object that lets you access each object in the array.
         */
        objectEnumerator():NSEnumerator;
        /**
         * Returns an enumerator object that lets you access each object in the array, in reverse order.
         */
        reverseObjectEnumerator():NSEnumerator;
        /**
         * Returns the lowest index whose corresponding array value is equal to a given object.
         */
        index(anObject: any):number;
        /**
         * Returns the lowest index whose corresponding array value is identical to a given object.
         */
        indexOfObjectIdentical(anObject: any):number;
        /**
         * Returns the index of the first object in the array that passes a test in a given block.
         */
        indexOfObject(predicate: ()=>any):number;
        /**
         * Returns the indexes of objects in the array that pass a test in a given block.
         */
        indexesOfObjects(predicate: ()=>any):IndexSet;
        /**
         * Executes a given closure using each object in the array, starting with the first object and continuing through the array to the last object.
         */
        enumerateObjects(block: ()=>any):void;
        /**
         * Returns the first object contained in the receiving array that’s equal to an object in another given array.
         */
        firstObjectCommon(otherArray: any[]):any;
        /**
         * Compares the receiving array to another array.
         */
        isEqual(otherArray: any[]):boolean;
        /**
         * Returns a new array that is a copy of the receiving array with a given object added to the end.
         */
        adding(anObject: any):any[];
        /**
         * Returns a new array that is a copy of the receiving array with the objects contained in another array added to the end.
         */
        addingObjects(otherArray: any[]):any[];
        /**
         * Evaluates a given predicate against each object in the receiving array and returns a new array containing the objects for which the predicate returns true.
         */
        filtered(predicate: NSPredicate):any[];
        /**
         * Returns a new array containing the receiving array’s elements that fall within the limits specified by a given range.
         */
        subarray(range: NSRange):any[];
        /**
         * Returns a new array that lists the receiving array’s elements in ascending order as defined by the comparison function 
         * .
         */
        sortedArray(comparator: ()=>any, context: UnsafeMutableRawPointer | null):any[];
        /**
         * Constructs and returns an 
         *  object that is the result of interposing a given separator between the elements of the array.
         */
        componentsJoined(separator: string):string;
        /**
         * Writes the contents of the array to a file at a given path.
         */
        write(path: string, useAuxiliaryFile: boolean):boolean;
        /**
         * Returns an array containing all the pathname elements in the receiving array that have filename extensions from a given array.
         */
        pathsMatchingExtensions(filterTypes: string[]):string[];
        /**
         * Raises an exception.
         */
        addObserver(observer: NSObject, keyPath: string, options: NSKeyValueObservingOptions, context: UnsafeMutableRawPointer | null):void;
        /**
         * Raises an exception.
         */
        removeObserver(observer: NSObject, keyPath: string):void;
        /**
         * Invokes 
         *  on each of the array's items using the specified 
         *  and 
         * .
         */
        setValue(value: any, key: string):void;
        /**
         * Returns an array containing the results of invoking 
         *  using 
         *  on each of the array's objects.
         */
        value(key: string):any;
        /**
         * Returns a new array that lists this array’s elements in a random order.
         */
        shuffled():any[];
        /**
         * Return an 
         *  over the elements of this 
         * .
         */
        makeIterator():NSFastEnumerationIterator;
    }
    export class NSOrderedCollectionDifference extends NSObject {
        /**
         * A Boolean value that indicates if the difference has changes.
         */
        hasChanges: boolean;
        /**
         * A collection of insertion change objects.
         */
        insertions: NSOrderedCollectionChange[];
        /**
         * A collection of removal change objects.
         */
        removals: NSOrderedCollectionChange[];
        /**
         * Creates an ordered collection difference using an array of ordered collection changes.
         */
        constructor (changes: NSOrderedCollectionChange[]);
        /**
         * Creates an ordered collection difference from arrays of inserted and removed objects with corresponding sets of indices.
         */
        constructor (inserts: IndexSet, insertedObjects: any[] | null, removes: IndexSet, removedObjects: any[] | null);
        /**
         * Creates an ordered collection difference from arrays of inserted and removed objects with corresponding sets of indices, in addition to an array of ordered collection changes.
         */
        constructor (inserts: IndexSet, insertedObjects: any[] | null, removes: IndexSet, removedObjects: any[] | null, changes: NSOrderedCollectionChange[]);
        /**
         * Calculate the difference between two objects in the reverse direction of comparison.
         */
        inverse():Self;
        /**
         * Create a new ordered collection difference by mapping over this difference’s members, processing the change objects with the block provided.
         */
        transformingChanges(block: ()=>any):CollectionDifference<Any>;
    }
    export class NSOrderedCollectionChange extends NSObject {
        /**
         * The type of change.
         */
        changeType: NSCollectionChangeType;
        /**
         * The index location of the change.
         */
        index: number;
        /**
         * An object the change inserts or removes.
         */
        object: any;
        /**
         * When this property is set to a value other than 
         * , the receiver is one half of a move, and this value is the index of the change’s counterpart of the opposite type in the diff.
         */
        associatedIndex: number;
        /**
         * Creates a change object that represents inserting or removing an object from an ordered collection at a specific index.
         */
        constructor (anObject: any, type: NSCollectionChangeType, index: number);
        /**
         * Creates a change object that represents inserting, removing, or moving an object from an ordered collection at a specific index.
         */
        constructor (anObject: any, type: NSCollectionChangeType, index: number, associatedIndex: number);
    }
    export enum NSCollectionChangeType {
        insert = 0,
        remove = 1,
    }
    export class NSOrderedCollectionDifferenceCalculationOptions {
        /**
         * An option that identifies insertions or removals as moves.
         */
        static inferMoves: NSOrderedCollectionDifferenceCalculationOptions;
        /**
         * An option that indicates that the difference should omit references to the insertions.
         */
        static omitInsertedObjects: NSOrderedCollectionDifferenceCalculationOptions;
        /**
         * An option that indicates that the difference should omit references to the removals.
         */
        static omitRemovedObjects: NSOrderedCollectionDifferenceCalculationOptions;
        /**
         * Creates a set of difference calculation options.
         */
        constructor (rawValue: number);
    }
    export class NSBinarySearchingOptions {
        /**
         * Specifies that the search should return the first object in the range that is equal to the given object.
         */
        static firstEqual: NSBinarySearchingOptions;
        /**
         * Specifies that the search should return the last object in the range that is equal to the given object.
         */
        static lastEqual: NSBinarySearchingOptions;
        /**
         * Returns the index at which you should insert the object in order to maintain a sorted array.
         */
        static insertionIndex: NSBinarySearchingOptions;
        constructor (rawValue: number);
    }
    export class NSMutableArray extends NSArray {
        /**
         * Initializes a newly allocated array.
         */
        constructor ();
        /**
         * Returns an array, initialized with enough memory to initially hold a given number of objects.
         */
        constructor (numItems: number);
        constructor (coder: NSCoder);
        /**
         * Inserts a given object at the end of the array.
         */
        add(anObject: any):void;
        /**
         * Adds the objects contained in another given array to the end of the receiving array’s content.
         */
        addObjects(otherArray: any[]):void;
        /**
         * Inserts a given object into the array’s contents at a given index.
         */
        insert(anObject: any, index: number):void;
        /**
         * Empties the array of all its elements.
         */
        removeAllObjects():void;
        /**
         * Removes the object with the highest-valued index in the array 
         */
        removeLastObject():void;
        /**
         * Removes all occurrences in the array of a given object.
         */
        remove(anObject: any):void;
        /**
         * Removes the object at 
         *  .
         */
        removeObject(index: number):void;
        /**
         * Removes the objects at the specified indexes from the array.
         */
        removeObjects(indexes: IndexSet):void;
        /**
         * Replaces the object at 
         *  with 
         * .
         */
        replaceObject(index: number, anObject: any):void;
        /**
         * Replaces the objects in the receiving array at locations specified with the objects from a given array.
         */
        replaceObjects(indexes: IndexSet, objects: any[]):void;
        /**
         * Sets the receiving array’s elements to those in another given array.
         */
        setArray(otherArray: any[]):void;
        /**
         * Evaluates a given predicate against the array’s content and leaves only objects that match.
         */
        filter(predicate: NSPredicate):void;
        /**
         * Exchanges the objects in the array at given indexes.
         */
        exchangeObject(idx1: number, idx2: number):void;
        /**
         * Sorts the receiver using a given array of sort descriptors.
         */
        sort(sortDescriptors: NSSortDescriptor[]):void;
    }
    export class NSDictionary extends NSObject {
        /**
         * The number of entries in the dictionary.
         */
        count: number;
        /**
         * A new array containing the dictionary’s keys, or an empty array if the dictionary has no entries.
         */
        allKeys: any[];
        /**
         * A new array containing the dictionary’s values, or an empty array if the dictionary has no entries.
         */
        allValues: any[];
        /**
         * A string that represents the contents of the dictionary, formatted as a property list.
         */
        description: string;
        /**
         * A string that represents the contents of the dictionary, formatted in 
         *  file format.
         */
        descriptionInStringsFileFormat: string;
        /**
         * A mirror that reflects the dictionary.
         */
        customMirror: Mirror;
        underestimatedCount: number;
        /**
         * Initializes a newly allocated dictionary.
         */
        constructor ();
        /**
         * Initializes a newly allocated dictionary with key-value pairs constructed from the provided arrays of keys and objects.
         */
        constructor (objects: any[], keys: NSCopying[]);
        /**
         * Initializes a newly allocated dictionary with the specified number of key-value pairs constructed from the provided C arrays of keys and objects.
         */
        constructor (objects: UnsafePointer<AnyObject> | null, keys: UnsafePointer<NSCopying> | null, cnt: number);
        /**
         * Creates a dictionary containing a given key and value.
         */
        constructor (object: any, key: NSCopying);
        /**
         * Initializes a newly allocated dictionary by placing in it the keys and values contained in another given dictionary.
         */
        constructor (otherDictionary: Record<AnyHashable, any>);
        /**
         * Initializes a newly allocated dictionary using the objects contained in another given dictionary.
         */
        constructor (otherDictionary: Record<AnyHashable, any>, flag: boolean);
        /**
         * Initializes a newly allocated dictionary from the given key-value pairs.
         */
        constructor (...elements: any);
        /**
         * Initializes a newly allocated dictionary using the keys and values found at a given URL.
         */
        constructor (url: URL, error: any);
        /**
         * Initializes a newly allocated dictionary using the keys and values found at a given URL.
         */
        constructor (url: URL);
        /**
         * Initializes a newly allocated dictionary using the keys and values found in a file at a given path.
         */
        constructor (path: string);
        /**
         * Creates a dictionary initialized from data in the provided unarchiver.
         */
        constructor (coder: NSCoder);
        /**
         * Creates a shared key set object for the specified keys.
         */
        static sharedKeySet(keys: NSCopying[]):any;
        /**
         * Returns a Boolean value that indicates whether the contents of the receiving dictionary are equal to the contents of another given dictionary.
         */
        isEqual(otherDictionary: Record<AnyHashable, any>):boolean;
        /**
         * Returns the value associated with a given key.
         */
        value(key: string):any;
        /**
         * Returns as a static array the set of objects from the dictionary that corresponds to the specified keys.
         */
        objects(keys: any[], marker: any):any[];
        /**
         * Returns the value associated with a given key.
         */
        object(aKey: any):any;
        /**
         * Provides an enumerator to access the keys in the dictionary.
         */
        keyEnumerator():NSEnumerator;
        /**
         * Returns an enumerator object that lets you access each value in the dictionary.
         */
        objectEnumerator():NSEnumerator;
        /**
         * Applies a given block object to the entries of the dictionary.
         */
        enumerateKeysAndObjects(block: ()=>any):void;
        /**
         * Returns an iterator over the elements of this sequence.
         */
        makeIterator():NSDictionary.Iterator;
        /**
         * Returns an array of the dictionary’s keys, in the order they would be in if the dictionary were sorted by its values.
         */
        keysSortedByValue(comparator: Selector):any[];
        /**
         * Returns the set of keys whose corresponding value satisfies a constraint described by a block object.
         */
        keysOfEntries(predicate: ()=>any):Set<AnyHashable>;
        /**
         * Writes a property list representation of the contents of the dictionary to a given URL.
         */
        write(url: URL):void;
        /**
         * Returns the file’s size, in bytes.
         */
        fileSize():UInt64;
        /**
         * Returns the file type.
         */
        fileType():string;
        /**
         * Returns the file’s creation date.
         */
        fileCreationDate():Date | null;
        /**
         * Returns file’s modification date.
         */
        fileModificationDate():Date | null;
        /**
         * Returns the file’s POSIX permissions.
         */
        filePosixPermissions():number;
        /**
         * Returns the file’s owner account ID.
         */
        fileOwnerAccountID():NSNumber | null;
        /**
         * Returns the file’s owner account name.
         */
        fileOwnerAccountName():string;
        /**
         * Returns file’s group owner account ID.
         */
        fileGroupOwnerAccountID():NSNumber | null;
        /**
         * Returns the file’s group owner account name.
         */
        fileGroupOwnerAccountName():string;
        /**
         * Returns a Boolean value indicating whether the file hides its extension.
         */
        fileExtensionHidden():boolean;
        /**
         * Returns a Boolean value indicating whether the file is immutable.
         */
        fileIsImmutable():boolean;
        /**
         * Returns a Boolean value indicating whether the file is append only.
         */
        fileIsAppendOnly():boolean;
        /**
         * Returns the filesystem file number.
         */
        fileSystemFileNumber():number;
        /**
         * Returns the filesystem number.
         */
        fileSystemNumber():number;
        /**
         * Returns file’s HFS type code.
         */
        fileHFSTypeCode():OSType;
        /**
         * Returns the file’s HFS creator code.
         */
        fileHFSCreatorCode():OSType;
    }
    /**
     * the same name namespace of class NSDictionary
     */
    export namespace NSDictionary {
        export class Iterator {
            /**
             * Returns the next key-value pair of a dictionary object as a tuple.
             */
            next():any;
        }
    }
    export class NSMutableDictionary extends NSDictionary {
        /**
         * Initializes a newly allocated mutable dictionary, allocating enough memory to hold 
         *  entries.
         */
        constructor (numItems: number);
        /**
         * Initializes a newly allocated mutable dictionary.
         */
        constructor ();
        /**
         * Creates a mutable dictionary which is optimized for dealing with a known set of keys.
         */
        constructor (keyset: any);
        constructor (coder: NSCoder);
        constructor (inHeadersData: Data!);
        constructor (inHeadersData: UnsafeRawPointer!, inDataSize: number);
        /**
         * Adds a given key-value pair to the dictionary.
         */
        setObject(anObject: any, aKey: NSCopying):void;
        /**
         * Adds a given key-value pair to the dictionary.
         */
        setValue(value: any, key: string):void;
        /**
         * Adds to the receiving dictionary the entries from another dictionary.
         */
        addEntries(otherDictionary: Record<AnyHashable, any>):void;
        /**
         * Sets the contents of the receiving dictionary to entries in a given dictionary.
         */
        setDictionary(otherDictionary: Record<AnyHashable, any>):void;
        /**
         * Removes a given key and its associated value from the dictionary.
         */
        removeObject(aKey: any):void;
        /**
         * Empties the dictionary of its entries.
         */
        removeAllObjects():void;
        /**
         * Removes from the dictionary entries specified by elements in a given array.
         */
        removeObjects(keyArray: any[]):void;
        addApplicationParameterHeader(inHeaderData: UnsafeRawPointer!, inHeaderDataLength: UInt32):OBEXError;
        addAuthorizationChallengeHeader(inHeaderData: UnsafeRawPointer!, inHeaderDataLength: UInt32):OBEXError;
        addAuthorizationResponseHeader(inHeaderData: UnsafeRawPointer!, inHeaderDataLength: UInt32):OBEXError;
        addBodyHeader(inHeaderData: UnsafeRawPointer!, inHeaderDataLength: UInt32, isEndOfBody: boolean):OBEXError;
        addByteSequenceHeader(inHeaderData: UnsafeRawPointer!, inHeaderDataLength: UInt32):OBEXError;
        addConnectionIDHeader(inHeaderData: UnsafeRawPointer!, inHeaderDataLength: UInt32):OBEXError;
        addCountHeader(inCount: UInt32):OBEXError;
        addDescriptionHeader(inDescriptionString: String!):OBEXError;
        addHTTPHeader(inHeaderData: UnsafeRawPointer!, inHeaderDataLength: UInt32):OBEXError;
        addImageDescriptorHeader(inHeaderData: UnsafeRawPointer!, inHeaderDataLength: UInt32):OBEXError;
        addImageHandleHeader(type: String!):OBEXError;
        addLengthHeader(length: UInt32):OBEXError;
        addNameHeader(inNameString: String!):OBEXError;
        addObjectClassHeader(inHeaderData: UnsafeRawPointer!, inHeaderDataLength: UInt32):OBEXError;
        addTargetHeader(inHeaderData: UnsafeRawPointer!, inHeaderDataLength: UInt32):OBEXError;
        addTime4ByteHeader(time4Byte: UInt32):OBEXError;
        addTimeISOHeader(inHeaderData: UnsafeRawPointer!, inHeaderDataLength: UInt32):OBEXError;
        addTypeHeader(type: String!):OBEXError;
        addUserDefinedHeader(inHeaderData: UnsafeRawPointer!, inHeaderDataLength: UInt32):OBEXError;
        addWhoHeader(inHeaderData: UnsafeRawPointer!, inHeaderDataLength: UInt32):OBEXError;
        getHeaderBytes():NSMutableData!;
    }
    export class NSData extends NSObject {
        /**
         * A pointer to the data object's contents.
         */
        bytes: UnsafeRawPointer;
        /**
         * The number of bytes contained by the data object.
         */
        length: number;
        /**
         * A string that contains a hexadecimal representation of the data object’s contents in a property list format.
         */
        description: string;
        endIndex: number;
        regions: Data[];
        startIndex: number;
        /**
         * Initializes a data object filled with a given number of bytes copied from a given buffer.
         */
        constructor (bytes: UnsafeRawPointer | null, length: number);
        /**
         * Initializes a data object filled with a given number of bytes of data from a given buffer.
         */
        constructor (bytes: UnsafeMutableRawPointer, length: number);
        /**
         * Initializes a data object filled with a given number of bytes of data from a given buffer, with a custom deallocator block.
         */
        constructor (bytes: UnsafeMutableRawPointer, length: number, deallocator: ()=>any | null);
        /**
         * Initializes a newly allocated data object by adding the given number of bytes from the given buffer.
         */
        constructor (bytes: UnsafeMutableRawPointer, length: number, b: boolean);
        /**
         * Initializes a data object with the contents of another data object.
         */
        constructor (data: Data);
        /**
         * Initializes a data object with the content of the file at a given path.
         */
        constructor (path: string);
        /**
         * Initializes a data object with the content of the file at a given path.
         */
        constructor (path: string, readOptionsMask: NSData.ReadingOptions);
        /**
         * Initializes a data object with the data from the location specified by a given URL.
         */
        constructor (url: URL);
        /**
         * Initializes a data object with the data from the location specified by a given URL.
         */
        constructor (url: URL, readOptionsMask: NSData.ReadingOptions);
        /**
         * Initializes a data object with the contents of the mapped file specified by a given path.
         */
        constructor (path: string);
        /**
         * Initializes a data object with the given Base64 encoded data.
         */
        constructor (base64Data: Data, options: NSData.Base64DecodingOptions);
        /**
         * Initializes a data object initialized with the given Base64 encoded string.
         */
        constructor (base64String: string);
        /**
         * Creates a data object from the mapped file at a given path.
         */
        static dataWithContentsOfMappedFile(path: string):any;
        /**
         * Writes the data object's bytes to the file specified by a given path.
         */
        write(path: string, useAuxiliaryFile: boolean):boolean;
        /**
         * Creates a Base64, UTF-8 encoded data object from the string using the given options.
         */
        base64EncodedData(options: NSData.Base64EncodingOptions):Data;
        /**
         * Creates a Base64 encoded string from the string using the given options.
         */
        base64EncodedString(options: NSData.Base64EncodingOptions):string;
        /**
         * Initializes a Base64 encoded string from the string.
         */
        base64Encoding():string;
        /**
         * Enumerates each range of bytes in the data object using a block.
         */
        enumerateBytes(block: ()=>any):void;
        /**
         * Copies a data object’s contents into a given buffer.
         */
        getBytes(buffer: UnsafeMutableRawPointer):void;
        /**
         * Returns a new data object containing the data object's bytes that fall within the limits specified by a given range.
         */
        subdata(range: NSRange):Data;
        /**
         * Finds and returns the range of the first occurrence of the given data, within the given range, subject to given options.
         */
        range(dataToFind: Data, mask: NSData.SearchOptions, searchRange: NSRange):NSRange;
        /**
         * Returns a Boolean value indicating whether this data object is the same as another.
         */
        isEqual(other: Data):boolean;
        /**
         * Returns a new data object by compressing the data object’s bytes.
         */
        compressed(algorithm: NSData.CompressionAlgorithm):Self;
        /**
         * Returns a new data object by decompressing data object’s bytes.
         */
        decompressed(algorithm: NSData.CompressionAlgorithm):Self;
    }
    /**
     * the same name namespace of class NSData
     */
    export namespace NSData {
        export class ReadingOptions {
            /**
             * A hint indicating the file should be mapped into virtual memory, if possible and safe.
             */
            static mappedIfSafe: NSData.ReadingOptions;
            /**
             * A hint indicating the file should not be stored in the file-system caches.
             */
            static uncached: NSData.ReadingOptions;
            /**
             * Hint to map the file in if possible.
             */
            static alwaysMapped: NSData.ReadingOptions;
            /**
             * Deprecated name for 
             * .
             */
            static dataReadingMapped: NSData.ReadingOptions;
            /**
             * Deprecated name for 
             * .
             */
            static mappedRead: NSData.ReadingOptions;
            /**
             * Deprecated name for 
             * .
             */
            static uncachedRead: NSData.ReadingOptions;
            constructor (rawValue: number);
        }
        export class WritingOptions {
            /**
             * An option to write data to an auxiliary file first and then replace the original file with the auxiliary file when the write completes.
             */
            static atomic: NSData.WritingOptions;
            /**
             * An option that attempts to write data to a file and fails with an error if the destination file already exists.
             */
            static withoutOverwriting: NSData.WritingOptions;
            /**
             * An option to not encrypt the file when writing it out.
             */
            static noFileProtection: NSData.WritingOptions;
            /**
             * An option to make the file accessible only while the device is unlocked. 
             */
            static completeFileProtection: NSData.WritingOptions;
            /**
             * An option to allow the file to be accessible while the device is unlocked or the file is already open.
             */
            static completeFileProtectionUnlessOpen: NSData.WritingOptions;
            /**
             * An option to allow the file to be accessible after a user first unlocks the device.
             */
            static completeFileProtectionUntilFirstUserAuthentication: NSData.WritingOptions;
            /**
             * An option the system uses when determining the file protection options that the system assigns to the data.
             */
            static fileProtectionMask: NSData.WritingOptions;
            /**
             * An option that attempts to write data to an auxiliary file first and then exchange the files.
             */
            static atomicWrite: NSData.WritingOptions;
            constructor (rawValue: number);
        }
        export class Base64EncodingOptions {
            /**
             * Set the maximum line length to 64 characters, after which a line ending is inserted.
             */
            static lineLength64Characters: NSData.Base64EncodingOptions;
            /**
             * Set the maximum line length to 76 characters, after which a line ending is inserted.
             */
            static lineLength76Characters: NSData.Base64EncodingOptions;
            /**
             * When a maximum line length is set, specify that the line ending to insert should include a carriage return.
             */
            static endLineWithCarriageReturn: NSData.Base64EncodingOptions;
            /**
             * When a maximum line length is set, specify that the line ending to insert should include a line feed.
             */
            static endLineWithLineFeed: NSData.Base64EncodingOptions;
            constructor (rawValue: number);
        }
        export class Base64DecodingOptions {
            /**
             * Modify the decoding algorithm so that it ignores unknown non-Base-64 bytes, including line ending characters.
             */
            static ignoreUnknownCharacters: NSData.Base64DecodingOptions;
            constructor (rawValue: number);
        }
        export class SearchOptions {
            /**
             * Search from the end of the data object.
             */
            static backwards: NSData.SearchOptions;
            /**
             * Search is limited to start (or end, if searching backwards) of the data object.
             */
            static anchored: NSData.SearchOptions;
            constructor (rawValue: number);
        }
        export enum CompressionAlgorithm {
            lz4 = 1,
            lzfse = 0,
            lzma = 2,
            zlib = 3,
        }
    }
    export class NSSet extends NSObject {
        /**
         * The number of members in the set.
         */
        count: number;
        /**
         * An array containing the set’s members, or an empty array if the set has no members.
         */
        allObjects: any[];
        /**
         * A string that represents the contents of the set, formatted as a property list.
         */
        description: string;
        customMirror: Mirror;
        underestimatedCount: number;
        /**
         * Creates and returns a set that contains a single given object.
         */
        constructor (object: any);
        /**
         * Creates and returns a set containing a specified number of objects from a given C array of objects.
         */
        constructor (objects: UnsafePointer<AnyObject>, cnt: number);
        /**
         * Initializes a newly allocated set with the objects that are contained in a given array.
         */
        constructor (array: any[]);
        /**
         * Initializes a newly allocated set and adds to it objects from another given set.
         */
        constructor (set: Set<AnyHashable>);
        /**
         * Initializes a newly allocated set and adds to it members of another given set.
         */
        constructor (set: Set<AnyHashable>, flag: boolean);
        /**
         * Initializes a newly allocated set.
         */
        constructor ();
        constructor (...elements: any);
        constructor (coder: NSCoder);
        constructor (indexPath: IndexPath);
        constructor (indexPaths: IndexPath[]);
        constructor (...elements: any);
        /**
         * Returns a new set formed by adding a given object to the receiving set.
         */
        adding(anObject: any):Set<AnyHashable>;
        /**
         * Returns a new set formed by adding the objects in a given set to the receiving set.
         */
        addingObjects(other: Set<AnyHashable>):Set<AnyHashable>;
        /**
         * Returns one of the objects in the set, or 
         *  if the set contains no objects.
         */
        anyObject():any;
        /**
         * Returns a Boolean value that indicates whether a given object is present in the set.
         */
        contains(anObject: any):boolean;
        /**
         * Evaluates a given predicate against each object in the receiving set and returns a new set containing the objects for which the predicate returns true.
         */
        filtered(predicate: NSPredicate):Set<AnyHashable>;
        /**
         * Determines whether a given object is present in the set, and returns that object if it is.
         */
        member(object: any):any;
        /**
         * Returns an enumerator object that lets you access each object in the set.
         */
        objectEnumerator():NSEnumerator;
        /**
         * Executes a given block using each object in the set.
         */
        enumerateObjects(block: ()=>any):void;
        /**
         * Returns a set of objects that pass a test in a given block.
         */
        objects(predicate: ()=>any):Set<AnyHashable>;
        /**
         * Returns a Boolean value that indicates whether every object in the receiving set is also present in another given set.
         */
        isSubset(otherSet: Set<AnyHashable>):boolean;
        /**
         * Returns a Boolean value that indicates whether at least one object in the receiving set is also present in another given set.
         */
        intersects(otherSet: Set<AnyHashable>):boolean;
        /**
         * Compares the receiving set to another set.
         */
        isEqual(otherSet: Set<AnyHashable>):boolean;
        /**
         * Return a set containing the results of invoking 
         *  on each of the receiving set's members.
         */
        value(key: string):any;
        /**
         * Invokes 
         *  on each of the set’s members.
         */
        setValue(value: any, key: string):void;
        /**
         * Returns an array of the set’s content sorted as specified by a given array of sort descriptors.
         */
        sortedArray(sortDescriptors: NSSortDescriptor[]):any[];
        /**
         * Raises an exception.
         */
        addObserver(observer: NSObject, keyPath: string, options: NSKeyValueObservingOptions, context: UnsafeMutableRawPointer | null):void;
        /**
         * Raises an exception.
         */
        removeObserver(observer: NSObject, keyPath: string, context: UnsafeMutableRawPointer | null):void;
        enumerateIndexPaths(opts: NSEnumerationOptions, block: ()=>any):void;
        /**
         * Return an 
         *  over the elements of this 
         * .
         */
        makeIterator():NSFastEnumerationIterator;
    }
    export class NSMutableSet extends NSSet {
        /**
         * Returns an initialized mutable set with a given initial capacity.
         */
        constructor (numItems: number);
        /**
         * Initializes a newly allocated set.
         */
        constructor ();
        constructor (coder: NSCoder);
        /**
         * Adds a given object to the set, if it is not already a member.
         */
        add(object: any):void;
        /**
         * Evaluates a given predicate against the set’s content and removes from the set those objects for which the predicate returns false.
         */
        filter(predicate: NSPredicate):void;
        /**
         * Removes a given object from the set.
         */
        remove(object: any):void;
        /**
         * Empties the set of all of its members.
         */
        removeAllObjects():void;
        /**
         * Adds to the set each object contained in a given array that is not already a member.
         */
        addObjects(array: any[]):void;
        /**
         * Adds each object in another given set to the receiving set, if not present.
         */
        union(otherSet: Set<AnyHashable>):void;
        /**
         * Removes each object in another given set from the receiving set, if present.
         */
        minus(otherSet: Set<AnyHashable>):void;
        /**
         * Removes from the receiving set each object that isn’t a member of another given set.
         */
        intersect(otherSet: Set<AnyHashable>):void;
        /**
         * Empties the receiving set, then adds each object contained in another given set.
         */
        setSet(otherSet: Set<AnyHashable>):void;
    }
    export class IndexPath {
        /**
         * A mirror that reflects the index path.
         */
        customMirror: Mirror;
        /**
         * A textual description of the index path suitable for debugging.
         */
        debugDescription: string;
        /**
         * A textual description of the index path.
         */
        description: string;
        /**
         * One past the index of the last node in the index path.
         */
        endIndex: IndexPath.Index;
        /**
         * The computed hash value for the index path.
         */
        hashValue: number;
        /**
         * A Boolean value indicating whether the collection is empty.
         */
        isEmpty: boolean;
        /**
         * The value of the item element of the index path.
         */
        item: number;
        /**
         * A sequence containing the same elements as this sequence, but on which some operations, such as 
         *  and 
         * , are implemented lazily.
         */
        lazy: LazySequence<IndexPath>;
        /**
         * The value of the row element of the index path.
         */
        row: number;
        /**
         * The value of the section element of the index path.
         */
        section: number;
        /**
         * The index of the first node in the index path.
         */
        startIndex: IndexPath.Index;
        count: number;
        /**
         * The first element of the collection.
         */
        first: number;
        /**
         * The indices that are valid for subscripting the collection, in ascending order.
         */
        indices: DefaultIndices<IndexPath>;
        /**
         * The last element of the collection.
         */
        last: number;
        publisher: any;
        /**
         * A value less than or equal to the number of elements in the collection.
         */
        underestimatedCount: number;
        /**
         * Creates an empty index path.
         */
        constructor ();
        /**
         * Creates an index path with a single element.
         */
        constructor (index: IndexPath.Element);
        /**
         * Creates an index path from an array literal.
         */
        constructor (...indexes: IndexPath.Element);
        /**
         * Creates an index path from an array of elements.
         */
        constructor (indexes: Array<IndexPath.Element>);
        constructor (decoder: Decoder);
        /**
         * Creates an index path that references an item in a particular section.
         */
        constructor (item: number, section: number);
        /**
         * Creates an index path that references a row in a particular section.
         */
        constructor (row: number, section: number);
        /**
         * Removes and returns the first element of the collection.
         */
        removeFirst():number;
        /**
         * Removes and returns the last element of the collection.
         */
        removeLast():number;
        /**
         * Removes and returns the first element of the collection.
         */
        popFirst():number;
        /**
         * Removes and returns the last element of the collection.
         */
        popLast():number;
        /**
         * Returns the maximum element in the sequence.
         */
        max():number;
        /**
         * Returns the minimum element in the sequence.
         */
        min():number;
        /**
         * Appends the nodes of another index path to this one.
         */
        append(other: IndexPath):void;
        /**
         * Returns a new index path containing the elements of this one plus the given element.
         */
        appending(other: IndexPath.Element):IndexPath;
        /**
         * Compares this index path to another in depth-first traversal order.
         */
        compare(other: IndexPath):ComparisonResult;
        /**
         * Returns a subsequence containing all but the given number of initial elements.
         */
        dropFirst(k: number):IndexPath;
        /**
         * Return a new index path containing all but the last element.
         */
        dropLast():IndexPath;
        /**
         * Returns a sequence of pairs (
         * , 
         * ), where 
         *  represents a consecutive integer starting at zero and 
         *  represents an element of the sequence.
         */
        enumerated():EnumeratedSequence<IndexPath>;
        /**
         * Returns the index that follows the given index.
         */
        index(i: IndexPath.Index):IndexPath.Index;
        /**
         * Returns an iterator over the nodes of the index path.
         */
        makeIterator():IndexingIterator<IndexPath>;
        /**
         * Returns a subsequence, up to the specified maximum length, containing the initial elements of the collection.
         */
        prefix(maxLength: number):IndexPath;
        /**
         * Returns a view presenting the elements of the collection in reverse order.
         */
        reversed():ReversedCollection<IndexPath>;
        /**
         * Returns the elements of the sequence, sorted.
         */
        sorted():number[];
        /**
         * Returns a subsequence, up to the given maximum length, containing the final elements of the collection.
         */
        suffix(maxLength: number):IndexPath;
        /**
         * Returns a Boolean value indicating whether every element of a sequence satisfies a given predicate.
         */
        allSatisfy(predicate: Bool):boolean;
        /**
         * Returns a Boolean value indicating whether the sequence contains the given element.
         */
        contains(element: number):boolean;
        /**
         * Returns the distance between two indices.
         */
        distance(start: any, end: any):number;
        /**
         * Returns a subsequence by skipping elements while 
         *  returns 
         *  and returning the remaining elements.
         */
        drop(predicate: Bool):IndexPath;
        encode(encoder: Encoder):void;
        /**
         * Returns an array containing, in order, the elements of the sequence that satisfy the given predicate.
         */
        filter(isIncluded: Bool):number[];
        /**
         * Returns the first index where the specified value appears in the collection.
         */
        firstIndex(element: number):any;
        /**
         * Calls the given closure on each element in the sequence in the same order as a 
         * -
         *  loop.
         */
        forEach(body: Void):void;
        /**
         * Offsets the given index by the specified distance.
         */
        formIndex(i: any, distance: number):void;
        hash(hasher: Hasher):void;
        /**
         * Returns the last index where the specified value appears in the collection.
         */
        lastIndex(element: number):any;
        /**
         * Reorders the elements of the collection such that all the elements that match the given predicate are after all the elements that don’t match.
         */
        partition(belongsInSecondPartition: Bool):any;
        /**
         * Returns a random element of the collection.
         */
        randomElement():number;
        /**
         * Reverses the elements of the collection in place.
         */
        reverse():void;
        /**
         * Shuffles the collection in place.
         */
        shuffle():void;
        /**
         * Returns the elements of the sequence, shuffled.
         */
        shuffled():number[];
        /**
         * Sorts the collection in place.
         */
        sort():void;
        /**
         * Returns the longest possible subsequences of the collection, in order, that don’t contain elements satisfying the given predicate.
         */
        split(maxSplits: number, omittingEmptySubsequences: boolean, isSeparator: Bool):IndexPath[];
        /**
         * Exchanges the values at the specified indices of the collection.
         */
        swapAt(i: any, j: any):void;
        trimPrefix(predicate: Bool):void;
        trimmingPrefix(predicate: Bool):IndexPath;
    }
    export class NSIndexPath extends NSObject {
        /**
         * An index number identifying a section in a table view or collection view.
         */
        section: number;
        /**
         * An index number identifying a row in a section of a table view.
         */
        row: number;
        /**
         * An index number identifying an item in a section of a collection view.
         */
        item: number;
        /**
         * The number of nodes in the index path.
         */
        length: number;
        /**
         * Initializes an index path with a single node.
         */
        constructor (index: number);
        /**
         * Initializes an index path with the given nodes and length.
         */
        constructor (indexes: UnsafePointer<Int> | null, length: number);
        /**
         * Initializes an index path with the indexes of a specific row and section in a table view.
         */
        constructor (row: number, section: number);
        /**
         * Initializes an index path with the indexes of a specific item and section in a collection view. 
         */
        constructor (item: number, section: number);
        /**
         * Returns an index path containing the nodes in the receiving index path plus another given index.
         */
        adding(index: number):IndexPath;
        /**
         * Returns an index path with the nodes in the receiving index path, excluding the last one.
         */
        removingLastIndex():IndexPath;
        /**
         * Indicates the depth-first traversal order of the receiving index path and another index path.
         */
        compare(otherObject: IndexPath):ComparisonResult;
        /**
         * Provides the value at a particular node in the index path.
         */
        index(position: number):number;
        /**
         * Copies the indexes stored in the index path from the positions specified by the position range into the specified indexes.
         */
        getIndexes(indexes: UnsafeMutablePointer<Int>, positionRange: NSRange):void;
    }
    export class IndexSet {
        /**
         * The beginning index in the set.
         */
        startIndex: IndexSet.Index;
        /**
         * The ending index in the set.
         */
        endIndex: IndexSet.Index;
        /**
         * A sequence containing the same elements as this sequence, but on which some operations, such as 
         *  and 
         * , are implemented lazily.
         */
        lazy: LazySequence<IndexSet>;
        /**
         * A textual description of the index set.
         */
        description: string;
        /**
         * A textual description of the index set suitable for debugging.
         */
        debugDescription: string;
        /**
         * A mirror that reflects the index set.
         */
        customMirror: Mirror;
        /**
         * The computed hash value for the index set.
         */
        hashValue: number;
        /**
         * The indices that are valid for subscripting the collection, in ascending order.
         */
        indices: DefaultIndices<IndexSet>;
        isEmpty: boolean;
        /**
         * The last integer in 
         * , or nil if 
         *  is empty.
         */
        last: IndexSet.Element | null;
        publisher: any;
        /**
         * A value less than or equal to the number of elements in the collection.
         */
        underestimatedCount: number;
        /**
         * Initializes an empty index set.
         */
        constructor ();
        /**
         * Creates a set containing the elements of the given array literal.
         */
        constructor (...arrayLiteral: number);
        /**
         * Initializes an index set with a single integer.
         */
        constructor (integer: IndexSet.Element);
        /**
         * Initializes an index set with a range of integers.
         */
        constructor (range: Range<IndexSet.Element>);
        constructor (decoder: Decoder);
        /**
         * Returns the count of integers in 
         *  that intersect 
         * .
         */
        count(range: Range<IndexSet.Element>):number;
        /**
         * Intersects the 
         *  with 
         * .
         */
        formIntersection(other: IndexSet):void;
        /**
         * Exclusive or the 
         *  with 
         * .
         */
        formSymmetricDifference(other: IndexSet):void;
        /**
         * Union the 
         *  with 
         * .
         */
        formUnion(other: IndexSet):void;
        /**
         * Intersects the 
         *  with 
         * .
         */
        intersection(other: IndexSet):IndexSet;
        /**
         * Exclusive or the 
         *  with 
         * .
         */
        symmetricDifference(other: IndexSet):IndexSet;
        /**
         * Union the 
         *  with 
         * .
         */
        union(other: IndexSet):IndexSet;
        /**
         * Insert an integer into the 
         * .
         */
        insert(integer: IndexSet.Element):any;
        /**
         * Insert an integer into the 
         * .
         */
        update(integer: IndexSet.Element):IndexSet.Element | null;
        /**
         * Remove an integer from the 
         * .
         */
        remove(integer: IndexSet.Element):IndexSet.Element | null;
        /**
         * Remove all values from the 
         * .
         */
        removeAll():void;
        /**
         * Removes the elements of the given set from this set.
         */
        subtract(other: IndexSet):void;
        /**
         * Returns a new set containing the elements of this set that do not occur in the given set.
         */
        subtracting(other: IndexSet):IndexSet;
        /**
         * Returns 
         *  if 
         *  contains 
         * .
         */
        contains(integer: IndexSet.Element):boolean;
        /**
         * Returns 
         *  if 
         *  intersects any of the integers in 
         * .
         */
        intersects(range: Range<IndexSet.Element>):boolean;
        /**
         * Returns a Boolean value that indicates whether the set has no members in common with the given set.
         */
        isDisjoint(other: IndexSet):boolean;
        /**
         * Returns a Boolean value that indicates whether this set is a strict subset of the given set.
         */
        isStrictSubset(other: IndexSet):boolean;
        /**
         * Returns a Boolean value that indicates whether this set is a strict superset of the given set.
         */
        isStrictSuperset(other: IndexSet):boolean;
        /**
         * Returns a Boolean value that indicates whether the set is a subset of another set.
         */
        isSubset(other: IndexSet):boolean;
        /**
         * Returns a Boolean value that indicates whether the set is a superset of the given set.
         */
        isSuperset(other: IndexSet):boolean;
        /**
         * Returns the index that follows the given index in the set.
         */
        index(i: IndexSet.Index):IndexSet.Index;
        /**
         * Modifies the given index to refer to the item after the one it currently refers to.
         */
        formIndex(i: IndexSet.Index):void;
        /**
         * Return a 
         *  which can be used to subscript the index set.
         */
        indexRange(range: Range<IndexSet.Element>):Range<IndexSet.Index>;
        /**
         * Returns the first element of the sequence that satisfies the given predicate.
         */
        first(predicate: Bool):number;
        /**
         * Returns the maximum element in the sequence.
         */
        max():number;
        /**
         * Returns the minimum element in the sequence.
         */
        min():number;
        /**
         * Returns an integer contained in 
         *  which is less than or equal to 
         * , or 
         *  if a result could not be found.
         */
        integerLessThanOrEqualTo(integer: IndexSet.Element):IndexSet.Element | null;
        /**
         * Returns an integer contained in 
         *  which is greater than 
         * , or 
         *  if a result could not be found.
         */
        integerGreaterThan(integer: IndexSet.Element):IndexSet.Element | null;
        /**
         * Returns an integer contained in 
         *  which is greater than or equal to 
         * , or 
         *  if a result could not be found.
         */
        integerGreaterThanOrEqualTo(integer: IndexSet.Element):IndexSet.Element | null;
        /**
         * Returns an integer contained in 
         *  which is less than 
         * , or 
         *  if a result could not be found.
         */
        integerLessThan(integer: IndexSet.Element):IndexSet.Element | null;
        /**
         * Returns an array containing, in order, the elements of the sequence that satisfy the given predicate.
         */
        filter(isIncluded: Bool):number[];
        /**
         * Returns an IndexSet filtered according to the result of 
         * .
         */
        filteredIndexSet(range: Range<IndexSet.Element>, includeInteger: Bool):IndexSet;
        /**
         * Returns a subsequence, up to the specified maximum length, containing the initial elements of the collection.
         */
        prefix(maxLength: number):Slice<IndexSet>;
        /**
         * Returns a subsequence, up to the given maximum length, containing the final elements of the collection.
         */
        suffix(maxLength: number):Slice<IndexSet>;
        /**
         * Returns a subsequence containing all but the specified number of final elements.
         */
        dropLast(k: number):Slice<IndexSet>;
        /**
         * Returns a subsequence containing all but the given number of initial elements.
         */
        dropFirst(k: number):Slice<IndexSet>;
        /**
         * Returns a subsequence by skipping elements while 
         *  returns 
         *  and returning the remaining elements.
         */
        drop(predicate: Bool):Slice<IndexSet>;
        /**
         * Calls the given closure on each element in the sequence in the same order as a 
         * -
         *  loop.
         */
        forEach(body: Void):void;
        /**
         * Returns a sequence of pairs (
         * , 
         * ), where 
         *  represents a consecutive integer starting at zero and 
         *  represents an element of the sequence.
         */
        enumerated():EnumeratedSequence<IndexSet>;
        /**
         * Returns an iterator over the elements of this sequence.
         */
        makeIterator():IndexingIterator<IndexSet>;
        /**
         * Returns the elements of the sequence, sorted.
         */
        sorted():number[];
        /**
         * Returns a view presenting the elements of the collection in reverse order.
         */
        reversed():ReversedCollection<IndexSet>;
        /**
         * For a positive delta, shifts the indexes in [index, INT_MAX] to the right, thereby inserting an “empty space” [index, delta], for a negative delta, shifts the indexes in [index, INT_MAX] to the left, thereby deleting the indexes in the range [index - delta, delta].
         */
        shift(integer: IndexSet.Element, delta: number):void;
        /**
         * Returns the longest possible subsequences of the collection, in order, that don’t contain elements satisfying the given predicate.
         */
        split(maxSplits: number, omittingEmptySubsequences: boolean, isSeparator: Bool):Slice<IndexSet>[];
        /**
         * Returns a 
         * -based view of 
         * .
         */
        rangeView(range: Range<IndexSet.Element>):IndexSet.RangeView;
        /**
         * Returns a Boolean value indicating whether every element of a sequence satisfies a given predicate.
         */
        allSatisfy(predicate: Bool):boolean;
        /**
         * Returns the distance between two indices.
         */
        distance(start: Index, end: Index):number;
        encode(encoder: Encoder):void;
        /**
         * Returns the first index where the specified value appears in the collection.
         */
        firstIndex(element: number):Index | null;
        hash(hasher: Hasher):void;
        /**
         * Returns the last index where the specified value appears in the collection.
         */
        lastIndex(element: number):Index | null;
        /**
         * Returns a random element of the collection.
         */
        randomElement():number;
        /**
         * Returns the elements of the sequence, shuffled.
         */
        shuffled():number[];
        trimmingPrefix(predicate: Bool):Slice<IndexSet>;
    }
    /**
     * the same name namespace of class IndexSet
     */
    export namespace IndexSet {
        export class RangeView {
            /**
             * The number of elements in the collection.
             */
            count: number;
            /**
             * The ending index in the range.
             */
            endIndex: IndexSet.RangeView.Index;
            /**
             * The first element of the collection.
             */
            first: Range<IndexSet.Element> | null;
            /**
             * A Boolean value indicating whether the collection is empty.
             */
            isEmpty: boolean;
            /**
             * The last element of the collection.
             */
            last: Range<IndexSet.Element> | null;
            /**
             * A sequence containing the same elements as this sequence, but on which some operations, such as 
             *  and 
             * , are implemented lazily.
             */
            lazy: LazySequence<IndexSet.RangeView>;
            /**
             * The beginning index in the range.
             */
            startIndex: IndexSet.RangeView.Index;
            /**
             * The indices that are valid for subscripting the collection, in ascending order.
             */
            indices: DefaultIndices<IndexSet.RangeView>;
            publisher: any;
            /**
             * A value less than or equal to the number of elements in the collection.
             */
            underestimatedCount: number;
            /**
             * Returns the index in the range after the specified one.
             */
            index(i: IndexSet.RangeView.Index):IndexSet.RangeView.Index;
            /**
             * Returns a subsequence, up to the specified maximum length, containing the initial elements of the collection.
             */
            prefix(maxLength: number):Slice<IndexSet.RangeView>;
            /**
             * Returns a subsequence, up to the given maximum length, containing the final elements of the collection.
             */
            suffix(maxLength: number):Slice<IndexSet.RangeView>;
            /**
             * Returns a subsequence containing all but the given number of initial elements.
             */
            dropFirst(k: number):Slice<IndexSet.RangeView>;
            /**
             * Returns a subsequence containing all but the specified number of final elements.
             */
            dropLast(k: number):Slice<IndexSet.RangeView>;
            /**
             * Returns a sequence of pairs (
             * , 
             * ), where 
             *  represents a consecutive integer starting at zero and 
             *  represents an element of the sequence.
             */
            enumerated():EnumeratedSequence<IndexSet.RangeView>;
            /**
             * Returns an iterator over the indexes of this range view.
             */
            makeIterator():IndexingIterator<IndexSet.RangeView>;
            /**
             * Returns a view presenting the elements of the collection in reverse order.
             */
            reversed():ReversedCollection<IndexSet.RangeView>;
            /**
             * Returns the longest possible subsequences of the collection, in order, around elements equal to the given element.
             */
            split(separator: Range<IndexSet.Element>, maxSplits: number, omittingEmptySubsequences: boolean):Slice<IndexSet.RangeView>[];
            /**
             * Returns a Boolean value indicating whether the sequence contains the given element.
             */
            contains(element: Range<IndexSet.Element>):boolean;
            /**
             * Returns a Boolean value indicating whether every element of a sequence satisfies a given predicate.
             */
            allSatisfy(predicate: Bool):boolean;
            /**
             * Returns the distance between two indices.
             */
            distance(start: number, end: number):number;
            /**
             * Returns a subsequence by skipping elements while 
             *  returns 
             *  and returning the remaining elements.
             */
            drop(predicate: Bool):Slice<IndexSet.RangeView>;
            /**
             * Returns an array containing, in order, the elements of the sequence that satisfy the given predicate.
             */
            filter(isIncluded: Bool):Range<IndexSet.Element>[];
            /**
             * Returns the first index where the specified value appears in the collection.
             */
            firstIndex(element: Range<IndexSet.Element>):number;
            /**
             * Calls the given closure on each element in the sequence in the same order as a 
             * -
             *  loop.
             */
            forEach(body: Void):void;
            /**
             * Offsets the given index by the specified distance.
             */
            formIndex(i: Int, distance: number):void;
            /**
             * Returns the elements of this sequence of sequences, concatenated.
             */
            joined():FlattenSequence<IndexSet.RangeView>;
            /**
             * Returns the last index where the specified value appears in the collection.
             */
            lastIndex(element: Range<IndexSet.Element>):number;
            /**
             * Returns the maximum element in the sequence, using the given predicate as the comparison between elements.
             */
            max(areInIncreasingOrder: Bool):Range<IndexSet.Element> | null;
            /**
             * Returns the minimum element in the sequence, using the given predicate as the comparison between elements.
             */
            min(areInIncreasingOrder: Bool):Range<IndexSet.Element> | null;
            /**
             * Returns a random element of the collection.
             */
            randomElement():Range<IndexSet.Element> | null;
            /**
             * Returns the elements of the sequence, shuffled.
             */
            shuffled():Range<IndexSet.Element>[];
            /**
             * Returns the elements of the sequence, sorted using the given predicate as the comparison between elements.
             */
            sorted(areInIncreasingOrder: Bool):Range<IndexSet.Element>[];
            trimmingPrefix(predicate: Bool):Slice<IndexSet.RangeView>;
        }
        export class Index {
            /**
             * A textual description of the index.
             */
            description: string;
        }
    }
    export class NSIndexSet extends NSObject {
        /**
         * The number of indexes in the index set.
         */
        count: number;
        /**
         * The first index in the index set.
         */
        firstIndex: number;
        /**
         * The last index in the index set.
         */
        lastIndex: number;
        underestimatedCount: number;
        /**
         * Initializes an allocated 
         *  object with an index.
         */
        constructor (value: number);
        /**
         * Initializes an allocated 
         *  object with an index range.
         */
        constructor (range: NSRange);
        /**
         * Initializes an allocated 
         *  object with an index set.
         */
        constructor (indexSet: IndexSet);
        /**
         * Indicates whether the index set contains a specific index.
         */
        contains(value: number):boolean;
        /**
         * Indicates whether the index set contains any of the indexes in a range.
         */
        intersects(range: NSRange):boolean;
        /**
         * Returns the number of indexes in the index set that are members of a given range.
         */
        countOfIndexes(range: NSRange):number;
        /**
         * Returns the index of the first object that passes the predicate Block test.
         */
        index(predicate: ()=>any):number;
        /**
         * Returns an 
         *  containing the receiving index set’s objects that pass the Block test.
         */
        indexes(predicate: ()=>any):IndexSet;
        /**
         * Enumerates over the ranges in the range of objects using the block 
         */
        enumerateRanges(range: NSRange, opts: NSEnumerationOptions, block: ()=>any):void;
        /**
         * Indicates whether the indexes in the receiving index set are the same indexes contained in another index set.
         */
        isEqual(indexSet: IndexSet):boolean;
        /**
         * Returns either the closest index in the index set that is less than a specific index or the not-found indicator.
         */
        indexLessThanIndex(value: number):number;
        /**
         * Returns either the closest index in the index set that is less than or equal to a specific index or the not-found indicator.
         */
        indexLessThanOrEqual(value: number):number;
        /**
         * Returns either the closest index in the index set that is greater than or equal to a specific index or the not-found indicator.
         */
        indexGreaterThanOrEqual(value: number):number;
        /**
         * Returns either the closest index in the index set that is greater than a specific index or the not-found indicator.
         */
        indexGreaterThanIndex(value: number):number;
        /**
         * The index set fills an index buffer with the indexes contained both in the index set and in an index range, returning the number of indexes copied.
         */
        getIndexes(indexBuffer: UnsafeMutablePointer<Int>, bufferSize: number, range: NSRangePointer | null):number;
        /**
         * Executes a given Block using each object in the index set.
         */
        enumerate(block: ()=>any):void;
        /**
         * Returns an 
         *  over the elements of this 
         * .
         */
        makeIterator():NSIndexSetIterator;
    }
    export class NSIndexSetIterator {
        next():number;
    }
    export class NSMutableIndexSet extends NSIndexSet {
        /**
         * Adds an index  to the receiver.
         */
        add(value: number):void;
        /**
         * Removes an index from the receiver.
         */
        remove(value: number):void;
        /**
         * Removes the receiver’s indexes.
         */
        removeAllIndexes():void;
        /**
         * Shifts a group of indexes to the left or the right within the receiver.
         */
        shiftIndexesStarting(index: number, delta: number):void;
    }
    export class NSMutableData extends NSData {
        /**
         * A pointer to the data contained by the mutable data object.
         */
        mutableBytes: UnsafeMutableRawPointer;
        /**
         * The number of bytes contained in the mutable data object.
         */
        length: number;
        /**
         * Returns an initialized mutable data object capable of holding the specified number of bytes.
         */
        constructor (capacity: number);
        /**
         * Initializes and returns a mutable data object containing a given number of zeroed bytes.
         */
        constructor (length: number);
        /**
         * Appends to the receiver a given number of bytes from a given buffer.
         */
        append(bytes: UnsafeRawPointer, length: number):void;
        /**
         * Increases the length of the receiver by a given number of bytes.
         */
        increaseLength(extraLength: number):void;
        /**
         * Replaces with a given set of bytes a given range within the contents of the receiver. 
         */
        replaceBytes(range: NSRange, bytes: UnsafeRawPointer):void;
        /**
         * Replaces with zeroes the contents of the receiver in a given range.
         */
        resetBytes(range: NSRange):void;
        /**
         * Replaces the entire contents of the receiver with the contents of another data object.
         */
        setData(data: Data):void;
        /**
         * Compresses the data object’s bytes using an algorithm that you specify.
         */
        compress(algorithm: NSData.CompressionAlgorithm):void;
        /**
         * Decompresses the data object’s bytes.
         */
        decompress(algorithm: NSData.CompressionAlgorithm):void;
    }
    export class NSCountedSet extends NSMutableSet {
        /**
         * Returns a counted set object initialized with the contents of a given array.
         */
        constructor (array: any[]);
        /**
         * Returns a counted set object initialized with the contents of a given set.
         */
        constructor (set: Set<AnyHashable>);
        /**
         * Returns a counted set object initialized with enough memory to hold a given number of objects.
         */
        constructor (numItems: number);
        /**
         * Adds a given object to the set.
         */
        add(object: any):void;
        /**
         * Removes a given object from the set.
         */
        remove(object: any):void;
        /**
         * Returns the count associated with a given object in the set.
         */
        count(object: any):number;
        /**
         * Returns an enumerator object that lets you access each object in the set once, independent of its count.
         */
        objectEnumerator():NSEnumerator;
    }
    export class NSOrderedSet extends NSObject {
        /**
         * The number of members in the set.
         */
        count: number;
        /**
         * The first object in the ordered set.
         */
        firstObject: any;
        /**
         * The last object in the ordered set.
         */
        lastObject: any;
        /**
         * An ordered set in the reverse order.
         */
        reversed: NSOrderedSet;
        /**
         * A string that represents the contents of the ordered set, formatted as a property list.
         */
        description: string;
        /**
         * A representation of the ordered set as an array.
         */
        array: any[];
        /**
         * A representation of the set containing the contents of the ordered set.
         */
        set: Set<AnyHashable>;
        underestimatedCount: number;
        /**
         * Initializes a newly allocated set with the objects that are contained in a given array.
         */
        constructor (array: any[]);
        /**
         * Initializes a newly allocated set with the objects that are contained in a given array, optionally copying the items.
         */
        constructor (set: any[], flag: boolean);
        /**
         * Initializes a newly allocated set with the objects that are contained in the specified range of an array, optionally copying the items.
         */
        constructor (set: any[], range: NSRange, flag: boolean);
        /**
         * Initializes a new ordered set with the object.
         */
        constructor (object: any);
        /**
         * Initializes a newly allocated set with a specified number of objects from a given C array of objects.
         */
        constructor (objects: UnsafePointer<AnyObject> | null, cnt: number);
        /**
         * Initializes a new ordered set with the contents of a set.
         */
        constructor (set: NSOrderedSet);
        /**
         * Initializes a new ordered set with the contents of a set, optionally copying the items.
         */
        constructor (set: NSOrderedSet, flag: boolean);
        /**
         * Initializes a new ordered set with the contents of an ordered set, optionally copying the items.
         */
        constructor (set: NSOrderedSet, range: NSRange, flag: boolean);
        /**
         * Initializes a new ordered set with the contents of a set.
         */
        constructor (set: Set<AnyHashable>);
        /**
         * Initializes a new ordered set with the contents of a set, optionally copying the objects in the set.
         */
        constructor (set: Set<AnyHashable>, flag: boolean);
        /**
         * Initializes a newly allocated ordered set.
         */
        constructor ();
        constructor (...elements: any);
        constructor (coder: NSCoder);
        constructor (...elements: any);
        /**
         * Returns a Boolean value that indicates whether a given object is present in the ordered set.
         */
        contains(object: any):boolean;
        /**
         * Executes a given block using the objects in the ordered set at the specified indexes.
         */
        enumerateObjects(s: IndexSet, opts: NSEnumerationOptions, block: ()=>any):void;
        /**
         * Returns the object at the specified index of the set.
         */
        object(idx: number):any;
        /**
         * Returns the objects in the ordered set at the specified indexes.
         */
        objects(indexes: IndexSet):any[];
        /**
         * Returns the index of the specified object.
         */
        index(object: any):number;
        /**
         * Returns the index, from a given set of indexes, of the object in the ordered set that passes a test in a given block for a given set of enumeration options.
         */
        indexes(s: IndexSet, opts: NSEnumerationOptions, predicate: ()=>any):IndexSet;
        /**
         * Returns an enumerator object that lets you access each object in the ordered set.
         */
        objectEnumerator():NSEnumerator;
        /**
         * Returns an enumerator object that lets you access each object in the ordered set.
         */
        reverseObjectEnumerator():NSEnumerator;
        /**
         * Invokes 
         *  on each of the receiver's members using the specified value and key
         */
        setValue(value: any, key: string):void;
        /**
         * Returns an ordered set containing the results of invoking 
         *  using key on each of the ordered set’s objects.
         */
        value(key: string):any;
        /**
         * Raises an exception.
         */
        addObserver(observer: NSObject, keyPath: string, options: NSKeyValueObservingOptions, context: UnsafeMutableRawPointer | null):void;
        /**
         * Raises an exception.
         */
        removeObserver(observer: NSObject, keyPath: string):void;
        /**
         * Compares the receiving ordered set to another ordered set.
         */
        isEqual(other: NSOrderedSet):boolean;
        /**
         * Returns a Boolean value that indicates whether at least one object in the receiving ordered set is also present in another given ordered set.
         */
        intersects(other: NSOrderedSet):boolean;
        /**
         * Returns a Boolean value that indicates whether at least one object in the receiving ordered set is also present in another given set.
         */
        intersectsSet(set: Set<AnyHashable>):boolean;
        /**
         * Returns a Boolean value that indicates whether every object in the receiving ordered set is also present in another given ordered set.
         */
        isSubset(other: NSOrderedSet):boolean;
        /**
         * Returns an array of the ordered set’s elements sorted as specified by a given array of sort descriptors.
         */
        sortedArray(sortDescriptors: NSSortDescriptor[]):any[];
        /**
         * Evaluates a given predicate against each object in the receiving ordered set and returns a new ordered set containing the objects for which the predicate returns true.
         */
        filtered(p: NSPredicate):NSOrderedSet;
        /**
         * Return an 
         *  over the elements of this 
         * .
         */
        makeIterator():NSFastEnumerationIterator;
    }
    export class NSMutableOrderedSet extends NSOrderedSet {
        /**
         * Returns an initialized mutable ordered set with a given initial capacity.
         */
        constructor (numItems: number);
        /**
         * Initializes a newly allocated mutable ordered set.
         */
        constructor ();
        constructor (coder: NSCoder);
        /**
         * Appends a given object to the end of the mutable ordered set, if it is not already a member.
         */
        add(object: any):void;
        /**
         * Appends to the end of the mutable ordered set each object contained in a given array that is not already a member.
         */
        addObjects(array: any[]):void;
        /**
         * Inserts the given object at the specified index of the mutable ordered set, if it is not already a member.
         */
        insert(object: any, idx: number):void;
        /**
         * Removes a given object from the mutable ordered set.
         */
        remove(object: any):void;
        /**
         * Removes a the object at the specified index from the mutable ordered set.
         */
        removeObject(idx: number):void;
        /**
         * Removes the objects at the specified indexes from the mutable ordered set.
         */
        removeObjects(indexes: IndexSet):void;
        /**
         * Removes all the objects from the mutable ordered set.
         */
        removeAllObjects():void;
        /**
         * Replaces the object at the specified index with the new object.
         */
        replaceObject(idx: number, object: any):void;
        /**
         * Replaces the objects at the specified indexes with the new objects.
         */
        replaceObjects(indexes: IndexSet, objects: any[]):void;
        /**
         * Appends or replaces the object at the specified index.
         */
        setObject(obj: any, idx: number):void;
        /**
         * Moves the objects at the specified indexes to the new location.
         */
        moveObjects(indexes: IndexSet, idx: number):void;
        /**
         * Exchanges the object at the specified index with the object at the other index.
         */
        exchangeObject(idx1: number, idx2: number):void;
        /**
         * Evaluates a given predicate against the mutable ordered set’s content and leaves only objects that match.
         */
        filter(p: NSPredicate):void;
        /**
         * Sorts the receiving ordered set using a given array of sort descriptors.
         */
        sort(sortDescriptors: NSSortDescriptor[]):void;
        /**
         * Sorts the specified range of the mutable ordered set using the specified options and the comparison method specified by a given comparator block.
         */
        sortRange(range: NSRange, opts: NSSortOptions, cmptr: ()=>any):void;
        /**
         * Removes from the receiving ordered set each object that isn’t a member of another ordered set.
         */
        intersect(other: NSOrderedSet):void;
        /**
         * Removes from the receiving ordered set each object that isn’t a member of another set.
         */
        intersectSet(other: Set<AnyHashable>):void;
        /**
         * Removes each object in another given ordered set from the receiving mutable ordered set, if present.
         */
        minus(other: NSOrderedSet):void;
        /**
         * Removes each object in another given set from the receiving mutable ordered set, if present.
         */
        minusSet(other: Set<AnyHashable>):void;
        /**
         * Adds each object in another given ordered set to the receiving mutable ordered set, if not present.
         */
        union(other: NSOrderedSet):void;
        /**
         * Adds each object in another given set to the receiving mutable ordered set, if not present.
         */
        unionSet(other: Set<AnyHashable>):void;
    }
    export interface NSDiscardableContent {
        /**
         * Returns a Boolean value indicating whether the discardable contents are still available and have been successfully accessed.
         */
        beginContentAccess():boolean;
        /**
         * Called if the discardable contents are no longer being accessed.
         */
        endContentAccess():void;
        /**
         * Called to discard the contents of the receiver if the value of the accessed counter is 0.
         */
        discardContentIfPossible():void;
        /**
         * Returns a Boolean value indicating whether the content has been discarded.
         */
        isContentDiscarded():boolean;
    }
    export interface NSCacheDelegate extends NSObjectProtocol {
        /**
         * Called when an object is about to be evicted or removed from the cache.
         */
        cache(cache: any, obj: any):void;
    }
    export class NSPurgeableData extends NSMutableData {}
    export class NSPointerArray extends NSObject {
        /**
         * The number of elements in the receiver.
         */
        count: number;
        /**
         * All the objects in the receiver.
         */
        allObjects: any[];
        /**
         * The functions in use by the receiver.
         */
        pointerFunctions: NSPointerFunctions;
        /**
         * Initializes the receiver to use the given options.
         */
        constructor (options: NSPointerFunctions.Options);
        /**
         * Initializes the receiver to use the given functions.
         */
        constructor (functions: NSPointerFunctions);
        /**
         * Returns a new pointer array that maintains strong references to its elements.
         */
        static strongObjects():NSPointerArray;
        /**
         * Returns a new pointer array that maintains weak references to its elements.
         */
        static weakObjects():NSPointerArray;
        /**
         * Returns the pointer at a given index.
         */
        pointer(index: number):UnsafeMutableRawPointer | null;
        /**
         * Adds a given pointer to the receiver.
         */
        addPointer(pointer: UnsafeMutableRawPointer | null):void;
        /**
         * Removes the pointer at a given index.
         */
        removePointer(index: number):void;
        /**
         * Inserts a pointer at a given index.
         */
        insertPointer(item: UnsafeMutableRawPointer | null, index: number):void;
        /**
         * Replaces the pointer at a given index.
         */
        replacePointer(index: number, item: UnsafeMutableRawPointer | null):void;
        /**
         * Removes 
         *  values from the receiver.
         */
        compact():void;
    }
    export class NSPointerFunctions extends NSObject {
        /**
         * The hash function.
         */
        hashFunction: ()=>any | null;
        /**
         * The function used to compare pointers.
         */
        isEqualFunction: ()=>any | null;
        /**
         * The function used to determine the size of pointers.
         */
        sizeFunction: ()=>any | null;
        /**
         * The function used to describe elements.
         */
        descriptionFunction: ()=>any | null;
        /**
         * The function used to acquire memory.
         */
        acquireFunction: ()=>any | null;
        /**
         * The function used to relinquish memory.
         */
        relinquishFunction: ()=>any | null;
        /**
         * Specifies whether, in a garbage collected environment, pointers should be assigned using a strong write barrier.
         */
        usesStrongWriteBarrier: boolean;
        /**
         * Specifies whether, in a garbage collected environment, pointers should use weak read and write barriers.
         */
        usesWeakReadAndWriteBarriers: boolean;
        /**
         * Returns an 
         *  object initialized with the given options.
         */
        constructor (options: NSPointerFunctions.Options);
    }
    /**
     * the same name namespace of class NSPointerFunctions
     */
    export namespace NSPointerFunctions {
        export class Options {
            /**
             * Use Mach memory.
             */
            static machVirtualMemory: NSPointerFunctions.Options;
            /**
             * Use 
             *  on removal, 
             *  on copy in.
             */
            static mallocMemory: NSPointerFunctions.Options;
            /**
             * Take no action when pointers are deleted.
             */
            static opaqueMemory: NSPointerFunctions.Options;
            /**
             * Use strong write-barriers to backing store; use garbage-collected memory on copy-in.
             */
            static strongMemory: NSPointerFunctions.Options;
            /**
             * Uses weak read and write barriers appropriate for ARC or GC. Using NSPointerFunctionsWeakMemory object references will turn to 
             *  on last release.
             */
            static weakMemory: NSPointerFunctions.Options;
            /**
             * Use a string hash and 
             * ; C-string '
             * ' style description.
             */
            static cStringPersonality: NSPointerFunctions.Options;
            /**
             * Use unshifted value as hash and equality.
             */
            static integerPersonality: NSPointerFunctions.Options;
            /**
             * Use 
             *  and 
             *  methods for hashing and equality comparisons, use the 
             *  method for a description.
             */
            static objectPersonality: NSPointerFunctions.Options;
            /**
             * Use shifted pointer for the hash value and direct comparison to determine equality; use the 
             *  method for a description.
             */
            static objectPointerPersonality: NSPointerFunctions.Options;
            /**
             * Use shifted pointer for the hash value and direct comparison to determine equality.
             */
            static opaquePersonality: NSPointerFunctions.Options;
            /**
             * Use a memory hash and 
             *  (using a size function that you must set—see 
             * ).
             */
            static structPersonality: NSPointerFunctions.Options;
            /**
             * Use the memory acquire function to allocate and copy items on input (see 
             * ).
             */
            static copyIn: NSPointerFunctions.Options;
            constructor (rawValue: number);
        }
    }
    export class NSMapEnumerator {
        constructor ();
        constructor (_pi: number, _si: number, _bs: UnsafeMutableRawPointer | null);
    }
    export class NSMapTableKeyCallBacks {
        /**
         * Points to the function which produces an autoreleased NSString * describing the given element. If 
         * , then the map table produces a generic string description.
         */
        describe: ()=>any | null;
        /**
         * Points to the function which must produce hash code for key elements of the map table. If 
         * , the pointer value is used as the hash code. Second parameter is the element for which hash code should be produced.
         */
        hash: ()=>any | null;
        /**
         * Points to the function which compares second and third parameters. If 
         * , then == is used for comparison.
         */
        isEqual: ()=>any | null;
        /**
         * No key put in map table can be this value. An exception is raised if attempt is made to use this value as a key
         */
        notAKeyMarker: UnsafeRawPointer | null;
        /**
         * Points to the function which decrements a reference count for the given element, and if the reference count becomes zero, frees the given element. If 
         * , then nothing is done for reference counting or releasing.
         */
        release: ()=>any | null;
        /**
         * Points to the function which increments a reference count for the given element. If 
         * , then nothing is done for reference counting.
         */
        retain: ()=>any | null;
        constructor ();
        constructor (hash: ()=>any | null, isEqual: ()=>any | null, retain: ()=>any | null, release: ()=>any | null, describe: ()=>any | null, notAKeyMarker: UnsafeRawPointer | null);
    }
    export class NSMapTableValueCallBacks {
        /**
         * Points to the function that produces an autoreleased NSString * describing the given element. If 
         * , then the map table produces a generic string description.
         */
        describe: ()=>any | null;
        /**
         * Points to the function that decrements a reference count for the given element, and if the reference count becomes zero, frees the given element. If 
         * , then nothing is done for reference counting or releasing.
         */
        release: ()=>any | null;
        /**
         * Points to the function that increments a reference count for the given element. If 
         * , then nothing is done for reference counting.
         */
        retain: ()=>any | null;
        constructor ();
        constructor (retain: ()=>any | null, release: ()=>any | null, describe: ()=>any | null);
    }
    export class NSHashEnumerator {
        constructor ();
        constructor (_pi: number, _si: number, _bs: UnsafeMutableRawPointer | null);
    }
    export class NSHashTableCallBacks {
        /**
         * Points to the function that produces an autoreleased NSString * describing the given element. If 
         * , then the hash table produces a generic string description.
         */
        describe: ()=>any | null;
        /**
         * Points to the function that must produce hash code for elements of the hash table. If 
         * , the pointer value is used as the hash code. Second parameter is the element for which hash code should be produced.
         */
        hash: ()=>any | null;
        /**
         * Points to the function that compares second and third parameters. If 
         * , then == is used for comparison.
         */
        isEqual: ()=>any | null;
        /**
         * Points to the function that decrements a reference count for the given element, and if the reference count becomes 0, frees the given element. If 
         * , then nothing is done for reference counting or releasing.
         */
        release: ()=>any | null;
        /**
         * Points to the function that increments a reference count for the given element. If 
         * , then nothing is done for reference counting.
         */
        retain: ()=>any | null;
        constructor ();
        constructor (hash: ()=>any | null, isEqual: ()=>any | null, retain: ()=>any | null, release: ()=>any | null, describe: ()=>any | null);
    }
    export class NSEnumerator extends NSObject {
        /**
         * The array of unenumerated objects.
         */
        allObjects: any[];
        /**
         * Returns the next object from the collection being enumerated.
         */
        nextObject():any;
        /**
         * Return an 
         *  over the 
         * .
         */
        makeIterator():NSFastEnumerationIterator;
    }
    export interface NSFastEnumeration {
        /**
         * Returns by reference a C array of objects over which the sender should iterate, and as the return value the number of objects in the array.
         */
        countByEnumerating(state: UnsafeMutablePointer<NSFastEnumerationState>, buffer: AutoreleasingUnsafeMutablePointer<AnyObject?>, len: number):number;
    }
    export class NSFastEnumerationState {
        /**
         * A C array that you can use to hold returned values.
         */
        extra: any;
        /**
         * A C array of objects.
         */
        itemsPtr: AutoreleasingUnsafeMutablePointer<AnyObject?> | null;
        /**
         * Arbitrary state information used to detect whether the collection has been mutated.
         */
        mutationsPtr: UnsafeMutablePointer<UInt> | null;
        /**
         * Arbitrary state information used by the iterator. Typically this is set to 
         *  at the beginning of the iteration.
         */
        state: number;
        constructor ();
        constructor (state: number, itemsPtr: AutoreleasingUnsafeMutablePointer<AnyObject?> | null, mutationsPtr: UnsafeMutablePointer<UInt> | null, extra: any);
    }
    export class NSFastEnumerationIterator {
        constructor (enumerable: NSFastEnumeration);
        next():any;
    }
    export class NSEnumerationOptions {
        /**
         * Specifies that the Block enumeration should be concurrent.
         */
        static concurrent: NSEnumerationOptions;
        /**
         * Specifies that the enumeration should be performed in reverse.
         */
        static reverse: NSEnumerationOptions;
        constructor (rawValue: number);
    }
    export class NSSortOptions {
        /**
         * Specifies that the Block sort operation should be concurrent.
         */
        static concurrent: NSSortOptions;
        /**
         * Specifies that the sorted results should return compared items having equal value in the order they occurred originally.
         */
        static stable: NSSortOptions;
        constructor (rawValue: number);
    }
    export class NSNull extends NSObject {}
    export class Date {
        /**
         * Returns a date instance that represents the current date and time, at the moment of access.
         */
        static now: Date;
        /**
         * A date value representing a date in the distant future.
         */
        static distantFuture: Date;
        /**
         * A date value representing a date in the distant past.
         */
        static distantPast: Date;
        /**
         * The number of seconds from 1 January 1970 to the reference date, 1 January 2001.
         */
        static timeIntervalBetween1970AndReferenceDate: number;
        static defaultResolverSpecification: EmptyResolverSpecification<Date>;
        /**
         * The time interval between the date value and the current date and time.
         */
        timeIntervalSinceNow: TimeInterval;
        /**
         * The interval between the date value and 00:00:00 UTC on 1 January 2001.
         */
        timeIntervalSinceReferenceDate: TimeInterval;
        /**
         * The interval between the date value and 00:00:00 UTC on 1 January 1970.
         */
        timeIntervalSince1970: TimeInterval;
        /**
         * A textual description of the date value.
         */
        description: string;
        /**
         * A textual description of the date suitable for debugging.
         */
        debugDescription: string;
        /**
         * A mirror that reflects the date.
         */
        customMirror: Mirror;
        /**
         * The computed hash value of the date.
         */
        hashValue: number;
        /**
         * A custom playground Quick Look for the date.
         */
        customPlaygroundQuickLook: PlaygroundQuickLook;
        /**
         * Creates a date value initialized to the current date and time.
         */
        constructor ();
        /**
         * Creates a date value initialized relative to the current date and time by a given number of seconds.
         */
        constructor (timeIntervalSinceNow: TimeInterval);
        /**
         * Creates a date value initialized relative to another given date by a given number of seconds.
         */
        constructor (timeInterval: TimeInterval, date: Date);
        /**
         * Creates a date value initialized relative to 00:00:00 UTC on 1 January 2001 by a given number of seconds.
         */
        constructor (ti: TimeInterval);
        /**
         * Creates a date value initialized relative to 00:00:00 UTC on 1 January 1970 by a given number of seconds.
         */
        constructor (timeIntervalSince1970: TimeInterval);
        /**
         * Creates a new instance by decoding from the given decoder.
         */
        constructor (decoder: Decoder);
        /**
         * Compares another date to this one.
         */
        compare(other: Date):ComparisonResult;
        /**
         * Returns the distance from this date to another date, specified as a time interval.
         */
        distance(other: Date):TimeInterval;
        /**
         * Returns the interval between this date and another given date.
         */
        timeIntervalSince(date: Date):TimeInterval;
        /**
         * Adds a time interval to this date.
         */
        addTimeInterval(timeInterval: TimeInterval):void;
        /**
         * Creates a new date value by adding a time interval to this date.
         */
        addingTimeInterval(timeInterval: TimeInterval):Date;
        /**
         * Returns a date offset the specified time interval from this date.
         */
        advanced(n: TimeInterval):Date;
        /**
         * Generates a locale-aware string representation of a date using the default date format style.
         */
        formatted():string;
        /**
         * Generates a locale-aware string representation of a date using the ISO 8601 date format.
         */
        ISO8601Format(style: Date.ISO8601FormatStyle):string;
        /**
         * Encodes this date into the given encoder.
         */
        encode(encoder: Encoder):void;
        /**
         * Hashes the essential components of this value by feeding them into the given hasher.
         */
        hash(hasher: Hasher):void;
    }
    /**
     * the same name namespace of class Date
     */
    export namespace Date {
        export class FormatStyle {
            /**
             * A factory variable used as a base for custom date format styles.
             */
            static dateTime: Date.FormatStyle;
            /**
             * The calendar to use when formatting the date.
             */
            calendar: Calendar;
            /**
             * The capitalization context to use when formatting the date.
             */
            capitalizationContext: FormatStyleCapitalizationContext;
            /**
             * An attributed format style created from the date format style.
             */
            attributed: Date.AttributedStyle;
            /**
             * The strategy used to parse a string into a date.
             */
            parseStrategy: Date.FormatStyle;
            /**
             * The hash value of the style format.
             */
            hashValue: number;
            regex: Regex<Date>;
            /**
             * Creates an instance using the provided date, time, locale, calendar, time zone, and capitalization context.
             */
            constructor (date: Date.FormatStyle.DateStyle | null, time: Date.FormatStyle.TimeStyle | null, locale: Locale, calendar: Calendar, timeZone: TimeZone, capitalizationContext: FormatStyleCapitalizationContext);
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            /**
             * Creates a fixed-format parse strategy.
             */
            static fixed(format: Date.FormatString, timeZone: TimeZone, locale: Locale | null):Date.ParseStrategy;
            static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
            /**
             * Modifies the date format style to use the specified day format style.
             */
            day(format: Date.FormatStyle.Symbol.Day):Date.FormatStyle;
            /**
             * Modifies the date format style to use the specified day of the year format style.
             */
            dayOfYear(format: Date.FormatStyle.Symbol.DayOfYear):Date.FormatStyle;
            /**
             * Modifies the date format style to use the specified era format style.
             */
            era(format: Date.FormatStyle.Symbol.Era):Date.FormatStyle;
            /**
             * Modifies the date format style to use the specified month format style.
             */
            month(format: Date.FormatStyle.Symbol.Month):Date.FormatStyle;
            /**
             * Modifies the date format style to use the specified quarter format style.
             */
            quarter(format: Date.FormatStyle.Symbol.Quarter):Date.FormatStyle;
            /**
             * Modifies the date format style to use the specified week format style.
             */
            week(format: Date.FormatStyle.Symbol.Week):Date.FormatStyle;
            /**
             * Modifies the date format style to use the specified weekday format style.
             */
            weekday(format: Date.FormatStyle.Symbol.Weekday):Date.FormatStyle;
            /**
             * Modifies the date format style to use the specified year format style.
             */
            year(format: Date.FormatStyle.Symbol.Year):Date.FormatStyle;
            /**
             * Modifies the date format style to use the specified hour format style.
             */
            hour(format: Date.FormatStyle.Symbol.Hour):Date.FormatStyle;
            /**
             * Modifies the date format style to use the specified minute format style.
             */
            minute(format: Date.FormatStyle.Symbol.Minute):Date.FormatStyle;
            /**
             * Modifies the date format style to use the specified second format style.
             */
            second(format: Date.FormatStyle.Symbol.Second):Date.FormatStyle;
            /**
             * Modifies the date format style to use the specified second fraction format style.
             */
            secondFraction(format: Date.FormatStyle.Symbol.SecondFraction):Date.FormatStyle;
            /**
             * Modifies the date format style to use the specified time zone format style.
             */
            timeZone(format: Date.FormatStyle.Symbol.TimeZone):Date.FormatStyle;
            /**
             * Modifies the date format style to use the specified locale.
             */
            locale(locale: Locale):Date.FormatStyle;
            /**
             * Creates a locale-aware string representation from a date value.
             */
            format(value: Date):string;
            /**
             * Parses a string into a date.
             */
            parse(value: string):Date;
            /**
             * Encodes the value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of the value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
            consuming(input: string, index: String.Index, bounds: Range<String.Index>):any;
        }
        /**
         * the same name namespace of class FormatStyle
         */
        export namespace FormatStyle {
            export class DateStyle {
                /**
                 * A date style with some components abbreviated for space-constrained applications.
                 */
                static abbreviated: Date.FormatStyle.DateStyle;
                /**
                 * A date style with all components represented.
                 */
                static complete: Date.FormatStyle.DateStyle;
                /**
                 * A lengthened date style with the full month, day of month, and year components represented.
                 */
                static long: Date.FormatStyle.DateStyle;
                /**
                 * A date style with the month, day of month, and year components represented as numeric values.
                 */
                static numeric: Date.FormatStyle.DateStyle;
                /**
                 * A date style with no date-related components represented.
                 */
                static omitted: Date.FormatStyle.DateStyle;
                /**
                 * The hash value of the style format.
                 */
                hashValue: number;
                /**
                 * Creates a new instance by decoding from the given decoder.
                 */
                constructor (decoder: Decoder);
                /**
                 * Encodes the value into the given encoder.
                 */
                encode(encoder: Encoder):void;
                /**
                 * Hashes the essential components of the value by feeding them into the given hasher.
                 */
                hash(hasher: Hasher):void;
            }
            export class TimeStyle {
                /**
                 * A time style with all components represented.
                 */
                static complete: Date.FormatStyle.TimeStyle;
                /**
                 * A time style with no time-related components represented.
                 */
                static omitted: Date.FormatStyle.TimeStyle;
                /**
                 * A shortened time style with only the hour, minute, and day period components represented.
                 */
                static shortened: Date.FormatStyle.TimeStyle;
                /**
                 * A time style with all components except the time zone represented.
                 */
                static standard: Date.FormatStyle.TimeStyle;
                /**
                 * The hash value of the style format.
                 */
                hashValue: number;
                /**
                 * Creates a new instance by decoding from the given decoder.
                 */
                constructor (decoder: Decoder);
                /**
                 * Encodes the value into the given encoder.
                 */
                encode(encoder: Encoder):void;
                /**
                 * Hashes the essential components of the value by feeding them into the given hasher.
                 */
                hash(hasher: Hasher):void;
            }
            export class Symbol {
                /**
                 * The hash value of the style format.
                 */
                hashValue: number;
                /**
                 * Hashes the essential components of the value by feeding them into the given hasher.
                 */
                hash(hasher: Hasher):void;
            }
            /**
             * the same name namespace of class Symbol
             */
            export namespace Symbol {
                export class CyclicYear {
                    /**
                     * Custom cyclic year format style that portrays a shortened cyclic year.
                     */
                    static abbreviated: Date.FormatStyle.Symbol.CyclicYear;
                    /**
                     * Custom cyclic year format style that portrays the shortest representation of a cyclic year.
                     */
                    static narrow: Date.FormatStyle.Symbol.CyclicYear;
                    /**
                     * Custom cyclic year format style that portrays a complete representation of a cyclic year.
                     */
                    static wide: Date.FormatStyle.Symbol.CyclicYear;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class Day {
                    /**
                     * Custom format style portraying the minimum number of digits that represents the numeric day of month.
                     */
                    static defaultDigits: Date.FormatStyle.Symbol.Day;
                    /**
                     * Custom format style portraying the ordinal of the day in the month.
                     */
                    static ordinalOfDayInMonth: Date.FormatStyle.Symbol.Day;
                    /**
                     * Custom format style portraying the two-digit numeric day of month, zero-padded if necessary.
                     */
                    static twoDigits: Date.FormatStyle.Symbol.Day;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Creates a custom day format style representing the modified Julian day.
                     */
                    static julianModified(minimumLength: number):Date.FormatStyle.Symbol.Day;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class DayOfYear {
                    /**
                     * Custom format style portraying the minimum number of digits that represents the numeric day of the year.
                     */
                    static defaultDigits: Date.FormatStyle.Symbol.DayOfYear;
                    /**
                     * Custom format style portraying the three-digit numeric day of the year, zero-padded if necessary.
                     */
                    static threeDigits: Date.FormatStyle.Symbol.DayOfYear;
                    /**
                     * Custom format style portraying the two-digit numeric day of the year, zero-padded if necessary.
                     */
                    static twoDigits: Date.FormatStyle.Symbol.DayOfYear;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class DayPeriod {
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Static factory method that creates a custom day period format style using a conversational style.
                     */
                    static conversational(width: Date.FormatStyle.Symbol.DayPeriod.Width):Date.FormatStyle.Symbol.DayPeriod;
                    /**
                     * Static factory method that creates a custom day period format style using a standard style.
                     */
                    static standard(width: Date.FormatStyle.Symbol.DayPeriod.Width):Date.FormatStyle.Symbol.DayPeriod;
                    /**
                     * Static factory method that creates a custom day period format style using a style that represents midday and midnight. 
                     */
                    static with12s(width: Date.FormatStyle.Symbol.DayPeriod.Width):Date.FormatStyle.Symbol.DayPeriod;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                /**
                 * the same name namespace of class DayPeriod
                 */
                export namespace DayPeriod {
                    export enum Width {
                        abbreviated,
                        narrow,
                        wide,
                    }
                }
                export class Era {
                    /**
                     * An abbreviated representation of an era. 
                     */
                    static abbreviated: Date.FormatStyle.Symbol.Era;
                    /**
                     * A narrow representation of an era.
                     */
                    static narrow: Date.FormatStyle.Symbol.Era;
                    /**
                     * A full representation of an era.
                     */
                    static wide: Date.FormatStyle.Symbol.Era;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class Hour {
                    /**
                     * Custom format style portraying the minimum number of digits that represents the numeric hour.
                     */
                    static defaultDigitsNoAMPM: Date.FormatStyle.Symbol.Hour;
                    /**
                     * Custom format style portraying the numeric hour using two digits.
                     */
                    static twoDigitsNoAMPM: Date.FormatStyle.Symbol.Hour;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Custom format style portraying the minimum number of digits that represents the hour and locale-dependent conversational day period formats.
                     */
                    static conversationalDefaultDigits(amPM: Date.FormatStyle.Symbol.Hour.AMPMStyle):Date.FormatStyle.Symbol.Hour;
                    /**
                     * Custom format style portraying two digits that represent the hour and locale-dependent conversational day period formats.
                     */
                    static conversationalTwoDigits(amPM: Date.FormatStyle.Symbol.Hour.AMPMStyle):Date.FormatStyle.Symbol.Hour;
                    /**
                     * Custom format style portraying the minimum number of digits that represents the hour and locale-dependent day period formats.
                     */
                    static defaultDigits(amPM: Date.FormatStyle.Symbol.Hour.AMPMStyle):Date.FormatStyle.Symbol.Hour;
                    /**
                     * Custom format style portraying two digits that represent the hour and locale-dependent day period formats.
                     */
                    static twoDigits(amPM: Date.FormatStyle.Symbol.Hour.AMPMStyle):Date.FormatStyle.Symbol.Hour;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                /**
                 * the same name namespace of class Hour
                 */
                export namespace Hour {
                    export class AMPMStyle {
                        /**
                         * A type that specifies the abbreviated day period for when the locale prefers using day period with hour.
                         */
                        static abbreviated: Date.FormatStyle.Symbol.Hour.AMPMStyle;
                        /**
                         * A type that specifies the narrow day period if the locale prefers using day period with hour.
                         */
                        static narrow: Date.FormatStyle.Symbol.Hour.AMPMStyle;
                        /**
                         * A type that hides the day period marker.
                         */
                        static omitted: Date.FormatStyle.Symbol.Hour.AMPMStyle;
                        /**
                         * A type that represents the wide day period if the locale prefers using day period with hour.
                         */
                        static wide: Date.FormatStyle.Symbol.Hour.AMPMStyle;
                        /**
                         * The hash value of the style format.
                         */
                        hashValue: number;
                        /**
                         * Creates a new instance by decoding from the given decoder.
                         */
                        constructor (decoder: Decoder);
                        /**
                         * Hashes the essential components of the value by feeding them into the given hasher.
                         */
                        hash(hasher: Hasher):void;
                        /**
                         * Encodes this value into the given encoder.
                         */
                        encode(encoder: Encoder):void;
                    }
                }
                export class Minute {
                    /**
                     * The custom minute format style showing the minimum number of digits that represents the numeric minute.
                     */
                    static defaultDigits: Date.FormatStyle.Symbol.Minute;
                    /**
                     * The custom format style that shows the two-digit numeric minute, zero-padded if necessary.
                     */
                    static twoDigits: Date.FormatStyle.Symbol.Minute;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class Month {
                    /**
                     * The abbreviated representation of a month.
                     */
                    static abbreviated: Date.FormatStyle.Symbol.Month;
                    /**
                     * Custom month format style showing the minimum number of digits that represents the numeric month.
                     */
                    static defaultDigits: Date.FormatStyle.Symbol.Month;
                    /**
                     * The shortest representation of a month.
                     */
                    static narrow: Date.FormatStyle.Symbol.Month;
                    /**
                     * The custom month format style that uses two digits to represent the numeric month.
                     */
                    static twoDigits: Date.FormatStyle.Symbol.Month;
                    /**
                     * The full representation of a month.
                     */
                    static wide: Date.FormatStyle.Symbol.Month;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class Quarter {
                    /**
                     * The abbreviated representation of a quarter.
                     */
                    static abbreviated: Date.FormatStyle.Symbol.Quarter;
                    /**
                     * The shortest representation of a quarter.
                     */
                    static narrow: Date.FormatStyle.Symbol.Quarter;
                    /**
                     * The one-digit representation of a quarter.
                     */
                    static oneDigit: Date.FormatStyle.Symbol.Quarter;
                    /**
                     * The two-digit representation of a quarter.
                     */
                    static twoDigits: Date.FormatStyle.Symbol.Quarter;
                    /**
                     * The full representation of a quarter.
                     */
                    static wide: Date.FormatStyle.Symbol.Quarter;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class Second {
                    /**
                     * The custom format style that conveys the minimum number of digits that represents the numeric second.
                     */
                    static defaultDigits: Date.FormatStyle.Symbol.Second;
                    /**
                     * The custom format style that conveys a two-digit numeric second, zero-padded if necessary.
                     */
                    static twoDigits: Date.FormatStyle.Symbol.Second;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class SecondFraction {
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Creates a custom format style representing the fractional seconds component of a date.
                     */
                    static fractional(val: number):Date.FormatStyle.Symbol.SecondFraction;
                    /**
                     * Creates a custom format style representing the milliseconds elapsed in a day.
                     */
                    static milliseconds(val: number):Date.FormatStyle.Symbol.SecondFraction;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class StandaloneMonth {
                    /**
                     * The abbreviated representation of a standalone month.
                     */
                    static abbreviated: Date.FormatStyle.Symbol.StandaloneMonth;
                    /**
                     * The custom month format style that shows the minimum number of digits to represent a standalone month.
                     */
                    static defaultDigits: Date.FormatStyle.Symbol.StandaloneMonth;
                    /**
                     * The shortest representation of a standalone month.
                     */
                    static narrow: Date.FormatStyle.Symbol.StandaloneMonth;
                    /**
                     * The two-digit representation of a standalone month.
                     */
                    static twoDigits: Date.FormatStyle.Symbol.StandaloneMonth;
                    /**
                     * The full representation of a standalone month.
                     */
                    static wide: Date.FormatStyle.Symbol.StandaloneMonth;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class StandaloneQuarter {
                    /**
                     * The abbreviated representation of a standalone quarter.
                     */
                    static abbreviated: Date.FormatStyle.Symbol.StandaloneQuarter;
                    /**
                     * The shortest representation of a standalone quarter.
                     */
                    static narrow: Date.FormatStyle.Symbol.StandaloneQuarter;
                    /**
                     * The one-digit representation of a standalone quarter.
                     */
                    static oneDigit: Date.FormatStyle.Symbol.StandaloneQuarter;
                    /**
                     * The two-digit representation of a standalone quarter.
                     */
                    static twoDigits: Date.FormatStyle.Symbol.StandaloneQuarter;
                    /**
                     * The full representation of a standalone quarter.
                     */
                    static wide: Date.FormatStyle.Symbol.StandaloneQuarter;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class StandaloneWeekday {
                    /**
                     * The abbreviated representation of a standalone weekday.
                     */
                    static abbreviated: Date.FormatStyle.Symbol.StandaloneWeekday;
                    /**
                     * The shortest representation of a standalone weekday.
                     */
                    static narrow: Date.FormatStyle.Symbol.StandaloneWeekday;
                    /**
                     * The one-digit representation of a standalone weekday.
                     */
                    static oneDigit: Date.FormatStyle.Symbol.StandaloneWeekday;
                    /**
                     * The short representation of a standalone weekday.
                     */
                    static short: Date.FormatStyle.Symbol.StandaloneWeekday;
                    /**
                     * The full representation of a standalone weekday.
                     */
                    static wide: Date.FormatStyle.Symbol.StandaloneWeekday;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class TimeZone {
                    /**
                     * The exemplar city for a timezone.
                     */
                    static exemplarLocation: Date.FormatStyle.Symbol.TimeZone;
                    /**
                     * The generic location representation of a timezone.
                     */
                    static genericLocation: Date.FormatStyle.Symbol.TimeZone;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Returns the specific, non-location representation of a timezone.
                     */
                    static specificName(width: Date.FormatStyle.Symbol.TimeZone.Width):Date.FormatStyle.Symbol.TimeZone;
                    /**
                     * Returns the generic, non-location representation of a timezone.
                     */
                    static genericName(width: Date.FormatStyle.Symbol.TimeZone.Width):Date.FormatStyle.Symbol.TimeZone;
                    /**
                     * Creates the ISO 8601 representation of the timezone with hours, minutes, and optional seconds.
                     */
                    static iso8601(width: Date.FormatStyle.Symbol.TimeZone.Width):Date.FormatStyle.Symbol.TimeZone;
                    /**
                     * Returns the localized GMT format representation of a timezone.
                     */
                    static localizedGMT(width: Date.FormatStyle.Symbol.TimeZone.Width):Date.FormatStyle.Symbol.TimeZone;
                    /**
                     * Returns the timezone identifier.
                     */
                    static identifier(width: Date.FormatStyle.Symbol.TimeZone.Width):Date.FormatStyle.Symbol.TimeZone;
                    /**
                     * Hashes the essential components of this value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                /**
                 * the same name namespace of class TimeZone
                 */
                export namespace TimeZone {
                    export enum Width {
                        short,
                        long,
                    }
                }
                export class VerbatimHour {
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Creates a custom format style portraying the minimum number of digits that represents the hour.
                     */
                    static defaultDigits(clock: Date.FormatStyle.Symbol.VerbatimHour.Clock, hourCycle: Date.FormatStyle.Symbol.VerbatimHour.HourCycle):Date.FormatStyle.Symbol.VerbatimHour;
                    /**
                     * Creates a custom format style portraying two digits that represent the hour.
                     */
                    static twoDigits(clock: Date.FormatStyle.Symbol.VerbatimHour.Clock, hourCycle: Date.FormatStyle.Symbol.VerbatimHour.HourCycle):Date.FormatStyle.Symbol.VerbatimHour;
                    /**
                     * Hashes the essential components of this value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                /**
                 * the same name namespace of class VerbatimHour
                 */
                export namespace VerbatimHour {
                    export class Clock {
                        /**
                         * A clock that portrays the hour using a 12-hour clock.
                         */
                        static twelveHour: Date.FormatStyle.Symbol.VerbatimHour.Clock;
                        /**
                         * A clock that portrays the hour using a 24-hour clock.
                         */
                        static twentyFourHour: Date.FormatStyle.Symbol.VerbatimHour.Clock;
                        /**
                         * The hash value of the style format.
                         */
                        hashValue: number;
                        /**
                         * Creates a new instance by decoding from the given decoder.
                         */
                        constructor (decoder: Decoder);
                        /**
                         * Encodes the value into the given encoder.
                         */
                        encode(encoder: Encoder):void;
                        /**
                         * Hashes the essential components of the value by feeding them into the given hasher.
                         */
                        hash(hasher: Hasher):void;
                    }
                    export class HourCycle {
                        /**
                         * A hour cycle that indicates a clock that starts at one.
                         */
                        static oneBased: Date.FormatStyle.Symbol.VerbatimHour.HourCycle;
                        /**
                         * A hour cycle that indicates a clock that starts at zero.
                         */
                        static zeroBased: Date.FormatStyle.Symbol.VerbatimHour.HourCycle;
                        /**
                         * The hash value of the style format.
                         */
                        hashValue: number;
                        /**
                         * Creates a new instance by decoding from the given decoder.
                         */
                        constructor (decoder: Decoder);
                        /**
                         * Encodes the value into the given encoder.
                         */
                        encode(encoder: Encoder):void;
                        /**
                         * Hashes the essential components of the value by feeding them into the given hasher.
                         */
                        hash(hasher: Hasher):void;
                    }
                }
                export class Week {
                    /**
                     * Custom week format style showing the minimum number of digits that represents the numeric week.
                     */
                    static defaultDigits: Date.FormatStyle.Symbol.Week;
                    /**
                     * Custom format style portraying the two-digit numeric week, zero-padded if necessary.
                     */
                    static twoDigits: Date.FormatStyle.Symbol.Week;
                    /**
                     * Custom format style portraying the numeric week of the month.
                     */
                    static weekOfMonth: Date.FormatStyle.Symbol.Week;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class Weekday {
                    /**
                     * A shortened weekday representation.
                     */
                    static abbreviated: Date.FormatStyle.Symbol.Weekday;
                    /**
                     * The shortest weekday representation.
                     */
                    static narrow: Date.FormatStyle.Symbol.Weekday;
                    /**
                     * The one-digit representation of a weekday.
                     */
                    static oneDigit: Date.FormatStyle.Symbol.Weekday;
                    /**
                     * The short weekday representation.
                     */
                    static short: Date.FormatStyle.Symbol.Weekday;
                    /**
                     * The two-digit representation of a standalone weekday, zero-padded if necessary.
                     */
                    static twoDigits: Date.FormatStyle.Symbol.Weekday;
                    /**
                     * The complete weekday representation.
                     */
                    static wide: Date.FormatStyle.Symbol.Weekday;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class Year {
                    /**
                     * The custom year format style showing the minimum number of digits that represents the numeric year.
                     */
                    static defaultDigits: Date.FormatStyle.Symbol.Year;
                    /**
                     * The custom format style portraying the two-digit numeric year, zero-padded if necessary.
                     */
                    static twoDigits: Date.FormatStyle.Symbol.Year;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Returns a custom format style that portrays the year of the calendar system of the provided length, zero-padded if necessary.
                     */
                    static padded(length: number):Date.FormatStyle.Symbol.Year;
                    /**
                     * Returns a custom format style that portrays the year of a non-Gregorian calendar system in the corresponding Gregorian year.
                     */
                    static relatedGregorian(minimumLength: number):Date.FormatStyle.Symbol.Year;
                    /**
                     * Returns a custom format style that portrays the year of the calendar system, encompassing all supra-year fields.
                     */
                    static extended(minimumLength: number):Date.FormatStyle.Symbol.Year;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
                export class YearForWeekOfYear {
                    /**
                     * Custom week of the year format style showing the minimum number of digits that represents the year in week-of-year calendars.
                     */
                    static defaultDigits: Date.FormatStyle.Symbol.YearForWeekOfYear;
                    /**
                     * The custom format style that represents the two-digit numeric year in week-of-year calendars, zero-padded or truncated if necessary.
                     */
                    static twoDigits: Date.FormatStyle.Symbol.YearForWeekOfYear;
                    /**
                     * The hash value of the style format.
                     */
                    hashValue: number;
                    /**
                     * Returns a custom format style that represents the three or more digits of the year in week-of-year calendars, zero-padded if necessary.
                     */
                    static padded(length: number):Date.FormatStyle.Symbol.YearForWeekOfYear;
                    /**
                     * Hashes the essential components of the value by feeding them into the given hasher.
                     */
                    hash(hasher: Hasher):void;
                }
            }
        }
        export class AttributedStyle {
            /**
             * The hash value of the style format.
             */
            hashValue: number;
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
            /**
             * Modifies the date attributed style to use the specified locale.
             */
            locale(locale: Locale):Date.AttributedStyle;
            /**
             * Creates a locale-aware attributed string representation from a date value.
             */
            format(value: Date):AttributedString;
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
        export class ParseStrategy {
            calendar: Calendar;
            format: string;
            hashValue: number;
            isLenient: boolean;
            locale: Locale | null;
            regex: Regex<Date>;
            timeZone: TimeZone;
            twoDigitStartDate: Date;
            constructor (format: Date.FormatString, locale: Locale | null, timeZone: TimeZone, calendar: Calendar, isLenient: boolean, twoDigitStartDate: Date);
            constructor (decoder: Decoder);
            static date(style: DateStyle, locale: Locale, timeZone: TimeZone, calendar: Calendar | null):Date.ParseStrategy;
            static dateTime(date: DateStyle, time: TimeStyle, locale: Locale, timeZone: TimeZone, calendar: Calendar | null):Date.ParseStrategy;
            static fixed(format: Date.FormatString, timeZone: TimeZone, locale: Locale | null):Date.ParseStrategy;
            consuming(input: string, index: String.Index, bounds: Range<String.Index>):any;
            encode(encoder: Encoder):void;
            hash(hasher: Hasher):void;
            parse(value: string):Date;
        }
        export class RelativeFormatStyle {
            /**
             * Specifies the style to use when describing a relative date, such as “1 day ago” or “yesterday”.
             */
            presentation: Date.RelativeFormatStyle.Presentation;
            /**
             * The style to use when formatting the quantity or the name of the unit, such as “1 day ago” or “one day ago”.
             */
            unitsStyle: Date.RelativeFormatStyle.UnitsStyle;
            /**
             * The calendar to use when formatting relative dates.
             */
            calendar: Calendar;
            /**
             * The capitalization context to use when formatting the relative dates.
             */
            capitalizationContext: FormatStyleCapitalizationContext;
            /**
             * The locale to use when formatting the relative date.
             */
            locale: Locale;
            /**
             * The hash value of the style format.
             */
            hashValue: number;
            /**
             * Creates a relative date format style with the specified presentation, units, locale, calendar, and capitalization context.
             */
            constructor (presentation: Date.RelativeFormatStyle.Presentation, unitsStyle: Date.RelativeFormatStyle.UnitsStyle, locale: Locale, calendar: Calendar, capitalizationContext: FormatStyleCapitalizationContext);
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            /**
             * Returns a relative date format style based on the provided presentation and unit style.
             */
            static relative(presentation: Presentation, unitsStyle: UnitsStyle):Date.RelativeFormatStyle;
            static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
            /**
             * Creates a locale-aware string representation from a relative date value.
             */
            format(destDate: Date):string;
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
        /**
         * the same name namespace of class RelativeFormatStyle
         */
        export namespace RelativeFormatStyle {
            export class Presentation {
                /**
                 * A style that uses named styles to describe relative dates, such as “yesterday”, “last week”, or “next week”.
                 */
                static named: Date.RelativeFormatStyle.Presentation;
                /**
                 * A style that uses a numeric style to describe relative dates, such as “1 day ago” or “in 3 weeks”.
                 */
                static numeric: Date.RelativeFormatStyle.Presentation;
                /**
                 * The hash value of the style format.
                 */
                hashValue: number;
                /**
                 * Creates a new instance by decoding from the given decoder.
                 */
                constructor (decoder: Decoder);
                /**
                 * Encodes this value into the given encoder.
                 */
                encode(encoder: Encoder):void;
                /**
                 * Hashes the essential components of this value by feeding them into the given hasher.
                 */
                hash(hasher: Hasher):void;
            }
            export class UnitsStyle {
                /**
                 * A style that uses abbreviated units, such as “2 mo. ago”.
                 */
                static abbreviated: Date.RelativeFormatStyle.UnitsStyle;
                /**
                 * A style that uses the shortest units, such as “2 mo. ago”.
                 */
                static narrow: Date.RelativeFormatStyle.UnitsStyle;
                /**
                 * A style that spells out units, such as “two months ago”.
                 */
                static spellOut: Date.RelativeFormatStyle.UnitsStyle;
                /**
                 * A style that uses full representation of units, such as “2 months ago”.
                 */
                static wide: Date.RelativeFormatStyle.UnitsStyle;
                /**
                 * The hash value of the style format.
                 */
                hashValue: number;
                /**
                 * Creates a new instance by decoding from the given decoder.
                 */
                constructor (decoder: Decoder);
                /**
                 * Encodes this value into the given encoder.
                 */
                encode(encoder: Encoder):void;
                /**
                 * Hashes the essential components of this value by feeding them into the given hasher.
                 */
                hash(hasher: Hasher):void;
            }
        }
        export class IntervalFormatStyle {
            /**
             * A convenience factory variable to use as a base for custom date interval format styles.
             */
            static interval: Date.IntervalFormatStyle;
            /**
             * The calendar for formatting the date interval.
             */
            calendar: Calendar;
            /**
             * The hash value of the style format.
             */
            hashValue: number;
            /**
             * Creates an instance using the provided date, time, locale, calendar, time zone, and capitalization context.
             */
            constructor (date: Date.IntervalFormatStyle.DateStyle | null, time: Date.IntervalFormatStyle.TimeStyle | null, locale: Locale, calendar: Calendar, timeZone: TimeZone);
            /**
             * Creates an instance by decoding from the specified decoder.
             */
            constructor (decoder: Decoder);
            static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
            /**
             * Modifies the date interval format style to use the specified time zone format.
             */
            timeZone(format: Date.IntervalFormatStyle.Symbol.TimeZone):Date.IntervalFormatStyle;
            /**
             * Modifies the date interval format style to use the specified locale.
             */
            locale(locale: Locale):Date.IntervalFormatStyle;
            /**
             * Modifies the date interval format style to include the day.
             */
            day():Date.IntervalFormatStyle;
            /**
             * Modifies the date interval format style to use the specified hour format style.
             */
            hour(format: Date.IntervalFormatStyle.Symbol.Hour):Date.IntervalFormatStyle;
            /**
             * Modifies the date interval format style to include the minutes.
             */
            minute():Date.IntervalFormatStyle;
            /**
             * Modifies the date interval format style to include the month.
             */
            month(format: Date.IntervalFormatStyle.Symbol.Month):Date.IntervalFormatStyle;
            /**
             * Modifies the date interval format style to include the seconds.
             */
            second():Date.IntervalFormatStyle;
            /**
             * Modifies the date interval format style to include the specified weekday style.
             */
            weekday(format: Date.IntervalFormatStyle.Symbol.Weekday):Date.IntervalFormatStyle;
            /**
             * Modifies the date interval format style to include the year.
             */
            year():Date.IntervalFormatStyle;
            /**
             * Creates a locale-aware string representation from a relative date value.
             */
            format(v: Range<Date>):string;
            /**
             * Encodes the value into the specified encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of the value by feeding them into the specified hasher.
             */
            hash(hasher: Hasher):void;
        }
        export class ISO8601FormatStyle {
            /**
             * A convenience factory variable that provides a base format for customizing ISO 8601 date format styles.
             */
            static iso8601: Date.ISO8601FormatStyle;
            /**
             * The character used to separate the components of a date.
             */
            dateSeparator: Date.ISO8601FormatStyle.DateSeparator;
            /**
             * The character used to separate the date and time components of an ISO 8601 string representation of a date.
             */
            dateTimeSeparator: Date.ISO8601FormatStyle.DateTimeSeparator;
            /**
             * The time zone used to create and parse date representations.
             */
            timeZone: TimeZone;
            /**
             * The strategy used to parse a string into a date.
             */
            parseStrategy: Date.ISO8601FormatStyle;
            /**
             * The hash value of the style format.
             */
            hashValue: number;
            includingFractionalSeconds: boolean;
            regex: Regex<Date>;
            /**
             * Creates an instance using the provided date separator, date and time components separator, and time zone.
             */
            constructor (dateSeparator: Date.ISO8601FormatStyle.DateSeparator, dateTimeSeparator: Date.ISO8601FormatStyle.DateTimeSeparator, timeZone: TimeZone);
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            constructor (dateSeparator: Date.ISO8601FormatStyle.DateSeparator, dateTimeSeparator: Date.ISO8601FormatStyle.DateTimeSeparator, timeSeparator: Date.ISO8601FormatStyle.TimeSeparator, timeZoneSeparator: Date.ISO8601FormatStyle.TimeZoneSeparator, includingFractionalSeconds: boolean, timeZone: TimeZone);
            /**
             * Creates a fixed format parse strategy.
             */
            static fixed(format: Date.FormatString, timeZone: TimeZone, locale: Locale | null):Date.ParseStrategy;
            static iso8601Date(timeZone: TimeZone, dateSeparator: DateSeparator):Date.ISO8601FormatStyle;
            static iso8601WithTimeZone(includingFractionalSeconds: boolean, dateSeparator: DateSeparator, dateTimeSeparator: DateTimeSeparator, timeSeparator: TimeSeparator, timeZoneSeparator: TimeZoneSeparator):Date.ISO8601FormatStyle;
            static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
            /**
             * Modifies the date format style to use the specified locale.
             */
            locale(locale: Locale):Date.ISO8601FormatStyle;
            /**
             * Modifies the ISO 8601 date format style to include the year in the formatted output.
             */
            year():Date.ISO8601FormatStyle;
            /**
             * Modifies the ISO 8601 date format style to include the month in the formatted output.
             */
            month():Date.ISO8601FormatStyle;
            /**
             * Modifies the ISO 8601 date format style to include the week of the year in the formatted output.
             */
            weekOfYear():Date.ISO8601FormatStyle;
            /**
             * Modifies the ISO 8601 date format style to include the day in the formatted output.
             */
            day():Date.ISO8601FormatStyle;
            /**
             * Modifies the ISO 8601 date format style to include the time in the formatted output.
             */
            time(includingFractionalSeconds: boolean):Date.ISO8601FormatStyle;
            /**
             * Modifies the ISO 8601 date format style to use the specified time separator.
             */
            timeSeparator(separator: Date.ISO8601FormatStyle.TimeSeparator):Date.ISO8601FormatStyle;
            /**
             * Modifies the ISO 8601 date format style to use the specified time zone separator.
             */
            timeZoneSeparator(separator: Date.ISO8601FormatStyle.TimeZoneSeparator):Date.ISO8601FormatStyle;
            /**
             * Parses a string into a date.
             */
            parse(value: string):Date;
            /**
             * Creates a locale-aware ISO 8601 string representation from a date value.
             */
            format(value: Date):string;
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
            consuming(input: string, index: String.Index, bounds: Range<String.Index>):any;
        }
        /**
         * the same name namespace of class ISO8601FormatStyle
         */
        export namespace ISO8601FormatStyle {
            export enum DateSeparator {
                dash,
                omitted,
            }
            export enum DateTimeSeparator {
                space,
                standard,
            }
            export enum TimeSeparator {
                colon,
                omitted,
            }
            export enum TimeZoneSeparator {
                colon,
                omitted,
            }
        }
        export class ComponentsFormatStyle {
            static timeDuration: Date.ComponentsFormatStyle;
            calendar: Calendar;
            fields: Set<Date.ComponentsFormatStyle.Field> | null;
            hashValue: number;
            locale: Locale;
            style: Date.ComponentsFormatStyle.Style;
            constructor (decoder: Decoder);
            constructor (style: Date.ComponentsFormatStyle.Style, locale: Locale, calendar: Calendar, fields: Set<Date.ComponentsFormatStyle.Field> | null);
            static components(style: Style, fields: Set<Field> | null):Date.ComponentsFormatStyle;
            static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
            encode(encoder: Encoder):void;
            format(v: Range<Date>):string;
            hash(hasher: Hasher):void;
        }
        /**
         * the same name namespace of class ComponentsFormatStyle
         */
        export namespace ComponentsFormatStyle {
            export class Field {
                static day: Date.ComponentsFormatStyle.Field;
                static hour: Date.ComponentsFormatStyle.Field;
                static minute: Date.ComponentsFormatStyle.Field;
                static month: Date.ComponentsFormatStyle.Field;
                static second: Date.ComponentsFormatStyle.Field;
                static week: Date.ComponentsFormatStyle.Field;
                static year: Date.ComponentsFormatStyle.Field;
                hashValue: number;
                constructor (decoder: Decoder);
                encode(encoder: Encoder):void;
                hash(hasher: Hasher):void;
            }
            export class Style {
                static abbreviated: Date.ComponentsFormatStyle.Style;
                static condensedAbbreviated: Date.ComponentsFormatStyle.Style;
                static narrow: Date.ComponentsFormatStyle.Style;
                static spellOut: Date.ComponentsFormatStyle.Style;
                static wide: Date.ComponentsFormatStyle.Style;
                hashValue: number;
                constructor (decoder: Decoder);
                encode(encoder: Encoder):void;
                hash(hasher: Hasher):void;
            }
        }
        export class FormatString {
            hashValue: number;
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            constructor (stringInterpolation: Date.FormatString.StringInterpolation);
            constructor (value: string);
            /**
             * Creates an instance initialized to the given value.
             */
            constructor (value: string);
            hash(hasher: Hasher):void;
        }
        /**
         * the same name namespace of class FormatString
         */
        export namespace FormatString {
            export class StringInterpolation {
                constructor (literalCapacity: number, interpolationCount: number);
                appendInterpolation(cyclicYear: Date.FormatStyle.Symbol.CyclicYear):void;
                appendLiteral(literal: string):void;
            }
        }
        export class VerbatimFormatStyle {
            attributed: Date.AttributedStyle;
            calendar: Calendar;
            hashValue: number;
            locale: Locale | null;
            parseStrategy: Date.ParseStrategy;
            regex: Regex<Date>;
            timeZone: TimeZone;
            constructor (format: Date.FormatString, locale: Locale | null, timeZone: TimeZone, calendar: Calendar);
            constructor (decoder: Decoder);
            static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
            static verbatim(format: Date.FormatString, locale: Locale | null, timeZone: TimeZone, calendar: Calendar):Date.VerbatimFormatStyle;
            consuming(input: string, index: String.Index, bounds: Range<String.Index>):any;
            encode(encoder: Encoder):void;
            format(value: Date):string;
            hash(hasher: Hasher):void;
        }
    }
    export class NSDate extends NSObject {
        /**
         * A date object representing a date in the distant future. 
         */
        static distantFuture: Date;
        /**
         * A date object representing a date in the distant past.
         */
        static distantPast: Date;
        /**
         * The current date and time, as of the time of access.
         */
        static now: Date;
        /**
         * The interval between the date object and the current date and time.
         */
        timeIntervalSinceNow: TimeInterval;
        /**
         * The interval between the date object and 00:00:00 UTC on 1 January 2001.
         */
        timeIntervalSinceReferenceDate: TimeInterval;
        /**
         * The interval between the date object and 00:00:00 UTC on 1 January 1970.
         */
        timeIntervalSince1970: TimeInterval;
        /**
         * A string representation of the date object.
         */
        description: string;
        /**
         * A custom playground Quick Look for this object.
         */
        customPlaygroundQuickLook: PlaygroundQuickLook;
        /**
         * Returns a date object initialized to the current date and time.
         */
        constructor ();
        /**
         * Returns a date object initialized relative to the current date and time by a given number of seconds.
         */
        constructor (secs: TimeInterval);
        /**
         * Returns a date object initialized relative to another given date by a given number of seconds.
         */
        constructor (secsToBeAdded: TimeInterval, date: Date);
        /**
         * Returns a date object initialized relative to 00:00:00 UTC on 1 January 2001 by a given number of seconds.
         */
        constructor (ti: TimeInterval);
        /**
         * Returns a date object initialized relative to 00:00:00 UTC on 1 January 1970 by a given number of seconds.
         */
        constructor (secs: TimeInterval);
        /**
         * Returns a date object initialized from data in the given unarchiver.
         */
        constructor (coder: NSCoder);
        /**
         * Returns a date object initialized with a date and time value specified by a given string in the international string representation format.
         */
        constructor (description: string);
        /**
         * Creates and returns a date object set to the date and time specified by a given string.
         */
        static date(_string: string):any;
        /**
         * Returns a Boolean value that indicates whether a given object is a date that is exactly equal the receiver.
         */
        isEqual(otherDate: Date):boolean;
        /**
         * Returns the earlier of the receiver and another given date.
         */
        earlierDate(anotherDate: Date):Date;
        /**
         * Returns the later of the receiver and another given date.
         */
        laterDate(anotherDate: Date):Date;
        /**
         * Indicates the temporal ordering of the receiver and another given date.
         */
        compare(other: Date):ComparisonResult;
        /**
         * Returns the interval between the receiver and another given date.
         */
        timeIntervalSince(anotherDate: Date):TimeInterval;
        /**
         * Returns a new date object that is set to a given number of seconds relative to the receiver.
         */
        addingTimeInterval(ti: TimeInterval):Self;
        /**
         * Returns a new date object that is set to a given number of seconds relative to the receiver.
         */
        addTimeInterval(seconds: TimeInterval):any;
    }
    export class DateInterval {
        /**
         * The start date.
         */
        start: Date;
        /**
         * The end date.
         */
        end: Date;
        /**
         * The duration.
         */
        duration: TimeInterval;
        /**
         * A textual description of the date interval.
         */
        description: string;
        /**
         * A textual description of the date interval suitable for debugging.
         */
        debugDescription: string;
        /**
         * A mirror that reflects the date interval.
         */
        customMirror: Mirror;
        /**
         * The computed hash value for the date interval.
         */
        hashValue: number;
        /**
         * Initializes an interval with start and end dates set to the current date and the duration set to 
         * .
         */
        constructor ();
        /**
         * Initializes an interval with the specified start date and duration.
         */
        constructor (start: Date, duration: TimeInterval);
        /**
         * Initializes an interval with the specified start and end date.
         */
        constructor (start: Date, end: Date);
        constructor (decoder: Decoder);
        /**
         * Returns an interval that represents the interval where the given date interval and the current instance intersect.
         */
        intersection(dateInterval: DateInterval):DateInterval | null;
        /**
         * Indicates whether this interval intersects the specified interval.
         */
        intersects(dateInterval: DateInterval):boolean;
        /**
         * Indicates whether this interval contains the given date.
         */
        contains(date: Date):boolean;
        /**
         * Compares two intervals.
         */
        compare(dateInterval: DateInterval):ComparisonResult;
        encode(encoder: Encoder):void;
        hash(hasher: Hasher):void;
    }
    export class NSDateInterval extends NSObject {
        /**
         * The start date of the date interval.
         */
        startDate: Date;
        /**
         * The end date of the date interval.
         */
        endDate: Date;
        /**
         * The duration of the date interval.
         */
        duration: TimeInterval;
        /**
         * Initializes a date interval by setting the start and end date to the current date.
         */
        constructor ();
        /**
         * Initializes a date interval with a given start date and duration.
         */
        constructor (startDate: Date, duration: TimeInterval);
        /**
         * Initializes a date interval from a given start date and end date.
         */
        constructor (startDate: Date, endDate: Date);
        /**
         * Returns a date interval initialized from data in the given unarchiver.
         */
        constructor (coder: NSCoder);
        /**
         * Compares the receiver with the specified date interval.
         */
        compare(dateInterval: DateInterval):ComparisonResult;
        /**
         * Indicates whether the receiver is equal to the specified date interval.
         */
        isEqual(dateInterval: DateInterval):boolean;
        /**
         * Indicates whether the receiver intersects with the specified date interval.
         */
        intersects(dateInterval: DateInterval):boolean;
        /**
         * Returns the intersection between the receiver and the specified date interval.
         */
        intersection(dateInterval: DateInterval):DateInterval | null;
        /**
         * Indicates whether the receiver contains the specified date.
         */
        contains(date: Date):boolean;
    }
    export class DateComponents {
        static defaultResolverSpecification: EmptyResolverSpecification<DateComponents>;
        /**
         * The calendar used to interpret the other values in this structure.
         */
        calendar: Calendar | null;
        /**
         * A time zone.
         */
        timeZone: TimeZone | null;
        /**
         * Indicates whether the current combination of properties represents a date which exists in the current calendar.
         */
        isValidDate: boolean;
        /**
         * The date calculated from the current components using the stored calendar.
         */
        date: Date | null;
        /**
         * An era or count of eras.
         */
        era: number;
        /**
         * A year or count of years.
         */
        year: number;
        /**
         * The year corresponding to a week-counting week.
         */
        yearForWeekOfYear: number;
        /**
         * A quarter or count of quarters.
         */
        quarter: number;
        /**
         * A month or count of months.
         */
        month: number;
        /**
         * Set to true if these components represent a leap month.
         */
        isLeapMonth: boolean;
        /**
         * A week of the month or a count of weeks of the month.
         */
        weekOfMonth: number;
        /**
         * A week of the year or count of the weeks of the year.
         */
        weekOfYear: number;
        /**
         * A weekday or count of weekdays.
         */
        weekday: number;
        /**
         * A weekday ordinal or count of weekday ordinals. 
         */
        weekdayOrdinal: number;
        /**
         * A day or count of days.
         */
        day: number;
        /**
         * An hour or count of hours.
         */
        hour: number;
        /**
         * A minute or count of minutes.
         */
        minute: number;
        /**
         * A second or count of seconds.
         */
        second: number;
        /**
         * A nanosecond or count of nanoseconds.
         */
        nanosecond: number;
        /**
         * A textual description of the date components.
         */
        description: string;
        /**
         * A textual description of the date components suitable for debugging.
         */
        debugDescription: string;
        /**
         * A mirror that reflects the date.
         */
        customMirror: Mirror;
        /**
         * The computed hash value of the date.
         */
        hashValue: number;
        /**
         * Initializes a date components value, optionally specifying values for its fields.
         */
        constructor (calendar: Calendar | null, timeZone: TimeZone | null, era: number, year: number, month: number, day: number, hour: number, minute: number, second: number, nanosecond: number, weekday: number, weekdayOrdinal: number, quarter: number, weekOfMonth: number, weekOfYear: number, yearForWeekOfYear: number);
        constructor (decoder: Decoder);
        constructor (period: Product.SubscriptionPeriod);
        /**
         * Returns the value of one of the properties, using an enumeration value instead of a property name.
         */
        value(component: Calendar.Component):number;
        /**
         * Set the value of one of the properties, using an enumeration value instead of a property name.
         */
        setValue(value: number, component: Calendar.Component):void;
        encode(encoder: Encoder):void;
        hash(hasher: Hasher):void;
    }
    export interface DataProtocol extends RandomAccessCollection {
        /**
         * A collection of buffers that make up the whole of the type conforming to a data protocol. 
         */
        regions: Self.Regions;
        /**
         * Copies the bytes of data from the type into a raw memory buffer.
         */
        copyBytes(ptr: UnsafeMutableRawBufferPointer):number;
    }
    export class NSDateComponents extends NSObject {
        /**
         * The calendar used to interpret the date components.
         */
        calendar: Calendar | null;
        /**
         * The time zone used to interpret the date components.
         */
        timeZone: TimeZone | null;
        /**
         * A Boolean value that indicates whether the current combination of properties represents a date which exists in the current calendar.
         */
        isValidDate: boolean;
        /**
         * The date calculated from the current components using the stored calendar.
         */
        date: Date | null;
        /**
         * The number of eras.
         */
        era: number;
        /**
         * The number of years.
         */
        year: number;
        /**
         * The ISO 8601 week-numbering year.
         */
        yearForWeekOfYear: number;
        /**
         * The number of quarters.
         */
        quarter: number;
        /**
         * The number of months.
         */
        month: number;
        /**
         * A Boolean value that indicates whether the month is a leap month.
         */
        isLeapMonth: boolean;
        /**
         * The number of the weekdays.
         */
        weekday: number;
        /**
         * The ordinal number of weekdays.
         */
        weekdayOrdinal: number;
        /**
         * The week number of the months.
         */
        weekOfMonth: number;
        /**
         * The ISO 8601 week date of the year.
         */
        weekOfYear: number;
        /**
         * The number of days.
         */
        day: number;
        /**
         * The number of hour units for the receiver.
         */
        hour: number;
        /**
         * The number of minute units for the receiver.
         */
        minute: number;
        /**
         * The number of second units for the receiver.
         */
        second: number;
        /**
         * The number of nanosecond units for the receiver.
         */
        nanosecond: number;
        /**
         * Returns the number of weeks.
         */
        week():number;
        /**
         * Sets the number of weeks.
         */
        setWeek(v: number):void;
        /**
         * Returns the value for a given calendar unit.
         */
        value(unit: NSCalendar.Unit):number;
        /**
         * Sets a value for a given calendar unit.
         */
        setValue(value: number, unit: NSCalendar.Unit):void;
    }
    export class NSCalendar extends NSObject {}
    /**
     * the same name namespace of class NSCalendar
     */
    export namespace NSCalendar {
        export class Unit {
            /**
             * Identifier for the era unit.
             */
            static era: NSCalendar.Unit;
            /**
             * Identifier for the year unit.
             */
            static year: NSCalendar.Unit;
            /**
             * Identifier for the week-counting year unit. 
             */
            static yearForWeekOfYear: NSCalendar.Unit;
            /**
             * Identifier for the quarter of the calendar.
             */
            static quarter: NSCalendar.Unit;
            /**
             * Identifier for the month unit. 
             */
            static month: NSCalendar.Unit;
            /**
             * Identifier for the week of the year calendar unit.
             */
            static weekOfYear: NSCalendar.Unit;
            /**
             * Identifier for the week of the month calendar unit.
             */
            static weekOfMonth: NSCalendar.Unit;
            /**
             * Identifier for the weekday unit.
             */
            static weekday: NSCalendar.Unit;
            /**
             * Identifier for the ordinal weekday unit.
             */
            static weekdayOrdinal: NSCalendar.Unit;
            /**
             * Identifier for the day unit. 
             */
            static day: NSCalendar.Unit;
            /**
             * Identifier for the hour unit.
             */
            static hour: NSCalendar.Unit;
            /**
             * Identifier for the minute unit.
             */
            static minute: NSCalendar.Unit;
            /**
             * Identifier for the second unit.
             */
            static second: NSCalendar.Unit;
            /**
             * Identifier for the nanosecond unit.
             */
            static nanosecond: NSCalendar.Unit;
            /**
             * Identifier for the calendar of a date components object.
             */
            static calendar: NSCalendar.Unit;
            /**
             * Identifier for the time zone of a date components object.
             */
            static timeZone: NSCalendar.Unit;
            /**
             * Specifies the era unit.
             */
            static NSEraCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the year unit.
             */
            static NSYearCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the month unit. 
             */
            static NSMonthCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the day unit. 
             */
            static NSDayCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the hour unit.
             */
            static NSHourCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the minute unit.
             */
            static NSMinuteCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the second unit.
             */
            static NSSecondCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the week unit.
             */
            static NSWeekCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the weekday unit.
             */
            static NSWeekdayCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the ordinal weekday unit. 
             */
            static NSWeekdayOrdinalCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the quarter of the calendar as a 
             * . In macOS 10.6 and earlier this was defined as equal to 
             * . In macOS 10.7 and later it is defined as 
             * .
             */
            static NSQuarterCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the original week of a month calendar unit.
             */
            static NSWeekOfMonthCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the original week of the year calendar unit.
             */
            static NSWeekOfYearCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the year when the calendar is being interpreted as a week-based calendar. 
             */
            static NSYearForWeekOfYearCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the calendar of the calendar.
             */
            static NSCalendarCalendarUnit: NSCalendar.Unit;
            /**
             * Specifies the time zone of the calendar as an 
             * .
             */
            static NSTimeZoneCalendarUnit: NSCalendar.Unit;
            /**
             * Creates a new calendar unit from the raw value.
             */
            constructor (rawValue: number);
        }
    }
    export interface MutableDataProtocol extends DataProtocol,MutableCollection,RangeReplaceableCollection {}
    export interface ContiguousBytes {}
    export class URL {
        static applicationDirectory: URL;
        static applicationSupportDirectory: URL;
        static cachesDirectory: URL;
        static defaultResolverSpecification: ResolverSpecification;
        static desktopDirectory: URL;
        static documentsDirectory: URL;
        static downloadsDirectory: URL;
        static homeDirectory: URL;
        static libraryDirectory: URL;
        static moviesDirectory: URL;
        static musicDirectory: URL;
        static picturesDirectory: URL;
        static sharedPublicDirectory: URL;
        static temporaryDirectory: URL;
        static transferRepresentation: TransferRepresentation;
        static trashDirectory: URL;
        static userDirectory: URL;
        /**
         * The data representation of the URL’s relativeString.
         */
        dataRepresentation: Data;
        /**
         * The absolute string for the URL.
         */
        absoluteString: string;
        /**
         * The absolute URL.
         */
        absoluteURL: URL;
        /**
         * The base URL.
         */
        baseURL: URL | null;
        /**
         * The fragment component of the URL if the URL conforms to RFC 1808 (the most common form of URL), otherwise nil.
         */
        fragment: string;
        /**
         * The host component of a URL if the URL conforms to RFC 1808 (the most common form of URL), otherwise nil.
         */
        host: string;
        /**
         * The last path component of the URL, or an empty string if the path is an empty string.
         */
        lastPathComponent: string;
        /**
         * The path component of the URL if the URL conforms to RFC 1808 (the most common form of URL), otherwise an empty string.
         */
        path: string;
        /**
         * The path components of the URL, or an empty array if the path is an empty string.
         */
        pathComponents: string[];
        /**
         * The path extension of the URL, or an empty string if the path is an empty string.
         */
        pathExtension: string;
        /**
         * The port component of the URL if the URL conforms to RFC 1808 (the most common form of URL), otherwise nil.
         */
        port: number;
        /**
         * The query of the URL if the URL conforms to RFC 1808 (the most common form of URL), otherwise nil.
         */
        query: string;
        /**
         * The relative path of the URL if the URL conforms to RFC 1808 (the most common form of URL), otherwise nil.
         */
        relativePath: string;
        /**
         * The relative portion of a URL.
         */
        relativeString: string;
        /**
         * The scheme of the URL.
         */
        scheme: string;
        /**
         * A version of the URL with any instances of “..” or “.” removed from its path.
         */
        standardized: URL;
        /**
         * A standardized version of the path of a file URL.
         */
        standardizedFileURL: URL;
        /**
         * The user component of the URL if the URL conforms to RFC 1808 (the most common form of URL), otherwise nil.
         */
        user: string;
        /**
         * The password component of the URL if the URL conforms to RFC 1808 (the most common form of URL), otherwise nil.
         */
        password: string;
        /**
         * A Boolean that is true if the scheme is 
         * .
         */
        isFileURL: boolean;
        /**
         * A Boolean that is true if the URL path represents a directory.
         */
        hasDirectoryPath: boolean;
        /**
         * The URL’s resource data, as an asynchronous sequence of bytes.
         */
        resourceBytes: URL.AsyncBytes;
        /**
         * The URL’s resource data, as an asynchronous sequence of lines of text.
         */
        lines: AsyncLineSequence<URL.AsyncBytes>;
        /**
         * A textual description of the URL.
         */
        description: string;
        /**
         * A textual description of the URL suitable for debugging.
         */
        debugDescription: string;
        /**
         * A playground quicklook for the URL.
         */
        customPlaygroundQuickLook: PlaygroundQuickLook;
        /**
         * The computed hash value for the URL.
         */
        hashValue: number;
        /**
         * Creates a URL instance from the provided string.
         */
        constructor (_string: string);
        /**
         * Creates a URL instance from the provided string, relative to another URL.
         */
        constructor (_string: string, url: URL | null);
        /**
         * Creates a file URL that references the local file or directory at path.
         */
        constructor (path: string);
        /**
         * Creates a file URL that references the local file or directory at path.
         */
        constructor (path: string, isDirectory: boolean);
        /**
         * Creates a file URL that references the local file or directory at path, relative to a base URL.
         */
        constructor (path: string, base: URL | null);
        /**
         * Creates a file URL that references the local file or directory at path, relative to a base URL.
         */
        constructor (path: string, isDirectory: boolean, base: URL | null);
        /**
         * Creates a file URL that references the local file or directory for the file system representation of the path.
         */
        constructor (path: UnsafePointer<Int8>, isDirectory: boolean, baseURL: URL | null);
        /**
         * Creates a URL from a playground file literal.
         */
        constructor (name: string);
        /**
         * Creates a file URL that references the local file or directory at the file path you specify.
         */
        constructor (path: FilePath);
        /**
         * Creates a file URL that references the local file or directory at the file path you specify.
         */
        constructor (path: FilePath, isDirectory: boolean);
        /**
         * Creates a URL that refers to a location specified by resolving bookmark data.
         */
        constructor (data: Data, options: URL.BookmarkResolutionOptions, url: URL | null, bookmarkDataIsStale: Bool);
        /**
         * Creates a URL that refers to the location specified by resolving an alias file.
         */
        constructor (url: URL, options: URL.BookmarkResolutionOptions);
        /**
         * Creates a URL from a resource.
         */
        constructor (resource: URLResource);
        /**
         * Creates a URL by decoding an encoded representation.
         */
        constructor (decoder: Decoder);
        /**
         * Initializes a newly created URL using the contents of the given data, relative to a base URL.
         */
        constructor (dataRepresentation: Data, url: URL | null, isAbsolute: boolean);
        constructor (path: FilePath, directoryHint: URL.DirectoryHint);
        constructor (path: string, directoryHint: URL.DirectoryHint, base: URL | null);
        constructor (directory: FileManager.SearchPathDirectory, domain: FileManager.SearchPathDomainMask, url: URL | null, shouldCreate: boolean);
        /**
         * Creates an alias file on disk at a specified location with specified bookmark data. 
         */
        static writeBookmarkData(data: Data, url: URL):void;
        static currentDirectory():URL;
        /**
         * Encodes the URL for later decoding.
         */
        encode(encoder: Encoder):void;
        /**
         * Returns a collection of resource values identified by the given resource keys.
         */
        resourceValues(keys: Set<URLResourceKey>):URLResourceValues;
        /**
         * Sets the resource value identified by a given resource key.
         */
        setResourceValues(values: URLResourceValues):void;
        /**
         * Removes the cached resource value identified by a given resource value key from the URL object.
         */
        removeCachedResourceValue(key: URLResourceKey):void;
        /**
         * Removes all cached resource values and all temporary resource values from the URL object.
         */
        removeAllCachedResourceValues():void;
        /**
         * Sets a temporary resource value on the URL object.
         */
        setTemporaryResourceValue(value: any, key: URLResourceKey):void;
        /**
         * Resolves any symlinks in the path of a file URL.
         */
        resolveSymlinksInPath():void;
        /**
         * Resolves any symlinks in the path of a file URL.
         */
        resolvingSymlinksInPath():URL;
        /**
         * Standardizes the path of a file URL.
         */
        standardize():void;
        /**
         * Appends a path component to the URL.
         */
        appendPathComponent(pathComponent: string):void;
        /**
         * Returns a URL by appending the specified path component to self.
         */
        appendingPathComponent(pathComponent: string):URL;
        /**
         * Appends the specified path extension to self.
         */
        appendPathExtension(pathExtension: string):void;
        /**
         * Returns a URL by appending the specified path extension to self.
         */
        appendingPathExtension(pathExtension: string):URL;
        /**
         * Returns a URL constructed by removing the last path component of self.
         */
        deleteLastPathComponent():void;
        /**
         * Returns a URL constructed by removing the last path component of self.
         */
        deletingLastPathComponent():URL;
        /**
         * Returns a URL constructed by removing any path extension.
         */
        deletePathExtension():void;
        /**
         * Returns a URL constructed by removing any path extension.
         */
        deletingPathExtension():URL;
        /**
         * Returns bookmark data for the URL, created with specified options and resource values.
         */
        bookmarkData(options: URL.BookmarkCreationOptions, keys: Set<URLResourceKey> | null, url: URL | null):Data;
        /**
         * Returns whether the URL’s resource exists and is reachable.
         */
        checkResourceIsReachable():boolean;
        /**
         * Returns whether the promised item URL’s resource exists and is reachable.
         */
        checkPromisedItemIsReachable():boolean;
        /**
         * Gets resource values from URLs of ‘promised’ items.
         */
        promisedItemResourceValues(keys: Set<URLResourceKey>):URLResourceValues;
        /**
         * Given a url created by resolving a bookmark data created with security scope, make the resource referenced by the url accessible to the process.
         */
        startAccessingSecurityScopedResource():boolean;
        /**
         * Revokes the access granted to the url by a prior successful call to the complementary start function.
         */
        stopAccessingSecurityScopedResource():void;
        /**
         * Hashes the essential components of this URL by feeding them into the given hasher.
         */
        hash(hasher: Hasher):void;
        append(queryItems: URLQueryItem[]):void;
        appending(queryItems: URLQueryItem[]):URL;
        formatted():string;
    }
    /**
     * the same name namespace of class URL
     */
    export namespace URL {
        export class AsyncBytes {
            /**
             * The bytes loaded from the resource, as an asychronous sequence of characters.
             */
            characters: AsyncCharacterSequence<URL.AsyncBytes>;
            /**
             * The bytes loaded from the resource, as an asychronous sequence of Unicode scalar values.
             */
            unicodeScalars: AsyncUnicodeScalarSequence<URL.AsyncBytes>;
            /**
             * The bytes loaded from the resource, as an asychronous sequence of lines of text.
             */
            lines: AsyncLineSequence<URL.AsyncBytes>;
            /**
             * Returns a Boolean value that indicates whether the asynchronous sequence contains the given element.
             */
            contains(search: UInt8):boolean;
            /**
             * Returns a Boolean value that indicates whether all elements produced by the asynchronous sequence satisfy the given predicate.
             */
            allSatisfy(predicate: Bool):boolean;
            /**
             * Returns the first element of the sequence that satisfies the given predicate.
             */
            first(predicate: Bool):UInt8 | null;
            /**
             * Returns the minimum element in an asynchronous sequence of comparable elements.
             */
            min():UInt8 | null;
            /**
             * Returns the maximum element in an asynchronous sequence of comparable elements.
             */
            max():UInt8 | null;
            /**
             * Returns an asynchronous sequence, up to the specified maximum length, containing the initial elements of the base asynchronous sequence.
             */
            prefix(count: number):AsyncPrefixSequence<URL.AsyncBytes>;
            /**
             * Omits a specified number of elements from the base asynchronous sequence, then passes through all remaining elements.
             */
            dropFirst(count: number):AsyncDropFirstSequence<URL.AsyncBytes>;
            /**
             * Omits elements from the base asynchronous sequence until a given closure returns false, after which it passes through all remaining elements.
             */
            drop(predicate: Bool):AsyncDropWhileSequence<URL.AsyncBytes>;
            /**
             * Creates an asynchronous sequence that contains, in order, the elements of the base sequence that satisfy the given predicate.
             */
            filter(isIncluded: Bool):AsyncFilterSequence<URL.AsyncBytes>;
            makeAsyncIterator():URL.AsyncBytes.AsyncIterator;
        }
        /**
         * the same name namespace of class AsyncBytes
         */
        export namespace AsyncBytes {
            export class AsyncIterator {
                next():UInt8 | null;
            }
        }
        export class FormatStyle {
            static url: URL.FormatStyle;
            hashValue: number;
            parseStrategy: URL.ParseStrategy;
            constructor (decoder: Decoder);
            constructor (scheme: URL.FormatStyle.ComponentDisplayOption, user: URL.FormatStyle.ComponentDisplayOption, password: URL.FormatStyle.ComponentDisplayOption, host: URL.FormatStyle.HostDisplayOption, port: URL.FormatStyle.ComponentDisplayOption, path: URL.FormatStyle.ComponentDisplayOption, query: URL.FormatStyle.ComponentDisplayOption, fragment: URL.FormatStyle.ComponentDisplayOption);
            static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
            encode(encoder: Encoder):void;
            format(value: URL):string;
            fragment(strategy: URL.FormatStyle.ComponentDisplayOption):URL.FormatStyle;
            hash(hasher: Hasher):void;
            host(strategy: URL.FormatStyle.HostDisplayOption):URL.FormatStyle;
            locale(locale: Locale):URL.FormatStyle;
            password(strategy: URL.FormatStyle.ComponentDisplayOption):URL.FormatStyle;
            path(strategy: URL.FormatStyle.ComponentDisplayOption):URL.FormatStyle;
            port(strategy: URL.FormatStyle.ComponentDisplayOption):URL.FormatStyle;
            query(strategy: URL.FormatStyle.ComponentDisplayOption):URL.FormatStyle;
            scheme(strategy: URL.FormatStyle.ComponentDisplayOption):URL.FormatStyle;
            user(strategy: URL.FormatStyle.ComponentDisplayOption):URL.FormatStyle;
        }
        /**
         * the same name namespace of class FormatStyle
         */
        export namespace FormatStyle {
            export class ComponentDisplayOption {
                static always: URL.FormatStyle.ComponentDisplayOption;
                static never: URL.FormatStyle.ComponentDisplayOption;
                static omitIfHTTPFamily: URL.FormatStyle.ComponentDisplayOption;
                description: string;
                hashValue: number;
                constructor (decoder: Decoder);
                static displayWhen(component: URL.FormatStyle.Component, requirements: Set<String>):URL.FormatStyle.ComponentDisplayOption;
                static omitWhen(component: URL.FormatStyle.Component, requirements: Set<String>):URL.FormatStyle.ComponentDisplayOption;
                encode(encoder: Encoder):void;
                hash(hasher: Hasher):void;
            }
            export class HostDisplayOption {
                static always: URL.FormatStyle.HostDisplayOption;
                static never: URL.FormatStyle.HostDisplayOption;
                static omitIfHTTPFamily: URL.FormatStyle.HostDisplayOption;
                description: string;
                hashValue: number;
                constructor (decoder: Decoder);
                static displayWhen(component: URL.FormatStyle.Component, requirements: Set<String>):URL.FormatStyle.HostDisplayOption;
                static omitSpecificSubdomains(subdomainsToOmit: Set<String>, omitMultiLevelSubdomains: boolean):URL.FormatStyle.HostDisplayOption;
                static omitWhen(component: URL.FormatStyle.Component, requirements: Set<String>):URL.FormatStyle.HostDisplayOption;
                encode(encoder: Encoder):void;
                hash(hasher: Hasher):void;
            }
            export enum Component {
                fragment,
                host,
                password,
                path,
                port,
                query,
                scheme,
                user,
            }
        }
        export class ParseStrategy {
            static url: URL.ParseStrategy;
            hashValue: number;
            regex: Regex<URL>;
            constructor (decoder: Decoder);
            constructor (scheme: URL.ParseStrategy.ComponentParseStrategy<String>, user: URL.ParseStrategy.ComponentParseStrategy<String>, password: URL.ParseStrategy.ComponentParseStrategy<String>, host: URL.ParseStrategy.ComponentParseStrategy<String>, port: URL.ParseStrategy.ComponentParseStrategy<Int>, path: URL.ParseStrategy.ComponentParseStrategy<String>, query: URL.ParseStrategy.ComponentParseStrategy<String>, fragment: URL.ParseStrategy.ComponentParseStrategy<String>);
            static fixed(format: Date.FormatString, timeZone: TimeZone, locale: Locale | null):Date.ParseStrategy;
            consuming(input: string, index: String.Index, bounds: Range<String.Index>):any;
            encode(encoder: Encoder):void;
            fragment(strategy: URL.ParseStrategy.ComponentParseStrategy<String>):URL.ParseStrategy;
            hash(hasher: Hasher):void;
            host(strategy: URL.ParseStrategy.ComponentParseStrategy<String>):URL.ParseStrategy;
            parse(value: string):URL;
            password(strategy: URL.ParseStrategy.ComponentParseStrategy<String>):URL.ParseStrategy;
            path(strategy: URL.ParseStrategy.ComponentParseStrategy<String>):URL.ParseStrategy;
            port(strategy: URL.ParseStrategy.ComponentParseStrategy<Int>):URL.ParseStrategy;
            query(strategy: URL.ParseStrategy.ComponentParseStrategy<String>):URL.ParseStrategy;
            scheme(strategy: URL.ParseStrategy.ComponentParseStrategy<String>):URL.ParseStrategy;
            user(strategy: URL.ParseStrategy.ComponentParseStrategy<String>):URL.ParseStrategy;
        }
        export enum DirectoryHint {
            checkFileSystem,
            inferFromPath,
            isDirectory,
            notDirectory,
        }
    }
    export class NSURL extends NSObject {}
    /**
     * the same name namespace of class NSURL
     */
    export namespace NSURL {
        export class BookmarkResolutionOptions {
            /**
             * Specifies that no UI feedback should accompany resolution of the bookmark data.
             */
            static withoutUI: NSURL.BookmarkResolutionOptions;
            /**
             * Specifies that no volume should be mounted during resolution of the bookmark data.
             */
            static withoutMounting: NSURL.BookmarkResolutionOptions;
            /**
             * Specifies that the security scope, applied to the bookmark when it was created, should be used during resolution of the bookmark data.
             */
            static withSecurityScope: NSURL.BookmarkResolutionOptions;
            static withoutImplicitStartAccessing: NSURL.BookmarkResolutionOptions;
            /**
             * Initializes a new resolution options structure.
             */
            constructor (rawValue: number);
        }
        export class BookmarkCreationOptions {
            /**
             * Specifies that when a bookmark created with this option is resolved, its embedded file ID should take precedence over other sources of information (file system path, for example) in the event of a conflict.
             */
            static preferFileIDResolution: NSURL.BookmarkCreationOptions;
            /**
             * Specifies that a bookmark created with this option should be created with minimal information.
             */
            static minimalBookmark: NSURL.BookmarkCreationOptions;
            /**
             * Specifies that the bookmark data should include properties required to create Finder alias files.
             */
            static suitableForBookmarkFile: NSURL.BookmarkCreationOptions;
            /**
             * Specifies that you want to create a security-scoped bookmark that, when resolved, provides a security-scoped URL allowing read/write access to a file-system resource.
             */
            static withSecurityScope: NSURL.BookmarkCreationOptions;
            /**
             * When combined with the 
             *  option, specifies that you want to create a security-scoped bookmark that, when resolved, provides a security-scoped URL allowing read-only access to a file-system resource.
             */
            static securityScopeAllowOnlyReadAccess: NSURL.BookmarkCreationOptions;
            static withoutImplicitSecurityScope: NSURL.BookmarkCreationOptions;
            /**
             * Initializes a new bookmark creation options structure.
             */
            constructor (rawValue: number);
        }
    }
    export class TimeZone {
        /**
         * The time zone currently used by the system, automatically updating to the user’s current preference.
         */
        static autoupdatingCurrent: TimeZone;
        /**
         * The time zone currently used by the system.
         */
        static current: TimeZone;
        /**
         * Returns an array of strings listing the identifier of all the time zones known to the system.
         */
        static knownTimeZoneIdentifiers: string[];
        /**
         * Returns the mapping of abbreviations to time zone identifiers.
         */
        static abbreviationDictionary: Record<string, string>;
        /**
         * Returns the time zone data version.
         */
        static timeZoneDataVersion: string;
        static gmt: TimeZone;
        /**
         * The geopolitical region identifier that identifies the time zone.
         */
        identifier: string;
        /**
         * The date of the next (after the current instant) daylight saving time transition for the time zone. 
         */
        nextDaylightSavingTimeTransition: Date | null;
        /**
         * A textual description of the time zone.
         */
        description: string;
        /**
         * A textual description of the time zone suitable for debugging.
         */
        debugDescription: string;
        /**
         * A mirror that reflects the time zone.
         */
        customMirror: Mirror;
        /**
         * The computed hash value of the time zone.
         */
        hashValue: number;
        /**
         * Returns a time zone initialized with a specific number of seconds from GMT.
         */
        constructor (seconds: number);
        constructor (abbreviation: string);
        constructor (decoder: Decoder);
        constructor (identifier: string);
        /**
         * Returns the abbreviation for the time zone at a given date.
         */
        abbreviation(date: Date):string;
        /**
         * The current difference in seconds between the time zone and Greenwich Mean Time.
         */
        secondsFromGMT(date: Date):number;
        /**
         * Returns a Boolean value that indicates whether the receiver uses daylight saving time at a given date.
         */
        isDaylightSavingTime(date: Date):boolean;
        /**
         * Returns the daylight saving time offset for a given date.
         */
        daylightSavingTimeOffset(date: Date):TimeInterval;
        /**
         * Returns the name of the receiver localized for a given locale.
         */
        localizedName(style: NSTimeZone.NameStyle, locale: Locale | null):string;
        encode(encoder: Encoder):void;
        hash(hasher: Hasher):void;
    }
    export class NSTimeZone extends NSObject {
        /**
         * An object that tracks the current system time zone.
         */
        static local: TimeZone;
        /**
         * The time zone currently used by the system.
         */
        static system: TimeZone;
        /**
         * The default time zone for the current app.
         */
        static default: TimeZone;
        /**
         * Returns an array of strings listing the IDs of all the time zones known to the system.
         */
        static knownTimeZoneNames: string[];
        /**
         * Returns a dictionary holding the mappings of time zone abbreviations to time zone names.
         */
        static abbreviationDictionary: Record<string, string>;
        /**
         * Returns the time zone data version.
         */
        static timeZoneDataVersion: string;
        /**
         * The geopolitical region ID that identifies the receiver.
         */
        name: string;
        /**
         * The abbreviation for the receiver, such as “EDT” (Eastern Daylight Time).
         */
        abbreviation: string;
        /**
         * The current difference in seconds between the receiver and Greenwich Mean Time.
         */
        secondsFromGMT: number;
        /**
         * The data that stores the information used by the receiver.
         */
        data: Data;
        /**
         * A Boolean value that indicates whether the receiver is currently using daylight saving time.
         */
        isDaylightSavingTime: boolean;
        /**
         * The current daylight saving time offset of the receiver.
         */
        daylightSavingTimeOffset: TimeInterval;
        /**
         * The date of the next daylight saving time transition for the receiver.
         */
        nextDaylightSavingTimeTransition: Date | null;
        /**
         * A textual description of the time zone including the name, abbreviation, offset from GMT, and whether or not daylight saving time is currently in effect.
         */
        description: string;
        /**
         * Returns a time zone initialized with a given identifier.
         */
        constructor (tzName: string);
        /**
         * Initializes a time zone with a given identifier and time zone data.
         */
        constructor (tzName: string, aData: Data | null);
        /**
         * Returns the time zone object identified by a given abbreviation.
         */
        constructor (abbreviation: string);
        /**
         * Returns a time zone object offset from Greenwich Mean Time by a given number of seconds.
         */
        constructor (seconds: number);
        /**
         * Clears any time zone value cached for the 
         *  property.
         */
        static resetSystemTimeZone():void;
        /**
         * Indicates whether the receiver has the same name and data as the specified time zone.
         */
        isEqual(aTimeZone: TimeZone):boolean;
        /**
         * Returns the localized name of the time zone.
         */
        localizedName(style: NSTimeZone.NameStyle, locale: Locale | null):string;
    }
    /**
     * the same name namespace of class NSTimeZone
     */
    export namespace NSTimeZone {
        export enum NameStyle {
            standard = 0,
            shortStandard = 1,
            daylightSaving = 2,
            shortDaylightSaving = 3,
            generic = 4,
            shortGeneric = 5,
        }
    }
    export class DateFormatter extends Formatter {
        /**
         * Returns the default formatting behavior for instances of the class.
         */
        static defaultFormatterBehavior: DateFormatter.Behavior;
        /**
         * The date style of the receiver.
         */
        dateStyle: DateFormatter.Style;
        /**
         * The time style of the receiver.
         */
        timeStyle: DateFormatter.Style;
        /**
         * The date format string used by the receiver.
         */
        dateFormat: String!;
        /**
         * The capitalization formatting context used when formatting a date.
         */
        formattingContext: Formatter.Context;
        /**
         * The calendar for the receiver.
         */
        calendar: Calendar!;
        /**
         * The default date for the receiver.
         */
        defaultDate: Date | null;
        /**
         * The locale for the receiver.
         */
        locale: Locale!;
        /**
         * The time zone for the receiver.
         */
        timeZone: TimeZone!;
        /**
         * The earliest date that can be denoted by a two-digit year specifier.
         */
        twoDigitStartDate: Date | null;
        /**
         * The start date of the Gregorian calendar for the receiver.
         */
        gregorianStartDate: Date | null;
        /**
         * The formatter behavior for the receiver.
         */
        formatterBehavior: DateFormatter.Behavior;
        /**
         * A Boolean value that indicates whether the receiver uses heuristics when parsing a string.
         */
        isLenient: boolean;
        /**
         * A Boolean value that indicates whether the receiver uses phrases such as “today” and “tomorrow” for the date component.
         */
        doesRelativeDateFormatting: boolean;
        /**
         * The AM symbol for the receiver.
         */
        amSymbol: String!;
        /**
         * The PM symbol for the receiver.
         */
        pmSymbol: String!;
        /**
         * The array of weekday symbols for the receiver.
         */
        weekdaySymbols: [String]!;
        /**
         * The array of short weekday symbols for the receiver.
         */
        shortWeekdaySymbols: [String]!;
        /**
         * The array of very short weekday symbols for the receiver.
         */
        veryShortWeekdaySymbols: [String]!;
        /**
         * The array of standalone weekday symbols for the receiver.
         */
        standaloneWeekdaySymbols: [String]!;
        /**
         * The array of short standalone weekday symbols for the receiver.
         */
        shortStandaloneWeekdaySymbols: [String]!;
        /**
         * The array of very short standalone weekday symbols for the receiver.
         */
        veryShortStandaloneWeekdaySymbols: [String]!;
        /**
         * The month symbols for the receiver.
         */
        monthSymbols: [String]!;
        /**
         * The array of short month symbols for the receiver.
         */
        shortMonthSymbols: [String]!;
        /**
         * The very short month symbols for the receiver.
         */
        veryShortMonthSymbols: [String]!;
        /**
         * The standalone month symbols for the receiver.
         */
        standaloneMonthSymbols: [String]!;
        /**
         * The short standalone month symbols for the receiver.
         */
        shortStandaloneMonthSymbols: [String]!;
        /**
         * The very short month symbols for the receiver.
         */
        veryShortStandaloneMonthSymbols: [String]!;
        /**
         * The quarter symbols for the receiver.
         */
        quarterSymbols: [String]!;
        /**
         * The short quarter symbols for the receiver.
         */
        shortQuarterSymbols: [String]!;
        /**
         * The standalone quarter symbols for the receiver.
         */
        standaloneQuarterSymbols: [String]!;
        /**
         * The short standalone quarter symbols for the receiver.
         */
        shortStandaloneQuarterSymbols: [String]!;
        /**
         * The era symbols for the receiver.
         */
        eraSymbols: [String]!;
        /**
         * The long era symbols for the receiver
         */
        longEraSymbols: [String]!;
        /**
         * Indicates whether the formatter generates the deprecated calendar date type.
         */
        generatesCalendarDates: boolean;
        /**
         * Returns a string representation of a specified date, that the system formats for the current locale using the specified date and time styles.
         */
        static localizedString(date: Date, dstyle: DateFormatter.Style, tstyle: DateFormatter.Style):string;
        /**
         * Returns a date representation of a specified string that the system interprets using the receiver’s current settings.
         */
        date(_string: string):Date | null;
        /**
         * Returns a string representation of a specified date that the system formats using the receiver’s current settings.
         */
        string(date: Date):string;
        /**
         * Returns by reference a date representation of a specified string and its date range, as well as a Boolean value that indicates whether the system can parse the string.
         */
        getObjectValue(obj: AutoreleasingUnsafeMutablePointer<AnyObject?> | null, _string: string, rangep: UnsafeMutablePointer<NSRange> | null):void;
        /**
         * Sets the date format from a template using the specified locale for the receiver.
         */
        setLocalizedDateFormatFromTemplate(dateFormatTemplate: string):void;
    }
    /**
     * the same name namespace of class DateFormatter
     */
    export namespace DateFormatter {
        export enum Style {
            none = 0,
            short = 1,
            medium = 2,
            long = 3,
            full = 4,
        }
        export enum Behavior {
            default = 0,
            behavior10_0 = 1000,
            behavior10_4 = 1040,
        }
    }
    export class DateComponentsFormatter extends Formatter {
        /**
         * The bitmask of calendrical units such as day and month to include in the output string.
         */
        allowedUnits: NSCalendar.Unit;
        /**
         * A Boolean indicating whether non-integer units may be used for values.
         */
        allowsFractionalUnits: boolean;
        /**
         * The default calendar to use when formatting date components. 
         */
        calendar: Calendar | null;
        /**
         * A Boolean value indicating whether to collapse the largest unit into smaller units when a certain threshold is met.
         */
        collapsesLargestUnit: boolean;
        /**
         * A Boolean value indicating whether the resulting phrase reflects an inexact time value.
         */
        includesApproximationPhrase: boolean;
        /**
         * A Boolean value indicating whether output strings reflect the amount of time remaining. 
         */
        includesTimeRemainingPhrase: boolean;
        /**
         * The maximum number of time units to include in the output string.
         */
        maximumUnitCount: number;
        /**
         * The formatting style for unit names. 
         */
        unitsStyle: DateComponentsFormatter.UnitsStyle;
        /**
         * The formatting style for units whose value is 0.
         */
        zeroFormattingBehavior: DateComponentsFormatter.ZeroFormattingBehavior;
        formattingContext: Formatter.Context;
        referenceDate: Date | null;
        /**
         * Returns a localized string based on the specified date components and style option.
         */
        static localizedString(components: DateComponents, unitsStyle: DateComponentsFormatter.UnitsStyle):string;
        /**
         * Returns a formatted string based on the specified date component information.
         */
        string(components: DateComponents):string;
        getObjectValue(obj: AutoreleasingUnsafeMutablePointer<AnyObject?> | null, _string: string, error: AutoreleasingUnsafeMutablePointer<NSString?> | null):boolean;
    }
    /**
     * the same name namespace of class DateComponentsFormatter
     */
    export namespace DateComponentsFormatter {
        export enum UnitsStyle {
            spellOut = 4,
            full = 3,
            short = 2,
            brief = 5,
            abbreviated = 1,
            positional = 0,
        }
        export class ZeroFormattingBehavior {
            /**
             * The default formatting behavior. When using positional units, this behavior drops leading zeroes but pads middle and trailing values with zeros as needed. For example, with hours, minutes, and seconds displayed, the value for one hour and 10 seconds is “1:00:10”. For all other unit styles, this behavior drops all units whose values are 0. For example, when days, hours, minutes, and seconds are allowed, the abbreviated version of one hour and 10 seconds is displayed as “1h 10s”.  
             */
            static default: DateComponentsFormatter.ZeroFormattingBehavior;
            /**
             * The drop leading zeroes formatting behavior. Units whose values are 0 are dropped starting at the beginning of the sequence. Units continue to be dropped until a non-zero value is encountered. For example, when days, hours, minutes, and seconds are allowed, the abbreviated version of ten minutes is displayed as “10m 0s”. 
             */
            static dropLeading: DateComponentsFormatter.ZeroFormattingBehavior;
            /**
             * The drop middle zero units behavior. Units whose values are 0 are dropped from anywhere in the middle of a sequence. For example, when days, hours, minutes, and seconds are allowed, the abbreviated version of one hour, zero minutes, and five seconds is displayed as “0d 1h 5s”.  
             */
            static dropMiddle: DateComponentsFormatter.ZeroFormattingBehavior;
            /**
             * The drop trailing zero units behavior. Units whose value is 0 are dropped starting at the end of the sequence. For example, when days, hours, minutes, and seconds are allowed, the abbreviated version of one hour is displayed as “0d 1h”. 
             */
            static dropTrailing: DateComponentsFormatter.ZeroFormattingBehavior;
            /**
             * The drop all zero units behavior. This behavior drops all units whose values are 0. For example, when days, hours, minutes, and seconds are allowed, the abbreviated version of one hour is displayed as “1h”. 
             */
            static dropAll: DateComponentsFormatter.ZeroFormattingBehavior;
            /**
             * The add padding zeroes behavior. This behavior pads values with zeroes as appropriate. For example, consider the value of one hour formatted using the positional and abbreviated unit styles. When days, hours, minutes, and seconds are allowed, the value is displayed as “0d 1:00:00” using the positional style, and as “0d 1h 0m 0s” using the abbreviated style. 
             */
            static pad: DateComponentsFormatter.ZeroFormattingBehavior;
            constructor (rawValue: number);
        }
    }
    export class DateIntervalFormatter extends Formatter {
        /**
         * The style to use when formatting day, month, and year information.
         */
        dateStyle: DateIntervalFormatter.Style;
        /**
         * The style to use when formatting hour, minute, and second information.
         */
        timeStyle: DateIntervalFormatter.Style;
        /**
         * The template for formatting one date and time value.
         */
        dateTemplate: String!;
        /**
         * The calendar to use for date values. 
         */
        calendar: Calendar!;
        /**
         * The locale to use when formatting date and time values.
         */
        locale: Locale!;
        /**
         * The time zone with which to specify time values. 
         */
        timeZone: TimeZone!;
        /**
         * Returns a formatted string based on the specified start and end dates.
         */
        string(fromDate: Date, toDate: Date):string;
    }
    /**
     * the same name namespace of class DateIntervalFormatter
     */
    export namespace DateIntervalFormatter {
        export enum Style {
            none = 0,
            short = 1,
            medium = 2,
            long = 3,
            full = 4,
        }
    }
    export class ISO8601DateFormatter extends Formatter {
        /**
         * Options for generating and parsing ISO 8601 date representations. See 
         *  for possible values.
         */
        formatOptions: ISO8601DateFormatter.Options;
        /**
         * The time zone used to create and parse date representations. When unspecified, GMT is used.
         */
        timeZone: TimeZone!;
        /**
         * Initializes an ISO 8601 date formatter with default format, time zone, and options.
         */
        constructor ();
        /**
         * Creates and returns an ISO 8601 formatted string representation of the specified date.
         */
        string(date: Date):string;
        /**
         * Creates and returns a date object from the specified ISO 8601 formatted string representation.
         */
        date(_string: string):Date | null;
    }
    /**
     * the same name namespace of class ISO8601DateFormatter
     */
    export namespace ISO8601DateFormatter {
        export class Options {
            static withYear: ISO8601DateFormatter.Options;
            static withMonth: ISO8601DateFormatter.Options;
            static withWeekOfYear: ISO8601DateFormatter.Options;
            static withDay: ISO8601DateFormatter.Options;
            static withTime: ISO8601DateFormatter.Options;
            static withTimeZone: ISO8601DateFormatter.Options;
            static withSpaceBetweenDateAndTime: ISO8601DateFormatter.Options;
            static withDashSeparatorInDate: ISO8601DateFormatter.Options;
            static withColonSeparatorInTime: ISO8601DateFormatter.Options;
            static withColonSeparatorInTimeZone: ISO8601DateFormatter.Options;
            static withFullDate: ISO8601DateFormatter.Options;
            static withFullTime: ISO8601DateFormatter.Options;
            static withInternetDateTime: ISO8601DateFormatter.Options;
            static withFractionalSeconds: ISO8601DateFormatter.Options;
            constructor (rawValue: number);
        }
    }
    export class URLResourceKey {
        static isApplicationKey: URLResourceKey;
        static applicationIsScriptableKey: URLResourceKey;
        /**
         * Key for determining whether the resource is a directory, returned as a Boolean 
         *  object (read-only).
         */
        static isDirectoryKey: URLResourceKey;
        /**
         * The parent directory of the resource, returned as an 
         *  object, or 
         *  if the resource is the root directory of its volume (read-only).
         */
        static parentDirectoryURLKey: URLResourceKey;
        /**
         * The key for the total size allocated on-disk for the file.
         */
        static fileAllocatedSizeKey: URLResourceKey;
        /**
         * The key for the protection level of the file.
         */
        static fileProtectionKey: URLResourceKey;
        /**
         * The key for a value that APFS assigns to identify a file’s content data stream.
         */
        static fileContentIdentifierKey: URLResourceKey;
        /**
         * The key for the resource’s unique identifier.
         */
        static fileResourceIdentifierKey: URLResourceKey;
        /**
         * The key for the resource’s object type.
         */
        static fileResourceTypeKey: URLResourceKey;
        /**
         * The key for the resource’s security information.
         */
        static fileSecurityKey: URLResourceKey;
        /**
         * The key for the file’s size, in bytes.
         */
        static fileSizeKey: URLResourceKey;
        /**
         * The key for determining whether the file is an alias.
         */
        static isAliasFileKey: URLResourceKey;
        /**
         * The key for determining whether the resource is a file package
         */
        static isPackageKey: URLResourceKey;
        /**
         * The key for determining whether the resource is a regular file rather than a directory or a symbolic link.
         */
        static isRegularFileKey: URLResourceKey;
        /**
         * The key for a Boolean value that indicates whether the file system can delete a file when the system needs to free space.
         */
        static isPurgeableKey: URLResourceKey;
        /**
         * The key for a Boolean value that indicates whether the file has sparse regions.
         */
        static isSparseKey: URLResourceKey;
        /**
         * The key for a Boolean value that indicates whether the file has extended attributes.
         */
        static mayHaveExtendedAttributesKey: URLResourceKey;
        /**
         * The key for a Boolean value that indicates whether cloned files and their original files may share data blocks.
         */
        static mayShareFileContentKey: URLResourceKey;
        /**
         * The key for the optimal block size to use when reading or writing the file’s data.
         */
        static preferredIOBlockSizeKey: URLResourceKey;
        /**
         * The key for the total allocated size of the file, in bytes.
         */
        static totalFileAllocatedSizeKey: URLResourceKey;
        /**
         * The key for the total displayable size of the file, in bytes.
         */
        static totalFileSizeKey: URLResourceKey;
        /**
         * Key for the volume’s available capacity in bytes (read-only).
         */
        static volumeAvailableCapacityKey: URLResourceKey;
        /**
         * Key for the volume’s available capacity in bytes for storing important resources (read-only).
         */
        static volumeAvailableCapacityForImportantUsageKey: URLResourceKey;
        /**
         * Key for the volume’s available capacity in bytes for storing nonessential resources (read-only).
         */
        static volumeAvailableCapacityForOpportunisticUsageKey: URLResourceKey;
        /**
         * Key for the volume’s total capacity in bytes (read-only).
         */
        static volumeTotalCapacityKey: URLResourceKey;
        /**
         * Key for determining whether the volume is automounted, returned as a Boolean 
         *  object (read-only).
         */
        static volumeIsAutomountedKey: URLResourceKey;
        /**
         * Key for determining whether the volume is visible in GUI-based file-browsing environments, such as the Desktop or the Finder application, returned as a Boolean 
         *  object (read-only).
         */
        static volumeIsBrowsableKey: URLResourceKey;
        /**
         * Key for determining whether the volume is ejectable from the drive mechanism under software control, returned as a Boolean 
         *  object (read-only).
         */
        static volumeIsEjectableKey: URLResourceKey;
        /**
         * Whether the volume is encrypted, returned as 
         *  containing a Boolean value (read-only).
         */
        static volumeIsEncryptedKey: URLResourceKey;
        /**
         * Key for determining whether the volume is connected to an internal bus, returned as a Boolean 
         *  object, or 
         *  if it cannot be determined (read-only).
         */
        static volumeIsInternalKey: URLResourceKey;
        /**
         * Key for determining whether the volume is currently journaling, returned as a Boolean 
         *  object (read-only).
         */
        static volumeIsJournalingKey: URLResourceKey;
        /**
         * Key for determining whether the volume is stored on a local device, returned as a Boolean 
         *  object (read-only).
         */
        static volumeIsLocalKey: URLResourceKey;
        /**
         * Key for determining whether the volume is read-only, returned as a Boolean 
         *  object (read-only).
         */
        static volumeIsReadOnlyKey: URLResourceKey;
        /**
         * Key for determining whether the volume is removable from the drive mechanism, returned as a Boolean 
         *  object (read-only).
         */
        static volumeIsRemovableKey: URLResourceKey;
        /**
         * Whether the volume is the root filesystem, returned as 
         *  containing a Boolean value (read-only).
         */
        static volumeIsRootFileSystemKey: URLResourceKey;
        /**
         * A Boolean value that indicates the volume supports data protection for files.
         */
        static volumeSupportsFileProtectionKey: URLResourceKey;
        /**
         * Key for determining whether the URL is a file system trigger directory, returned as a Boolean 
         *  object (read-only). Traversing or opening a file system trigger directory causes an attempt to mount a file system on the directory.
         */
        static isMountTriggerKey: URLResourceKey;
        /**
         * Key for determining whether the resource is the root directory of a volume, returned as a Boolean 
         *  object (read-only).
         */
        static isVolumeKey: URLResourceKey;
        /**
         * Key for the volume’s creation date, returned as an 
         *  object, or 
         *  if it cannot be determined (read-only).
         */
        static volumeCreationDateKey: URLResourceKey;
        /**
         * The unique identifier of the resource’s volume, returned as an 
         *  (read-only).
         */
        static volumeIdentifierKey: URLResourceKey;
        /**
         * Key for the volume’s descriptive format name, returned as an 
         *  object (read-only).
         */
        static volumeLocalizedFormatDescriptionKey: URLResourceKey;
        /**
         * The name of the volume as it should be displayed in the user interface, returned as an 
         *  object (read-only).
         */
        static volumeLocalizedNameKey: URLResourceKey;
        /**
         * Key for the largest file size supported by the volume in bytes, returned as a Boolean 
         *  object, or 
         *  if it cannot be determined (read-only).
         */
        static volumeMaximumFileSizeKey: URLResourceKey;
        /**
         * The name of the volume, returned as an 
         *  object (read-write). Settable only if 
         *  is 
         * .
         */
        static volumeNameKey: URLResourceKey;
        /**
         * Key for the total number of resources on the volume, returned as an 
         *  object (read-only).
         */
        static volumeResourceCountKey: URLResourceKey;
        static volumeSupportsAccessPermissionsKey: URLResourceKey;
        /**
         * Key for determining whether the volume implements whole-file advisory locks in the style of flock, along with the 
         *  and 
         *  flags of the open function, returned as a Boolean 
         *  object (read-only).
         */
        static volumeSupportsAdvisoryFileLockingKey: URLResourceKey;
        /**
         * Key for determining whether the volume supports case-preserved names, returned as a Boolean 
         *  object (read-only).
         */
        static volumeSupportsCasePreservedNamesKey: URLResourceKey;
        /**
         * Key for determining whether the volume supports case-sensitive names, returned as a Boolean 
         *  object (read-only).
         */
        static volumeSupportsCaseSensitiveNamesKey: URLResourceKey;
        /**
         * Whether the volume supports transparent decompression of compressed files using 
         * , returned as 
         *  containing a Boolean value (read-only).
         */
        static volumeSupportsCompressionKey: URLResourceKey;
        /**
         * Whether the volume supports exclusive renaming using 
         *  with the 
         *  option, returned as 
         *  containing a Boolean value (read-only).
         */
        static volumeSupportsExclusiveRenamingKey: URLResourceKey;
        /**
         * Key for determining whether the volume supports extended security (access control lists), returned as a Boolean 
         *  object (read-only).
         */
        static volumeSupportsExtendedSecurityKey: URLResourceKey;
        /**
         * Whether the volume supports cloning using 
         * , returned as 
         *  containing a Boolean value (read-only).
         */
        static volumeSupportsFileCloningKey: URLResourceKey;
        /**
         * Key for determining whether the volume supports hard links, returned as a Boolean 
         *  object (read-only).
         */
        static volumeSupportsHardLinksKey: URLResourceKey;
        static volumeSupportsImmutableFilesKey: URLResourceKey;
        /**
         * Key for determining whether the volume supports journaling, returned as a Boolean 
         *  object (read-only).
         */
        static volumeSupportsJournalingKey: URLResourceKey;
        /**
         * Key for determining whether the volume supports persistent IDs, returned as a Boolean 
         *  object (read-only).
         */
        static volumeSupportsPersistentIDsKey: URLResourceKey;
        /**
         * Key for determining whether the volume can be renamed, returned as a Boolean 
         *  object (read-only).
         */
        static volumeSupportsRenamingKey: URLResourceKey;
        /**
         * Key for determining whether the volume supports reliable storage of times for the root directory, returned as a Boolean 
         *  object (read-only).
         */
        static volumeSupportsRootDirectoryDatesKey: URLResourceKey;
        /**
         * Key for determining whether the volume supports sparse files, returned as a Boolean 
         *  object (read-only).
         */
        static volumeSupportsSparseFilesKey: URLResourceKey;
        /**
         * Whether the volume supports renaming using 
         *  with the 
         *  option, returned as 
         *  containing a Boolean value (read-only).
         */
        static volumeSupportsSwapRenamingKey: URLResourceKey;
        /**
         * Key for determining whether the volume supports symbolic links, returned as a Boolean 
         *  object (read-only).
         */
        static volumeSupportsSymbolicLinksKey: URLResourceKey;
        /**
         * Key for determining whether the volume supports returning volume size information, returned as a Boolean 
         *  object (read-only). If 
         * , volume size information is available as values of the 
         *  and
         *  keys.
         */
        static volumeSupportsVolumeSizesKey: URLResourceKey;
        /**
         * Key for determining whether the volume supports zero runs, returned as a Boolean 
         *  object (read-only).
         */
        static volumeSupportsZeroRunsKey: URLResourceKey;
        /**
         * Key for the URL needed to remount the network volume, returned as an 
         *  object, or 
         *  if a URL is not available (read-only).
         */
        static volumeURLForRemountingKey: URLResourceKey;
        /**
         * The root directory of the resource’s volume, returned as an 
         *  object (read-only).
         */
        static volumeURLKey: URLResourceKey;
        /**
         * Key for the volume’s persistent UUID, returned as an 
         *  object, or 
         *  if a persistent UUID is not available (read-only).
         */
        static volumeUUIDStringKey: URLResourceKey;
        /**
         * The key for a Boolean value that indicates whether the item is in iCloud storage.
         */
        static isUbiquitousItemKey: URLResourceKey;
        /**
         * The key for the name components of the most recent editor.
         */
        static ubiquitousSharedItemMostRecentEditorNameComponentsKey: URLResourceKey;
        /**
         * The key for a Boolean value that indicates whether the system has already made a call 
         *  to download the item.
         */
        static ubiquitousItemDownloadRequestedKey: URLResourceKey;
        /**
         * The key for a Boolean value that indicates whether the system is downloading the item from iCloud.
         */
        static ubiquitousItemIsDownloadingKey: URLResourceKey;
        /**
         * The key for an error object that indicates why downloading the item from iCloud fails.
         */
        static ubiquitousItemDownloadingErrorKey: URLResourceKey;
        /**
         * The key for the current download state for the item.
         */
        static ubiquitousItemDownloadingStatusKey: URLResourceKey;
        /**
         * The key of a Boolean value that indicates whether the system excludes the item from syncing.
         */
        static ubiquitousItemIsExcludedFromSyncKey: URLResourceKey;
        /**
         * The key for a Boolean value that indicates whether the system uploads the item’s data to iCloud storage.
         */
        static ubiquitousItemIsUploadedKey: URLResourceKey;
        /**
         * The key for a Boolean value that indicates whether the system is uploading the item to iCloud.
         */
        static ubiquitousItemIsUploadingKey: URLResourceKey;
        /**
         * The key for an error object that indicates why uploading the item to iCloud fails.
         */
        static ubiquitousItemUploadingErrorKey: URLResourceKey;
        /**
         * The key for a Boolean value that indicates whether this item has outstanding conflicts.
         */
        static ubiquitousItemHasUnresolvedConflictsKey: URLResourceKey;
        /**
         * The key for a string that contains the name of the item’s container as it appears to the user.
         */
        static ubiquitousItemContainerDisplayNameKey: URLResourceKey;
        /**
         * The key for the name components of the item’s owner.
         */
        static ubiquitousSharedItemOwnerNameComponentsKey: URLResourceKey;
        /**
         * The key for the current user’s permissions.
         */
        static ubiquitousSharedItemCurrentUserPermissionsKey: URLResourceKey;
        /**
         * The key for the role of the current user.
         */
        static ubiquitousSharedItemCurrentUserRoleKey: URLResourceKey;
        /**
         * The key for a Boolean value that indicates a shared item.
         */
        static ubiquitousItemIsSharedKey: URLResourceKey;
        /**
         * All thumbnails as a single NSImage (read-write).
         */
        static thumbnailKey: URLResourceKey;
        /**
         * A dictionary of NSImage/UIImage objects keyed by size (read-write). See 
         *  for a list of possible keys.
         */
        static thumbnailDictionaryKey: URLResourceKey;
        /**
         * Key for the resource properties that have not been set after the 
         *  method returns an error, returned as an array of 
         *  objects.
         */
        static keysOfUnsetValuesKey: URLResourceKey;
        static quarantinePropertiesKey: URLResourceKey;
        /**
         * The time at which the resource’s was created or renamed into or within its parent directory, returned as an 
         * . Inconsistent behavior may be observed when this attribute is requested on hard-linked items. This property is not supported by all volumes. (read-only)
         */
        static addedToDirectoryDateKey: URLResourceKey;
        /**
         * The time at which the resource’s attributes were most recently modified, returned as an 
         *  object if the volume supports attribute modification dates, or 
         *  if attribute modification dates are unsupported (read-only).
         */
        static attributeModificationDateKey: URLResourceKey;
        /**
         * The time at which the resource was most recently accessed.
         */
        static contentAccessDateKey: URLResourceKey;
        /**
         * The time at which the resource was most recently modified.
         */
        static contentModificationDateKey: URLResourceKey;
        /**
         * The time at which the resource was created.
         */
        static creationDateKey: URLResourceKey;
        /**
         * The icon stored with the resource, returned as an 
         *  object, or 
         *  if the resource has no custom icon.
         */
        static customIconKey: URLResourceKey;
        /**
         * The document identifier returned as an 
         *  (read-only). 
         */
        static documentIdentifierKey: URLResourceKey;
        /**
         * The resource’s normal icon, returned as an 
         *  object (read-only).
         */
        static effectiveIconKey: URLResourceKey;
        /**
         * An opaque generation identifier, returned as an 
         *  (read-only)
         */
        static generationIdentifierKey: URLResourceKey;
        /**
         * Key for determining whether the resource’s extension is normally removed from its localized name, returned as a Boolean 
         *  object (read-write).
         */
        static hasHiddenExtensionKey: URLResourceKey;
        /**
         * A key for indicating whether the system excludes the resource from all backups of app data.
         */
        static isExcludedFromBackupKey: URLResourceKey;
        /**
         * Key for determining whether the current process (as determined by the EUID) can execute the resource (if it is a file) or search the resource (if it is a directory), returned as a Boolean 
         *  object (read-only).
         */
        static isExecutableKey: URLResourceKey;
        /**
         * Key for determining whether the resource is normally not displayed to users, returned as a Boolean 
         *  object (read-write).
         */
        static isHiddenKey: URLResourceKey;
        /**
         * Key for determining whether the current process (as determined by the EUID) can read the resource, returned as a Boolean 
         *  object (read-only).
         */
        static isReadableKey: URLResourceKey;
        /**
         * Key for determining whether the resource is a symbolic link, returned as a Boolean 
         *  object (read-only).
         */
        static isSymbolicLinkKey: URLResourceKey;
        /**
         * Key for determining whether the resource's system immutable bit is set, returned as a Boolean 
         *  object (read-write).
         */
        static isSystemImmutableKey: URLResourceKey;
        /**
         * Key for determining whether the resource's user immutable bit is set, returned as a Boolean 
         *  object (read-write).
         */
        static isUserImmutableKey: URLResourceKey;
        /**
         * Key for determining whether the current process (as determined by the EUID) can write to the resource, returned as a Boolean 
         *  object (read-only).
         */
        static isWritableKey: URLResourceKey;
        /**
         * The resource’s label color, returned as an 
         *  object, or 
         *  if the resource has no label color (read-only).
         */
        static labelColorKey: URLResourceKey;
        /**
         * The resource’s label number, returned as an 
         *  object (read-write).
         */
        static labelNumberKey: URLResourceKey;
        /**
         * The number of hard links to the resource, returned as an 
         *  object (read-only).
         */
        static linkCountKey: URLResourceKey;
        /**
         * The resource’s localized label text, returned as an 
         *  object, or 
         *  if the resource has no localized label text (read-only).
         */
        static localizedLabelKey: URLResourceKey;
        /**
         * The resource’s localized or extension-hidden name, returned as an 
         *  object (read-only).
         */
        static localizedNameKey: URLResourceKey;
        /**
         * The resource’s localized type description, returned as an 
         *  object (read-only).
         */
        static localizedTypeDescriptionKey: URLResourceKey;
        /**
         * The resource’s name in the file system, returned as an 
         *  object (read-write).
         */
        static nameKey: URLResourceKey;
        /**
         * The file system path for the URL, returned as an 
         *  object (read-only).
         */
        static pathKey: URLResourceKey;
        static canonicalPathKey: URLResourceKey;
        /**
         * The names of tags attached to the resource, returned as an array of 
         *  values (read-write).
         */
        static tagNamesKey: URLResourceKey;
        /**
         * The resource’s uniform type identifier (UTI), returned as an 
         *  object (read-only).
         */
        static typeIdentifierKey: URLResourceKey;
        /**
         * The resource’s type.
         */
        static contentTypeKey: URLResourceKey;
        /**
         * The key for a Boolean value that indicates whether the system downloaded this item’s data from iCloud storage.
         */
        static ubiquitousItemIsDownloadedKey: URLResourceKey;
        /**
         * The key for a value that indicates the percentage of data that the system downloaded from iCloud storage.
         */
        static ubiquitousItemPercentDownloadedKey: URLResourceKey;
        /**
         * The key for a value that indicates the percentage of data that the system uploaded to iCloud storage.
         */
        static ubiquitousItemPercentUploadedKey: URLResourceKey;
        constructor (rawValue: string);
        constructor (rawValue: string);
    }
    export class URLFileProtection {
        /**
         * The file is stored in an encrypted format on disk and cannot be read from or written to while the device is locked or booting.
         */
        static complete: URLFileProtection;
        /**
         * The file is stored in an encrypted format on disk after it is closed.
         */
        static completeUnlessOpen: URLFileProtection;
        /**
         * The file is stored in an encrypted format on disk and cannot be accessed until after the device has booted.
         */
        static completeUntilFirstUserAuthentication: URLFileProtection;
        /**
         * The file has no special protections associated with it.
         */
        static none: URLFileProtection;
        /**
         * Creates a URL file protection type value.
         */
        constructor (rawValue: string);
    }
    export class URLFileResourceType {
        /**
         * The resource is a named pipe.
         */
        static namedPipe: URLFileResourceType;
        /**
         * The resource is a character special file.
         */
        static characterSpecial: URLFileResourceType;
        /**
         * The resource is a directory.
         */
        static directory: URLFileResourceType;
        /**
         * The resource is a block special file.
         */
        static blockSpecial: URLFileResourceType;
        /**
         * The resource is a regular file.
         */
        static regular: URLFileResourceType;
        /**
         * The resource is a symbolic link.
         */
        static symbolicLink: URLFileResourceType;
        /**
         * The resource is a socket.
         */
        static socket: URLFileResourceType;
        /**
         * The resource’s type is unknown.
         */
        static unknown: URLFileResourceType;
        /**
         * Creates a file resource type from the provided constant string.
         */
        constructor (rawValue: string);
    }
    export class URLUbiquitousItemDownloadingStatus {
        /**
         * A local copy of this item exists and is the most up-to-date version known to the device.
         */
        static current: URLUbiquitousItemDownloadingStatus;
        /**
         * A local copy of this item exists, but it is stale. The most recent version will be downloaded as soon as possible.
         */
        static downloaded: URLUbiquitousItemDownloadingStatus;
        /**
         * This item has not been downloaded yet. Use 
         *  to download it.
         */
        static notDownloaded: URLUbiquitousItemDownloadingStatus;
        /**
         * Creates a download status instance from the provided constant string.
         */
        constructor (rawValue: string);
    }
    export class URLUbiquitousSharedItemRole {
        static owner: URLUbiquitousSharedItemRole;
        static participant: URLUbiquitousSharedItemRole;
        /**
         * Creates a shared item role instance from the provided constant string.
         */
        constructor (rawValue: string);
    }
    export class URLUbiquitousSharedItemPermissions {
        static readOnly: URLUbiquitousSharedItemPermissions;
        static readWrite: URLUbiquitousSharedItemPermissions;
        /**
         * Creates a shared item permissions instance from the provided constant string.
         */
        constructor (rawValue: string);
    }
    export class URLThumbnailDictionaryItem {
        /**
         * A 1024 x 1024 pixel thumbnail as a 
         *  on iOS or an 
         *  in macOS.
         */
        static NSThumbnail1024x1024SizeKey: URLThumbnailDictionaryItem;
        /**
         * Creates a thumbnail dictionary item key from the provided constant string.
         */
        constructor (rawValue: string);
        /**
         * Creates a thumbnail dictionary item key from the provided raw value string.
         */
        constructor (rawValue: string);
    }
    export class URLResourceValues {
        /**
         * A Boolean value that indicates whether the application is scriptable.
         */
        applicationIsScriptable: boolean;
        /**
         * A Boolean value that indicates whether the resource is an application.
         */
        isApplication: boolean;
        /**
         * A Boolean value that indicates whether the resource is a directory.
         */
        isDirectory: boolean;
        /**
         * A value that the kernel assigns to identify a document.
         */
        documentIdentifier: number;
        /**
         * A value APFS assigns that identifies a file’s content data stream.
         */
        fileContentIdentifier: Int64 | null;
        /**
         * The total allocated size on-disk for the file, in bytes.
         */
        fileAllocatedSize: number;
        /**
         * The protection level for the file.
         */
        fileProtection: URLFileProtection | null;
        /**
         * An identifier for comparing two file system objects for equality.
         */
        fileResourceIdentifier: ((NSCopying & NSSecureCoding & NSObjectProtocol));
        /**
         * The type of the file system object.
         */
        fileResourceType: URLFileResourceType | null;
        /**
         * The file system object’s security information.
         */
        fileSecurity: NSFileSecurity | null;
        /**
         * The total file size, in bytes.
         */
        fileSize: number;
        /**
         * A Boolean value that indicates whether the file system can delete a file when the system needs to free space.
         */
        isPurgeable: boolean;
        /**
         * A Boolean value that indicates whether the file has sparse regions.
         */
        isSparse: boolean;
        /**
         * A Boolean value that indicates the file may have extended attributes.
         */
        mayHaveExtendedAttributes: boolean;
        /**
         * A Boolean value that indicates whether you can execute the file resource or search a directory resource.
         */
        isExecutable: boolean;
        /**
         * A Boolean value that indicates whether the resource is a regular file.
         */
        isRegularFile: boolean;
        /**
         * A Boolean value that indicates whether the cloned files and their original files may share data blocks.
         */
        mayShareFileContent: boolean;
        /**
         * The total allocated size of the file, in bytes.
         */
        totalFileAllocatedSize: number;
        /**
         * The total displayable size of the file, in bytes.
         */
        totalFileSize: number;
        /**
         * The volume’s available capacity, in bytes.
         */
        volumeAvailableCapacity: number;
        /**
         * The volume’s available capacity for storing important resources, in bytes.
         */
        volumeAvailableCapacityForImportantUsage: Int64 | null;
        /**
         * The volume’s available capacity for storing nonessential resources, in bytes.
         */
        volumeAvailableCapacityForOpportunisticUsage: Int64 | null;
        /**
         * The volume’s total capacity, in bytes.
         */
        volumeTotalCapacity: number;
        /**
         * A Boolean value that indicates whether the volume is automounted.
         */
        volumeIsAutomounted: boolean;
        /**
         * A Boolean value that indicates whether the volume is visible through the user interface.
         */
        volumeIsBrowsable: boolean;
        /**
         * A Boolean value that indicates whether the volume’s media is ejectable from the drive mechanism under software control.
         */
        volumeIsEjectable: boolean;
        /**
         * A Boolean value that indicates whether the volume is encrypted.
         */
        volumeIsEncrypted: boolean;
        /**
         * A Boolean value that indicates whether the volume’s device is connected to an internal bus, or 
         *  if not available.
         */
        volumeIsInternal: boolean;
        /**
         * A Boolean value that indicates whether the volume is currently using a journal for speedy recovery after an unplanned restart.
         */
        volumeIsJournaling: boolean;
        /**
         * A Boolean value that indicates whether the volume is stored on a local device.
         */
        volumeIsLocal: boolean;
        /**
         * A Boolean value that indicates whether the volume is read-only.
         */
        volumeIsReadOnly: boolean;
        /**
         * A Boolean value that indicates whether the volume’s media is removable from the drive mechanism.
         */
        volumeIsRemovable: boolean;
        /**
         * A Boolean value that indicates whether the volume is the root filesystem.
         */
        volumeIsRootFileSystem: boolean;
        /**
         * A Boolean value that indicates whether this URL is a file system trigger directory.
         */
        isMountTrigger: boolean;
        /**
         * A Boolean value that indicates whether the root directory is a volume.
         */
        isVolume: boolean;
        /**
         * URL of the volume on which the resource is stored.
         */
        volume: URL | null;
        /**
         * The volume’s creation date, or 
         *  if this cannot be determined.
         */
        volumeCreationDate: Date | null;
        /**
         * An identifier that identifies the volume the file system object is on.
         */
        volumeIdentifier: ((NSCopying & NSSecureCoding & NSObjectProtocol));
        /**
         * The volume format that’s visible to the user.
         */
        volumeLocalizedFormatDescription: string;
        /**
         * The name of the volume that’s visible to the user.
         */
        volumeLocalizedName: string;
        /**
         * The largest file size supported by this file system, in bytes, or 
         *  if this cannot be determined.
         */
        volumeMaximumFileSize: number;
        /**
         * The name of the volume.
         */
        volumeName: string;
        /**
         * The total number of resources on the volume.
         */
        volumeResourceCount: number;
        /**
         * A Boolean value that indicates whether the volume supports setting standard access permissions.
         */
        volumeSupportsAccessPermissions: boolean;
        /**
         * A Boolean value that indicates whether the volume implements whole-file flock(2) style advisory locks, and the O_EXLOCK and O_SHLOCK flags of the open(2) call.
         */
        volumeSupportsAdvisoryFileLocking: boolean;
        /**
         * A Boolean value that indicates whether the volume format preserves the case of file and directory names.
         */
        volumeSupportsCasePreservedNames: boolean;
        /**
         * A Boolean value that indicates whether the volume format treats upper and lower case characters in file and directory names as different.
         */
        volumeSupportsCaseSensitiveNames: boolean;
        /**
         * A Boolean value that indicates whether the volume supports transparent decompression of compressed files using decmpfs.
         */
        volumeSupportsCompression: boolean;
        /**
         * A Boolean value that indicates whether the volume warns of a pre-existing destination when renaming a file.
         */
        volumeSupportsExclusiveRenaming: boolean;
        /**
         * A Boolean value that indicates whether the volume implements extended security (ACLs).
         */
        volumeSupportsExtendedSecurity: boolean;
        /**
         * A Boolean value that indicates whether the volume supports file cloning.
         */
        volumeSupportsFileCloning: boolean;
        /**
         * A Boolean value that indicates whether the volume format supports hard links.
         */
        volumeSupportsHardLinks: boolean;
        /**
         * A Boolean value that indicates whether the volume supports making files immutable.
         */
        volumeSupportsImmutableFiles: boolean;
        /**
         * A Boolean value that indicates whether the volume format supports a journal used to speed recovery in case of unplanned restart (such as a power outage or crash).
         */
        volumeSupportsJournaling: boolean;
        /**
         * A Boolean value that indicates whether the volume format supports persistent object identifiers and can look up file system objects by their IDs.
         */
        volumeSupportsPersistentIDs: boolean;
        /**
         * A Boolean value that indicates whether the volume can be renamed.
         */
        volumeSupportsRenaming: boolean;
        /**
         * A Boolean value that indicates whether the volume supports reliable storage of times for the root directory.
         */
        volumeSupportsRootDirectoryDates: boolean;
        /**
         * A Boolean value that indicates whether the volume format supports sparse files.
         */
        volumeSupportsSparseFiles: boolean;
        /**
         * A Boolean value that indicates whether the volume supports swapping source and target files when both exist.
         */
        volumeSupportsSwapRenaming: boolean;
        /**
         * A Boolean value that indicates whether the volume format supports symbolic links.
         */
        volumeSupportsSymbolicLinks: boolean;
        /**
         * A Boolean value that indicates whether the volume supports returning volume size values.
         */
        volumeSupportsVolumeSizes: boolean;
        /**
         * A Boolean value that indicates whether the volume keeps track of allocated but unwritten parts of a file so that it can substitute zeroes without actually writing zeroes to the media.
         */
        volumeSupportsZeroRuns: boolean;
        /**
         * The 
         *  needed to remount a network volume, or 
         *  if not available.
         */
        volumeURLForRemounting: URL | null;
        /**
         * The volume’s persistent 
         *  as a string, or 
         *  if a persistent 
         *  is not available for the volume.
         */
        volumeUUIDString: string;
        /**
         * A Boolean value that indicates whether the item is in the iCloud storage.
         */
        isUbiquitousItem: boolean;
        /**
         * A Boolean value that indicates a shared item.
         */
        ubiquitousItemIsShared: boolean;
        /**
         * A Boolean value that indicates the system excludes the item from syncing.
         */
        ubiquitousItemIsExcludedFromSync: boolean;
        /**
         * The current user’s permissions for the shared item.
         */
        ubiquitousSharedItemCurrentUserPermissions: URLUbiquitousSharedItemPermissions | null;
        /**
         * The current user’s role for the shared item.
         */
        ubiquitousSharedItemCurrentUserRole: URLUbiquitousSharedItemRole | null;
        /**
         * The name components of the most recent editor of the shared item.
         */
        ubiquitousSharedItemMostRecentEditorNameComponents: PersonNameComponents | null;
        /**
         * The name components of the owner of the shared item.
         */
        ubiquitousSharedItemOwnerNameComponents: PersonNameComponents | null;
        /**
         * The name of the item’s container as the system displays it to users.
         */
        ubiquitousItemContainerDisplayName: string;
        /**
         * A Boolean value that indicates whether the user or the system requests a download of the item.
         */
        ubiquitousItemDownloadRequested: boolean;
        /**
         * The error when downloading the item from iCloud fails.
         */
        ubiquitousItemDownloadingError: NSError | null;
        /**
         * The download status of the item.
         */
        ubiquitousItemDownloadingStatus: URLUbiquitousItemDownloadingStatus | null;
        /**
         * A Boolean value that indicates whether the item has outstanding conflicts.
         */
        ubiquitousItemHasUnresolvedConflicts: boolean;
        /**
         * A Boolean value that indicates whether the system is downloading the item.
         */
        ubiquitousItemIsDownloading: boolean;
        /**
         * A Boolean value that indicates whether data is present in the cloud for the item.
         */
        ubiquitousItemIsUploaded: boolean;
        /**
         * A Boolean value that indicates whether the system is uploading the item.
         */
        ubiquitousItemIsUploading: boolean;
        /**
         * The error when uploading the item to iCloud fails.
         */
        ubiquitousItemUploadingError: NSError | null;
        /**
         * A thumbnail image of the URL.
         */
        thumbnail: NSImage | null;
        /**
         * A dictionary of UIKit image objects keyed by size.
         */
        thumbnailDictionary: Record<URLThumbnailDictionaryItem, UIImage> | null;
        /**
         * The date the resource was created, or renamed into or within its parent directory. 
         */
        addedToDirectoryDate: Date | null;
        /**
         * A loosely-typed dictionary containing all keys and values.
         */
        allValues: Record<URLResourceKey, any>;
        /**
         * The time the resource’s attributes were last modified.
         */
        attributeModificationDate: Date | null;
        /**
         * The URL’s path as a canonical absolute file system path.
         */
        canonicalPath: string;
        /**
         * The date the resource was last accessed.
         */
        contentAccessDate: Date | null;
        /**
         * The time the resource content was last modified.
         */
        contentModificationDate: Date | null;
        /**
         * The date the resource was created.
         */
        creationDate: Date | null;
        customIcon: NSImage | null;
        effectiveIcon: AnyObject | null;
        /**
         * An opaque generation identifier which can be compared using 
         *  to determine if the data in a document has been modified.
         */
        generationIdentifier: ((NSCopying & NSSecureCoding & NSObjectProtocol));
        /**
         * True for resources whose filename extension is removed from the localized name property.
         */
        hasHiddenExtension: boolean;
        /**
         * true if the resource is a Finder alias file or a symlink, false otherwise
         */
        isAliasFile: boolean;
        /**
         * True if resource should be excluded from backups, false otherwise.
         */
        isExcludedFromBackup: boolean;
        /**
         * True for resources normally not displayed to users.
         */
        isHidden: boolean;
        /**
         * True for packaged directories.
         */
        isPackage: boolean;
        /**
         * True if this process (as determined by EUID) can read the resource.
         */
        isReadable: boolean;
        /**
         * True for symlinks.
         */
        isSymbolicLink: boolean;
        /**
         * True for system-immutable resources.
         */
        isSystemImmutable: boolean;
        /**
         * True for user-immutable resources
         */
        isUserImmutable: boolean;
        /**
         * True if this process (as determined by EUID) can write to the resource.
         */
        isWritable: boolean;
        labelColor: NSColor | null;
        /**
         * The label number assigned to the resource.
         */
        labelNumber: number;
        /**
         * Number of hard links to the resource.
         */
        linkCount: number;
        /**
         * The user-visible label text.
         */
        localizedLabel: string;
        /**
         * Localized or extension-hidden name as displayed to users.
         */
        localizedName: string;
        /**
         * User-visible type or “kind” description.
         */
        localizedTypeDescription: string;
        /**
         * The resource name provided by the file system.
         */
        name: string;
        /**
         * The resource’s parent directory, if any.
         */
        parentDirectory: URL | null;
        /**
         * The URL’s path as a file system path.
         */
        path: string;
        /**
         * The optimal block size when reading or writing this file’s data, or nil if not available.
         */
        preferredIOBlockSize: number;
        /**
         * The quarantine properties as defined in LSQuarantine.h. To remove quarantine information from a file, pass 
         *  as the value when setting this property.
         */
        quarantineProperties: Record<string, any> | null;
        /**
         * The array of Tag names.
         */
        tagNames: string[] | null;
        /**
         * A string that represents the identifier for the type of the resource.
         */
        typeIdentifier: string;
        /**
         * The resource’s type.
         */
        contentType: UTType | null;
        /**
         * Initializes a new resource values structure.
         */
        constructor ();
    }
    export class NSMeasurement extends NSObject {
        /**
         * The unit of measure.
         */
        unit: Unit;
        /**
         * The measurement value, represented as a double-precision floating-point number.
         */
        doubleValue: number;
        /**
         * Initializes a new measurement with a specified double-precision floating-point value and unit.
         */
        constructor (doubleValue: number, unit: Unit);
        /**
         * Indicates whether the measurement can be converted to the given unit.
         */
        canBeConverted(unit: Unit):boolean;
        /**
         * Returns a measurement created by converting the receiver to the specified unit.
         */
        converting(unit: Unit):Measurement<Unit>;
        /**
         * Returns a new measurement by adding the receiver to the specified measurement.
         */
        adding(measurement: Measurement<Unit>):Measurement<Unit>;
        /**
         * Returns a new measurement by subtracting the specified measurement from the receiver.
         */
        subtracting(measurement: Measurement<Unit>):Measurement<Unit>;
    }
    export class Unit extends NSObject {
        /**
         * The symbolic representation of the unit.
         */
        symbol: string;
        /**
         * Initializes a new unit with the specified symbol.
         */
        constructor (symbol: string);
    }
    export class Dimension extends Unit {
        /**
         * The unit converter that represents the unit in terms of the dimension’s base unit.
         */
        converter: UnitConverter;
        /**
         * Initializes a dimensional unit with the symbol and unit converter you specify.
         */
        constructor (symbol: string, converter: UnitConverter);
        constructor (forLocale: Locale);
        /**
         * Returns the base unit.
         */
        static baseUnit():Self;
    }
    export class UnitConverter extends NSObject {
        /**
         * For a given unit, returns the specified value of that unit in terms of the base unit of its dimension.
         */
        baseUnitValue(value: number):number;
        /**
         * For a given unit, returns the specified value of the base unit in terms of that unit.
         */
        value(baseUnitValue: number):number;
    }
    export class UnitConverterLinear extends UnitConverter {
        /**
         * The coefficient to use in the linear unit conversion calculation.
         */
        coefficient: number;
        /**
         * The constant to use in the linear unit conversion calculation.
         */
        constant: number;
        /**
         * Initializes the unit converter with the coefficient you specify.
         */
        constructor (coefficient: number);
        /**
         * Creates a unit converter with the coefficient and constant you specify.
         */
        constructor (coefficient: number, constant: number);
    }
    export class UnitArea extends Dimension {
        /**
         * The square megameters unit of area.
         */
        static squareMegameters: UnitArea;
        /**
         * The square kilometers unit of area.
         */
        static squareKilometers: UnitArea;
        /**
         * The square meters unit of area.
         */
        static squareMeters: UnitArea;
        /**
         * The square centimeters unit of area.
         */
        static squareCentimeters: UnitArea;
        /**
         * The square millimeters unit of area.
         */
        static squareMillimeters: UnitArea;
        /**
         * The square micrometers unit of area.
         */
        static squareMicrometers: UnitArea;
        /**
         * The square nanometers unit of area.
         */
        static squareNanometers: UnitArea;
        /**
         * The square inches unit of area.
         */
        static squareInches: UnitArea;
        /**
         * The square feet unit of area.
         */
        static squareFeet: UnitArea;
        /**
         * The square yards unit of area.
         */
        static squareYards: UnitArea;
        /**
         * The square miles unit of area.
         */
        static squareMiles: UnitArea;
        /**
         * The acres unit of area.
         */
        static acres: UnitArea;
        /**
         * The ares unit of area.
         */
        static ares: UnitArea;
        /**
         * The hectares unit of area.
         */
        static hectares: UnitArea;
    }
    export class UnitLength extends Dimension {
        /**
         * The megameters unit of length.
         */
        static megameters: UnitLength;
        /**
         * The kilometers unit of length.
         */
        static kilometers: UnitLength;
        /**
         * The hectometers unit of length.
         */
        static hectometers: UnitLength;
        /**
         * The decameters unit of length.
         */
        static decameters: UnitLength;
        /**
         * The meters unit of length.
         */
        static meters: UnitLength;
        /**
         * The decimeters unit of length.
         */
        static decimeters: UnitLength;
        /**
         * The centimeters unit of length.
         */
        static centimeters: UnitLength;
        /**
         * The millimeters unit of length.
         */
        static millimeters: UnitLength;
        /**
         * The micrometers unit of length.
         */
        static micrometers: UnitLength;
        /**
         * The nanometers unit of length.
         */
        static nanometers: UnitLength;
        /**
         * The picometers unit of length.
         */
        static picometers: UnitLength;
        /**
         * The inches unit of length.
         */
        static inches: UnitLength;
        /**
         * The feet unit of length.
         */
        static feet: UnitLength;
        /**
         * The yards unit of length.
         */
        static yards: UnitLength;
        /**
         * The miles unit of length.
         */
        static miles: UnitLength;
        /**
         * The Scandinavian miles unit of length.
         */
        static scandinavianMiles: UnitLength;
        /**
         * The light years unit of length.
         */
        static lightyears: UnitLength;
        /**
         * The nautical miles unit of length.
         */
        static nauticalMiles: UnitLength;
        /**
         * The fathoms unit of length.
         */
        static fathoms: UnitLength;
        /**
         * The furlongs unit of length.
         */
        static furlongs: UnitLength;
        /**
         * The astronomical units unit of length.
         */
        static astronomicalUnits: UnitLength;
        /**
         * The parsecs unit of length.
         */
        static parsecs: UnitLength;
        constructor (forLocale: Locale, usage: MeasurementFormatUnitUsage<UnitLength>);
    }
    export class UnitVolume extends Dimension {
        /**
         * The megaliters unit of volume.
         */
        static megaliters: UnitVolume;
        /**
         * The kiloliters unit of volume.
         */
        static kiloliters: UnitVolume;
        /**
         * The liters unit of volume.
         */
        static liters: UnitVolume;
        /**
         * The deciliters unit of volume.
         */
        static deciliters: UnitVolume;
        /**
         * The centiliters unit of volume.
         */
        static centiliters: UnitVolume;
        /**
         * The milliliters unit of volume.
         */
        static milliliters: UnitVolume;
        /**
         * The cubic kilometers unit of volume.
         */
        static cubicKilometers: UnitVolume;
        /**
         * The cubic meters unit of volume.
         */
        static cubicMeters: UnitVolume;
        /**
         * The cubic decimeters unit of volume.
         */
        static cubicDecimeters: UnitVolume;
        /**
         * The cubic centimeters unit of volume.
         */
        static cubicCentimeters: UnitVolume;
        /**
         * The cubic millimeters unit of volume.
         */
        static cubicMillimeters: UnitVolume;
        /**
         * The cubic inches unit of volume.
         */
        static cubicInches: UnitVolume;
        /**
         * The cubic feet unit of volume.
         */
        static cubicFeet: UnitVolume;
        /**
         * The cubic yards unit of volume.
         */
        static cubicYards: UnitVolume;
        /**
         * The cubic miles unit of volume.
         */
        static cubicMiles: UnitVolume;
        /**
         * The acre feet unit of volume.
         */
        static acreFeet: UnitVolume;
        /**
         * The bushels unit of volume.
         */
        static bushels: UnitVolume;
        /**
         * The teaspoons unit of volume.
         */
        static teaspoons: UnitVolume;
        /**
         * The tablespoons unit of volume.
         */
        static tablespoons: UnitVolume;
        /**
         * The fluid ounces unit of volume.
         */
        static fluidOunces: UnitVolume;
        /**
         * The cups unit of volume.
         */
        static cups: UnitVolume;
        /**
         * The pints unit of volume.
         */
        static pints: UnitVolume;
        /**
         * The quarts unit of volume.
         */
        static quarts: UnitVolume;
        /**
         * The gallons unit of volume.
         */
        static gallons: UnitVolume;
        /**
         * The imperial teaspoons unit of volume.
         */
        static imperialTeaspoons: UnitVolume;
        /**
         * The imperial tablespoons unit of volume.
         */
        static imperialTablespoons: UnitVolume;
        /**
         * The imperial fluid ounces unit of volume.
         */
        static imperialFluidOunces: UnitVolume;
        /**
         * The imperial pints unit of volume.
         */
        static imperialPints: UnitVolume;
        /**
         * The imperial quarts unit of volume.
         */
        static imperialQuarts: UnitVolume;
        /**
         * The imperial gallons unit of volume.
         */
        static imperialGallons: UnitVolume;
        /**
         * The metric cups unit of volume.
         */
        static metricCups: UnitVolume;
        constructor (forLocale: Locale, usage: MeasurementFormatUnitUsage<UnitVolume>);
    }
    export class UnitAngle extends Dimension {
        /**
         * The degrees unit of angle.
         */
        static degrees: UnitAngle;
        /**
         * The arc minutes unit of angle.
         */
        static arcMinutes: UnitAngle;
        /**
         * The arc seconds unit of angle.
         */
        static arcSeconds: UnitAngle;
        /**
         * The radians unit of angle.
         */
        static radians: UnitAngle;
        /**
         * The gradians unit of angle.
         */
        static gradians: UnitAngle;
        /**
         * The revolutions unit of angle.
         */
        static revolutions: UnitAngle;
    }
    export class UnitMass extends Dimension {
        /**
         * The kilograms unit of mass.
         */
        static kilograms: UnitMass;
        /**
         * The grams unit of mass.
         */
        static grams: UnitMass;
        /**
         * The decigrams unit of mass.
         */
        static decigrams: UnitMass;
        /**
         * The centigrams unit of mass.
         */
        static centigrams: UnitMass;
        /**
         * The milligrams unit of mass.
         */
        static milligrams: UnitMass;
        /**
         * The micrograms unit of mass.
         */
        static micrograms: UnitMass;
        /**
         * The nanograms unit of mass.
         */
        static nanograms: UnitMass;
        /**
         * The picograms unit of mass.
         */
        static picograms: UnitMass;
        /**
         * The ounces unit of mass.
         */
        static ounces: UnitMass;
        /**
         * The pounds unit of mass.
         */
        static pounds: UnitMass;
        /**
         * The stone unit of mass.
         */
        static stones: UnitMass;
        /**
         * The metric tons unit of mass.
         */
        static metricTons: UnitMass;
        /**
         * The short tons unit of mass.
         */
        static shortTons: UnitMass;
        /**
         * The carats unit of mass.
         */
        static carats: UnitMass;
        /**
         * The ounces troy unit of mass.
         */
        static ouncesTroy: UnitMass;
        /**
         * The slugs unit of mass.
         */
        static slugs: UnitMass;
        constructor (forLocale: Locale, usage: MeasurementFormatUnitUsage<UnitMass>);
    }
    export class UnitPressure extends Dimension {
        /**
         * The gigapascals unit of pressure.
         */
        static gigapascals: UnitPressure;
        /**
         * The megapascals unit of pressure.
         */
        static megapascals: UnitPressure;
        /**
         * The kilopascals unit of pressure.
         */
        static kilopascals: UnitPressure;
        /**
         * The hectopascals unit of pressure.
         */
        static hectopascals: UnitPressure;
        /**
         * The inches of mercury unit of pressure.
         */
        static inchesOfMercury: UnitPressure;
        /**
         * The bars unit of pressure.
         */
        static bars: UnitPressure;
        /**
         * The millibars unit of pressure.
         */
        static millibars: UnitPressure;
        /**
         * The millimeters of mercury unit of pressure.
         */
        static millimetersOfMercury: UnitPressure;
        /**
         * The newtons per square meter unit of pressure.
         */
        static newtonsPerMetersSquared: UnitPressure;
        /**
         * The pounds per square inch unit of pressure.
         */
        static poundsForcePerSquareInch: UnitPressure;
    }
    export class UnitAcceleration extends Dimension {
        /**
         * Returns the meter per second squared unit of acceleration.
         */
        static metersPerSecondSquared: UnitAcceleration;
        /**
         * Returns the gravity unit of acceleration.
         */
        static gravity: UnitAcceleration;
    }
    export class UnitDuration extends Dimension {
        /**
         * The hour unit of duration.
         */
        static hours: UnitDuration;
        /**
         * The minute unit of duration.
         */
        static minutes: UnitDuration;
        /**
         * The second unit of duration.
         */
        static seconds: UnitDuration;
        /**
         * The millisecond unit of duration.
         */
        static milliseconds: UnitDuration;
        /**
         * The microsecond unit of duration.
         */
        static microseconds: UnitDuration;
        /**
         * The nanosecond unit of duration.
         */
        static nanoseconds: UnitDuration;
        /**
         * The picosecond unit of duration.
         */
        static picoseconds: UnitDuration;
    }
    export class UnitFrequency extends Dimension {
        /**
         * The terahertz unit of frequency.
         */
        static terahertz: UnitFrequency;
        /**
         * The gigahertz unit of frequency.
         */
        static gigahertz: UnitFrequency;
        /**
         * The megahertz unit of frequency.
         */
        static megahertz: UnitFrequency;
        /**
         * The kilohertz unit of frequency.
         */
        static kilohertz: UnitFrequency;
        /**
         * The hertz unit of frequency.
         */
        static hertz: UnitFrequency;
        /**
         * The millihertz unit of frequency.
         */
        static millihertz: UnitFrequency;
        /**
         * The microhertz unit of frequency.
         */
        static microhertz: UnitFrequency;
        /**
         * The nanohertz unit of frequency.
         */
        static nanohertz: UnitFrequency;
        /**
         * The frames per second unit of frequency.
         */
        static framesPerSecond: UnitFrequency;
    }
    export class UnitSpeed extends Dimension {
        /**
         * The meter per second unit of speed.
         */
        static metersPerSecond: UnitSpeed;
        /**
         * The kilometers per hour unit of speed.
         */
        static kilometersPerHour: UnitSpeed;
        /**
         * The miles per hour unit of speed.
         */
        static milesPerHour: UnitSpeed;
        /**
         * The knots unit of speed.
         */
        static knots: UnitSpeed;
    }
    export class UnitEnergy extends Dimension {
        /**
         * The kilojoules unit of energy.
         */
        static kilojoules: UnitEnergy;
        /**
         * The joules unit of energy.
         */
        static joules: UnitEnergy;
        /**
         * The kilocalories unit of energy.
         */
        static kilocalories: UnitEnergy;
        /**
         * The calories unit of energy.
         */
        static calories: UnitEnergy;
        /**
         * The kilowatt hours unit of energy.
         */
        static kilowattHours: UnitEnergy;
        constructor (forLocale: Locale, usage: MeasurementFormatUnitUsage<UnitEnergy>);
    }
    export class UnitPower extends Dimension {
        /**
         * The terawatts unit of power.
         */
        static terawatts: UnitPower;
        /**
         * The gigawatts unit of power.
         */
        static gigawatts: UnitPower;
        /**
         * The megawatts unit of power.
         */
        static megawatts: UnitPower;
        /**
         * The kilowatts unit of power.
         */
        static kilowatts: UnitPower;
        /**
         * The watts unit of power.
         */
        static watts: UnitPower;
        /**
         * The milliwatts unit of power.
         */
        static milliwatts: UnitPower;
        /**
         * The microwatts unit of power.
         */
        static microwatts: UnitPower;
        /**
         * The nanowatts unit of power.
         */
        static nanowatts: UnitPower;
        /**
         * The picowatts unit of power.
         */
        static picowatts: UnitPower;
        /**
         * The femtowatts unit of power.
         */
        static femtowatts: UnitPower;
        /**
         * The horsepower unit of power.
         */
        static horsepower: UnitPower;
    }
    export class UnitTemperature extends Dimension {
        /**
         * The kelvin unit of temperature.
         */
        static kelvin: UnitTemperature;
        /**
         * The degree Celsius unit of temperature.
         */
        static celsius: UnitTemperature;
        /**
         * The degree Fahrenheit unit of temperature.
         */
        static fahrenheit: UnitTemperature;
        constructor (forLocale: Locale, usage: MeasurementFormatUnitUsage<UnitTemperature>);
    }
    export class UnitIlluminance extends Dimension {
        /**
         * The lux unit of illuminance.
         */
        static lux: UnitIlluminance;
    }
    export class UnitElectricCharge extends Dimension {
        /**
         * The coulombs unit of electric charge.
         */
        static coulombs: UnitElectricCharge;
        /**
         * The megaampere hours unit of electric charge.
         */
        static megaampereHours: UnitElectricCharge;
        /**
         * The kiloampere hours unit of electric charge.
         */
        static kiloampereHours: UnitElectricCharge;
        /**
         * The ampere hours unit of electric charge.
         */
        static ampereHours: UnitElectricCharge;
        /**
         * The milliampere hours unit of electric charge.
         */
        static milliampereHours: UnitElectricCharge;
        /**
         * The microampere hours unit of electric charge.
         */
        static microampereHours: UnitElectricCharge;
    }
    export class UnitElectricCurrent extends Dimension {
        /**
         * The megaamperes unit of electric current.
         */
        static megaamperes: UnitElectricCurrent;
        /**
         * The kiloamperes unit of electric current.
         */
        static kiloamperes: UnitElectricCurrent;
        /**
         * The amperes unit of electric current.
         */
        static amperes: UnitElectricCurrent;
        /**
         * The milliamperes unit of electric current.
         */
        static milliamperes: UnitElectricCurrent;
        /**
         * The microamperes unit of electric current.
         */
        static microamperes: UnitElectricCurrent;
    }
    export class UnitElectricPotentialDifference extends Dimension {
        /**
         * The megavolts unit of electric potential difference.
         */
        static megavolts: UnitElectricPotentialDifference;
        /**
         * The kilovolts unit of electric potential difference.
         */
        static kilovolts: UnitElectricPotentialDifference;
        /**
         * The volts unit of electric potential difference.
         */
        static volts: UnitElectricPotentialDifference;
        /**
         * The millivolts unit of electric potential difference.
         */
        static millivolts: UnitElectricPotentialDifference;
        /**
         * The microvolts unit of electric potential difference.
         */
        static microvolts: UnitElectricPotentialDifference;
    }
    export class UnitElectricResistance extends Dimension {
        /**
         * The megaohms unit of electric resistance.
         */
        static megaohms: UnitElectricResistance;
        /**
         * The kiloohms unit of electric resistance.
         */
        static kiloohms: UnitElectricResistance;
        /**
         * The ohms unit of electric resistance.
         */
        static ohms: UnitElectricResistance;
        /**
         * The milliohms unit of electric resistance.
         */
        static milliohms: UnitElectricResistance;
        /**
         * The microohms unit of electric resistance.
         */
        static microohms: UnitElectricResistance;
    }
    export class UnitConcentrationMass extends Dimension {
        /**
         * The grams per liter unit of concentration.
         */
        static gramsPerLiter: UnitConcentrationMass;
        /**
         * The milligrams per deciliter unit of concentration.
         */
        static milligramsPerDeciliter: UnitConcentrationMass;
        /**
         * Returns the millimoles per liter unit with the specified number of grams per mole.
         */
        static millimolesPerLiter(gramsPerMole: number):UnitConcentrationMass;
    }
    export class UnitDispersion extends Dimension {
        /**
         * The parts per million unit.
         */
        static partsPerMillion: UnitDispersion;
    }
    export class UnitFuelEfficiency extends Dimension {
        /**
         * The miles per imperial gallon unit of fuel efficiency.
         */
        static milesPerImperialGallon: UnitFuelEfficiency;
        /**
         * The liters per 100 kilometers unit of fuel efficiency.
         */
        static litersPer100Kilometers: UnitFuelEfficiency;
        /**
         * The miles per gallon unit of fuel efficiency.
         */
        static milesPerGallon: UnitFuelEfficiency;
    }
    export class UnitInformationStorage extends Dimension {
        /**
         * The bits unit of information.
         */
        static bits: UnitInformationStorage;
        /**
         * The nibbles unit of information.
         */
        static nibbles: UnitInformationStorage;
        /**
         * The bytes unit of information.
         */
        static bytes: UnitInformationStorage;
        /**
         * The kibibits unit of information.
         */
        static kibibits: UnitInformationStorage;
        /**
         * The kibibytes unit of information.
         */
        static kibibytes: UnitInformationStorage;
        /**
         * The mebibits unit of information.
         */
        static mebibits: UnitInformationStorage;
        /**
         * The mebibytes unit of information.
         */
        static mebibytes: UnitInformationStorage;
        /**
         * The gibibits unit of information.
         */
        static gibibits: UnitInformationStorage;
        /**
         * The gibibytes unit of information.
         */
        static gibibytes: UnitInformationStorage;
        /**
         * The tebibits unit of information.
         */
        static tebibits: UnitInformationStorage;
        /**
         * The tebibytes unit of information.
         */
        static tebibytes: UnitInformationStorage;
        /**
         * The pebibits unit of information.
         */
        static pebibits: UnitInformationStorage;
        /**
         * The pebibytes unit of information.
         */
        static pebibytes: UnitInformationStorage;
        /**
         * The exbibits unit of information.
         */
        static exbibits: UnitInformationStorage;
        /**
         * The exbibytes unit of information.
         */
        static exbibytes: UnitInformationStorage;
        /**
         * The zebibits unit of information.
         */
        static zebibits: UnitInformationStorage;
        /**
         * The zebibytes unit of information.
         */
        static zebibytes: UnitInformationStorage;
        /**
         * The yobibits unit of information.
         */
        static yobibits: UnitInformationStorage;
        /**
         * The yobibytes unit of information.
         */
        static yobibytes: UnitInformationStorage;
        /**
         * The kilobits unit of information.
         */
        static kilobits: UnitInformationStorage;
        /**
         * The kilobytes unit of information.
         */
        static kilobytes: UnitInformationStorage;
        /**
         * The megabits unit of information.
         */
        static megabits: UnitInformationStorage;
        /**
         * The megabytes unit of information.
         */
        static megabytes: UnitInformationStorage;
        /**
         * The gigabits unit of information.
         */
        static gigabits: UnitInformationStorage;
        /**
         * The gigabytes unit of information.
         */
        static gigabytes: UnitInformationStorage;
        /**
         * The terabits unit of information.
         */
        static terabits: UnitInformationStorage;
        /**
         * The terrabytes unit of information.
         */
        static terabytes: UnitInformationStorage;
        /**
         * The petabits unit of information.
         */
        static petabits: UnitInformationStorage;
        /**
         * The petabytes unit of information.
         */
        static petabytes: UnitInformationStorage;
        /**
         * The exabits unit of information.
         */
        static exabits: UnitInformationStorage;
        /**
         * The exabytes unit of information.
         */
        static exabytes: UnitInformationStorage;
        /**
         * The zettabits unit of information.
         */
        static zettabits: UnitInformationStorage;
        /**
         * The zettabytes unit of information.
         */
        static zettabytes: UnitInformationStorage;
        /**
         * The yottabits unit of information.
         */
        static yottabits: UnitInformationStorage;
        /**
         * The yottabytes unit of information.
         */
        static yottabytes: UnitInformationStorage;
    }
    export class PersonNameComponentsFormatter extends Formatter {
        /**
         * The formatting style of the receiver.
         */
        style: PersonNameComponentsFormatter.Style;
        /**
         * A Boolean value that specifies whether the receiver should use only the phonetic representations of name components. 
         *  by default.
         */
        isPhonetic: boolean;
        locale: Locale!;
        /**
         * Returns a string formatted for a given 
         *  object using the provided style and options.
         */
        static localizedString(components: PersonNameComponents, nameFormatStyle: PersonNameComponentsFormatter.Style, nameOptions: PersonNameComponentsFormatter.Options):string;
        /**
         * Returns a string formatted for a given 
         *  object.
         */
        string(components: PersonNameComponents):string;
        /**
         * Returns an attributed string formatted for a given 
         *  object, with attribute annotations for each component.
         */
        annotatedString(components: PersonNameComponents):NSAttributedString;
        /**
         * Returns a person name components object from a given string.
         */
        personNameComponents(_string: string):PersonNameComponents | null;
        /**
         * Returns by reference a person name components object after creating it from a given string.
         */
        getObjectValue(obj: AutoreleasingUnsafeMutablePointer<AnyObject?> | null, _string: string, error: AutoreleasingUnsafeMutablePointer<NSString?> | null):boolean;
    }
    /**
     * the same name namespace of class PersonNameComponentsFormatter
     */
    export namespace PersonNameComponentsFormatter {
        export enum Style {
            default = 0,
            short = 1,
            medium = 2,
            long = 3,
            abbreviated = 4,
        }
        export class Options {
            /**
             * The formatter should format the component object’s 
             *  components instead of its own components.
             */
            static phonetic: PersonNameComponentsFormatter.Options;
            constructor (rawValue: number);
        }
    }
    export class PersonNameComponents {
        /**
         * The portion of a name’s full form of address that precedes the name itself.
         */
        namePrefix: string;
        /**
         * Name bestowed upon an individual to differentiate them from other members of a group that share a family name.
         */
        givenName: string;
        /**
         * Secondary name bestowed upon an individual to differentiate them from others that have the same given name.
         */
        middleName: string;
        /**
         * Name bestowed upon an individual to denote membership in a group or family.
         */
        familyName: string;
        /**
         * The portion of a name’s full form of address that follows the name itself.
         */
        nameSuffix: string;
        /**
         * Name substituted for the purposes of familiarity.
         */
        nickname: string;
        /**
         * The phonetic representation name components of the receiver. 
         */
        phoneticRepresentation: PersonNameComponents | null;
        /**
         * A textual description of the structure.
         */
        description: string;
        /**
         * A textual description of the structure suitable for debugging.
         */
        debugDescription: string;
        /**
         * A mirror that reflects the date interval.
         */
        customMirror: Mirror;
        /**
         * The computed hash value for the date interval.
         */
        hashValue: number;
        /**
         * Initializes a new person name components structure.
         */
        constructor ();
        /**
         * Creates a new instance by decoding from the given decoder.
         */
        constructor (decoder: Decoder);
        /**
         * Creates a person name components object from a given string.
         */
        constructor (value: string);
        constructor (namePrefix: string, givenName: string, middleName: string, familyName: string, nameSuffix: string, nickname: string, phoneticRepresentation: PersonNameComponents | null);
        /**
         * Generates a locale-aware string representation of an instance of person name components using the default format style.
         */
        formatted():string;
        /**
         * Encodes this value into the given encoder.
         */
        encode(encoder: Encoder):void;
        /**
         * Hashes the essential components of this value by feeding them into the given hasher.
         */
        hash(hasher: Hasher):void;
    }
    /**
     * the same name namespace of class PersonNameComponents
     */
    export namespace PersonNameComponents {
        export class FormatStyle {
            /**
             * The strategy used to parse a string into person name components.
             */
            parseStrategy: PersonNameComponents.ParseStrategy;
            /**
             * Specifies the style of the formatted result.
             */
            style: PersonNameComponents.FormatStyle.Style;
            /**
             * The locale to use when formatting the person name components.
             */
            locale: Locale;
            /**
             * The style used to create a locale-aware attributed string representation of an instance of person name components.
             */
            attributed: PersonNameComponents.AttributedStyle;
            /**
             * The hash value of the style format.
             */
            hashValue: number;
            /**
             * Creates an instance using the provided format style and locale.
             */
            constructor (style: PersonNameComponents.FormatStyle.Style, locale: Locale);
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            /**
             * Creates a person name components format style using the provided format style.
             */
            static name(style: Style):PersonNameComponents.FormatStyle;
            static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
            /**
             * Creates a string representation from a person name components value.
             */
            format(value: PersonNameComponents):string;
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
        /**
         * the same name namespace of class FormatStyle
         */
        export namespace FormatStyle {
            export enum Style {
                abbreviated,
                long,
                medium,
                short,
            }
        }
        export class AttributedStyle {
            hashValue: number;
            locale: Locale;
            style: PersonNameComponents.FormatStyle.Style;
            constructor (decoder: Decoder);
            constructor (style: PersonNameComponents.FormatStyle.Style, locale: Locale);
            static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
            encode(encoder: Encoder):void;
            format(value: PersonNameComponents):AttributedString;
            hash(hasher: Hasher):void;
        }
        export class ParseStrategy {
            static name: PersonNameComponents.ParseStrategy;
            hashValue: number;
            constructor ();
            constructor (decoder: Decoder);
            static fixed(format: Date.FormatString, timeZone: TimeZone, locale: Locale | null):Date.ParseStrategy;
            encode(encoder: Encoder):void;
            hash(hasher: Hasher):void;
            parse(value: string):PersonNameComponents;
        }
    }
    export class NSPersonNameComponents extends NSObject {
        /**
         * The portion of a name’s full form of address that precedes the name itself 
         * .
         */
        namePrefix: string;
        /**
         * Name bestowed upon an individual to differentiate them from other members of a group that share a family name 
         * .
         */
        givenName: string;
        /**
         * Secondary name bestowed upon an individual to differentiate them from others that have the same given name 
         * .
         */
        middleName: string;
        /**
         * Name bestowed upon an individual to denote membership in a group or family. 
         * .
         */
        familyName: string;
        /**
         * The portion of a name’s full form of address that follows the name itself 
         * .
         */
        nameSuffix: string;
        /**
         * Name substituted for the purposes of familiarity 
         * .
         */
        nickname: string;
        /**
         * The phonetic representation name components of the receiver. 
         */
        phoneticRepresentation: PersonNameComponents | null;
    }
    export class RelativeDateTimeFormatter extends Formatter {
        /**
         * The calendar to use for formatting values that don’t have an inherent calendar of their own.
         */
        calendar: Calendar!;
        /**
         * The locale to use when formatting the date.
         */
        locale: Locale!;
        /**
         * The style to use when describing a relative date, for example “yesterday” or “1 day ago”.
         */
        dateTimeStyle: RelativeDateTimeFormatter.DateTimeStyle;
        /**
         * The style to use when formatting the quantity or the name of the unit, such as “1 day ago” or “one day ago”.
         */
        unitsStyle: RelativeDateTimeFormatter.UnitsStyle;
        /**
         * A description of where the formatted string will appear, allowing the formatter to capitalize the output appropriately.
         */
        formattingContext: Formatter.Context;
        /**
         * Formats the date interval from the reference date to the specified date using the formatter’s calendar.
         */
        localizedString(date: Date, referenceDate: Date):string;
        /**
         * Creates a formatted string for a date relative to the current date and time.
         */
        string(obj: any):string;
    }
    /**
     * the same name namespace of class RelativeDateTimeFormatter
     */
    export namespace RelativeDateTimeFormatter {
        export enum DateTimeStyle {
            named = 1,
            numeric = 0,
        }
        export enum UnitsStyle {
            abbreviated = 3,
            full = 0,
            short = 2,
            spellOut = 1,
        }
    }
    export class ByteCountFormatter extends Formatter {
        /**
         * Specify the formatting context for the formatted string.
         */
        formattingContext: Formatter.Context;
        /**
         * Specify the number of bytes to be used for kilobytes.
         */
        countStyle: ByteCountFormatter.CountStyle;
        /**
         * Determines whether to allow more natural display of some values.
         */
        allowsNonnumericFormatting: boolean;
        /**
         * Determines whether to include the number of bytes after the formatted string.
         */
        includesActualByteCount: boolean;
        /**
         * Determines the display style of the size representation.
         */
        isAdaptive: boolean;
        /**
         * Specify the units that can be used in the output.
         */
        allowedUnits: ByteCountFormatter.Units;
        /**
         * Determines whether to include the count in the resulting formatted string.
         */
        includesCount: boolean;
        /**
         * Determines whether to include the units in the resulting formatted string.
         */
        includesUnit: boolean;
        /**
         * Determines whether to zero pad fraction digits so a consistent number of characters is displayed in a representation.
         */
        zeroPadsFractionDigits: boolean;
        /**
         * Converts a byte count into the specified string format without creating an 
         *  object.
         */
        static string(byteCount: Int64, countStyle: ByteCountFormatter.CountStyle):string;
    }
    /**
     * the same name namespace of class ByteCountFormatter
     */
    export namespace ByteCountFormatter {
        export class Units {
            /**
             * Displays bytes in the formatter content.
             */
            static useBytes: ByteCountFormatter.Units;
            /**
             * Displays kilobytes in the formatter content.
             */
            static useKB: ByteCountFormatter.Units;
            /**
             * Displays megabytes in the formatter content.
             */
            static useMB: ByteCountFormatter.Units;
            /**
             * Displays gigabytes in the formatter content.
             */
            static useGB: ByteCountFormatter.Units;
            /**
             * Displays terabytes in the formatter content.
             */
            static useTB: ByteCountFormatter.Units;
            /**
             * Displays petabyte in the formatter content.
             */
            static usePB: ByteCountFormatter.Units;
            /**
             * Displays exabytes in the formatter content.
             */
            static useEB: ByteCountFormatter.Units;
            /**
             * Displays zettabytes in the formatter content.
             */
            static useZB: ByteCountFormatter.Units;
            /**
             * Displays yottabytes in the formatter content.
             */
            static useYBOrHigher: ByteCountFormatter.Units;
            /**
             * Can use any unit in the formatter content.
             */
            static useAll: ByteCountFormatter.Units;
            constructor (rawValue: number);
        }
        export enum CountStyle {
            file = 0,
            memory = 1,
            decimal = 2,
            binary = 3,
        }
    }
    export class MeasurementFormatter extends Formatter {
        /**
         * The options for how the unit is formatted.
         */
        unitOptions: MeasurementFormatter.UnitOptions;
        /**
         * The unit style.
         */
        unitStyle: Formatter.UnitStyle;
        /**
         * The locale of the formatter.
         */
        locale: Locale!;
        /**
         * The number formatter used to format the quantity of a measurement.
         */
        numberFormatter: NumberFormatter!;
        /**
         * Creates and returns a localized string representation of the provided measurement.
         */
        string(measurement: Measurement<Unit>):string;
    }
    /**
     * the same name namespace of class MeasurementFormatter
     */
    export namespace MeasurementFormatter {
        export class UnitOptions {
            static providedUnit: MeasurementFormatter.UnitOptions;
            static naturalScale: MeasurementFormatter.UnitOptions;
            static temperatureWithoutUnit: MeasurementFormatter.UnitOptions;
            constructor (rawValue: number);
        }
    }
    export class ListFormatter extends Formatter {
        /**
         * An object that formats each item in the list.
         */
        itemFormatter: Formatter | null;
        /**
         * The locale to use when formatting items in the list.
         */
        locale: Locale!;
        /**
         * Constructs a formatted string from an array of strings that uses the list format specific to the current locale.
         */
        static localizedString(strings: string[]):string;
        /**
         * Creates a formatted string for an array of items.
         */
        string(items: any[]):string;
    }
    export class URLComponents {
        /**
         * A URL created from the components.
         */
        url: URL | null;
        /**
         * A URL derived from the components object, in string form.
         */
        string: string;
        /**
         * The fragment subcomponent.
         */
        fragment: string;
        /**
         * The host subcomponent.
         */
        host: string;
        /**
         * The password subcomponent of the URL.
         */
        password: string;
        /**
         * The path subcomponent.
         */
        path: string;
        /**
         * The port subcomponent.
         */
        port: number;
        /**
         * The query subcomponent.
         */
        query: string;
        /**
         * An array of query items for the URL in the order in which they appear in the original query string.
         */
        queryItems: URLQueryItem[] | null;
        /**
         * The scheme subcomponent of the URL.
         */
        scheme: string;
        /**
         * The user subcomponent of the URL.
         */
        user: string;
        /**
         * The fragment subcomponent, percent-encoded.
         */
        percentEncodedFragment: string;
        /**
         * The host subcomponent, percent-encoded.
         */
        percentEncodedHost: string;
        /**
         * The password subcomponent, percent-encoded.
         */
        percentEncodedPassword: string;
        /**
         * The path subcomponent, percent-encoded.
         */
        percentEncodedPath: string;
        /**
         * The query subcomponent, percent-encoded.
         */
        percentEncodedQuery: string;
        /**
         * The user subcomponent, percent-encoded.
         */
        percentEncodedUser: string;
        /**
         * Returns the character range of the fragment in the string returned by the string property.
         */
        rangeOfFragment: Range<String.Index> | null;
        /**
         * Returns the character range of the host in the string returned by the string property.
         */
        rangeOfHost: Range<String.Index> | null;
        /**
         * Returns the character range of the password in the string returned by the string property.
         */
        rangeOfPassword: Range<String.Index> | null;
        /**
         * Returns the character range of the path in the string returned by the string property.
         */
        rangeOfPath: Range<String.Index> | null;
        /**
         * Returns the character range of the port in the string returned by the string property.
         */
        rangeOfPort: Range<String.Index> | null;
        /**
         * Returns the character range of the query in the string returned by the string property.
         */
        rangeOfQuery: Range<String.Index> | null;
        /**
         * Returns the character range of the scheme in the string returned by the string property.
         */
        rangeOfScheme: Range<String.Index> | null;
        /**
         * Returns the character range of the user in the string returned by the string property.
         */
        rangeOfUser: Range<String.Index> | null;
        /**
         * A textual description of the URL components.
         */
        description: string;
        /**
         * A textual description of the URL components suitable for debugging.
         */
        debugDescription: string;
        /**
         * A mirror that reflects the URL components.
         */
        customMirror: Mirror;
        /**
         * The computed hash value for the URL components.
         */
        hashValue: number;
        encodedHost: string;
        percentEncodedQueryItems: URLQueryItem[] | null;
        /**
         * Initializes with all components undefined.
         */
        constructor ();
        constructor (decoder: Decoder);
        constructor (_string: string);
        constructor (url: URL, resolve: boolean);
        encode(encoder: Encoder):void;
        hash(hasher: Hasher):void;
    }
    export class NSURLComponents extends NSObject {
        /**
         * A URL derived from the components object, in string form.
         */
        string: string;
        /**
         * A URL object derived from the components object.
         */
        url: URL | null;
        /**
         * The fragment URL component (the part after a 
         *  symbol), or 
         *  if not present.
         */
        fragment: string;
        /**
         * The host URL subcomponent, or 
         *  if not present.
         */
        host: string;
        /**
         * The password URL subcomponent, or 
         *  if not present.
         */
        password: string;
        /**
         * The path URL component, or 
         *  if not present.
         */
        path: string;
        /**
         * The port number URL component, or 
         *  if not present.
         */
        port: NSNumber | null;
        /**
         * The query URL component as a string, or 
         *  if not present.
         */
        query: string;
        /**
         * The query URL component as an array of name/value pairs.
         */
        queryItems: URLQueryItem[] | null;
        /**
         * The scheme URL component, or 
         *  if not present.
         */
        scheme: string;
        /**
         * The username URL subcomponent, or 
         *  if not present.
         */
        user: string;
        /**
         * The fragment URL component (the part after a 
         *  symbol) expressed as a URL-encoded string, or 
         *  if not present.
         */
        percentEncodedFragment: string;
        /**
         * The host URL subcomponent expressed as a URL-encoded string, or 
         *  if not present.
         */
        percentEncodedHost: string;
        /**
         * The password URL subcomponent expressed as a URL-encoded string, or 
         *  if not present.
         */
        percentEncodedPassword: string;
        /**
         * The path URL component expressed as a URL-encoded string, or 
         *  if not present.
         */
        percentEncodedPath: string;
        /**
         * The query URL component expressed as a URL-encoded string, or 
         *  if not present.
         */
        percentEncodedQuery: string;
        /**
         * The username URL subcomponent expressed as a URL-encoded string, or 
         *  if not present.
         */
        percentEncodedUser: string;
        percentEncodedQueryItems: URLQueryItem[] | null;
        /**
         * Returns the character range of the fragment in the string returned by the string property.
         */
        rangeOfFragment: NSRange;
        /**
         * Returns the character range of the host in the string returned by the string property.
         */
        rangeOfHost: NSRange;
        /**
         * Returns the character range of the password in the string returned by the string property.
         */
        rangeOfPassword: NSRange;
        /**
         * Returns the character range of the path in the string returned by the string property.
         */
        rangeOfPath: NSRange;
        /**
         * Returns the character range of the port in the string returned by the string property.
         */
        rangeOfPort: NSRange;
        /**
         * Returns the character range of the query in the string returned by the string property.
         */
        rangeOfQuery: NSRange;
        /**
         * Returns the character range of the scheme in the string returned by the string property.
         */
        rangeOfScheme: NSRange;
        /**
         * Returns the character range of the user in the string returned by the string property.
         */
        rangeOfUser: NSRange;
        encodedHost: string;
        /**
         * Initializes a URL components object with 
         *  for every component.
         */
        constructor ();
        /**
         * Initializes a URL components object by parsing a URL in string form.
         */
        constructor (URLString: string);
        /**
         * Initializes a URL components object by parsing the URL from an 
         *  object.
         */
        constructor (url: URL, resolve: boolean);
    }
    export class URLQueryItem {
        /**
         * The name of the query item.
         */
        name: string;
        /**
         * The value for the query item.
         */
        value: string;
        /**
         * A textual description of the query item.
         */
        description: string;
        /**
         * A textual description of the query item, suitable for debugging.
         */
        debugDescription: string;
        /**
         * A mirror that reflects the query item.
         */
        customMirror: Mirror;
        /**
         * The computed hash value for the query item.
         */
        hashValue: number;
        /**
         * Creates a new query item with the name and value you specify.
         */
        constructor (name: string, value: string);
        /**
         * Hashes the essential components of this value by feeding them into the specified hasher.
         */
        hash(hasher: Hasher):void;
    }
    export class NSURLQueryItem extends NSObject {
        /**
         * The name of the query item.
         */
        name: string;
        /**
         * The value for the query item.
         */
        value: string;
        /**
         * Initializes a newly allocated query item with the specified name and value.
         */
        constructor (name: string, value: string);
    }
    export class Formatter extends NSObject {
        /**
         * The default implementation of this method raises an exception.
         */
        string(obj: any):string;
        /**
         * The default implementation returns 
         *  to indicate that the formatter object does not provide an attributed string.
         */
        attributedString(obj: any, attrs: Record<NSAttributedString.Key, any> | null):NSAttributedString | null;
        /**
         * The default implementation of this method invokes 
         * .
         */
        editingString(obj: any):string;
        /**
         * The default implementation of this method raises an exception.
         */
        getObjectValue(obj: AutoreleasingUnsafeMutablePointer<AnyObject?> | null, _string: string, error: AutoreleasingUnsafeMutablePointer<NSString?> | null):boolean;
        /**
         * Returns a Boolean value that indicates whether a partial string is valid.
         */
        isPartialStringValid(partialString: string, newString: AutoreleasingUnsafeMutablePointer<NSString?> | null, error: AutoreleasingUnsafeMutablePointer<NSString?> | null):boolean;
    }
    /**
     * the same name namespace of class Formatter
     */
    export namespace Formatter {
        export enum Context {
            unknown = 0,
            dynamic = 1,
            standalone = 2,
            listItem = 3,
            beginningOfSentence = 4,
            middleOfSentence = 5,
        }
        export enum UnitStyle {
            short = 1,
            medium = 2,
            long = 3,
        }
    }
    export interface FormatStyle extends Decodable,Encodable,Hashable {
        format(value: Self.FormatInput):Self.FormatOutput;
        locale(locale: Locale):Self;
        static number: Decimal.FormatStyle;
        static percent: Decimal.FormatStyle.Percent;
        static currency(code: string):Self;
        static interval: Date.IntervalFormatStyle;
        static iso8601: Date.ISO8601FormatStyle;
        static timeDuration: Date.ComponentsFormatStyle;
        static dateTime: Date.FormatStyle;
        static url: URL.FormatStyle;
        static byteCount(style: ByteCountFormatStyle.Style, allowedUnits: ByteCountFormatStyle.Units, spellsOutZero: boolean, includesActualByteCount: boolean):Self;
        static components(style: Date.ComponentsFormatStyle.Style, fields: Set<Date.ComponentsFormatStyle.Field> | null):Self;
        static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
        static name(style: PersonNameComponents.FormatStyle.Style):Self;
        static relative(presentation: Date.RelativeFormatStyle.Presentation, unitsStyle: Date.RelativeFormatStyle.UnitsStyle):Self;
        static time(pattern: Duration.TimeFormatStyle.Pattern):Self;
        static units(units: Set<Duration.UnitsFormatStyle.Unit>, width: Duration.UnitsFormatStyle.UnitWidth, maximumUnitCount: number, zeroValueUnits: Duration.UnitsFormatStyle.ZeroValueUnitsDisplayStrategy, valueLength: number, fractionalPart: Duration.UnitsFormatStyle.FractionalPartDisplayStrategy):Self;
        static verbatim(format: Date.FormatString, locale: Locale | null, timeZone: TimeZone, calendar: Calendar):Date.VerbatimFormatStyle;
    }
    export class UUID {
        /**
         * Returns the UUID as bytes.
         */
        uuid: any;
        /**
         * Returns a string created from the UUID, such as “E621E1F8-C36C-495A-93FC-0C247A3E6E5F”
         */
        uuidString: string;
        /**
         * A textual description of the UUID.
         */
        description: string;
        /**
         * A textual description of the UUID suitable for debugging.
         */
        debugDescription: string;
        /**
         * A mirror that reflects the UUID.
         */
        customMirror: Mirror;
        /**
         * The computed hash value of the UUID.
         */
        hashValue: number;
        entityIdentifierString: string;
        /**
         * Creates a UUID with RFC 4122 version 4 random bytes.
         */
        constructor ();
        /**
         * Creates a UUID from the uuid C-language structure.
         */
        constructor (uuid: uuid_t);
        /**
         * Creates a UUID from a string representation.
         */
        constructor (_string: string);
        /**
         * Creates a UUID by decoding from the specified decoder.
         */
        constructor (decoder: Decoder);
        static entityIdentifier(entityIdentifierString: string):UUID | null;
        /**
         * Encodes the UUID into the specified encoder.
         */
        encode(encoder: Encoder):void;
        /**
         * Hashes the essential components of the value by feeding them into the specified hasher.
         */
        hash(hasher: Hasher):void;
    }
    export class FormatStyleCapitalizationContext {
        static unknown: FormatStyleCapitalizationContext;
        static standalone: FormatStyleCapitalizationContext;
        static listItem: FormatStyleCapitalizationContext;
        static beginningOfSentence: FormatStyleCapitalizationContext;
        static middleOfSentence: FormatStyleCapitalizationContext;
        hashValue: number;
        constructor (decoder: Decoder);
        encode(encoder: Encoder):void;
        hash(hasher: Hasher):void;
    }
    export class NSUUID extends NSObject {
        /**
         * The UUID as a string.
         */
        uuidString: string;
        /**
         * Initializes a new UUID with RFC 4122 version 4 random bytes.
         */
        constructor ();
        /**
         * Initializes a new UUID with the formatted string.
         */
        constructor (_string: string);
        /**
         * Initializes a new UUID with the given bytes.
         */
        constructor (bytes: UnsafePointer<UInt8> | null);
        /**
         * Returns the UUID as bytes.
         */
        getBytes(uuid: UnsafeMutablePointer<UInt8>):void;
        compare(otherUUID: UUID):ComparisonResult;
    }
    export class ByteCountFormatStyle {
        allowedUnits: ByteCountFormatStyle.Units;
        attributed: ByteCountFormatStyle.Attributed;
        hashValue: number;
        includesActualByteCount: boolean;
        locale: Locale;
        spellsOutZero: boolean;
        style: ByteCountFormatStyle.Style;
        constructor (decoder: Decoder);
        constructor (style: ByteCountFormatStyle.Style, allowedUnits: ByteCountFormatStyle.Units, spellsOutZero: boolean, includesActualByteCount: boolean, locale: Locale);
        static byteCount(style: Style, allowedUnits: Units, spellsOutZero: boolean, includesActualByteCount: boolean):ByteCountFormatStyle;
        static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
        encode(encoder: Encoder):void;
        format(value: Int64):string;
        hash(hasher: Hasher):void;
    }
    /**
     * the same name namespace of class ByteCountFormatStyle
     */
    export namespace ByteCountFormatStyle {
        export class Attributed {
            allowedUnits: ByteCountFormatStyle.Units;
            hashValue: number;
            includesActualByteCount: boolean;
            locale: Locale;
            spellsOutZero: boolean;
            style: ByteCountFormatStyle.Style;
            constructor (decoder: Decoder);
            static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
            encode(encoder: Encoder):void;
            format(value: Int64):AttributedString;
            hash(hasher: Hasher):void;
        }
        export class Units {
            static all: ByteCountFormatStyle.Units;
            static bytes: ByteCountFormatStyle.Units;
            static default: ByteCountFormatStyle.Units;
            static eb: ByteCountFormatStyle.Units;
            static gb: ByteCountFormatStyle.Units;
            static kb: ByteCountFormatStyle.Units;
            static mb: ByteCountFormatStyle.Units;
            static pb: ByteCountFormatStyle.Units;
            static tb: ByteCountFormatStyle.Units;
            static ybOrHigher: ByteCountFormatStyle.Units;
            static zb: ByteCountFormatStyle.Units;
            hashValue: number;
            /**
             * A Boolean value that indicates whether the set has no elements.
             */
            isEmpty: boolean;
            rawValue: number;
            /**
             * Creates an empty option set.
             */
            constructor ();
            /**
             * Creates a set containing the elements of the given array literal.
             */
            constructor (...arrayLiteral: ByteCountFormatStyle.Units);
            /**
             * Creates a new instance by decoding from the given decoder, when the type’s 
             *  is 
             * .
             */
            constructor (decoder: Decoder);
            constructor (rawValue: number);
            /**
             * Returns a Boolean value that indicates whether a given element is a member of the option set.
             */
            contains(member: ByteCountFormatStyle.Units):boolean;
            /**
             * Encodes this value into the given encoder, when the type’s 
             *  is 
             * .
             */
            encode(encoder: Encoder):void;
            /**
             * Removes all elements of this option set that are not also present in the given set.
             */
            formIntersection(other: ByteCountFormatStyle.Units):void;
            /**
             * Replaces this set with a new set containing all elements contained in either this set or the given set, but not in both.
             */
            formSymmetricDifference(other: ByteCountFormatStyle.Units):void;
            /**
             * Inserts the elements of another set into this option set.
             */
            formUnion(other: ByteCountFormatStyle.Units):void;
            hash(hasher: Hasher):void;
            /**
             * Adds the given element to the option set if it is not already a member.
             */
            insert(newMember: ByteCountFormatStyle.Units):any;
            /**
             * Returns a new option set with only the elements contained in both this set and the given set.
             */
            intersection(other: ByteCountFormatStyle.Units):ByteCountFormatStyle.Units;
            /**
             * Returns a Boolean value that indicates whether the set has no members in common with the given set.
             */
            isDisjoint(other: ByteCountFormatStyle.Units):boolean;
            /**
             * Returns a Boolean value that indicates whether this set is a strict subset of the given set.
             */
            isStrictSubset(other: ByteCountFormatStyle.Units):boolean;
            /**
             * Returns a Boolean value that indicates whether this set is a strict superset of the given set.
             */
            isStrictSuperset(other: ByteCountFormatStyle.Units):boolean;
            /**
             * Returns a Boolean value that indicates whether the set is a subset of another set.
             */
            isSubset(other: ByteCountFormatStyle.Units):boolean;
            /**
             * Returns a Boolean value that indicates whether the set is a superset of the given set.
             */
            isSuperset(other: ByteCountFormatStyle.Units):boolean;
            /**
             * Removes the given element and all elements subsumed by it.
             */
            remove(member: ByteCountFormatStyle.Units):ByteCountFormatStyle.Units | null;
            /**
             * Removes the elements of the given set from this set.
             */
            subtract(other: ByteCountFormatStyle.Units):void;
            /**
             * Returns a new set containing the elements of this set that do not occur in the given set.
             */
            subtracting(other: ByteCountFormatStyle.Units):ByteCountFormatStyle.Units;
            /**
             * Returns a new option set with the elements contained in this set or in the given set, but not in both.
             */
            symmetricDifference(other: ByteCountFormatStyle.Units):ByteCountFormatStyle.Units;
            /**
             * Returns a new option set of the elements contained in this set, in the given set, or in both.
             */
            union(other: ByteCountFormatStyle.Units):ByteCountFormatStyle.Units;
            /**
             * Inserts the given element into the set.
             */
            update(newMember: ByteCountFormatStyle.Units):ByteCountFormatStyle.Units | null;
        }
        export enum Style {
            binary,
            decimal,
            file,
            memory,
        }
    }
    export class StringStyle {
        hashValue: number;
        constructor (decoder: Decoder);
        static measurement(width: any, usage: MeasurementFormatUnitUsage<UnitTemperature>, hidesScaleName: boolean, numberFormatStyle: FloatingPointFormatStyle<Double> | null):any;
        format(value: string):string;
        encode(encoder: Encoder):void;
        locale(locale: Locale):StringStyle;
        hash(hasher: Hasher):void;
    }
    export enum NumberFormatStyleConfiguration {}
    export enum CurrencyFormatStyleConfiguration {}
    export enum DescriptiveNumberFormatConfiguration {}
    export interface ParseableFormatStyle extends FormatStyle {
        parseStrategy: Self.Strategy;
        static dateTime: Date.FormatStyle;
        static iso8601: Date.ISO8601FormatStyle;
        static number: Decimal.FormatStyle;
        static percent: Decimal.FormatStyle.Percent;
        static url: URL.FormatStyle;
        static currency(code: string):Self;
        static name(style: PersonNameComponents.FormatStyle.Style):Self;
    }
    export interface ParseStrategy extends Decodable,Encodable,Hashable {
        parse(value: Self.ParseInput):Self.ParseOutput;
        static dateTime: Date.FormatStyle;
        static iso8601: Date.ISO8601FormatStyle;
        static name: PersonNameComponents.ParseStrategy;
        static url: URL.ParseStrategy;
        static fixed(format: Date.FormatString, timeZone: TimeZone, locale: Locale | null):Date.ParseStrategy;
    }
    export enum InflectionRule {
        automatic,
        explicit,
    }
    export class Morphology {
        /**
         * The addressing preferences of the current user.
         */
        static user: Morphology;
        /**
         * A Boolean value that indicates whether this instance specifies no particular grammar.
         */
        isUnspecified: boolean;
        /**
         * The grammatical gender used for inflecting strings with this morphology.
         */
        grammaticalGender: Morphology.GrammaticalGender | null;
        /**
         * The grammatical number used for inflecting strings with this morphology.
         */
        number: Morphology.GrammaticalNumber | null;
        /**
         * The grammatical part of speech used for inflecting strings with this morphology.
         */
        partOfSpeech: Morphology.PartOfSpeech | null;
        /**
         * The hash value.
         */
        hashValue: number;
        /**
         * Creates an empty morphology instance.
         */
        constructor ();
        /**
         * Creates a new instance by decoding from the given decoder.
         */
        constructor (decoder: Decoder);
        /**
         * Sets a custom pronoun behavior for this morphology to apply to the given language.
         */
        setCustomPronoun(pronoun: Morphology.CustomPronoun | null, language: string):void;
        /**
         * Returns any custom pronoun behavior this morphology applies to the given language.
         */
        customPronoun(language: string):Morphology.CustomPronoun | null;
        /**
         * Encodes this value into the given encoder.
         */
        encode(encoder: Encoder):void;
        /**
         * Hashes the essential components of this value by feeding them into the given hasher.
         */
        hash(hasher: Hasher):void;
    }
    /**
     * the same name namespace of class Morphology
     */
    export namespace Morphology {
        export enum GrammaticalGender {
            feminine,
            masculine,
            neuter,
        }
        export enum GrammaticalNumber {
            zero,
            singular,
            plural,
            pluralTwo,
            pluralFew,
            pluralMany,
        }
        export enum PartOfSpeech {
            determiner,
            pronoun,
            letter,
            adverb,
            particle,
            adjective,
            adposition,
            verb,
            noun,
            conjunction,
            numeral,
            interjection,
            preposition,
            abbreviation,
        }
        export class CustomPronoun {
            /**
             * The subject pronoun form to apply when using this custom pronoun behavior.
             */
            subjectForm: string;
            /**
             * The object pronoun form to apply when using this custom pronoun behavior.
             */
            objectForm: string;
            /**
             * The posessive pronoun form to apply when using this custom pronoun behavior.
             */
            possessiveForm: string;
            /**
             * The posessive adjective pronoun form to apply when using this custom pronoun behavior.
             */
            possessiveAdjectiveForm: string;
            /**
             * The reflexive pronoun form to apply when using this custom pronoun behavior.
             */
            reflexiveForm: string;
            /**
             * The hash value.
             */
            hashValue: number;
            /**
             * Creates an empty custom pronoun.
             */
            constructor ();
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            /**
             * Returns a Boolean value that indicates whether the given language supports setting custom pronouns.
             */
            static isSupported(language: string):boolean;
            /**
             * Returns a collection of the custom pronoun keys required by this language.
             */
            static requiredKeys(language: string):PartialKeyPath<Morphology.CustomPronoun>[];
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
    }
    export class LengthFormatter extends Formatter {
        /**
         * A Boolean value that indicates whether the resulting string represents a person’s height.
         */
        isForPersonHeightUse: boolean;
        /**
         * The number formatter used to format the numbers in length strings.
         */
        numberFormatter: NumberFormatter!;
        /**
         * The unit style used by this formatter.
         */
        unitStyle: Formatter.UnitStyle;
        /**
         * This method is not supported for the 
         *  class. 
         */
        getObjectValue(obj: AutoreleasingUnsafeMutablePointer<AnyObject?> | null, _string: string, error: AutoreleasingUnsafeMutablePointer<NSString?> | null):boolean;
        /**
         * Returns a length string for the provided value.
         */
        string(numberInMeters: number):string;
        /**
         * Returns the unit string for the provided value.
         */
        unitString(numberInMeters: number, unitp: UnsafeMutablePointer<LengthFormatter.Unit> | null):string;
    }
    /**
     * the same name namespace of class LengthFormatter
     */
    export namespace LengthFormatter {
        export enum Unit {
            millimeter = 8,
            centimeter = 9,
            meter = 11,
            kilometer = 14,
            inch = 1281,
            foot = 1282,
            yard = 1283,
            mile = 1284,
        }
    }
    export class MassFormatter extends Formatter {
        /**
         * A Boolean value that indicates whether the resulting string represents a person’s mass.
         */
        isForPersonMassUse: boolean;
        /**
         * The number formatter used to format the numbers in a mass strings.
         */
        numberFormatter: NumberFormatter!;
        /**
         * The unit style used by this formatter.
         */
        unitStyle: Formatter.UnitStyle;
        /**
         * This method is not supported for the 
         *  class. 
         */
        getObjectValue(obj: AutoreleasingUnsafeMutablePointer<AnyObject?> | null, _string: string, error: AutoreleasingUnsafeMutablePointer<NSString?> | null):boolean;
        /**
         * Returns a mass string for the provided value.
         */
        string(numberInKilograms: number):string;
        /**
         * Returns the unit string for the provided value.
         */
        unitString(numberInKilograms: number, unitp: UnsafeMutablePointer<MassFormatter.Unit> | null):string;
    }
    /**
     * the same name namespace of class MassFormatter
     */
    export namespace MassFormatter {
        export enum Unit {
            gram = 11,
            kilogram = 14,
            ounce = 1537,
            pound = 1538,
            stone = 1539,
        }
    }
    export class EnergyFormatter extends Formatter {
        /**
         * A Boolean value that indicates whether the energy value is used to measure food energy.
         */
        isForFoodEnergyUse: boolean;
        /**
         * The number formatter used to format the numbers in energy strings.
         */
        numberFormatter: NumberFormatter!;
        /**
         * The unit style used by this formatter.
         */
        unitStyle: Formatter.UnitStyle;
        /**
         * This method is not supported for the 
         *  class. 
         */
        getObjectValue(obj: AutoreleasingUnsafeMutablePointer<AnyObject?> | null, _string: string, error: AutoreleasingUnsafeMutablePointer<NSString?> | null):boolean;
        /**
         * Returns an energy string for the provided value.
         */
        string(numberInJoules: number):string;
        /**
         * Returns the unit string for the provided value.
         */
        unitString(numberInJoules: number, unitp: UnsafeMutablePointer<EnergyFormatter.Unit> | null):string;
    }
    /**
     * the same name namespace of class EnergyFormatter
     */
    export namespace EnergyFormatter {
        export enum Unit {
            joule = 11,
            kilojoule = 14,
            calorie = 1793,
            kilocalorie = 1794,
        }
    }
    export class NSPredicate extends NSObject {
        /**
         * The predicate's format string.
         */
        predicateFormat: string;
        /**
         * Creates a predicate by substituting the values in a specified array into a format string and parsing the result.
         */
        constructor (predicateFormat: string, arguments: any[] | null);
        /**
         * Creates a predicate by substituting the values in an argument list into a format string and parsing the result.
         */
        constructor (predicateFormat: string, argList: CVaListPointer);
        /**
         * Creates a predicate by substituting the values in an argument list into a format string and parsing the result.
         */
        constructor (predicateFormat: string, ...args: CVarArg);
        /**
         * Creates and returns a predicate that always evaluates to a specified Boolean value.
         */
        constructor (value: boolean);
        /**
         * Creates a predicate that evaluates using a specified block object and bindings dictionary.
         */
        constructor (block: ()=>any);
        /**
         * Creates a predicate with a metadata query string.
         */
        constructor (queryString: string);
        /**
         * Returns a copy of the predicate and substitutes the predicates variables with specified values from a specified substitution variables dictionary.
         */
        withSubstitutionVariables(variables: Record<string, any>):Self;
        /**
         * Returns a Boolean value that indicates whether the specified object matches the conditions that the predicate specifies.
         */
        evaluate(object: any):boolean;
        /**
         * Forces a securely decoded predicate to allow evaluation.
         */
        allowEvaluation():void;
    }
    export class NSExpression extends NSObject {
        /**
         * The arguments for the expression.
         */
        arguments: NSExpression[] | null;
        /**
         * The collection of expressions in an aggregate expression, or the collection element of a subquery expression.
         */
        collection: any;
        /**
         * The constant value of the expression.
         */
        constantValue: any;
        /**
         * The expression type for the expression.
         */
        expressionType: NSExpression.ExpressionType;
        /**
         * The function for the expression.
         */
        function: string;
        /**
         * The key path for the expression.
         */
        keyPath: string;
        /**
         * The operand for the expression.
         */
        operand: NSExpression;
        /**
         * The predicate of a subquery expression.
         */
        predicate: NSPredicate;
        /**
         * The left expression of an aggregate expression.
         */
        left: NSExpression;
        /**
         * The right expression of an aggregate expression.
         */
        right: NSExpression;
        /**
         * The variable for the expression.
         */
        variable: string;
        /**
         * An expression to evalutate if a conditional expression’s predicate evaluates to false.
         */
        false: NSExpression;
        /**
         * An expression to evalutate if a conditional expression’s predicate evaluates to true.
         */
        true: NSExpression;
        /**
         * The block that executes to evaluate the expression.
         */
        expressionBlock: ()=>any;
        /**
         * Creates the expression with the specified expression type.
         */
        constructor (type: NSExpression.ExpressionType);
        /**
         * Creates the expression with the specified expression format and array of arguments.
         */
        constructor (expressionFormat: string, arguments: any[]);
        /**
         * Creates the expression with the specified expression format and arguments list.
         */
        constructor (expressionFormat: string, argList: CVaListPointer);
        /**
         * Creates the expression with the expression format and arguments list you specify.
         */
        constructor (expressionFormat: string, ...args: CVarArg);
        /**
         * Creates an expression by decoding from the coder you specify.
         */
        constructor (coder: NSCoder);
        /**
         * Creates an expression that represents a specified constant value.
         */
        constructor (obj: any);
        /**
         * Creates an expression that invokes the value function with a specified key path.
         */
        constructor (keyPath: string);
        /**
         * Creates an expression that extracts a value from the variable bindings dictionary for a specified key.
         */
        constructor (_string: string);
        /**
         * Creates an aggregate expression for a specified collection.
         */
        constructor (subexpressions: NSExpression[]);
        /**
         * Creates an expression object that represents the union of a specified set and collection.
         */
        constructor (left: NSExpression, right: NSExpression);
        /**
         * Creates an expression object that represents the intersection of a specified set and collection.
         */
        constructor (left: NSExpression, right: NSExpression);
        /**
         * Creates an expression object that represents the subtraction of a specified collection from a specified set.
         */
        constructor (left: NSExpression, right: NSExpression);
        /**
         * Creates an expression that filters a collection by storing elements in the collection in a specified variable and keeping the elements that the qualifier returns as true.
         */
        constructor (expression: NSExpression, variable: string, predicate: NSPredicate);
        /**
         * Creates an expression that returns a result, depending on the value of predicate.
         */
        constructor (predicate: NSPredicate, trueExpression: NSExpression, falseExpression: NSExpression);
        /**
         * Creates an expression object that uses the block for evaluating objects.
         */
        constructor (block: ()=>any, arguments: NSExpression[] | null);
        /**
         * Creates an expression that invokes one of the predefined functions.
         */
        constructor (name: string, parameters: any[]);
        /**
         * Creates an expression that returns the result of invoking a selector with a specified name using specified arguments.
         */
        constructor (target: NSExpression, name: string, parameters: any[] | null);
        /**
         * Creates an expression that represents the object you’re evaluating.
         */
        static expressionForEvaluatedObject():NSExpression;
        /**
         * Creates an expression that represents any key for a Spotlight query.
         */
        static expressionForAnyKey():NSExpression;
        /**
         * Evaluates an expression using a specified object and context.
         */
        expressionValue(object: any, context: NSMutableDictionary | null):any;
        /**
         * Forces a securely decoded expression to allow evaluation.
         */
        allowEvaluation():void;
    }
    /**
     * the same name namespace of class NSExpression
     */
    export namespace NSExpression {
        export enum ExpressionType {
            constantValue = 0,
            evaluatedObject = 1,
            variable = 2,
            keyPath = 3,
            function = 4,
            unionSet = 5,
            intersectSet = 6,
            minusSet = 7,
            subquery = 13,
            aggregate = 14,
            anyKey = 15,
            block = 19,
            conditional = 20,
        }
    }
    export class NSComparisonPredicate extends NSPredicate {
        /**
         * The comparison predicate modifier for the receiver.
         */
        comparisonPredicateModifier: NSComparisonPredicate.Modifier;
        /**
         * The selector for the receiver.
         */
        customSelector: Selector | null;
        /**
         * The right expression for the receiver.
         */
        rightExpression: NSExpression;
        /**
         * The left expression for the receiver.
         */
        leftExpression: NSExpression;
        /**
         * The options to use for the receiver.
         */
        options: NSComparisonPredicate.Options;
        /**
         * The predicate type for the receiver.
         */
        predicateOperatorType: NSComparisonPredicate.Operator;
        /**
         * Creates a predicate that you form by combining specified left and right expressions using a specified selector.
         */
        constructor (lhs: NSExpression, rhs: NSExpression, selector: Selector);
        /**
         * Creates a predicate to a specified type that you form by combining specified left and right expressions using a specified modifier and options.
         */
        constructor (lhs: NSExpression, rhs: NSExpression, modifier: NSComparisonPredicate.Modifier, type: NSComparisonPredicate.Operator, options: NSComparisonPredicate.Options);
        /**
         * Creates a predicate by decoding from the coder you specify.
         */
        constructor (coder: NSCoder);
    }
    /**
     * the same name namespace of class NSComparisonPredicate
     */
    export namespace NSComparisonPredicate {
        export enum Modifier {
            direct = 0,
            all = 1,
            any = 2,
        }
        export class Options {
            /**
             * A case-insensitive predicate.
             */
            static caseInsensitive: NSComparisonPredicate.Options;
            /**
             * A diacritic-insensitive predicate.
             */
            static diacriticInsensitive: NSComparisonPredicate.Options;
            /**
             * A predicate that indicates you’ve preprocessed the strings to compare.
             */
            static normalized: NSComparisonPredicate.Options;
            constructor (rawValue: number);
        }
        export enum Operator {
            lessThan = 0,
            lessThanOrEqualTo = 1,
            greaterThan = 2,
            greaterThanOrEqualTo = 3,
            equalTo = 4,
            notEqualTo = 5,
            matches = 6,
            like = 7,
            beginsWith = 8,
            endsWith = 9,
            in = 10,
            customSelector = 11,
            contains = 99,
            between = 100,
        }
    }
    export class NSCompoundPredicate extends NSPredicate {
        /**
         * The predicate type for the receiver.
         */
        compoundPredicateType: NSCompoundPredicate.LogicalType;
        /**
         * The receiver’s subpredicates.
         */
        subpredicates: any[];
        /**
         * Returns a new predicate that you form using an AND operation on the predicates in a specified array.
         */
        constructor (subpredicates: NSPredicate[]);
        /**
         * Returns a new predicate that you form using a NOT operation on a specified predicate.
         */
        constructor (predicate: NSPredicate);
        /**
         * Returns a new predicate that you form using an OR operation on the predicates in a specified array.
         */
        constructor (subpredicates: NSPredicate[]);
        /**
         * Returns the receiver that a specified type initializes using predicates from a specified array.
         */
        constructor (type: NSCompoundPredicate.LogicalType, subpredicates: NSPredicate[]);
        /**
         * Creates a predicate by decoding from the coder you specify.
         */
        constructor (coder: NSCoder);
    }
    /**
     * the same name namespace of class NSCompoundPredicate
     */
    export namespace NSCompoundPredicate {
        export enum LogicalType {
            not = 0,
            and = 1,
            or = 2,
        }
    }
    export class NSSortDescriptor extends NSObject {
        /**
         * A Boolean value that indicates whether the receiver specifies sorting in ascending order.
         */
        ascending: boolean;
        /**
         * The key that specifies the property to compare during sorting.
         */
        key: string;
        /**
         * The key path that specifies the property to compare during sorting.
         */
        keyPath: AnyKeyPath | null;
        /**
         * The selector for comparing objects.
         */
        selector: Selector | null;
        /**
         * The comparator for the sort descriptor.
         */
        comparator: Comparator;
        /**
         * Returns a sort descriptor that reverses the sort order.
         */
        reversedSortDescriptor: any;
        /**
         * Creates a sort descriptor with a specified string key path and sort order.
         */
        constructor (key: string, ascending: boolean);
        /**
         * Creates a sort descriptor with a specified string key path, ordering, and comparison selector.
         */
        constructor (key: string, ascending: boolean, selector: Selector | null);
        /**
         * Creates a sort descriptor with a specified string key path and ordering, and a comparator block.
         */
        constructor (key: string, ascending: boolean, cmptr: Comparator);
        /**
         * Creates a sort descriptor by decoding from the coder you specify.
         */
        constructor (coder: NSCoder);
        /**
         * Returns a comparison result value that indicates the sort order of two objects.
         */
        compare(object1: any, object2: any):ComparisonResult;
        /**
         * Forces a securely decoded sort descriptor to allow evaluation.
         */
        allowEvaluation():void;
    }
    export enum ComparisonResult {
        orderedAscending = -1,
        orderedSame = 0,
        orderedDescending = 1,
    }
    export interface SortComparator extends Hashable {
        /**
         * The sort order that the comparator uses to compare.
         */
        order: SortOrder;
        /**
         * A comparator that compares a string using a localized comparison in the current locale.
         */
        static localized: String.Comparator;
        /**
         * A comparator that compares a string using a localized, numeric comparison in the current locale.
         */
        static localizedStandard: String.Comparator;
        /**
         * Provides the relative ordering of two elements based on the sort order of the comparator.
         */
        compare(lhs: Self.Compared, rhs: Self.Compared):ComparisonResult;
    }
    export enum SortOrder {
        forward,
        reverse,
    }
    export class UndoManager extends NSObject {
        /**
         * A Boolean value that indicates whether the receiver has any actions to undo.
         */
        canUndo: boolean;
        /**
         * A Boolean value that indicates whether the receiver has any actions to redo.
         */
        canRedo: boolean;
        /**
         * The maximum number of top-level undo groups the receiver holds.
         */
        levelsOfUndo: number;
        /**
         * A Boolean value that indicates whether the receiver automatically creates undo groups around each pass of the run loop.
         */
        groupsByEvent: boolean;
        /**
         * The number of nested undo groups (or redo groups, if Redo was invoked last) in the current event loop.
         */
        groupingLevel: number;
        /**
         * A Boolean value that indicates whether the recording of undo operations is enabled.
         */
        isUndoRegistrationEnabled: boolean;
        /**
         * Returns a Boolean value that indicates whether the receiver is in the process of performing its 
         *  or 
         *  method.
         */
        isUndoing: boolean;
        /**
         * Returns a Boolean value that indicates whether the receiver is in the process of performing its 
         *  method.
         */
        isRedoing: boolean;
        /**
         * The name identifying the undo action.
         */
        undoActionName: string;
        /**
         * The name identifying the redo action.
         */
        redoActionName: string;
        /**
         * The complete title of the Undo menu command, for example, “Undo Paste.”
         */
        undoMenuItemTitle: string;
        /**
         * The complete title of the Redo menu command, for example, “Redo Paste.”
         */
        redoMenuItemTitle: string;
        /**
         * The modes governing the types of input handled during a cycle of the run loop.
         */
        runLoopModes: RunLoop.Mode[];
        /**
         * Boolean value that indicates whether the next undo action is discardable.
         */
        undoActionIsDiscardable: boolean;
        /**
         * Boolean value that indicates whether the next redo action is discardable.
         */
        redoActionIsDiscardable: boolean;
        /**
         * Registers the selector of the specified target to implement a single undo operation that the target receives.
         */
        registerUndo(target: any, selector: Selector, anObject: any):void;
        /**
         * Prepares the undo manager for invocation-based undo with the given target as the subject of the next undo operation.
         */
        prepare(target: any):any;
        /**
         * Closes the top-level undo group if necessary and invokes 
         * .
         */
        undo():void;
        /**
         * Performs the undo operations in the last undo group (whether top-level or nested), recording the operations on the redo stack as a single group.
         */
        undoNestedGroup():void;
        /**
         * Performs the operations in the last group on the redo stack, if there are any, recording them on the undo stack as a single group.
         */
        redo():void;
        /**
         * Marks the beginning of an undo group.
         */
        beginUndoGrouping():void;
        /**
         * Marks the end of an undo group.
         */
        endUndoGrouping():void;
        /**
         * Disables the recording of undo operations, whether by 
         *  or by invocation-based undo.
         */
        disableUndoRegistration():void;
        /**
         * Enables the recording of undo operations.
         */
        enableUndoRegistration():void;
        /**
         * Clears the undo and redo stacks and re-enables the receiver.
         */
        removeAllActions():void;
        /**
         * Sets the name of the action associated with the Undo or Redo command.
         */
        setActionName(actionName: string):void;
        /**
         * Returns the complete, localized title of the Undo menu command for the action identified by the given name.
         */
        undoMenuTitle(actionName: string):string;
        /**
         * Returns the complete, localized title of the Redo menu command for the action identified by the given name.
         */
        redoMenuTitle(actionName: string):string;
        /**
         * Sets whether the next undo or redo action is discardable.
         */
        setActionIsDiscardable(discardable: boolean):void;
    }
    export interface ProgressReporting extends NSObjectProtocol {
        /**
         * The progress object returned by the class.
         */
        progress: Progress;
    }
    export class Progress extends NSObject {
        /**
         * The total number of tracked units of work for the current progress.
         */
        totalUnitCount: Int64;
        /**
         * The number of completed units of work for the current job.
         */
        completedUnitCount: Int64;
        /**
         * A localized description of tracked progress for the receiver.
         */
        localizedDescription: String!;
        /**
         * A more specific localized description of tracked progress for the receiver.
         */
        localizedAdditionalDescription: String!;
        /**
         * A Boolean value that indicates whether the receiver is tracking work that you can cancel.
         */
        isCancellable: boolean;
        /**
         * A Boolean value that Indicates whether the receiver is tracking canceled work.
         */
        isCancelled: boolean;
        /**
         * The block to invoke when canceling progress.
         */
        cancellationHandler: ()=>any | null;
        /**
         * A Boolean value that indicates whether the receiver is tracking work that you can pause.
         */
        isPausable: boolean;
        /**
         * A Boolean value that indicates whether the receiver is tracking paused work.
         */
        isPaused: boolean;
        /**
         * The block to invoke when pausing progress.
         */
        pausingHandler: ()=>any | null;
        /**
         * A Boolean value that indicates whether the tracked progress is indeterminate.
         */
        isIndeterminate: boolean;
        /**
         * The fraction of the overall work that the progress object completes, including work from its suboperations.
         */
        fractionCompleted: number;
        /**
         * A Boolean value that indicates the progress object is complete.
         */
        isFinished: boolean;
        /**
         * The block to invoke when progress resumes.
         */
        resumingHandler: ()=>any | null;
        /**
         * An object that represents the kind of progress for the progress object.
         */
        kind: ProgressKind | null;
        /**
         * A value that indicates the estimated amount of time remaining to complete the progress.
         */
        estimatedTimeRemaining: TimeInterval | null;
        /**
         * A value that represents the speed of data processing, in bytes per second.
         */
        throughput: number;
        /**
         * A dictionary of arbitrary values for the receiver.
         */
        userInfo: Record<ProgressUserInfoKey, any>;
        /**
         * The kind of file operation for the progress object.
         */
        fileOperationKind: Progress.FileOperationKind | null;
        /**
         * A URL that represents the file for the current progress object.
         */
        fileURL: URL | null;
        /**
         * The total number of files for a file progress object.
         */
        fileTotalCount: number;
        /**
         * The number of completed files for a file progress object.
         */
        fileCompletedCount: number;
        /**
         * A Boolean value that indicates when the observed progress object invokes the publish method before you subscribe to it.
         */
        isOld: boolean;
        /**
         * Creates a new progress instance.
         */
        constructor (parentProgressOrNil: Progress | null, userInfoOrNil: Record<ProgressUserInfoKey, any> | null);
        /**
         * Creates and returns a progress instance.
         */
        constructor (unitCount: Int64);
        /**
         * Creates a progress instance for the specified progress object with a unit count that’s a portion of the containing object’s total unit count.
         */
        constructor (unitCount: Int64, parent: Progress, portionOfParentTotalUnitCount: Int64);
        /**
         * Creates and returns a progress instance with the specified unit count that isn’t part of any existing progress tree.
         */
        static discreteProgress(unitCount: Int64):Progress;
        /**
         * Returns the progress instance, if any.
         */
        static current():Progress | null;
        /**
         * Registers a file URL to hear about the progress of a file operation.
         */
        static addSubscriber(url: URL, publishingHandler: Progress.PublishingHandler):any;
        /**
         * Removes a proxy progress object that the add subscriber method returns.
         */
        static removeSubscriber(subscriber: any):void;
        /**
         * Sets the progress object as the current object of the current thread, and assigns the amount of work for the next suboperation progress object to perform.
         */
        becomeCurrent(unitCount: Int64):void;
        /**
         * Adds a process object as a suboperation of a progress tree.
         */
        addChild(child: Progress, inUnitCount: Int64):void;
        /**
         * Restores the previous progress object to become the current progress object on the thread.
         */
        resignCurrent():void;
        /**
         * Cancels progress tracking.
         */
        cancel():void;
        /**
         * Pauses progress tracking.
         */
        pause():void;
        /**
         * Resumes progress tracking.
         */
        resume():void;
        /**
         * Sets a value in the user info dictionary.
         */
        setUserInfoObject(objectOrNil: any, key: ProgressUserInfoKey):void;
        /**
         * Publishes the progress object for other processes to observe it.
         */
        publish():void;
        /**
         * Removes a progress object from publication, making it unobservable by other processes.
         */
        unpublish():void;
    }
    /**
     * the same name namespace of class Progress
     */
    export namespace Progress {
        export class FileOperationKind {
            /**
             * The progress is tracking the copying of a file from source to destination.
             */
            static copying: Progress.FileOperationKind;
            /**
             * The progress is tracking file decompression after a download.
             */
            static decompressingAfterDownloading: Progress.FileOperationKind;
            /**
             * The progress is tracking a file download operation.
             */
            static downloading: Progress.FileOperationKind;
            /**
             * The progress is tracking a file upload operation.
             */
            static uploading: Progress.FileOperationKind;
            /**
             * The progress is tracking the receipt of a file from another source.
             */
            static receiving: Progress.FileOperationKind;
            static duplicating: Progress.FileOperationKind;
            /**
             * Creates a new kind of file operation using the specified string.
             */
            constructor (rawValue: string);
            /**
             * Creates a new kind of file operation using the raw value of a string you specify.
             */
            constructor (rawValue: string);
        }
    }
    export class ProgressKind {
        /**
         * The value that indicates that the progress is tracking a file operation.
         */
        static file: ProgressKind;
        /**
         * Creates a new kind of progress using the specified string.
         */
        constructor (rawValue: string);
        /**
         * Creates a new kind of progress using the raw value of a string you specify.
         */
        constructor (rawValue: string);
    }
    export class ProgressUserInfoKey {
        /**
         * A key with a corresponding value that represents the time remaining, in seconds.
         */
        static estimatedTimeRemainingKey: ProgressUserInfoKey;
        /**
         * A key with a corresponding value that indicates the speed of data processing, in bytes per second.
         */
        static throughputKey: ProgressUserInfoKey;
        /**
         * A key with a corresponding value that is an image, typically an icon to represent the file.
         */
        static fileAnimationImageKey: ProgressUserInfoKey;
        /**
         * A key with a corresponding value that indicates the starting location of the image onscreen.
         */
        static fileAnimationImageOriginalRectKey: ProgressUserInfoKey;
        /**
         * A key with a corresponding value that represents the number of completed files.
         */
        static fileCompletedCountKey: ProgressUserInfoKey;
        /**
         * A key with a corresponding value that must be an image, typically an icon to represent the file.
         */
        static fileIconKey: ProgressUserInfoKey;
        /**
         * A key with a corresponding value that indicates the kind of file operation a progress object represents.
         */
        static fileOperationKindKey: ProgressUserInfoKey;
        /**
         * A key with a corresponding value that represents the total number of files within a file operation.
         */
        static fileTotalCountKey: ProgressUserInfoKey;
        /**
         * A key with a corresponding value that represents the file URL of a file operation for the progress object.
         */
        static fileURLKey: ProgressUserInfoKey;
        /**
         * Creates a new key using the specified string.
         */
        constructor (rawValue: string);
        /**
         * Creates a new key using the raw value of a string you specify.
         */
        constructor (rawValue: string);
    }
    export class Operation extends NSObject {
        /**
         * The block to execute after the operation’s main task is completed.
         */
        completionBlock: ()=>any | null;
        /**
         * A Boolean value indicating whether the operation has been cancelled
         */
        isCancelled: boolean;
        /**
         * A Boolean value indicating whether the operation is currently executing. 
         */
        isExecuting: boolean;
        /**
         * A Boolean value indicating whether the operation has finished executing its task.
         */
        isFinished: boolean;
        /**
         * A Boolean value indicating whether the operation executes its task asynchronously.
         */
        isConcurrent: boolean;
        /**
         * A Boolean value indicating whether the operation executes its task asynchronously.
         */
        isAsynchronous: boolean;
        /**
         * A Boolean value indicating whether the operation can be performed now. 
         */
        isReady: boolean;
        /**
         * The name of the operation.
         */
        name: string;
        /**
         * An array of the operation objects that must finish executing before the current object can begin executing.
         */
        dependencies: Operation[];
        /**
         * The relative amount of importance for granting system resources to the operation. 
         */
        qualityOfService: QualityOfService;
        /**
         * The thread priority to use when executing the operation
         */
        threadPriority: number;
        /**
         * The execution priority of the operation in an operation queue. 
         */
        queuePriority: Operation.QueuePriority;
        /**
         * Begins the execution of the operation. 
         */
        start():void;
        /**
         * Performs the receiver’s non-concurrent task.
         */
        main():void;
        /**
         * Advises the operation object that it should stop executing its task.
         */
        cancel():void;
        /**
         * Makes the receiver dependent on the completion of the specified operation.
         */
        addDependency(op: Operation):void;
        /**
         * Removes the receiver’s dependence on the specified operation.
         */
        removeDependency(op: Operation):void;
        /**
         * Blocks execution of the current thread until the operation object finishes its task.
         */
        waitUntilFinished():void;
    }
    /**
     * the same name namespace of class Operation
     */
    export namespace Operation {
        export enum QueuePriority {
            veryLow = -8,
            low = -4,
            normal = 0,
            high = 4,
            veryHigh = 8,
        }
    }
    export enum QualityOfService {
        userInteractive = 33,
        userInitiated = 25,
        utility = 17,
        background = 9,
        default = -1,
    }
    export class OperationQueue extends NSObject {
        /**
         * Returns the operation queue associated with the main thread. 
         */
        static main: OperationQueue;
        /**
         * Returns the operation queue that launched the current operation.
         */
        static current: OperationQueue | null;
        /**
         * The default maximum number of operations to invoke concurrently in a queue.
         */
        static defaultMaxConcurrentOperationCount: number;
        /**
         * The operations currently in the queue.
         */
        operations: Operation[];
        /**
         * The number of operations currently in the queue.
         */
        operationCount: number;
        /**
         * The default service level to apply to operations that the queue invokes.
         */
        qualityOfService: QualityOfService;
        /**
         * The maximum number of queued operations that can run at the same time.
         */
        maxConcurrentOperationCount: number;
        /**
         * An object that represents the total progress of the operations executing in the queue.
         */
        progress: Progress;
        /**
         * A Boolean value indicating whether the queue is actively scheduling operations for execution.
         */
        isSuspended: boolean;
        /**
         * The name of the operation queue.
         */
        name: string;
        /**
         * The dispatch queue that the operation queue uses to invoke operations.
         */
        underlyingQueue: DispatchQueue | null;
        /**
         * The operation queue’s definition of the current moment in time.
         */
        now: OperationQueue.SchedulerTimeType;
        /**
         * The minimum tolerance the dispatch queue scheduler allows.
         */
        minimumTolerance: OperationQueue.SchedulerTimeType.Stride;
        /**
         * Adds the specified operation to the receiver.
         */
        addOperation(op: Operation):void;
        /**
         * Adds the specified operations to the queue.
         */
        addOperations(ops: Operation[], wait: boolean):void;
        /**
         * Invokes a block when the queue finishes all enqueued operations, and prevents subsequent operations from starting until the block has completed.
         */
        addBarrierBlock(barrier: ()=>any):void;
        /**
         * Cancels all queued and executing operations.
         */
        cancelAllOperations():void;
        /**
         * Blocks the current thread until all the receiver’s queued and executing operations finish executing.
         */
        waitUntilAllOperationsAreFinished():void;
        /**
         * Performs the action at some time after the specified date, optionally taking into account tolerance if possible.
         */
        schedule(date: OperationQueue.SchedulerTimeType, tolerance: OperationQueue.SchedulerTimeType.Stride, options: OperationQueue.SchedulerOptions | null, action: ()=>any):void;
    }
    /**
     * the same name namespace of class OperationQueue
     */
    export namespace OperationQueue {
        export class SchedulerTimeType {
            /**
             * The date this type represents.
             */
            date: Date;
            hashValue: number;
            /**
             * Creates an operation queue scheduler time with the given date.
             */
            constructor (date: Date);
            /**
             * Creates a scheduler time type using data in an unarchiver.
             */
            constructor (decoder: Decoder);
            /**
             * Calculates an operation queue scheduler time by advancing the scheduler time type’s date by the given interval.
             */
            advanced(n: OperationQueue.SchedulerTimeType.Stride):OperationQueue.SchedulerTimeType;
            /**
             * The distance to another operation queue scheduler time.
             */
            distance(other: OperationQueue.SchedulerTimeType):OperationQueue.SchedulerTimeType.Stride;
            hash(hasher: Hasher):void;
            /**
             * Encodes a scheduler time type to an archiver.
             */
            encode(encoder: Encoder):void;
        }
        /**
         * the same name namespace of class SchedulerTimeType
         */
        export namespace SchedulerTimeType {
            export class Stride {
                /**
                 * The zero value.
                 */
                static zero: OperationQueue.SchedulerTimeType.Stride;
                /**
                 * The value of this time interval, in seconds.
                 */
                timeInterval: TimeInterval;
                /**
                 * The value of this time interval, in seconds.
                 */
                magnitude: TimeInterval;
                /**
                 * Creates a stride using the specified time interval.
                 */
                constructor (timeInterval: TimeInterval);
                /**
                 * Creates a stride using the specified floating-point value.
                 */
                constructor (value: TimeInterval);
                /**
                 * Creates a stride using the specified integer value.
                 */
                constructor (value: TimeInterval);
                /**
                 * Creates a scheduler time type stride using data in an unarchiver.
                 */
                constructor (decoder: Decoder);
                /**
                 * Converts the specified number of seconds into an instance of this scheduler time type.
                 */
                static seconds(s: number):OperationQueue.SchedulerTimeType.Stride;
                /**
                 * Converts the specified number of milliseconds into an instance of this scheduler time type.
                 */
                static milliseconds(ms: number):OperationQueue.SchedulerTimeType.Stride;
                /**
                 * Converts the specified number of microseconds into an instance of this scheduler time type.
                 */
                static microseconds(us: number):OperationQueue.SchedulerTimeType.Stride;
                /**
                 * Converts the specified number of nanoseconds into an instance of this scheduler time type.
                 */
                static nanoseconds(ns: number):OperationQueue.SchedulerTimeType.Stride;
                /**
                 * Replaces this value with its additive inverse.
                 */
                negate():void;
                /**
                 * Encodes a scheduler time type stride to an archiver.
                 */
                encode(encoder: Encoder):void;
            }
        }
        export class SchedulerOptions {}
    }
    export class BlockOperation extends Operation {
        /**
         * The blocks associated with the receiver.
         */
        executionBlocks: ()=>any;
        /**
         * Creates and returns an 
         *  object and adds the specified block to it.
         */
        constructor (block: ()=>any);
        /**
         * Adds the specified block to the receiver’s list of blocks to perform.
         */
        addExecutionBlock(block: ()=>any):void;
    }
    export class Timer extends NSObject {
        /**
         * A Boolean value that indicates whether the timer is currently valid.
         */
        isValid: boolean;
        /**
         * The date at which the timer will fire.
         */
        fireDate: Date;
        /**
         * The timer’s time interval, in seconds.
         */
        timeInterval: TimeInterval;
        /**
         * The receiver's 
         *  object.
         */
        userInfo: any;
        /**
         * The amount of time after the scheduled fire date that the timer may fire.
         */
        tolerance: TimeInterval;
        /**
         * Initializes a timer object with the specified time interval and block. 
         */
        constructor (interval: TimeInterval, repeats: boolean, block: ()=>any);
        /**
         * Initializes a timer object with the specified object and selector.
         */
        constructor (ti: TimeInterval, aTarget: any, aSelector: Selector, userInfo: any, yesOrNo: boolean);
        /**
         * Initializes a timer object with the specified invocation object.
         */
        constructor (ti: TimeInterval, invocation: NSInvocation, yesOrNo: boolean);
        /**
         * Initializes a timer for the specified date and time interval with the specified block.  
         */
        constructor (date: Date, interval: TimeInterval, repeats: boolean, block: ()=>any);
        /**
         * Initializes a timer using the specified object and selector. 
         */
        constructor (date: Date, ti: TimeInterval, t: any, s: Selector, ui: any, rep: boolean);
        /**
         * Creates a timer and schedules it on the current run loop in the default mode.
         */
        static scheduledTimer(interval: TimeInterval, repeats: boolean, block: ()=>any):Timer;
        /**
         * Returns a publisher that repeatedly emits the current date on the given interval.
         */
        static publish(interval: TimeInterval, tolerance: TimeInterval | null, runLoop: RunLoop, mode: RunLoop.Mode, options: RunLoop.SchedulerOptions | null):Timer.TimerPublisher;
        /**
         * Causes the timer's message to be sent to its target.
         */
        fire():void;
        /**
         * Stops the timer from ever firing again and requests its removal from its run loop.
         */
        invalidate():void;
    }
    /**
     * the same name namespace of class Timer
     */
    export namespace Timer {
        export class TimerPublisher {
            /**
             * Creates a publisher that repeatedly emits the current date on the given interval.
             */
            constructor (interval: TimeInterval, tolerance: TimeInterval | null, runLoop: RunLoop, mode: RunLoop.Mode, options: RunLoop.SchedulerOptions | null);
        }
    }
    export interface NSUserActivityDelegate extends NSObjectProtocol {
        /**
         * Notifies the user activity delegate that an input and output streams are available to open.
         */
        userActivity(userActivity: NSUserActivity, inputStream: InputStream, outputStream: OutputStream):void;
        /**
         * Notifies the delegate that the user activity was continued on another device.
         */
        userActivityWasContinued(userActivity: NSUserActivity):void;
        /**
         * Notifies the delegate that the user activity will be saved to be continued or persisted.
         */
        userActivityWillSave(userActivity: NSUserActivity):void;
    }
    export class ProcessInfo extends NSObject {
        /**
         * Returns the process information agent for the process.
         */
        static processInfo: ProcessInfo;
        /**
         * Posts when the thermal state of the system changes.
         */
        static thermalStateDidChangeNotification: NSNotification.Name;
        /**
         * Array of strings with the command-line arguments for the process.
         */
        arguments: string[];
        /**
         * The variable names (keys) and their values in the environment from which the process was launched.
         */
        environment: Record<string, string>;
        /**
         * Global unique identifier for the process.
         */
        globallyUniqueString: string;
        /**
         * A Boolean value that indicates whether the process originated as an iOS app and runs on macOS.
         */
        isMacCatalystApp: boolean;
        /**
         * A Boolean value that indicates whether the process is an iPhone or iPad app running on a Mac.
         */
        isiOSAppOnMac: boolean;
        /**
         * The identifier of the process (often called process ID).
         */
        processIdentifier: Int32;
        /**
         * The name of the process.
         */
        processName: string;
        /**
         * Returns the account name of the current user.
         */
        userName: string;
        /**
         * Returns the full name of the current user.
         */
        fullUserName: string;
        /**
         * A Boolean value indicating whether the app supports automatic termination.
         */
        automaticTerminationSupportEnabled: boolean;
        /**
         * The name of the host computer on which the process is executing.
         */
        hostName: string;
        /**
         * A string containing the version of the operating system on which the process is executing.
         */
        operatingSystemVersionString: string;
        /**
         * The version of the operating system on which the process is executing.
         */
        operatingSystemVersion: OperatingSystemVersion;
        /**
         * The number of processing cores available on the computer.
         */
        processorCount: number;
        /**
         * The number of active processing cores available on the computer.
         */
        activeProcessorCount: number;
        /**
         * The amount of physical memory on the computer in bytes.
         */
        physicalMemory: UInt64;
        /**
         * The amount of time the system has been awake since the last time it was restarted.
         */
        systemUptime: TimeInterval;
        /**
         * The current thermal state of the system.
         */
        thermalState: ProcessInfo.ThermalState;
        /**
         * A Boolean value that indicates the current state of Low Power Mode.
         */
        isLowPowerModeEnabled: boolean;
        /**
         * Disables the application for quickly killing using sudden termination.
         */
        disableSuddenTermination():void;
        /**
         * Enables the application for quick killing using sudden termination.
         */
        enableSuddenTermination():void;
        /**
         * Disables automatic termination for the application.
         */
        disableAutomaticTermination(reason: string):void;
        /**
         * Enables automatic termination for the application.
         */
        enableAutomaticTermination(reason: string):void;
        /**
         * Returns a Boolean value indicating whether the version of the operating system on which the process is executing is the same or later than the given version.
         */
        isOperatingSystemAtLeast(version: OperatingSystemVersion):boolean;
        /**
         * Returns a constant to indicate the operating system on which the process is executing.
         */
        operatingSystem():number;
        /**
         * Returns a string containing the name of the operating system on which the process is executing.
         */
        operatingSystemName():string;
        /**
         * Begin an activity using the given options and reason.
         */
        beginActivity(options: ProcessInfo.ActivityOptions, reason: string):NSObjectProtocol;
        /**
         * Ends the given activity.
         */
        endActivity(activity: NSObjectProtocol):void;
        /**
         * Synchronously perform an activity defined by a given block using the given options.
         */
        performActivity(options: ProcessInfo.ActivityOptions, reason: string, block: ()=>any):void;
        /**
         * Performs the specified block asynchronously and notifies you if the process is about to be suspended.
         */
        performExpiringActivity(reason: string, block: ()=>any):void;
    }
    /**
     * the same name namespace of class ProcessInfo
     */
    export namespace ProcessInfo {
        export class ActivityOptions {
            /**
             * A flag to require the screen to stay powered on.
             */
            static idleDisplaySleepDisabled: ProcessInfo.ActivityOptions;
            /**
             * A flag to prevent idle sleep.
             */
            static idleSystemSleepDisabled: ProcessInfo.ActivityOptions;
            /**
             * A flag to prevent sudden termination.
             */
            static suddenTerminationDisabled: ProcessInfo.ActivityOptions;
            /**
             * A flag to prevent automatic termination.
             */
            static automaticTerminationDisabled: ProcessInfo.ActivityOptions;
            /**
             * A flag to indicate the app is performing a user-requested action.
             */
            static userInitiated: ProcessInfo.ActivityOptions;
            /**
             * A flag to indicate the app is responding to user interaction.
             */
            static userInteractive: ProcessInfo.ActivityOptions;
            /**
             * A flag to indicate the app is performing a user-requested action, but that the system can sleep on idle.
             */
            static userInitiatedAllowingIdleSystemSleep: ProcessInfo.ActivityOptions;
            /**
             * A flag to indicate the app has initiated some kind of work, but not as the direct result of user request.
             */
            static background: ProcessInfo.ActivityOptions;
            /**
             * A flag to indicate the activity requires the highest amount of timer and I/O precision available.
             */
            static latencyCritical: ProcessInfo.ActivityOptions;
            /**
             * A flag to track the activity with an animation signpost interval.
             */
            static animationTrackingEnabled: ProcessInfo.ActivityOptions;
            /**
             * A flag to track the activity with a signpost interval.
             */
            static trackingEnabled: ProcessInfo.ActivityOptions;
            constructor (rawValue: UInt64);
        }
        export enum ThermalState {
            nominal = 0,
            fair = 1,
            serious = 2,
            critical = 3,
        }
    }
    export class OperatingSystemVersion {
        /**
         * The major release number, such as 10 in version 10.9.3.
         */
        majorVersion: number;
        /**
         * The minor release number, such as 9 in version 10.9.3.
         */
        minorVersion: number;
        /**
         * The update release number, such as 3 in version 10.9.3.
         */
        patchVersion: number;
        /**
         * Creates an empty operating system version.
         */
        constructor ();
        /**
         * Creates an operating system version with the provided values.
         */
        constructor (majorVersion: number, minorVersion: number, patchVersion: number);
    }
    export class NSBackgroundActivityScheduler extends NSObject {
        /**
         * A unique reverse DNS notation string, such as 
         * , that identifies the activity.
         */
        identifier: string;
        /**
         * A Boolean value indicating whether the activity should be rescheduled after it completes.
         */
        repeats: boolean;
        /**
         * An integer providing a suggested interval between scheduling and invoking the activity.
         */
        interval: TimeInterval;
        /**
         * A value of type 
         * , which controls how aggressively the system schedules the activity.
         */
        qualityOfService: QualityOfService;
        /**
         * A Boolean value indicating whether your app should stop performing background activity and resume at a more optimal time.
         */
        shouldDefer: boolean;
        /**
         * A value of type 
         * , which specifies a range of time during which the background activity may occur.
         */
        tolerance: TimeInterval;
        /**
         * Initializes a background activity scheduler object with a specified unique identifier.
         */
        constructor (identifier: string);
        /**
         * Begins scheduling the background activity.
         */
        schedule(block: ()=>any):void;
        /**
         * Prevents the background activity from being scheduled again.
         */
        invalidate():void;
    }
    /**
     * the same name namespace of class NSBackgroundActivityScheduler
     */
    export namespace NSBackgroundActivityScheduler {
        export enum Result {
            finished = 1,
            deferred = 2,
        }
    }
    export class NSUserNotification extends NSObject {
        /**
         * Specifies the title of the notification.
         */
        title: string;
        /**
         * Specifies the subtitle of the notification.
         */
        subtitle: string;
        /**
         * The body text of the notification.
         */
        informativeText: string;
        /**
         * Image shown in the content of the notification.
         */
        contentImage: NSImage | null;
        /**
         * A string that uniquely identifies a notification.
         */
        identifier: string;
        /**
         * The response with which the user responded to a notification.
         */
        response: NSAttributedString | null;
        /**
         * Optional placeholder string for inline reply field.
         */
        responsePlaceholder: string;
        /**
         * A Boolean value that specifies whether the notification displays an action button.
         */
        hasActionButton: boolean;
        /**
         * Specifies the title of the action button displayed in the notification.
         */
        actionButtonTitle: string;
        /**
         * Specifies a custom title for the close button in an alert-style notification.
         */
        otherButtonTitle: string;
        /**
         * A Boolean value that specifies whether the notification displays a reply button.
         */
        hasReplyButton: boolean;
        /**
         * Specifies when the notification should be delivered.
         */
        deliveryDate: Date | null;
        /**
         * The date this notification was actually delivered.
         */
        actualDeliveryDate: Date | null;
        /**
         * Specifies the date components that control how often a user notification is repeated.
         */
        deliveryRepeatInterval: DateComponents | null;
        /**
         * Specify the time zone to interpret the delivery date in.
         */
        deliveryTimeZone: TimeZone | null;
        /**
         * Specifies whether the user notification has been presented.
         */
        isPresented: boolean;
        /**
         * Specifies whether the remote was generated by a push notification.
         */
        isRemote: boolean;
        /**
         * Specifies the name of the sound to play when the notification is delivered.
         */
        soundName: string;
        /**
         * Specifies what caused a user notification to occur.
         */
        activationType: NSUserNotification.ActivationType;
        /**
         * An additional action selected by the user.
         */
        additionalActivationAction: NSUserNotificationAction | null;
        /**
         * The actions that can be taken on a notification in addition to the default action.
         */
        additionalActions: NSUserNotificationAction[] | null;
        /**
         * Application-specific user info that can be attached to the notification.
         */
        userInfo: Record<string, any> | null;
        constructor ();
    }
    /**
     * the same name namespace of class NSUserNotification
     */
    export namespace NSUserNotification {
        export enum ActivationType {
            none = 0,
            contentsClicked = 1,
            actionButtonClicked = 2,
            replied = 3,
            additionalActionClicked = 4,
        }
    }
    export class NSUserNotificationAction extends NSObject {
        /**
         * The identifier for the user notification action.
         */
        identifier: string;
        /**
         * The localized title shown to the user. 
         */
        title: string;
        /**
         * Creates a user notification action with a specified identifier and title.
         */
        constructor (identifier: string, title: string);
    }
    export class NSUserNotificationCenter extends NSObject {
        /**
         * Returns the default user notification center.
         */
        static default: NSUserNotificationCenter;
        /**
         * Specifies an array of scheduled user notifications that have not yet been delivered.
         */
        scheduledNotifications: NSUserNotification[];
        /**
         * An array of all user notifications delivered to the notification center.
         */
        deliveredNotifications: NSUserNotification[];
        /**
         * Specifies the notification center delegate.
         */
        delegate: NSUserNotificationCenterDelegate | null;
        /**
         * Schedules the specified user notification.
         */
        scheduleNotification(notification: NSUserNotification):void;
        /**
         * Removes the specified user notification for the scheduled notifications.
         */
        removeScheduledNotification(notification: NSUserNotification):void;
        /**
         * Deliver the specified user notification.
         */
        deliver(notification: NSUserNotification):void;
        /**
         * Remove a delivered user notification from the user notification center.
         */
        removeDeliveredNotification(notification: NSUserNotification):void;
        /**
         * Remove all delivered user notifications from the user notification center.
         */
        removeAllDeliveredNotifications():void;
    }
    export interface NSUserNotificationCenterDelegate extends NSObjectProtocol {
        /**
         * Sent to the delegate when a notification delivery date has arrived.
         */
        userNotificationCenter(center: NSUserNotificationCenter, notification: NSUserNotification):void;
    }
    export class Bundle extends NSObject {
        /**
         * Returns the bundle object that contains the current executable.
         */
        static main: Bundle;
        /**
         * Returns an array of all of the application’s bundles that represent frameworks.
         */
        static allFrameworks: Bundle[];
        /**
         * Returns an array of all the application’s non-framework bundles.
         */
        static allBundles: Bundle[];
        /**
         * A notification that lets observers know when classes are dynamically loaded.
         */
        static didLoadNotification: NSNotification.Name;
        /**
         * The file URL of the bundle’s subdirectory containing resource files.
         */
        resourceURL: URL | null;
        /**
         * The file URL of the receiver's executable file.
         */
        executableURL: URL | null;
        /**
         * The file URL of the bundle’s subdirectory containing private frameworks.
         */
        privateFrameworksURL: URL | null;
        /**
         * The file URL of the receiver's subdirectory containing shared frameworks.
         */
        sharedFrameworksURL: URL | null;
        /**
         * The file URL of the receiver's subdirectory containing plug-ins.
         */
        builtInPlugInsURL: URL | null;
        /**
         * The file URL of the bundle’s subdirectory containing shared support files.
         */
        sharedSupportURL: URL | null;
        /**
         * The file URL for the bundle’s App Store receipt.
         */
        appStoreReceiptURL: URL | null;
        /**
         * The full pathname of the bundle’s subdirectory containing resources.
         */
        resourcePath: string;
        /**
         * The full pathname of the receiver's executable file.
         */
        executablePath: string;
        /**
         * The full pathname of the bundle’s subdirectory containing private frameworks.
         */
        privateFrameworksPath: string;
        /**
         * The full pathname of the bundle’s subdirectory containing shared frameworks.
         */
        sharedFrameworksPath: string;
        /**
         * The full pathname of the receiver's subdirectory containing plug-ins.
         */
        builtInPlugInsPath: string;
        /**
         * The full pathname of the bundle’s subdirectory containing shared support files.
         */
        sharedSupportPath: string;
        /**
         * The full URL of the receiver’s bundle directory.
         */
        bundleURL: URL;
        /**
         * The full pathname of the receiver’s bundle directory.
         */
        bundlePath: string;
        /**
         * The receiver’s bundle identifier.
         */
        bundleIdentifier: string;
        /**
         * A dictionary, constructed from the bundle’s 
         *  file, that contains information about the receiver.
         */
        infoDictionary: Record<string, any> | null;
        /**
         * A list of all the localizations contained in the bundle.
         */
        localizations: string[];
        /**
         * An ordered list of preferred localizations contained in the bundle.
         */
        preferredLocalizations: string[];
        /**
         * The localization for the development language.
         */
        developmentLocalization: string;
        /**
         * A dictionary with the keys from the bundle’s localized property list.
         */
        localizedInfoDictionary: Record<string, any> | null;
        /**
         * The bundle’s principal class.
         */
        principalClass: AnyClass | null;
        /**
         * An array of numbers indicating the architecture types supported by the bundle’s executable.
         */
        executableArchitectures: NSNumber[] | null;
        /**
         * The load status of a bundle.
         */
        isLoaded: boolean;
        /**
         * Returns the 
         *  object with which the specified class is associated.
         */
        constructor (aClass: AnyClass);
        /**
         * Returns the 
         *  instance that has the specified bundle identifier.
         */
        constructor (identifier: string);
        /**
         * Returns an 
         *  object initialized to correspond to the specified file URL.
         */
        constructor (url: URL);
        /**
         * Returns an 
         *  object initialized to correspond to the specified directory.
         */
        constructor (path: string);
        /**
         * Unarchives the contents of a nib file located in the receiver's bundle.
         */
        loadNibNamed(name: string, owner: any, options: Record<UINib.OptionsKey, any> | null):any[] | null;
        /**
         * Returns the file URL for the resource file identified by the specified name and extension and residing in a given bundle directory. 
         */
        url(name: string, ext: string, subpath: string):URL | null;
        /**
         * Returns an array of file URLs for all resources identified by the specified file extension and located in the specified bundle subdirectory.
         */
        urls(ext: string, subpath: string):URL[] | null;
        /**
         * Returns the full pathname for the resource identified by the specified name and file extension.
         */
        path(name: string, ext: string):string;
        /**
         * Returns an array containing the pathnames for all bundle resources having the specified filename extension and residing in the resource subdirectory.
         */
        paths(ext: string, subpath: string):string[];
        /**
         * Returns the location of the specified image resource as an NSURL.
         */
        urlForImageResource(name: NSImage.Name):URL | null;
        /**
         * Returns the location of the specified image resource file. 
         */
        pathForImageResource(name: NSImage.Name):string;
        /**
         * Returns an 
         *  instance associated with the specified name, which can be backed by multiple files representing different resolution versions of the image.
         */
        image(name: NSImage.Name):NSImage | null;
        /**
         * Returns a localized version of the string designated by the specified key and residing in the specified table.
         */
        localizedString(key: string, value: string, tableName: string):string;
        /**
         * Returns the context-sensitive help for the specified key from the bundle's help file. 
         */
        contextHelp(key: NSHelpManager.ContextHelpKey):NSAttributedString | null;
        /**
         * Returns the value associated with the specified key in the receiver's information property list.
         */
        object(key: string):any;
        /**
         * A hint to the system of the relative order for purging tagged sets of resources in the bundle.
         */
        setPreservationPriority(priority: number, tags: Set<String>):void;
        /**
         * Returns the current preservation priority for the specified tag.
         */
        preservationPriority(tag: string):number;
        /**
         * Returns the 
         *  object for the specified name.
         */
        classNamed(className: string):AnyClass | null;
        /**
         * Returns a Boolean value indicating whether the bundle’s executable code could be loaded successfully. 
         */
        preflight():void;
        /**
         * Dynamically loads the bundle’s executable code into a running program, if the code has not already been loaded.
         */
        load():boolean;
        /**
         * Loads the bundle’s executable code and returns any errors.
         */
        loadAndReturnError():void;
        /**
         * Unloads the code associated with the receiver. 
         */
        unload():boolean;
    }
    export class NSBundleResourceRequest extends NSObject {
        /**
         * A reference to the bundle used for storing the downloaded resources. (read-only)
         */
        bundle: Bundle;
        /**
         * A set of strings, with each string specifying a tag used to mark on-demand resources managed by the request. (read-only)
         */
        tags: Set<String>;
        /**
         * A hint to the system of the relative priority of the resource request.
         */
        loadingPriority: number;
        /**
         * A reference to the progress object associated with the specified resource request. (read-only)
         */
        progress: Progress;
        /**
         * Initializes a resource request for managing the on-demand resources marked with any of the set of specified tags. The managed resources are loaded into the main bundle.
         */
        constructor (tags: Set<String>);
        /**
         * Initializes a resource request for managing the on-demand resources marked with any of the set of specified tags. The managed resources are loaded into the specified bundle.
         */
        constructor (tags: Set<String>, bundle: Bundle);
        /**
         * Requests access to the resources marked with the managed tags. If any of the resources are not on the device, they are requested from the App Store.
         */
        beginAccessingResources(completionHandler: ()=>any):void;
        /**
         * Checks whether the resources marked with the tags managed by the request are already on the device. If all of the resources are on the device, you can begin accessing those resources.
         */
        conditionallyBeginAccessingResources(completionHandler: ()=>any):void;
        /**
         * Informs the system that you have finished accessing the resources marked with the tags managed by the request.
         */
        endAccessingResources():void;
    }
    export interface NSKeyValueObservingCustomization extends NSObjectProtocol {
        static automaticallyNotifiesObservers(key: AnyKeyPath):boolean;
        static keyPathsAffectingValue(key: AnyKeyPath):Set<AnyKeyPath>;
    }
    export class NSKeyValueObservation extends NSObject {
        invalidate():void;
    }
    export enum NSKeyValueChange {
        setting = 1,
        insertion = 2,
        removal = 3,
        replacement = 4,
    }
    export class NSKeyValueObservingOptions {
        /**
         * Indicates that the change dictionary should provide the new attribute value, if applicable.
         */
        static new: NSKeyValueObservingOptions;
        /**
         * Indicates that the change dictionary should contain the old attribute value, if applicable.
         */
        static old: NSKeyValueObservingOptions;
        /**
         * If specified, a notification should be sent to the observer immediately, before the observer registration method even returns. 
         */
        static initial: NSKeyValueObservingOptions;
        /**
         * Whether separate notifications should be sent to the observer before and after each change, instead of a single notification after the change. 
         */
        static prior: NSKeyValueObservingOptions;
        constructor (rawValue: number);
    }
    export class NSKeyValueChangeKey {
        /**
         * If the value of the 
         *  entry is 
         * , 
         * , or 
         * , the value of this key is an 
         *  object that contains the indexes of the inserted, removed, or replaced objects.
         */
        static indexesKey: NSKeyValueChangeKey;
        /**
         * An 
         *  object that contains a value corresponding to one of the 
         *  enums, indicating what sort of change has occurred.
         */
        static kindKey: NSKeyValueChangeKey;
        /**
         * If the value of the 
         *  entry is 
         * , and 
         *  was specified when the observer was registered, the value of this key is the new value for the attribute.
         */
        static newKey: NSKeyValueChangeKey;
        /**
         * If the 
         *  option was specified when the observer was registered this notification is sent prior to a change.
         */
        static notificationIsPriorKey: NSKeyValueChangeKey;
        /**
         * If the value of the 
         *  entry is 
         * , and 
         *  was specified when the observer was registered, the value of this key is the value before the attribute was changed.
         */
        static oldKey: NSKeyValueChangeKey;
        constructor (rawValue: string);
    }
    export enum NSKeyValueSetMutationKind {
        union = 1,
        minus = 2,
        intersect = 3,
        set = 4,
    }
    export class Notification {
        /**
         * A tag identifying the notification.
         */
        name: Notification.Name;
        /**
         * An object that the poster wishes to send to observers.
         */
        object: any;
        /**
         * Storage for values or objects related to this notification.
         */
        userInfo: Record<AnyHashable, any> | null;
        /**
         * A textual description of the notification.
         */
        description: string;
        /**
         * A textual description of the notification suitable for debugging.
         */
        debugDescription: string;
        /**
         * A mirror that reflects the notification.
         */
        customMirror: Mirror;
        /**
         * The computed hash value for the notification.
         */
        hashValue: number;
        /**
         * Initializes a new notification.
         */
        constructor (name: Notification.Name, object: any, userInfo: Record<AnyHashable, any> | null);
        hash(hasher: Hasher):void;
    }
    export enum NSRectEdge {
        maxX = 2,
        maxY = 3,
        minX = 0,
        minY = 1,
    }
    export class AlignmentOptions {
        /**
         * Specifies that alignment of the minimum X coordinate should be to the nearest inward integral value.
         */
        static alignMinXInward: AlignmentOptions;
        /**
         * Specifies that alignment of the minimum Y coordinate should be to the nearest inward integral value.
         */
        static alignMinYInward: AlignmentOptions;
        /**
         * Specifies that alignment of the maximum X coordinate should be to the nearest inward integral value.
         */
        static alignMaxXInward: AlignmentOptions;
        /**
         * Specifies that alignment of the maximum X coordinate should be to the nearest inward integral value.
         */
        static alignMaxYInward: AlignmentOptions;
        /**
         * Specifies that alignment of the width should be to the nearest inward integral value.
         */
        static alignWidthInward: AlignmentOptions;
        /**
         * Specifies that alignment of the height should be to the nearest inward integral value.
         */
        static alignHeightInward: AlignmentOptions;
        /**
         * Specifies that alignment of the minimum X coordinate should be to the nearest outward integral value.
         */
        static alignMinXOutward: AlignmentOptions;
        /**
         * Specifies that alignment of the minimum Y coordinate should be to the nearest outward integral value.
         */
        static alignMinYOutward: AlignmentOptions;
        /**
         * Specifies that alignment of the maximum X coordinate should be to the nearest outward integral value.
         */
        static alignMaxXOutward: AlignmentOptions;
        /**
         * Specifies that alignment of the maximum Y coordinate should be to the nearest outward integral value.
         */
        static alignMaxYOutward: AlignmentOptions;
        /**
         * Specifies that alignment of the width should be to the nearest outward integral value.
         */
        static alignWidthOutward: AlignmentOptions;
        /**
         * Specifies that alignment of the height should be to the nearest outward integral value.
         */
        static alignHeightOutward: AlignmentOptions;
        /**
         * Specifies that alignment of the minimum X coordinate should be to the nearest integral value.
         */
        static alignMinXNearest: AlignmentOptions;
        /**
         * Specifies that alignment of the minimum Y coordinate should be to the nearest integral value.
         */
        static alignMinYNearest: AlignmentOptions;
        /**
         * Specifies that alignment of the maximum X coordinate should be to the nearest integral value.
         */
        static alignMaxXNearest: AlignmentOptions;
        /**
         * Specifies that alignment of the maximum Y coordinate should be to the nearest integral value.
         */
        static alignMaxYNearest: AlignmentOptions;
        /**
         * Specifies that alignment of the width should be to the nearest integral value.
         */
        static alignWidthNearest: AlignmentOptions;
        /**
         * Specifies that alignment of the height should be to the nearest integral value.
         */
        static alignHeightNearest: AlignmentOptions;
        /**
         * This option should be included  if the rectangle is in a flipped coordinate system. This allows 0.5 to be treated in a visually consistent way.
         */
        static alignRectFlipped: AlignmentOptions;
        /**
         * Aligns all edges inward. This is the same as 
         * .
         */
        static alignAllEdgesInward: AlignmentOptions;
        /**
         * Aligns all edges outwards. This is the same as 
         * .
         */
        static alignAllEdgesOutward: AlignmentOptions;
        /**
         * Aligns all edges to the nearest value. This is the same as 
         * .
         */
        static alignAllEdgesNearest: AlignmentOptions;
        constructor (rawValue: UInt64);
    }
    export class AffineTransform {
        /**
         * An identity affine transformation matrix.
         */
        static identity: AffineTransform;
        /**
         * An element of the transform matrix that contributes scaling, rotation, and shear.
         */
        m11: CGFloat;
        /**
         * An element of the transform matrix that contributes scaling, rotation, and shear.
         */
        m12: CGFloat;
        /**
         * An element of the transform matrix that contributes scaling, rotation, and shear.
         */
        m21: CGFloat;
        /**
         * An element of the transform matrix that contributes scaling, rotation, and shear.
         */
        m22: CGFloat;
        /**
         * An element of the transform matrix that contributes translation.
         */
        tX: CGFloat;
        /**
         * An element of the transform matrix that contributes translation.
         */
        tY: CGFloat;
        /**
         * A textual description of the transform.
         */
        description: string;
        /**
         * A textual description of the transform, suitable for debugging.
         */
        debugDescription: string;
        /**
         * The computed hash value for the transform.
         */
        hashValue: number;
        /**
         * Creates an affine transformation matrix with identity values.
         */
        constructor ();
        /**
         * Creates an affine transformation matrix from a rotation angle.
         */
        constructor (angle: CGFloat);
        /**
         * Creates an affine transformation matrix from a rotation angle.
         */
        constructor (angle: CGFloat);
        /**
         * Creates an affine transformation matrix from scaling a single value.
         */
        constructor (factor: CGFloat);
        /**
         * Creates an affine transformation matrix from scaling values.
         */
        constructor (x: CGFloat, y: CGFloat);
        /**
         * Creates an affine transformation matrix from translation values.
         */
        constructor (x: CGFloat, y: CGFloat);
        /**
         * Creates an affine transformation.
         */
        constructor (m11: CGFloat, m12: CGFloat, m21: CGFloat, m22: CGFloat, tX: CGFloat, tY: CGFloat);
        /**
         * Creates a new transform by decoding from the specified decoder.
         */
        constructor (decoder: Decoder);
        /**
         * Mutates an affine transformation matrix to apply a rotation.
         */
        rotate(angle: CGFloat):void;
        /**
         * Mutates an affine transformation matrix to apply scaling in both x and y dimensions.
         */
        scale(scale: CGFloat):void;
        /**
         * Mutates an affine transformation matrix to perform the specified translation.
         */
        translate(x: CGFloat, y: CGFloat):void;
        /**
         * Mutates an affine transformation by appending another affine transform.
         */
        append(transform: AffineTransform):void;
        /**
         * Mutates an affine transformation by prepending another affine transform.
         */
        prepend(transform: AffineTransform):void;
        /**
         * Inverts the transformation matrix, if possible.
         */
        invert():void;
        /**
         * Returns an inverted version of the matrix, if possible, or nil if not.
         */
        inverted():AffineTransform | null;
        /**
         * Applies the affine transform to the specified point.
         */
        transform(point: NSPoint):NSPoint;
        /**
         * Encodes this transform into the given encoder.
         */
        encode(encoder: Encoder):void;
        /**
         * Hashes the essential components of this value by feeding them into the specified hasher.
         */
        hash(hasher: Hasher):void;
    }
    export class NSAffineTransform extends NSObject {
        /**
         * The matrix coefficients stored as the transformation matrix.
         */
        transformStruct: NSAffineTransformStruct;
        /**
         * Initializes an affine transform matrix to the identity matrix.
         */
        constructor ();
        /**
         * Initializes the receiver’s matrix using another transform object.
         */
        constructor (transform: AffineTransform);
        /**
         * Applies a rotation factor (measured in degrees) to the receiver’s transformation matrix.
         */
        rotate(angle: number):void;
        /**
         * Applies the specified scaling factor along both x and y axes to the receiver’s transformation matrix.
         */
        scale(scale: number):void;
        /**
         * Applies scaling factors to each axis of the receiver’s transformation matrix.
         */
        scaleX(scaleX: number, scaleY: number):void;
        /**
         * Applies the specified translation factors to the receiver’s transformation matrix.
         */
        translateX(deltaX: number, deltaY: number):void;
        /**
         * Appends the specified matrix to the receiver’s matrix.
         */
        append(transform: AffineTransform):void;
        /**
         * Prepends the specified matrix to the receiver’s matrix.
         */
        prepend(transform: AffineTransform):void;
        /**
         * Replaces the receiver’s matrix with its inverse matrix.
         */
        invert():void;
        /**
         * Applies the receiver’s transform to the specified point and returns the result.
         */
        transform(aPoint: NSPoint):NSPoint;
    }
    export class NSAffineTransformStruct {
        /**
         * An element of the transform matrix that contributes scaling, rotation, and shear.
         */
        m11: number;
        /**
         * An element of the transform matrix that contributes scaling, rotation, and shear.
         */
        m12: number;
        /**
         * An element of the transform matrix that contributes scaling, rotation, and shear.
         */
        m21: number;
        /**
         * An element of the transform matrix that contributes scaling, rotation, and shear.
         */
        m22: number;
        /**
         * An element of the transform matrix that contributes translation.
         */
        tX: number;
        /**
         * An element of the transform matrix that contributes translation.
         */
        tY: number;
        /**
         * Initializes a zero-filled transformation matrix.
         */
        constructor ();
        constructor (m11: number, m12: number, m21: number, m22: number, tX: number, tY: number);
    }
    export class NotificationCenter extends NSObject {
        /**
         * The app’s default notification center.
         */
        static default: NotificationCenter;
        /**
         * Adds an entry to the notification center to receive notifications that passed to the provided block.
         */
        addObserver(name: NSNotification.Name | null, obj: any, queue: OperationQueue | null, block: ()=>any):NSObjectProtocol;
        /**
         * Removes matching entries from the notification center's dispatch table.
         */
        removeObserver(observer: any, aName: NSNotification.Name | null, anObject: any):void;
        /**
         * Posts a given notification to the notification center.
         */
        post(notification: Notification):void;
        /**
         * Returns an asynchronous sequence of notifications produced by this center for a given notification name and optional source object.
         */
        notifications(name: Notification.Name, object: AnyObject | null):NotificationCenter.Notifications;
        /**
         * Returns a publisher that emits events when broadcasting notifications.
         */
        publisher(name: Notification.Name, object: AnyObject | null):NotificationCenter.Publisher;
    }
    /**
     * the same name namespace of class NotificationCenter
     */
    export namespace NotificationCenter {
        export class Notifications {
            /**
             * Returns a Boolean value that indicates whether the asynchronous sequence contains the given element.
             */
            contains(search: Notification):boolean;
            /**
             * Returns a Boolean value that indicates whether all elements produced by the asynchronous sequence satisfy the given predicate.
             */
            allSatisfy(predicate: Bool):boolean;
            /**
             * Returns the first element of the sequence that satisfies the given predicate.
             */
            first(predicate: Bool):Notification | null;
            /**
             * Returns the minimum element in the asynchronous sequence, using the given predicate as the comparison between elements.
             */
            min(areInIncreasingOrder: Bool):Notification | null;
            /**
             * Returns the maximum element in the asynchronous sequence, using the given predicate as the comparison between elements.
             */
            max(areInIncreasingOrder: Bool):Notification | null;
            /**
             * Returns an asynchronous sequence, up to the specified maximum length, containing the initial elements of the base asynchronous sequence.
             */
            prefix(count: number):AsyncPrefixSequence<NotificationCenter.Notifications>;
            /**
             * Omits a specified number of elements from the base asynchronous sequence, then passes through all remaining elements.
             */
            dropFirst(count: number):AsyncDropFirstSequence<NotificationCenter.Notifications>;
            /**
             * Omits elements from the base asynchronous sequence until a given closure returns false, after which it passes through all remaining elements.
             */
            drop(predicate: Bool):AsyncDropWhileSequence<NotificationCenter.Notifications>;
            /**
             * Creates an asynchronous sequence that contains, in order, the elements of the base sequence that satisfy the given predicate.
             */
            filter(isIncluded: Bool):AsyncFilterSequence<NotificationCenter.Notifications>;
            /**
             * Creates the asynchronous iterator that produces elements of this asynchronous sequence.
             */
            makeAsyncIterator():NotificationCenter.Notifications.Iterator;
        }
        /**
         * the same name namespace of class Notifications
         */
        export namespace Notifications {
            export class Iterator {
                /**
                 * Asynchronously advances to the next element and returns it, or ends the sequence if there is no next element.
                 */
                next():Notification | null;
            }
        }
        export class Publisher {
            /**
             * The notification center this publisher uses as a source.
             */
            center: NotificationCenter;
            /**
             * The name of notifications published by this publisher.
             */
            name: Notification.Name;
            /**
             * The object posting the named notfication.
             */
            object: AnyObject | null;
            values: AsyncThrowingPublisher<NotificationCenter.Publisher>;
            /**
             * Creates a publisher that emits events when broadcasting notifications.
             */
            constructor (center: NotificationCenter, name: Notification.Name, object: AnyObject | null);
            allSatisfy(predicate: ()=>any):Publishers.AllSatisfy<NotificationCenter.Publisher>;
            append(...elements: Notification):any;
            assertNoFailure(prefix: string, file: StaticString, line: number):Publishers.AssertNoFailure<NotificationCenter.Publisher>;
            assign(published: any):void;
            breakpoint(receiveSubscription: ()=>any | null, receiveOutput: ()=>any | null, receiveCompletion: ()=>any | null):Publishers.Breakpoint<NotificationCenter.Publisher>;
            breakpointOnError():Publishers.Breakpoint<NotificationCenter.Publisher>;
            buffer(size: number, prefetch: Publishers.PrefetchStrategy, whenFull: Publishers.BufferingStrategy<Never>):Publishers.Buffer<NotificationCenter.Publisher>;
            collect():Publishers.Collect<NotificationCenter.Publisher>;
            contains(output: Notification):Publishers.Contains<NotificationCenter.Publisher>;
            count():Publishers.Count<NotificationCenter.Publisher>;
            drop(predicate: ()=>any):Publishers.DropWhile<NotificationCenter.Publisher>;
            dropFirst(count: number):Publishers.Drop<NotificationCenter.Publisher>;
            eraseToAnyPublisher():any;
            filter(isIncluded: ()=>any):Publishers.Filter<NotificationCenter.Publisher>;
            first():Publishers.First<NotificationCenter.Publisher>;
            handleEvents(receiveSubscription: ()=>any | null, receiveOutput: ()=>any | null, receiveCompletion: ()=>any | null, receiveCancel: ()=>any | null, receiveRequest: ()=>any | null):Publishers.HandleEvents<NotificationCenter.Publisher>;
            ignoreOutput():Publishers.IgnoreOutput<NotificationCenter.Publisher>;
            last():Publishers.Last<NotificationCenter.Publisher>;
            makeConnectable():Publishers.MakeConnectable<NotificationCenter.Publisher>;
            max(areInIncreasingOrder: ()=>any):Publishers.Comparison<NotificationCenter.Publisher>;
            merge(other: NotificationCenter.Publisher):Publishers.MergeMany<NotificationCenter.Publisher>;
            min(areInIncreasingOrder: ()=>any):Publishers.Comparison<NotificationCenter.Publisher>;
            output(index: number):Publishers.Output<NotificationCenter.Publisher>;
            prefix(maxLength: number):Publishers.Output<NotificationCenter.Publisher>;
            prepend(...elements: Notification):any;
            print(prefix: string, stream: TextOutputStream | null):Publishers.Print<NotificationCenter.Publisher>;
            removeDuplicates():Publishers.RemoveDuplicates<NotificationCenter.Publisher>;
            replaceEmpty(output: Notification):Publishers.ReplaceEmpty<NotificationCenter.Publisher>;
            replaceError(output: Notification):Publishers.ReplaceError<NotificationCenter.Publisher>;
            retry(retries: number):Publishers.Retry<NotificationCenter.Publisher>;
            share():Publishers.Share<NotificationCenter.Publisher>;
            sink(receiveCompletion: ()=>any, receiveValue: ()=>any):AnyCancellable;
            tryAllSatisfy(predicate: Bool):Publishers.TryAllSatisfy<NotificationCenter.Publisher>;
            tryContains(predicate: Bool):Publishers.TryContainsWhere<NotificationCenter.Publisher>;
            tryDrop(predicate: Bool):Publishers.TryDropWhile<NotificationCenter.Publisher>;
            tryFilter(isIncluded: Bool):Publishers.TryFilter<NotificationCenter.Publisher>;
            tryFirst(predicate: Bool):Publishers.TryFirstWhere<NotificationCenter.Publisher>;
            tryLast(predicate: Bool):Publishers.TryLastWhere<NotificationCenter.Publisher>;
            tryMax(areInIncreasingOrder: Bool):Publishers.TryComparison<NotificationCenter.Publisher>;
            tryMin(areInIncreasingOrder: Bool):Publishers.TryComparison<NotificationCenter.Publisher>;
            tryPrefix(predicate: Bool):Publishers.TryPrefixWhile<NotificationCenter.Publisher>;
            tryRemoveDuplicates(predicate: Bool):Publishers.TryRemoveDuplicates<NotificationCenter.Publisher>;
        }
    }
    export class NSEdgeInsets {
        /**
         * The distance from the bottom of the source rectangle to the bottom of the result rectangle.
         */
        bottom: number;
        /**
         * The distance from the left side of the source rectangle to the left side of the result rectangle.
         */
        left: number;
        /**
         * The distance from the right side of the source rectangle to the right side of the result rectangle.
         */
        right: number;
        /**
         * The distance from the top of the source rectangle to the top of the result rectangle.
         */
        top: number;
        /**
         * Creates an edge insets structure with the specified inset values.
         */
        constructor (top: number, left: number, bottom: number, right: number);
        /**
         * Creates an edge insets structure.
         */
        constructor ();
    }
    export class NotificationQueue extends NSObject {
        /**
         * Returns the default notification queue for the current thread.
         */
        static default: NotificationQueue;
        /**
         * Initializes and returns a notification queue for the specified notification center.
         */
        constructor (notificationCenter: NotificationCenter);
        /**
         * Adds a notification to the notification queue with a specified posting style, criteria for coalescing, and run loop mode.
         */
        enqueue(notification: Notification, postingStyle: NotificationQueue.PostingStyle, coalesceMask: NotificationQueue.NotificationCoalescing, modes: RunLoop.Mode[] | null):void;
        /**
         * Removes all notifications from the queue that match a provided notification using provided matching criteria.
         */
        dequeueNotifications(notification: Notification, coalesceMask: number):void;
    }
    /**
     * the same name namespace of class NotificationQueue
     */
    export namespace NotificationQueue {
        export class NotificationCoalescing {
            /**
             * Do not coalesce notifications in the queue.
             */
            static none: NotificationQueue.NotificationCoalescing;
            /**
             * Coalesce notifications with the same name.
             */
            static onName: NotificationQueue.NotificationCoalescing;
            /**
             * Coalesce notifications with the same object.
             */
            static onSender: NotificationQueue.NotificationCoalescing;
            constructor (rawValue: number);
        }
        export enum PostingStyle {
            asap = 2,
            whenIdle = 1,
            now = 3,
        }
    }
    export class DistributedNotificationCenter extends NotificationCenter {
        /**
         * Suspends or resumes notification delivery.
         */
        suspended: boolean;
        /**
         * Returns the default distributed notification center, representing the local notification center for the computer.
         */
        static default():DistributedNotificationCenter;
        /**
         * Returns the distributed notification center for a particular notification center type.
         */
        static forType(notificationCenterType: DistributedNotificationCenter.CenterType):DistributedNotificationCenter;
        /**
         * Adds an entry to the notification center's dispatch table with an observer, a selector, and an optional notification name and sender.
         */
        addObserver(observer: any, aSelector: Selector, aName: NSNotification.Name | null, anObject: string):void;
        /**
         * Removes matching entries from the receiver’s dispatch table.
         */
        removeObserver(observer: any, aName: NSNotification.Name | null, anObject: string):void;
        /**
         * Creates a notification, and posts it to the receiver.
         */
        post(aName: NSNotification.Name, anObject: string):void;
        /**
         * Creates a notification with information and an immediate-delivery specifier, and posts it to the receiver.
         */
        postNotificationName(name: NSNotification.Name, object: string, userInfo: Record<AnyHashable, any> | null, deliverImmediately: boolean):void;
    }
    /**
     * the same name namespace of class DistributedNotificationCenter
     */
    export namespace DistributedNotificationCenter {
        export class Options {
            static deliverImmediately: DistributedNotificationCenter.Options;
            static postToAllSessions: DistributedNotificationCenter.Options;
            constructor (rawValue: number);
        }
        export class CenterType {
            /**
             * Distributes notifications to all tasks on the sender’s computer.
             */
            static localNotificationCenterType: DistributedNotificationCenter.CenterType;
            constructor (rawValue: string);
            constructor (rawValue: string);
        }
        export enum SuspensionBehavior {
            drop = 1,
            coalesce = 2,
            hold = 3,
            deliverImmediately = 4,
        }
    }
    export interface NSExtensionRequestHandling extends NSObjectProtocol {
        /**
         * Tells the extension to prepare for a host app’s request.
         */
        beginRequest(context: NSExtensionContext):void;
    }
    export class NSExtensionContext extends NSObject {
        /**
         * The list of input 
         *  objects associated with the context.
         */
        inputItems: any[];
        /**
         * Metadata for populating your share extensions interface.
         */
        intent: INIntent | null;
        /**
         * The minimum size for a Siri hosted view.
         */
        hostedViewMinimumAllowedSize: CGSize;
        /**
         * The maximum size for a Siri hosted view.
         */
        hostedViewMaximumAllowedSize: CGSize;
        notificationActions: UNNotificationAction[];
        /**
         * The active display mode of the widget.
         */
        widgetActiveDisplayMode: NCWidgetDisplayMode;
        /**
         * The largest display mode the widget supports.
         */
        widgetLargestAvailableDisplayMode: NCWidgetDisplayMode;
        /**
         * Tells the host app to complete the app extension request with an array of result items.
         */
        completeRequest(items: any[] | null, completionHandler: ()=>any | null):void;
        /**
         * Tells the host app to cancel the app extension request, with a supplied error.
         */
        cancelRequest(error: Error):void;
        /**
         * Asks the system to open a URL on behalf of the currently running app extension.
         */
        open(URL: URL, completionHandler: ()=>any | null):void;
        /**
         * Tells the system that the Notification Content app extension began playing a media file.
         */
        mediaPlayingStarted():void;
        /**
         * Tells the system that the Notification Content app extension stopped playing a media file. 
         */
        mediaPlayingPaused():void;
        /**
         * Returns a human-readable string describing the data that SiriKit displays to the user when you handle an intent. 
         */
        interfaceParametersDescription():string;
        loadBroadcastingApplicationInfo(handler: ()=>any):void;
        performNotificationDefaultAction():void;
        dismissNotificationContentExtension():void;
        /**
         * Returns the maximum size for the specified widget display mode.
         */
        widgetMaximumSize(displayMode: NCWidgetDisplayMode):CGSize;
    }
    export class NSItemProvider extends NSObject {
        /**
         * The error domain associated with the 
         *  class.
         */
        static errorDomain: string;
        /**
         * The ideal presentation size of the item.
         */
        preferredPresentationSize: CGSize;
        /**
         * The preferred style for presenting the item provider's data.
         */
        preferredPresentationStyle: NSItemProvider.PreferredPresentationStyle;
        /**
         * The filename to use when writing the provided data to a file on disk.
         */
        suggestedName: string;
        teamData: Data | null;
        /**
         * Returns the array of type identifiers for the item provider, listed in the same order in which they were registered.
         */
        registeredTypeIdentifiers: string[];
        /**
         * The custom preview image handler block for the item provider.
         */
        previewImageHandler: NSItemProvider.LoadHandler | null;
        registeredContentTypes: UTType[];
        registeredContentTypesForOpenInPlace: UTType[];
        /**
         * The rectangle (in screen coordinates) occupied by the item in the host app’s source window.
         */
        sourceFrame: NSRect;
        /**
         * The rectangle (in screen coordinates) of the item’s visible content.
         */
        containerFrame: NSRect;
        /**
         * Provides data backed by the contents of an existing file. The system uses the URL’s filename extension to select an appropriate UTI.
         */
        constructor (fileURL: URL!);
        constructor (fileURL: URL, contentType: UTType | null, openInPlace: boolean, coordinated: boolean, visibility: NSItemProviderRepresentationVisibility);
        /**
         * Initializes an item provider with an object, according to the 
         *  type coercion policy.
         */
        constructor (item: NSSecureCoding | null, typeIdentifier: string);
        /**
         * Instantiates an empty item provider to which you can later register a data or file representation.
         */
        constructor ();
        /**
         * Initializes a new item provider, employing a specified object’s type identifiers to specify the data representations eligible to be loaded by the provider.
         */
        constructor (object: NSItemProviderWriting);
        /**
         * Returns a Boolean value indicating whether an item provider can load objects of a given class.
         */
        canLoadObject(aClass: NSItemProviderReading.Type):boolean;
        /**
         * A convenience method equivalent to the 
         *  method, but with the 
         *  parameter set to a value of 
         * .
         */
        hasItemConformingToTypeIdentifier(typeIdentifier: string):boolean;
        /**
         * Returns a Boolean value indicating whether an item provider contains a data representation conforming to a specified uniform type identifier (UTI) and to specified open-in-place behavior.
         */
        hasRepresentationConforming(typeIdentifier: string, fileOptions: NSItemProviderFileOptions):boolean;
        /**
         * Loads the item’s data and coerces it (as needed) to the specified type.
         */
        loadItem(typeIdentifier: string, options: Record<AnyHashable, any> | null, completionHandler: NSItemProvider.CompletionHandler | null):void;
        /**
         * Asynchronously copies the provided, typed data into a 
         *  object, returning a 
         *  object.
         */
        loadDataRepresentation(typeIdentifier: string, completionHandler: ()=>any):Progress;
        /**
         * Asynchronously writes a copy of the provided, typed data to a temporary file, returning a 
         *  object.
         */
        loadFileRepresentation(typeIdentifier: string, completionHandler: ()=>any):Progress;
        /**
         * Asynchronously opens a file in place, if doing so is possible, returning a 
         *  object.
         */
        loadInPlaceFileRepresentation(typeIdentifier: string, completionHandler: ()=>any):Progress;
        /**
         * Asynchronously loads an object of a specified class to an item provider, returning a 
         *  object.
         */
        loadObject(aClass: NSItemProviderReading.Type, completionHandler: ()=>any):Progress;
        /**
         * Loads the preview image for the item that is represented by the item provider.
         */
        loadPreviewImage(options: Record<AnyHashable, any>, completionHandler: NSItemProvider.CompletionHandler!):void;
        /**
         * Registers a CloudKit share for the user to modify.
         */
        registerCloudKitShare(share: CKShare, container: CKContainer):void;
        registerCKShare(share: CKShare, container: CKContainer, allowedSharingOptions: CKAllowedSharingOptions):void;
        /**
         * Registers a data-backed representation for an item, specifiying item visibility and a load handler.
         */
        registerDataRepresentation(typeIdentifier: string, visibility: NSItemProviderRepresentationVisibility, loadHandler: ()=>any | null):void;
        /**
         * Lazily registers an item, according to the 
         *  type coercion policy.
         */
        registerItem(typeIdentifier: string, loadHandler: NSItemProvider.LoadHandler):void;
        /**
         * Registers a file-backed representation for an item, specifying file options, item visibility, and a load handler.
         */
        registerFileRepresentation(typeIdentifier: string, fileOptions: NSItemProviderFileOptions, visibility: NSItemProviderRepresentationVisibility, loadHandler: ()=>any | null):void;
        /**
         * Adds representations of a specified object to an item provider, based on the object’s implementation of the 
         *  protocol and adhering to a visibility specification.
         */
        registerObject(object: NSItemProviderWriting, visibility: NSItemProviderRepresentationVisibility):void;
    }
    /**
     * the same name namespace of class NSItemProvider
     */
    export namespace NSItemProvider {
        export enum PreferredPresentationStyle {
            unspecified = 0,
            inline = 1,
            attachment = 2,
        }
        export enum ErrorCode {
            itemUnavailableError = -1000,
            unavailableCoercionError = -1200,
            unexpectedValueClassError = -1100,
            unknownError = -1,
        }
    }
    export class NSItemProviderFileOptions {
        /**
         * A data-access specification declaring that items should open in place, rather than being copied.
         */
        static openInPlace: NSItemProviderFileOptions;
        constructor (rawValue: number);
    }
    export interface NSItemProviderReading extends NSObjectProtocol {
        /**
         * Creates a new instance of a class using the given data and UTI string.
         */
        static object(data: Data, typeIdentifier: string):Self;
        /**
         * An array of UTI strings representing the data types supported by the class.
         */
        static readableTypeIdentifiersForItemProvider: string[];
    }
    export interface NSItemProviderWriting extends NSObjectProtocol {
        /**
         * Loads data of a particular type, identified by the given UTI.
         */
        loadData(typeIdentifier: string, completionHandler: ()=>any):Progress | null;
        /**
         * An array of UTI strings representing the types of data that can be loaded for an item provider.
         */
        static writableTypeIdentifiersForItemProvider: string[];
        /**
         * Asks the item provider for the default representation visibility specification for the given UTI.
         */
        static itemProviderVisibilityForRepresentation(typeIdentifier: string):NSItemProviderRepresentationVisibility;
    }
    export enum NSItemProviderRepresentationVisibility {
        all = 0,
        group = 2,
        ownProcess = 3,
        team = 1,
    }
    export class NSExtensionItem extends NSObject {
        /**
         * An optional title for the item.
         */
        attributedTitle: NSAttributedString | null;
        /**
         * An optional dictionary of keys and values corresponding to the extension item’s properties.
         */
        userInfo: Record<AnyHashable, any> | null;
        /**
         * An optional array of media data associated with the extension item.
         */
        attachments: NSItemProvider[] | null;
        /**
         * An optional string describing the extension item content.
         */
        attributedContentText: NSAttributedString | null;
    }
    export class NSError extends NSObject {
        /**
         * The error code.
         */
        code: number;
        /**
         * A string containing the error domain.
         */
        domain: string;
        /**
         * The user info dictionary. 
         */
        userInfo: Record<string, any>;
        /**
         * A string containing the localized description of the error.
         */
        localizedDescription: string;
        /**
         * An array containing the localized titles of buttons appropriate for displaying in an alert panel.
         */
        localizedRecoveryOptions: string[] | null;
        /**
         * A string containing the localized recovery suggestion for the error.
         */
        localizedRecoverySuggestion: string;
        /**
         * A string containing the localized explanation of the reason for the error.
         */
        localizedFailureReason: string;
        /**
         * The object in the user info dictionary corresponding to the 
         *  key.
         */
        recoveryAttempter: any;
        /**
         * A string to display in response to an alert panel help anchor button being pressed.
         */
        helpAnchor: string;
        underlyingErrors: Error[];
        /**
         * Returns an 
         *  object initialized for a given domain and code with a given 
         *  dictionary. 
         */
        constructor (domain: string, code: number, dict: Record<string, any> | null);
        /**
         * Specifies a block to call when the corresponding property is not present in the user info dictionary.
         */
        static setUserInfoValueProvider(errorDomain: string, provider: ()=>any | null):void;
        /**
         * Returns any user info provider specified for a given error domain.
         */
        static userInfoValueProvider(errorDomain: string):()=>any | null;
        /**
         * Returns a properly formatted error object with a 
         *  error code.
         */
        static fileProviderErrorForCollision(existingItem: NSFileProviderItem):Self;
        static fileProviderErrorForNonExistentItem(itemIdentifier: NSFileProviderItemIdentifier):Self;
        static fileProviderErrorForRejectedDeletion(updatedVersion: NSFileProviderItem):Self;
    }
    export class UserInfoKey {
        static NSURLErrorKey: ErrorUserInfoKey;
        static filePathErrorKey: ErrorUserInfoKey;
        static helpAnchorErrorKey: ErrorUserInfoKey;
        static localizedDescriptionKey: ErrorUserInfoKey;
        static localizedFailureReasonErrorKey: ErrorUserInfoKey;
        static localizedRecoveryOptionsErrorKey: ErrorUserInfoKey;
        static localizedRecoverySuggestionErrorKey: ErrorUserInfoKey;
        static recoveryAttempterErrorKey: ErrorUserInfoKey;
        static stringEncodingErrorKey: ErrorUserInfoKey;
        static underlyingErrorKey: ErrorUserInfoKey;
        hashValue: number;
        rawValue: string;
        constructor (rawValue: string);
        hash(hasher: Hasher):void;
    }
    export class CocoaError {
        static ServiceApplicationLaunchFailedError: CocoaError.Code;
        static ServiceApplicationNotFoundError: CocoaError.Code;
        static ServiceInvalidPasteboardDataError: CocoaError.Code;
        static ServiceMalformedServiceDictionaryError: CocoaError.Code;
        static ServiceMiscellaneousError: CocoaError.Code;
        static ServiceRequestTimedOutError: CocoaError.Code;
        static SharingServiceNotConfiguredError: CocoaError.Code;
        static TextReadInapplicableDocumentTypeError: CocoaError.Code;
        static TextWriteInapplicableDocumentTypeError: CocoaError.Code;
        static coderInvalidValue: CocoaError.Code;
        static coderReadCorrupt: CocoaError.Code;
        static coderReadCorruptError: CocoaError.Code;
        static coderValueNotFound: CocoaError.Code;
        static coderValueNotFoundError: CocoaError.Code;
        static coreData: CocoaError.Code;
        static coreDataError: CocoaError.Code;
        static entityMigrationPolicy: CocoaError.Code;
        static entityMigrationPolicyError: CocoaError.Code;
        static errorDomain: string;
        static executableArchitectureMismatch: CocoaError.Code;
        static executableArchitectureMismatchError: CocoaError.Code;
        static executableLink: CocoaError.Code;
        static executableLinkError: CocoaError.Code;
        static executableLoad: CocoaError.Code;
        static executableLoadError: CocoaError.Code;
        static executableNotLoadable: CocoaError.Code;
        static executableNotLoadableError: CocoaError.Code;
        static executableRuntimeMismatch: CocoaError.Code;
        static executableRuntimeMismatchError: CocoaError.Code;
        static externalRecordImport: CocoaError.Code;
        static externalRecordImportError: CocoaError.Code;
        static featureUnsupported: CocoaError.Code;
        static featureUnsupportedError: CocoaError.Code;
        static fileLocking: CocoaError.Code;
        static fileLockingError: CocoaError.Code;
        static fileManagerUnmountBusy: CocoaError.Code;
        static fileManagerUnmountBusyError: CocoaError.Code;
        static fileManagerUnmountUnknown: CocoaError.Code;
        static fileManagerUnmountUnknownError: CocoaError.Code;
        static fileNoSuchFile: CocoaError.Code;
        static fileNoSuchFileError: CocoaError.Code;
        static fileReadCorruptFile: CocoaError.Code;
        static fileReadCorruptFileError: CocoaError.Code;
        static fileReadInapplicableStringEncoding: CocoaError.Code;
        static fileReadInapplicableStringEncodingError: CocoaError.Code;
        static fileReadInvalidFileName: CocoaError.Code;
        static fileReadInvalidFileNameError: CocoaError.Code;
        static fileReadNoPermission: CocoaError.Code;
        static fileReadNoPermissionError: CocoaError.Code;
        static fileReadNoSuchFile: CocoaError.Code;
        static fileReadNoSuchFileError: CocoaError.Code;
        static fileReadTooLarge: CocoaError.Code;
        static fileReadTooLargeError: CocoaError.Code;
        static fileReadUnknown: CocoaError.Code;
        static fileReadUnknownError: CocoaError.Code;
        static fileReadUnknownStringEncoding: CocoaError.Code;
        static fileReadUnknownStringEncodingError: CocoaError.Code;
        static fileReadUnsupportedScheme: CocoaError.Code;
        static fileReadUnsupportedSchemeError: CocoaError.Code;
        static fileWriteFileExists: CocoaError.Code;
        static fileWriteFileExistsError: CocoaError.Code;
        static fileWriteInapplicableStringEncoding: CocoaError.Code;
        static fileWriteInapplicableStringEncodingError: CocoaError.Code;
        static fileWriteInvalidFileName: CocoaError.Code;
        static fileWriteInvalidFileNameError: CocoaError.Code;
        static fileWriteNoPermission: CocoaError.Code;
        static fileWriteNoPermissionError: CocoaError.Code;
        static fileWriteOutOfSpace: CocoaError.Code;
        static fileWriteOutOfSpaceError: CocoaError.Code;
        static fileWriteUnknown: CocoaError.Code;
        static fileWriteUnknownError: CocoaError.Code;
        static fileWriteUnsupportedScheme: CocoaError.Code;
        static fileWriteUnsupportedSchemeError: CocoaError.Code;
        static fileWriteVolumeReadOnly: CocoaError.Code;
        static fileWriteVolumeReadOnlyError: CocoaError.Code;
        static fontAssetDownloadError: CocoaError.Code;
        static formatting: CocoaError.Code;
        static formattingError: CocoaError.Code;
        static inferredMappingModel: CocoaError.Code;
        static inferredMappingModelError: CocoaError.Code;
        static keyValueValidation: CocoaError.Code;
        static keyValueValidationError: CocoaError.Code;
        static managedObjectConstraintMerge: CocoaError.Code;
        static managedObjectConstraintMergeError: CocoaError.Code;
        static managedObjectContextLocking: CocoaError.Code;
        static managedObjectContextLockingError: CocoaError.Code;
        static managedObjectExternalRelationship: CocoaError.Code;
        static managedObjectExternalRelationshipError: CocoaError.Code;
        static managedObjectMerge: CocoaError.Code;
        static managedObjectMergeError: CocoaError.Code;
        static managedObjectReferentialIntegrity: CocoaError.Code;
        static managedObjectReferentialIntegrityError: CocoaError.Code;
        static managedObjectValidation: CocoaError.Code;
        static managedObjectValidationError: CocoaError.Code;
        static migration: CocoaError.Code;
        static migrationCancelled: CocoaError.Code;
        static migrationCancelledError: CocoaError.Code;
        static migrationError: CocoaError.Code;
        static migrationManagerDestinationStore: CocoaError.Code;
        static migrationManagerDestinationStoreError: CocoaError.Code;
        static migrationManagerSourceStore: CocoaError.Code;
        static migrationManagerSourceStoreError: CocoaError.Code;
        static migrationMissingMappingModel: CocoaError.Code;
        static migrationMissingMappingModelError: CocoaError.Code;
        static migrationMissingSourceModel: CocoaError.Code;
        static migrationMissingSourceModelError: CocoaError.Code;
        static persistentStoreCoordinatorLocking: CocoaError.Code;
        static persistentStoreCoordinatorLockingError: CocoaError.Code;
        static persistentStoreIncompatibleSchema: CocoaError.Code;
        static persistentStoreIncompatibleSchemaError: CocoaError.Code;
        static persistentStoreIncompatibleVersionHash: CocoaError.Code;
        static persistentStoreIncompatibleVersionHashError: CocoaError.Code;
        static persistentStoreIncompleteSave: CocoaError.Code;
        static persistentStoreIncompleteSaveError: CocoaError.Code;
        static persistentStoreInvalidType: CocoaError.Code;
        static persistentStoreInvalidTypeError: CocoaError.Code;
        static persistentStoreOpen: CocoaError.Code;
        static persistentStoreOpenError: CocoaError.Code;
        static persistentStoreOperation: CocoaError.Code;
        static persistentStoreOperationError: CocoaError.Code;
        static persistentStoreSave: CocoaError.Code;
        static persistentStoreSaveConflictsError: CocoaError.Code;
        static persistentStoreSaveError: CocoaError.Code;
        static persistentStoreTimeout: CocoaError.Code;
        static persistentStoreTimeoutError: CocoaError.Code;
        static persistentStoreTypeMismatch: CocoaError.Code;
        static persistentStoreTypeMismatchError: CocoaError.Code;
        static persistentStoreUnsupportedRequestType: CocoaError.Code;
        static persistentStoreUnsupportedRequestTypeError: CocoaError.Code;
        static propertyListReadCorrupt: CocoaError.Code;
        static propertyListReadCorruptError: CocoaError.Code;
        static propertyListReadStream: CocoaError.Code;
        static propertyListReadStreamError: CocoaError.Code;
        static propertyListReadUnknownVersion: CocoaError.Code;
        static propertyListReadUnknownVersionError: CocoaError.Code;
        static propertyListWriteInvalid: CocoaError.Code;
        static propertyListWriteInvalidError: CocoaError.Code;
        static propertyListWriteStream: CocoaError.Code;
        static propertyListWriteStreamError: CocoaError.Code;
        static serviceApplicationLaunchFailed: CocoaError.Code;
        static serviceApplicationLaunchFailedError: CocoaError.Code;
        static serviceApplicationNotFound: CocoaError.Code;
        static serviceApplicationNotFoundError: CocoaError.Code;
        static serviceInvalidPasteboardData: CocoaError.Code;
        static serviceInvalidPasteboardDataError: CocoaError.Code;
        static serviceMalformedServiceDictionary: CocoaError.Code;
        static serviceMalformedServiceDictionaryError: CocoaError.Code;
        static serviceMiscellaneous: CocoaError.Code;
        static serviceMiscellaneousError: CocoaError.Code;
        static serviceRequestTimedOut: CocoaError.Code;
        static serviceRequestTimedOutError: CocoaError.Code;
        static sharingServiceNotConfigured: CocoaError.Code;
        static sharingServiceNotConfiguredError: CocoaError.Code;
        static sqlite: CocoaError.Code;
        static sqliteError: CocoaError.Code;
        static textReadInapplicableDocumentType: CocoaError.Code;
        static textReadInapplicableDocumentTypeError: CocoaError.Code;
        static textWriteInapplicableDocumentType: CocoaError.Code;
        static textWriteInapplicableDocumentTypeError: CocoaError.Code;
        static ubiquitousFileNotUploadedDueToQuota: CocoaError.Code;
        static ubiquitousFileNotUploadedDueToQuotaError: CocoaError.Code;
        static ubiquitousFileUbiquityServerNotAvailable: CocoaError.Code;
        static ubiquitousFileUnavailable: CocoaError.Code;
        static ubiquitousFileUnavailableError: CocoaError.Code;
        static userActivityConnectionUnavailable: CocoaError.Code;
        static userActivityConnectionUnavailableError: CocoaError.Code;
        static userActivityHandoffFailed: CocoaError.Code;
        static userActivityHandoffFailedError: CocoaError.Code;
        static userActivityHandoffUserInfoTooLarge: CocoaError.Code;
        static userActivityHandoffUserInfoTooLargeError: CocoaError.Code;
        static userActivityRemoteApplicationTimedOut: CocoaError.Code;
        static userActivityRemoteApplicationTimedOutError: CocoaError.Code;
        static userCancelled: CocoaError.Code;
        static userCancelledError: CocoaError.Code;
        static validationDateTooLate: CocoaError.Code;
        static validationDateTooLateError: CocoaError.Code;
        static validationDateTooSoon: CocoaError.Code;
        static validationDateTooSoonError: CocoaError.Code;
        static validationInvalidDate: CocoaError.Code;
        static validationInvalidDateError: CocoaError.Code;
        static validationMissingMandatoryProperty: CocoaError.Code;
        static validationMissingMandatoryPropertyError: CocoaError.Code;
        static validationMultipleErrors: CocoaError.Code;
        static validationMultipleErrorsError: CocoaError.Code;
        static validationNumberTooLarge: CocoaError.Code;
        static validationNumberTooLargeError: CocoaError.Code;
        static validationNumberTooSmall: CocoaError.Code;
        static validationNumberTooSmallError: CocoaError.Code;
        static validationRelationshipDeniedDelete: CocoaError.Code;
        static validationRelationshipDeniedDeleteError: CocoaError.Code;
        static validationRelationshipExceedsMaximumCount: CocoaError.Code;
        static validationRelationshipExceedsMaximumCountError: CocoaError.Code;
        static validationRelationshipLacksMinimumCount: CocoaError.Code;
        static validationRelationshipLacksMinimumCountError: CocoaError.Code;
        static validationStringPatternMatching: CocoaError.Code;
        static validationStringPatternMatchingError: CocoaError.Code;
        static validationStringTooLong: CocoaError.Code;
        static validationStringTooLongError: CocoaError.Code;
        static validationStringTooShort: CocoaError.Code;
        static validationStringTooShortError: CocoaError.Code;
        static xpcConnectionInterrupted: CocoaError.Code;
        static xpcConnectionInvalid: CocoaError.Code;
        static xpcConnectionReplyInvalid: CocoaError.Code;
        affectedObjects: AnyObject[] | null;
        affectedStores: AnyObject[] | null;
        code: Code;
        errorCode: number;
        errorUserInfo: Record<string, any>;
        /**
         * The file path associated with the error, if any.
         */
        filePath: string;
        hashValue: number;
        isCoderError: boolean;
        isExecutableError: boolean;
        isFileError: boolean;
        isFontError: boolean;
        isFormattingError: boolean;
        isPropertyListError: boolean;
        isServiceError: boolean;
        isSharingServiceError: boolean;
        isTextReadWriteError: boolean;
        isUbiquitousFileError: boolean;
        isUserActivityError: boolean;
        isValidationError: boolean;
        isXPCConnectionError: boolean;
        localizedDescription: string;
        persistentStoreSaveConflicts: NSMergeConflict[] | null;
        /**
         * The string encoding associated with this error, if any.
         */
        stringEncoding: String.Encoding | null;
        /**
         * The underlying error behind this error, if any.
         */
        underlying: Error | null;
        underlyingErrors: Error[];
        /**
         * The URL associated with this error, if any.
         */
        url: URL | null;
        userInfo: Record<string, any>;
        validationKey: string;
        validationObject: any;
        validationPredicate: NSPredicate | null;
        validationValue: any;
        constructor (code: Code, userInfo: Record<string, any>);
        static error(code: CocoaError.Code, userInfo: Record<AnyHashable, any> | null, url: URL | null):Error;
        hash(hasher: Hasher):void;
    }
    /**
     * the same name namespace of class CocoaError
     */
    export namespace CocoaError {
        export class Code {
            static ServiceApplicationLaunchFailedError: CocoaError.Code;
            static ServiceApplicationNotFoundError: CocoaError.Code;
            static ServiceInvalidPasteboardDataError: CocoaError.Code;
            static ServiceMalformedServiceDictionaryError: CocoaError.Code;
            static ServiceMiscellaneousError: CocoaError.Code;
            static ServiceRequestTimedOutError: CocoaError.Code;
            static SharingServiceNotConfiguredError: CocoaError.Code;
            static TextReadInapplicableDocumentTypeError: CocoaError.Code;
            static TextWriteInapplicableDocumentTypeError: CocoaError.Code;
            static coderInvalidValue: CocoaError.Code;
            static coderReadCorrupt: CocoaError.Code;
            static coderReadCorruptError: CocoaError.Code;
            static coderValueNotFound: CocoaError.Code;
            static coderValueNotFoundError: CocoaError.Code;
            static coreData: CocoaError.Code;
            static coreDataError: CocoaError.Code;
            static entityMigrationPolicy: CocoaError.Code;
            static entityMigrationPolicyError: CocoaError.Code;
            static executableArchitectureMismatch: CocoaError.Code;
            static executableArchitectureMismatchError: CocoaError.Code;
            static executableLink: CocoaError.Code;
            static executableLinkError: CocoaError.Code;
            static executableLoad: CocoaError.Code;
            static executableLoadError: CocoaError.Code;
            static executableNotLoadable: CocoaError.Code;
            static executableNotLoadableError: CocoaError.Code;
            static executableRuntimeMismatch: CocoaError.Code;
            static executableRuntimeMismatchError: CocoaError.Code;
            static externalRecordImport: CocoaError.Code;
            static externalRecordImportError: CocoaError.Code;
            static featureUnsupported: CocoaError.Code;
            static featureUnsupportedError: CocoaError.Code;
            static fileLocking: CocoaError.Code;
            static fileLockingError: CocoaError.Code;
            static fileManagerUnmountBusy: CocoaError.Code;
            static fileManagerUnmountBusyError: CocoaError.Code;
            static fileManagerUnmountUnknown: CocoaError.Code;
            static fileManagerUnmountUnknownError: CocoaError.Code;
            static fileNoSuchFile: CocoaError.Code;
            static fileNoSuchFileError: CocoaError.Code;
            static fileReadCorruptFile: CocoaError.Code;
            static fileReadCorruptFileError: CocoaError.Code;
            static fileReadInapplicableStringEncoding: CocoaError.Code;
            static fileReadInapplicableStringEncodingError: CocoaError.Code;
            static fileReadInvalidFileName: CocoaError.Code;
            static fileReadInvalidFileNameError: CocoaError.Code;
            static fileReadNoPermission: CocoaError.Code;
            static fileReadNoPermissionError: CocoaError.Code;
            static fileReadNoSuchFile: CocoaError.Code;
            static fileReadNoSuchFileError: CocoaError.Code;
            static fileReadTooLarge: CocoaError.Code;
            static fileReadTooLargeError: CocoaError.Code;
            static fileReadUnknown: CocoaError.Code;
            static fileReadUnknownError: CocoaError.Code;
            static fileReadUnknownStringEncoding: CocoaError.Code;
            static fileReadUnknownStringEncodingError: CocoaError.Code;
            static fileReadUnsupportedScheme: CocoaError.Code;
            static fileReadUnsupportedSchemeError: CocoaError.Code;
            static fileWriteFileExists: CocoaError.Code;
            static fileWriteFileExistsError: CocoaError.Code;
            static fileWriteInapplicableStringEncoding: CocoaError.Code;
            static fileWriteInapplicableStringEncodingError: CocoaError.Code;
            static fileWriteInvalidFileName: CocoaError.Code;
            static fileWriteInvalidFileNameError: CocoaError.Code;
            static fileWriteNoPermission: CocoaError.Code;
            static fileWriteNoPermissionError: CocoaError.Code;
            static fileWriteOutOfSpace: CocoaError.Code;
            static fileWriteOutOfSpaceError: CocoaError.Code;
            static fileWriteUnknown: CocoaError.Code;
            static fileWriteUnknownError: CocoaError.Code;
            static fileWriteUnsupportedScheme: CocoaError.Code;
            static fileWriteUnsupportedSchemeError: CocoaError.Code;
            static fileWriteVolumeReadOnly: CocoaError.Code;
            static fileWriteVolumeReadOnlyError: CocoaError.Code;
            static fontAssetDownloadError: CocoaError.Code;
            static formatting: CocoaError.Code;
            static formattingError: CocoaError.Code;
            static inferredMappingModel: CocoaError.Code;
            static inferredMappingModelError: CocoaError.Code;
            static keyValueValidation: CocoaError.Code;
            static keyValueValidationError: CocoaError.Code;
            static managedObjectConstraintMerge: CocoaError.Code;
            static managedObjectConstraintMergeError: CocoaError.Code;
            static managedObjectContextLocking: CocoaError.Code;
            static managedObjectContextLockingError: CocoaError.Code;
            static managedObjectExternalRelationship: CocoaError.Code;
            static managedObjectExternalRelationshipError: CocoaError.Code;
            static managedObjectMerge: CocoaError.Code;
            static managedObjectMergeError: CocoaError.Code;
            static managedObjectReferentialIntegrity: CocoaError.Code;
            static managedObjectReferentialIntegrityError: CocoaError.Code;
            static managedObjectValidation: CocoaError.Code;
            static managedObjectValidationError: CocoaError.Code;
            static migration: CocoaError.Code;
            static migrationCancelled: CocoaError.Code;
            static migrationCancelledError: CocoaError.Code;
            static migrationError: CocoaError.Code;
            static migrationManagerDestinationStore: CocoaError.Code;
            static migrationManagerDestinationStoreError: CocoaError.Code;
            static migrationManagerSourceStore: CocoaError.Code;
            static migrationManagerSourceStoreError: CocoaError.Code;
            static migrationMissingMappingModel: CocoaError.Code;
            static migrationMissingMappingModelError: CocoaError.Code;
            static migrationMissingSourceModel: CocoaError.Code;
            static migrationMissingSourceModelError: CocoaError.Code;
            static persistentStoreCoordinatorLocking: CocoaError.Code;
            static persistentStoreCoordinatorLockingError: CocoaError.Code;
            static persistentStoreIncompatibleSchema: CocoaError.Code;
            static persistentStoreIncompatibleSchemaError: CocoaError.Code;
            static persistentStoreIncompatibleVersionHash: CocoaError.Code;
            static persistentStoreIncompatibleVersionHashError: CocoaError.Code;
            static persistentStoreIncompleteSave: CocoaError.Code;
            static persistentStoreIncompleteSaveError: CocoaError.Code;
            static persistentStoreInvalidType: CocoaError.Code;
            static persistentStoreInvalidTypeError: CocoaError.Code;
            static persistentStoreOpen: CocoaError.Code;
            static persistentStoreOpenError: CocoaError.Code;
            static persistentStoreOperation: CocoaError.Code;
            static persistentStoreOperationError: CocoaError.Code;
            static persistentStoreSave: CocoaError.Code;
            static persistentStoreSaveConflicts: CocoaError.Code;
            static persistentStoreSaveConflictsError: CocoaError.Code;
            static persistentStoreSaveError: CocoaError.Code;
            static persistentStoreTimeout: CocoaError.Code;
            static persistentStoreTimeoutError: CocoaError.Code;
            static persistentStoreTypeMismatch: CocoaError.Code;
            static persistentStoreTypeMismatchError: CocoaError.Code;
            static persistentStoreUnsupportedRequestType: CocoaError.Code;
            static persistentStoreUnsupportedRequestTypeError: CocoaError.Code;
            static propertyListReadCorrupt: CocoaError.Code;
            static propertyListReadCorruptError: CocoaError.Code;
            static propertyListReadStream: CocoaError.Code;
            static propertyListReadStreamError: CocoaError.Code;
            static propertyListReadUnknownVersion: CocoaError.Code;
            static propertyListReadUnknownVersionError: CocoaError.Code;
            static propertyListWriteInvalid: CocoaError.Code;
            static propertyListWriteInvalidError: CocoaError.Code;
            static propertyListWriteStream: CocoaError.Code;
            static propertyListWriteStreamError: CocoaError.Code;
            static serviceApplicationLaunchFailed: CocoaError.Code;
            static serviceApplicationLaunchFailedError: CocoaError.Code;
            static serviceApplicationNotFound: CocoaError.Code;
            static serviceApplicationNotFoundError: CocoaError.Code;
            static serviceInvalidPasteboardData: CocoaError.Code;
            static serviceInvalidPasteboardDataError: CocoaError.Code;
            static serviceMalformedServiceDictionary: CocoaError.Code;
            static serviceMalformedServiceDictionaryError: CocoaError.Code;
            static serviceMiscellaneous: CocoaError.Code;
            static serviceMiscellaneousError: CocoaError.Code;
            static serviceRequestTimedOut: CocoaError.Code;
            static serviceRequestTimedOutError: CocoaError.Code;
            static sharingServiceNotConfigured: CocoaError.Code;
            static sharingServiceNotConfiguredError: CocoaError.Code;
            static sqlite: CocoaError.Code;
            static sqliteError: CocoaError.Code;
            static textReadInapplicableDocumentType: CocoaError.Code;
            static textReadInapplicableDocumentTypeError: CocoaError.Code;
            static textWriteInapplicableDocumentType: CocoaError.Code;
            static textWriteInapplicableDocumentTypeError: CocoaError.Code;
            static ubiquitousFileNotUploadedDueToQuota: CocoaError.Code;
            static ubiquitousFileNotUploadedDueToQuotaError: CocoaError.Code;
            static ubiquitousFileUbiquityServerNotAvailable: CocoaError.Code;
            static ubiquitousFileUnavailable: CocoaError.Code;
            static ubiquitousFileUnavailableError: CocoaError.Code;
            static userActivityConnectionUnavailable: CocoaError.Code;
            static userActivityConnectionUnavailableError: CocoaError.Code;
            static userActivityHandoffFailed: CocoaError.Code;
            static userActivityHandoffFailedError: CocoaError.Code;
            static userActivityHandoffUserInfoTooLarge: CocoaError.Code;
            static userActivityHandoffUserInfoTooLargeError: CocoaError.Code;
            static userActivityRemoteApplicationTimedOut: CocoaError.Code;
            static userActivityRemoteApplicationTimedOutError: CocoaError.Code;
            static userCancelled: CocoaError.Code;
            static userCancelledError: CocoaError.Code;
            static validationDateTooLate: CocoaError.Code;
            static validationDateTooLateError: CocoaError.Code;
            static validationDateTooSoon: CocoaError.Code;
            static validationDateTooSoonError: CocoaError.Code;
            static validationInvalidDate: CocoaError.Code;
            static validationInvalidDateError: CocoaError.Code;
            static validationMissingMandatoryProperty: CocoaError.Code;
            static validationMissingMandatoryPropertyError: CocoaError.Code;
            static validationMultipleErrors: CocoaError.Code;
            static validationMultipleErrorsError: CocoaError.Code;
            static validationNumberTooLarge: CocoaError.Code;
            static validationNumberTooLargeError: CocoaError.Code;
            static validationNumberTooSmall: CocoaError.Code;
            static validationNumberTooSmallError: CocoaError.Code;
            static validationRelationshipDeniedDelete: CocoaError.Code;
            static validationRelationshipDeniedDeleteError: CocoaError.Code;
            static validationRelationshipExceedsMaximumCount: CocoaError.Code;
            static validationRelationshipExceedsMaximumCountError: CocoaError.Code;
            static validationRelationshipLacksMinimumCount: CocoaError.Code;
            static validationRelationshipLacksMinimumCountError: CocoaError.Code;
            static validationStringPatternMatching: CocoaError.Code;
            static validationStringPatternMatchingError: CocoaError.Code;
            static validationStringTooLong: CocoaError.Code;
            static validationStringTooLongError: CocoaError.Code;
            static validationStringTooShort: CocoaError.Code;
            static validationStringTooShortError: CocoaError.Code;
            static xpcConnectionInterrupted: CocoaError.Code;
            static xpcConnectionInvalid: CocoaError.Code;
            static xpcConnectionReplyInvalid: CocoaError.Code;
            hashValue: number;
            rawValue: number;
            constructor (rawValue: number);
            hash(hasher: Hasher):void;
        }
    }
    export class MachError {
        /**
         * The operation was aborted.  Ipc code will catch this and reflect it as a message error.
         */
        static aborted: MachError.Code;
        /**
         * The receive right is already a member of the portset.
         */
        static alreadyInSet: MachError.Code;
        /**
         * A thread is attempting to wait for an event for which there is already a waiting thread.
         */
        static alreadyWaiting: MachError.Code;
        /**
         * During a page fault, indicates that the page was rejected as a result of a signature check.
         */
        static codesignError: MachError.Code;
        /**
         * An attempt was made to destroy the default processor set.
         */
        static defaultSet: MachError.Code;
        static errorDomain: string;
        /**
         * An attempt was made to fetch an exception port that is protected, or to abort a thread while processing a protected exception.
         */
        static exceptionProtected: MachError.Code;
        /**
         * The function could not be performed.  A catch-all.
         */
        static failure: MachError.Code;
        /**
         * Specified address is not currently valid.
         */
        static invalidAddress: MachError.Code;
        /**
         * The function requested was not applicable to this type of argument, or an argument is invalid.
         */
        static invalidArgument: MachError.Code;
        /**
         * The supplied (port) capability is improper.
         */
        static invalidCapability: MachError.Code;
        /**
         * Target host isn’t actually a host.
         */
        static invalidHost: MachError.Code;
        /**
         * A ledger was required but not supplied.
         */
        static invalidLedger: MachError.Code;
        /**
         * The port was not a memory cache control port.
         */
        static invalidMemoryControl: MachError.Code;
        /**
         * The name doesn’t denote a right in the task.
         */
        static invalidName: MachError.Code;
        /**
         * The external memory manager failed to initialize the memory object.
         */
        static invalidObject: MachError.Code;
        /**
         * The specified scheduling policy is not currently enabled for the processor set.
         */
        static invalidPolicy: MachError.Code;
        /**
         * An argument applied to assert processor set privilege was not a processor set control port.
         */
        static invalidProcessorSet: MachError.Code;
        /**
         * The name denotes a right, but not an appropriate right.
         */
        static invalidRight: MachError.Code;
        /**
         * An argument supplied to assert security privilege was not a host security port.
         */
        static invalidSecurity: MachError.Code;
        /**
         * Target task isn’t an active task.
         */
        static invalidTask: MachError.Code;
        /**
         * A blatant range error.
         */
        static invalidValue: MachError.Code;
        /**
         * The lock is already owned by another thread.
         */
        static lockOwned: MachError.Code;
        /**
         * The lock is already owned by the calling thread.
         */
        static lockOwnedSelf: MachError.Code;
        /**
         * Lock set has been destroyed and is no longer available.
         */
        static lockSetDestroyed: MachError.Code;
        /**
         * The thread holding the lock terminated before releasing the lock.
         */
        static lockUnstable: MachError.Code;
        /**
         * A page was requested of a memory manager via memory_object_data_request for an object using a MEMORY_OBJECT_COPY_CALL strategy, with the VM_PROT_WANTS_COPY flag being used to specify that the page desired is for a copy of the object, and the memory manager has detected the page was pushed into a copy of the object while the kernel was walking the shadow chain from the copy to the object. This error code is delivered via memory_object_data_error and is handled by the kernel (it forces the kernel to restart the fault). It will not be seen by users.
         */
        static memoryDataMoved: MachError.Code;
        /**
         * During a page fault, the memory object indicated that the data could not be returned.  This failure may be temporary; future attempts to access this same data may succeed, as defined by the memory object.
         */
        static memoryError: MachError.Code;
        /**
         * During a page fault, the target address refers to a memory object that has been destroyed.  This failure is permanent.
         */
        static memoryFailure: MachError.Code;
        /**
         * An attempt was made to supply “precious” data for memory that is already present in a memory object.
         */
        static memoryPresent: MachError.Code;
        /**
         * A strategic copy was attempted of an object upon which a quicker copy is now possible.  The caller should retry the copy using vm_object_copy_quickly. This error code is seen only by the kernel.
         */
        static memoryRestartCopy: MachError.Code;
        /**
         * The name already denotes a right in the task.
         */
        static nameExists: MachError.Code;
        /**
         * Bogus access restriction.
         */
        static noAccess: MachError.Code;
        /**
         * The address range specified is already in use, or no address range of the size specified could be found.
         */
        static noSpace: MachError.Code;
        /**
         * Remote node down or inaccessible.
         */
        static nodeDown: MachError.Code;
        /**
         * thread_depress_abort was called on a thread which was not currently depressed.
         */
        static notDepressed: MachError.Code;
        /**
         * The receive right is not a member of a port set.
         */
        static notInSet: MachError.Code;
        /**
         * The task in question does not hold receive rights for the port argument.
         */
        static notReceiver: MachError.Code;
        /**
         * Empty thread activation (No thread linked to it).
         */
        static notSupported: MachError.Code;
        /**
         * A signalled thread was not actually waiting.
         */
        static notWaiting: MachError.Code;
        /**
         * Some thread-oriented operation (semaphore_wait) timed out.
         */
        static operationTimedOut: MachError.Code;
        /**
         * The specified scheduling attributes exceed the thread’s limits.
         */
        static policyLimit: MachError.Code;
        /**
         * The requested property cannot be changed at this time.
         */
        static policyStatic: MachError.Code;
        /**
         * Specified memory is valid, but does not permit the required forms of access.
         */
        static protectionFailure: MachError.Code;
        /**
         * A system resource could not be allocated to fulfill this request.  This failure may not be permanent.
         */
        static resourceShortage: MachError.Code;
        /**
         * The task already has send or receive rights for the port under another name.
         */
        static rightExists: MachError.Code;
        /**
         * Allow an orphaned activation to continue executing.
         */
        static rpcContinueOrphan: MachError.Code;
        /**
         * Return from RPC indicating the target server was terminated before it successfully replied.
         */
        static rpcServerTerminated: MachError.Code;
        /**
         * Terminate an orphaned activation.
         */
        static rpcTerminateOrphan: MachError.Code;
        /**
         * Semaphore has been destroyed and is no longer available.
         */
        static semaphoreDestroyed: MachError.Code;
        static success: MachError.Code;
        /**
         * Object has been terminated and is no longer available.
         */
        static terminated: MachError.Code;
        /**
         * Operation would overflow limit on user-references.
         */
        static userReferencesOverflow: MachError.Code;
        code: MachErrorCode;
        errorCode: number;
        errorUserInfo: Record<string, any>;
        hashValue: number;
        localizedDescription: string;
        userInfo: Record<string, any>;
        constructor (code: MachErrorCode, userInfo: Record<string, any>);
        hash(hasher: Hasher):void;
    }
    export class POSIXError {
        /**
         * Argument list too long.
         */
        static E2BIG: POSIXErrorCode;
        /**
         * Permission denied.
         */
        static EACCES: POSIXErrorCode;
        /**
         * Address already in use.
         */
        static EADDRINUSE: POSIXErrorCode;
        /**
         * Can’t assign requested address.
         */
        static EADDRNOTAVAIL: POSIXErrorCode;
        /**
         * Address family not supported by protocol family.
         */
        static EAFNOSUPPORT: POSIXErrorCode;
        /**
         * non-blocking and interrupt i/o. Resource temporarily unavailable.
         */
        static EAGAIN: POSIXErrorCode;
        /**
         * Operation already in progress.
         */
        static EALREADY: POSIXErrorCode;
        /**
         * Authentication error.
         */
        static EAUTH: POSIXErrorCode;
        /**
         * Bad CPU type in executable.
         */
        static EBADARCH: POSIXErrorCode;
        /**
         * Program loading errors. Bad executable.
         */
        static EBADEXEC: POSIXErrorCode;
        /**
         * Bad file descriptor.
         */
        static EBADF: POSIXErrorCode;
        /**
         * Malformed Macho file.
         */
        static EBADMACHO: POSIXErrorCode;
        /**
         * Bad message.
         */
        static EBADMSG: POSIXErrorCode;
        /**
         * RPC struct is bad.
         */
        static EBADRPC: POSIXErrorCode;
        /**
         * Device / Resource busy.
         */
        static EBUSY: POSIXErrorCode;
        /**
         * Operation canceled.
         */
        static ECANCELED: POSIXErrorCode;
        /**
         * No child processes.
         */
        static ECHILD: POSIXErrorCode;
        /**
         * Software caused connection abort.
         */
        static ECONNABORTED: POSIXErrorCode;
        /**
         * Connection refused.
         */
        static ECONNREFUSED: POSIXErrorCode;
        /**
         * Connection reset by peer.
         */
        static ECONNRESET: POSIXErrorCode;
        /**
         * Resource deadlock avoided.
         */
        static EDEADLK: POSIXErrorCode;
        /**
         * Destination address required.
         */
        static EDESTADDRREQ: POSIXErrorCode;
        /**
         * Device error, for example paper out.
         */
        static EDEVERR: POSIXErrorCode;
        /**
         * math software. Numerical argument out of domain.
         */
        static EDOM: POSIXErrorCode;
        /**
         * Disc quota exceeded.
         */
        static EDQUOT: POSIXErrorCode;
        /**
         * File exists.
         */
        static EEXIST: POSIXErrorCode;
        /**
         * Bad address.
         */
        static EFAULT: POSIXErrorCode;
        /**
         * File too large.
         */
        static EFBIG: POSIXErrorCode;
        /**
         * Inappropriate file type or format.
         */
        static EFTYPE: POSIXErrorCode;
        /**
         * Host is down.
         */
        static EHOSTDOWN: POSIXErrorCode;
        /**
         * No route to host.
         */
        static EHOSTUNREACH: POSIXErrorCode;
        /**
         * Identifier removed.
         */
        static EIDRM: POSIXErrorCode;
        /**
         * Illegal byte sequence.
         */
        static EILSEQ: POSIXErrorCode;
        /**
         * Operation now in progress.
         */
        static EINPROGRESS: POSIXErrorCode;
        /**
         * Interrupted system call.
         */
        static EINTR: POSIXErrorCode;
        /**
         * Invalid argument.
         */
        static EINVAL: POSIXErrorCode;
        /**
         * Input/output error.
         */
        static EIO: POSIXErrorCode;
        /**
         * Socket is already connected.
         */
        static EISCONN: POSIXErrorCode;
        /**
         * Is a directory.
         */
        static EISDIR: POSIXErrorCode;
        /**
         * Too many levels of symbolic links.
         */
        static ELOOP: POSIXErrorCode;
        /**
         * Too many open files.
         */
        static EMFILE: POSIXErrorCode;
        /**
         * Too many links.
         */
        static EMLINK: POSIXErrorCode;
        /**
         * Message too long.
         */
        static EMSGSIZE: POSIXErrorCode;
        /**
         * Reserved.
         */
        static EMULTIHOP: POSIXErrorCode;
        /**
         * File name too long.
         */
        static ENAMETOOLONG: POSIXErrorCode;
        /**
         * Need authenticator.
         */
        static ENEEDAUTH: POSIXErrorCode;
        /**
         * ipc/network software – operational errors Network is down.
         */
        static ENETDOWN: POSIXErrorCode;
        /**
         * Network dropped connection on reset.
         */
        static ENETRESET: POSIXErrorCode;
        /**
         * Network is unreachable.
         */
        static ENETUNREACH: POSIXErrorCode;
        /**
         * Too many open files in system.
         */
        static ENFILE: POSIXErrorCode;
        /**
         * Attribute not found.
         */
        static ENOATTR: POSIXErrorCode;
        /**
         * No buffer space available.
         */
        static ENOBUFS: POSIXErrorCode;
        /**
         * No message available on STREAM.
         */
        static ENODATA: POSIXErrorCode;
        /**
         * Operation not supported by device.
         */
        static ENODEV: POSIXErrorCode;
        /**
         * No such file or directory.
         */
        static ENOENT: POSIXErrorCode;
        /**
         * Exec format error.
         */
        static ENOEXEC: POSIXErrorCode;
        /**
         * No locks available.
         */
        static ENOLCK: POSIXErrorCode;
        /**
         * Reserved.
         */
        static ENOLINK: POSIXErrorCode;
        /**
         * Cannot allocate memory.
         */
        static ENOMEM: POSIXErrorCode;
        /**
         * No message of desired type.
         */
        static ENOMSG: POSIXErrorCode;
        /**
         * No such policy registered.
         */
        static ENOPOLICY: POSIXErrorCode;
        /**
         * Protocol not available.
         */
        static ENOPROTOOPT: POSIXErrorCode;
        /**
         * No space left on device.
         */
        static ENOSPC: POSIXErrorCode;
        /**
         * No STREAM resources.
         */
        static ENOSR: POSIXErrorCode;
        /**
         * Not a STREAM.
         */
        static ENOSTR: POSIXErrorCode;
        /**
         * Function not implemented.
         */
        static ENOSYS: POSIXErrorCode;
        /**
         * Block device required.
         */
        static ENOTBLK: POSIXErrorCode;
        /**
         * Socket is not connected.
         */
        static ENOTCONN: POSIXErrorCode;
        /**
         * Not a directory.
         */
        static ENOTDIR: POSIXErrorCode;
        /**
         * Directory not empty.
         */
        static ENOTEMPTY: POSIXErrorCode;
        /**
         * State not recoverable.
         */
        static ENOTRECOVERABLE: POSIXErrorCode;
        /**
         * ipc/network software – argument errors. Socket operation on non-socket.
         */
        static ENOTSOCK: POSIXErrorCode;
        /**
         * Operation not supported.
         */
        static ENOTSUP: POSIXErrorCode;
        /**
         * Inappropriate ioctl for device.
         */
        static ENOTTY: POSIXErrorCode;
        /**
         * Device not configured.
         */
        static ENXIO: POSIXErrorCode;
        /**
         * Value too large to be stored in data type.
         */
        static EOVERFLOW: POSIXErrorCode;
        /**
         * Previous owner died.
         */
        static EOWNERDEAD: POSIXErrorCode;
        static EPERM: POSIXErrorCode;
        /**
         * Protocol family not supported.
         */
        static EPFNOSUPPORT: POSIXErrorCode;
        /**
         * Broken pipe.
         */
        static EPIPE: POSIXErrorCode;
        /**
         * quotas & mush. Too many processes.
         */
        static EPROCLIM: POSIXErrorCode;
        /**
         * Bad procedure for program.
         */
        static EPROCUNAVAIL: POSIXErrorCode;
        /**
         * Program version wrong.
         */
        static EPROGMISMATCH: POSIXErrorCode;
        /**
         * RPC prog. not avail.
         */
        static EPROGUNAVAIL: POSIXErrorCode;
        /**
         * Protocol error.
         */
        static EPROTO: POSIXErrorCode;
        /**
         * Protocol not supported.
         */
        static EPROTONOSUPPORT: POSIXErrorCode;
        /**
         * Protocol wrong type for socket.
         */
        static EPROTOTYPE: POSIXErrorCode;
        /**
         * Intelligent device errors. Device power is off.
         */
        static EPWROFF: POSIXErrorCode;
        /**
         * Interface output queue is full.
         */
        static EQFULL: POSIXErrorCode;
        /**
         * Result too large.
         */
        static ERANGE: POSIXErrorCode;
        /**
         * Too many levels of remote in path.
         */
        static EREMOTE: POSIXErrorCode;
        /**
         * Read-only file system.
         */
        static EROFS: POSIXErrorCode;
        /**
         * RPC version wrong.
         */
        static ERPCMISMATCH: POSIXErrorCode;
        /**
         * Shared library version mismatch.
         */
        static ESHLIBVERS: POSIXErrorCode;
        /**
         * Can’t send after socket shutdown.
         */
        static ESHUTDOWN: POSIXErrorCode;
        /**
         * Socket type not supported.
         */
        static ESOCKTNOSUPPORT: POSIXErrorCode;
        /**
         * Illegal seek.
         */
        static ESPIPE: POSIXErrorCode;
        /**
         * No such process.
         */
        static ESRCH: POSIXErrorCode;
        /**
         * Network File System. Stale NFS file handle.
         */
        static ESTALE: POSIXErrorCode;
        /**
         * STREAM ioctl timeout.
         */
        static ETIME: POSIXErrorCode;
        /**
         * Operation timed out.
         */
        static ETIMEDOUT: POSIXErrorCode;
        /**
         * Too many references: can’t splice.
         */
        static ETOOMANYREFS: POSIXErrorCode;
        /**
         * Text file busy.
         */
        static ETXTBSY: POSIXErrorCode;
        /**
         * Too many users.
         */
        static EUSERS: POSIXErrorCode;
        /**
         * Operation would block.
         */
        static EWOULDBLOCK: POSIXErrorCode;
        /**
         * Cross-device link.
         */
        static EXDEV: POSIXErrorCode;
        static errorDomain: string;
        code: POSIXErrorCode;
        errorCode: number;
        errorUserInfo: Record<string, any>;
        hashValue: number;
        localizedDescription: string;
        userInfo: Record<string, any>;
        constructor (code: POSIXErrorCode, userInfo: Record<string, any>);
        hash(hasher: Hasher):void;
    }
    export interface LocalizedError extends Error {
        errorDescription: string;
        failureReason: string;
        helpAnchor: string;
        recoverySuggestion: string;
    }
    export interface RecoverableError extends Error {
        /**
         * Provides a set of possible recovery options to present to the user.
         */
        recoveryOptions: string[];
        /**
         * Attempt to recover from this error when the user selected the option at the given index. Returns true to indicate successful recovery, and false otherwise.
         */
        attemptRecovery(recoveryOptionIndex: number):boolean;
    }
    export interface CustomNSError extends Error {
        errorCode: number;
        errorUserInfo: Record<string, any>;
        static errorDomain: string;
    }
    export class NSAssertionHandler extends NSObject {
        /**
         * Returns the 
         *  object associated with the current thread.
         */
        static current: NSAssertionHandler;
    }
    export class NSException extends NSObject {
        /**
         * A string used to uniquely identify the receiver.
         */
        name: NSExceptionName;
        /**
         * A string containing a “human-readable” reason for the receiver.
         */
        reason: string;
        /**
         * A dictionary containing application-specific data pertaining to the receiver.
         */
        userInfo: Record<AnyHashable, any> | null;
        /**
         * The call return addresses related to a raised exception.
         */
        callStackReturnAddresses: NSNumber[];
        /**
         * An array containing the current call stack symbols.
         */
        callStackSymbols: string[];
        /**
         * Initializes and returns a newly allocated exception object.
         */
        constructor (aName: NSExceptionName, aReason: string, aUserInfo: Record<AnyHashable, any> | null);
        /**
         * Creates and raises an exception with the specified name, reason, and arguments.
         */
        static raise(name: NSExceptionName, format: string, argList: CVaListPointer):void;
    }
    export class NSExceptionName {
        /**
         * 
         *  raises an 
         *  if a string cannot be represented in a file-system or string encoding.
         */
        static characterConversionException: NSExceptionName;
        /**
         * The exception raised on divide by zero.
         */
        static decimalNumberDivideByZeroException: NSExceptionName;
        /**
         * The exception raised if there is an exactness error.
         */
        static decimalNumberExactnessException: NSExceptionName;
        /**
         * The exception raised on overflow.
         */
        static decimalNumberOverflowException: NSExceptionName;
        /**
         * The exception raised on underflow.
         */
        static decimalNumberUnderflowException: NSExceptionName;
        /**
         * Name of an exception that occurs when an internal assertion fails and implies an unexpected condition within the distributed objects.
         */
        static destinationInvalidException: NSExceptionName;
        /**
         * Raised by 
         *  if attempts to determine file-handle type fail or if attempts to read from a file or channel fail.
         */
        static fileHandleOperationException: NSExceptionName;
        /**
         * A generic name for an exception.
         */
        static genericException: NSExceptionName;
        /**
         * Name of an exception that occurs when an internal assertion fails and implies an unexpected condition within the called code.
         */
        static internalInconsistencyException: NSExceptionName;
        /**
         * The name of the exception raised by 
         *  if there is a problem creating an archive.
         */
        static invalidArchiveOperationException: NSExceptionName;
        /**
         * Name of an exception that occurs when you pass an invalid argument to a method, such as a 
         *  pointer where a non-
         *  object is required.
         */
        static invalidArgumentException: NSExceptionName;
        /**
         * Name of an exception that occurs when the receive port of an 
         *  has become invalid. 
         */
        static invalidReceivePortException: NSExceptionName;
        /**
         * Name of an exception that occurs when the send port of an 
         *  has become invalid.
         */
        static invalidSendPortException: NSExceptionName;
        /**
         * The name of the exception raised by 
         *  if there is a problem extracting an archive.
         */
        static invalidUnarchiveOperationException: NSExceptionName;
        /**
         * The name of the exception raised if the 
         *  method is called after the operation was cancelled. 
         */
        static invocationOperationCancelledException: NSExceptionName;
        /**
         * The name of the exception raised if the 
         *  method is called for an invocation method with a 
         *  return type. 
         */
        static invocationOperationVoidResultException: NSExceptionName;
        /**
         * Obsolete; not currently used.
         */
        static mallocException: NSExceptionName;
        /**
         * Name of an exception that occurs when a remote object is accessed from a thread that should not access it. 
         */
        static objectInaccessibleException: NSExceptionName;
        /**
         * Name of an exception that occurs when the remote side of the 
         *  refused to send the message to the object because the object has never been vended.
         */
        static objectNotAvailableException: NSExceptionName;
        /**
         * No longer used.
         */
        static oldStyleException: NSExceptionName;
        /**
         * 
         *  raises an 
         *  if a string cannot be parsed as a property list.
         */
        static parseErrorException: NSExceptionName;
        /**
         * Generic error occurred on receive. 
         */
        static portReceiveException: NSExceptionName;
        /**
         * Generic error occurred on send. 
         */
        static portSendException: NSExceptionName;
        /**
         * Name of an exception that occurs when a timeout set on a port expires during a send or receive operation. 
         */
        static portTimeoutException: NSExceptionName;
        /**
         * Name of an exception that occurs when attempting to access outside the bounds of some data, such as beyond the end of a string.
         */
        static rangeException: NSExceptionName;
        /**
         * Raised when a key value coding operation fails. 
         *  keys are described in 
         */
        static undefinedKeyException: NSExceptionName;
        /**
         * The name of an exception raised by 
         *  if there are problems initializing or encoding.
         */
        static inconsistentArchiveException: NSExceptionName;
        static NSPPDIncludeNotFoundException: NSExceptionName;
        static NSPPDIncludeStackOverflowException: NSExceptionName;
        static NSPPDIncludeStackUnderflowException: NSExceptionName;
        static NSPPDParseException: NSExceptionName;
        static NSRTFPropertyStackOverflowException: NSExceptionName;
        static NSTIFFException: NSExceptionName;
        static abortModalException: NSExceptionName;
        static abortPrintingException: NSExceptionName;
        static accessibilityException: NSExceptionName;
        static appKitIgnoredException: NSExceptionName;
        static appKitVirtualMemoryException: NSExceptionName;
        static badBitmapParametersException: NSExceptionName;
        static badComparisonException: NSExceptionName;
        static badRTFColorTableException: NSExceptionName;
        static badRTFDirectiveException: NSExceptionName;
        static badRTFFontTableException: NSExceptionName;
        static badRTFStyleSheetException: NSExceptionName;
        static browserIllegalDelegateException: NSExceptionName;
        static colorListIOException: NSExceptionName;
        static colorListNotEditableException: NSExceptionName;
        static draggingException: NSExceptionName;
        static fontUnavailableException: NSExceptionName;
        static illegalSelectorException: NSExceptionName;
        static imageCacheException: NSExceptionName;
        static nibLoadingException: NSExceptionName;
        static pasteboardCommunicationException: NSExceptionName;
        /**
         * The name of an exception raised when there is already a print operation in process.
         */
        static printOperationExistsException: NSExceptionName;
        static printPackageException: NSExceptionName;
        static printingCommunicationException: NSExceptionName;
        /**
         * Exception generated if a line is too long in a 
         *  object.
         */
        static textLineTooLongException: NSExceptionName;
        static textNoSelectionException: NSExceptionName;
        static textReadException: NSExceptionName;
        static textWriteException: NSExceptionName;
        static typedStreamVersionException: NSExceptionName;
        static windowServerCommunicationException: NSExceptionName;
        static wordTablesReadException: NSExceptionName;
        static wordTablesWriteException: NSExceptionName;
        constructor (rawValue: string);
        constructor (rawValue: string);
    }
    export class NSAppleScript extends NSObject {
        /**
         * An 
         *  that supplies a detailed description of the error condition.
         */
        static errorMessage: string;
        /**
         * An 
         *  that specifies the error number.
         */
        static errorNumber: string;
        /**
         * An 
         *  that specifies the name of the application that generated the error.
         */
        static errorAppName: string;
        /**
         * An 
         *  that provides a brief description of the error.
         */
        static errorBriefMessage: string;
        /**
         * An 
         *  that specifies a range.
         */
        static errorRange: string;
        /**
         * A Boolean value that indicates whether the receiver's script has been compiled.
         */
        isCompiled: boolean;
        /**
         * The script source for the receiver.
         */
        source: string;
        /**
         * Initializes a newly allocated script instance from the source identified by the passed URL.
         */
        constructor (url: URL, errorInfo: AutoreleasingUnsafeMutablePointer<NSDictionary?> | null);
        /**
         * Initializes a newly allocated script instance from the passed source.
         */
        constructor (source: string);
        /**
         * Compiles the receiver, if it is not already compiled.
         */
        compileAndReturnError(errorInfo: AutoreleasingUnsafeMutablePointer<NSDictionary?> | null):boolean;
        /**
         * Executes the receiver, compiling it first if it is not already compiled.
         */
        executeAndReturnError(errorInfo: AutoreleasingUnsafeMutablePointer<NSDictionary?> | null):NSAppleEventDescriptor;
        /**
         * Executes an Apple event in the context of the receiver, as a means of allowing the application to invoke a handler in the script.
         */
        executeAppleEvent(event: NSAppleEventDescriptor, errorInfo: AutoreleasingUnsafeMutablePointer<NSDictionary?> | null):NSAppleEventDescriptor;
    }
    export class NSAppleEventDescriptor extends NSObject {
        /**
         * The 
         *  structure encapsulated by the receiver, if it has one.
         */
        aeDesc: UnsafePointer<AEDesc> | null;
        /**
         * The contents of the receiver as a Boolean value, coercing (to 
         * ) if necessary.
         */
        booleanValue: boolean;
        /**
         * The receiver’s data.
         */
        data: Data;
        /**
         * The descriptor type of the receiver.
         */
        descriptorType: DescType;
        /**
         * The contents of the receiver as an enumeration type, coercing to 
         *  if necessary.
         */
        enumCodeValue: OSType;
        /**
         * The contents of the receiver as an integer, coercing (to 
         * ) if necessary.
         */
        int32Value: Int32;
        /**
         * The number of descriptors in the receiver’s descriptor list.
         */
        numberOfItems: number;
        /**
         * The contents of the receiver as a Unicode text string, coercing to 
         *  if necessary.
         */
        stringValue: string;
        /**
         * The contents of the receiver as a type, coercing to 
         *  if necessary.
         */
        typeCodeValue: OSType;
        /**
         * The event class for the receiver.
         */
        eventClass: AEEventClass;
        /**
         * The event ID for the receiver.
         */
        eventID: AEEventID;
        /**
         * The receiver’s return ID (the ID for a reply Apple event).
         */
        returnID: AEReturnID;
        /**
         * The receiver’s transaction ID, if any.
         */
        transactionID: AETransactionID;
        dateValue: Date | null;
        doubleValue: number;
        fileURLValue: URL | null;
        isRecordDescriptor: boolean;
        /**
         * Creates a descriptor initialized with type 
         *  that stores the specified Boolean value.
         */
        constructor (_boolean: boolean);
        /**
         * Creates a descriptor initialized with type 
         *  that stores the specified enumerator data type value.
         */
        constructor (enumerator: OSType);
        /**
         * Creates a descriptor initialized with Apple event type 
         *  that stores the specified integer value.
         */
        constructor (signedInt: Int32);
        /**
         * Creates a descriptor initialized with type 
         *  that stores the text from the specified string.
         */
        constructor (_string: string);
        /**
         * Creates a descriptor initialized with type 
         *  that stores the specified type value.
         */
        constructor (typeCode: OSType);
        /**
         * Initializes a newly allocated instance as an empty list descriptor.
         */
        constructor (listDescriptor: any);
        /**
         * Initializes a newly allocated instance as a descriptor that is an Apple event record.
         */
        constructor (recordDescriptor: any);
        /**
         * Initializes a newly allocated instance as a descriptor for the specified Carbon 
         *  structure.
         */
        constructor (aeDesc: UnsafePointer<AEDesc>);
        /**
         * Initializes a newly allocated instance as a descriptor with the specified descriptor type and data (from an arbitrary sequence of bytes and a length count).
         */
        constructor (descriptorType: DescType, bytes: UnsafeRawPointer | null, byteCount: number);
        /**
         * Initializes a newly allocated instance as a descriptor with the specified descriptor type and data (from an instance of 
         * ).
         */
        constructor (descriptorType: DescType, data: Data | null);
        /**
         * Initializes a newly allocated instance as a descriptor for an Apple event, initialized with the specified values.
         */
        constructor (eventClass: AEEventClass, eventID: AEEventID, targetDescriptor: NSAppleEventDescriptor | null, returnID: AEReturnID, transactionID: AETransactionID);
        constructor (applicationURL: URL);
        constructor (bundleIdentifier: string);
        constructor (date: Date);
        constructor (doubleValue: number);
        constructor (fileURL: URL);
        constructor (processIdentifier: pid_t);
        /**
         * Creates a descriptor that represents an Apple event, initialized according to the specified information.
         */
        static appleEvent(eventClass: AEEventClass, eventID: AEEventID, targetDescriptor: NSAppleEventDescriptor | null, returnID: AEReturnID, transactionID: AETransactionID):NSAppleEventDescriptor;
        /**
         * Creates and initializes an empty list descriptor.
         */
        static list():NSAppleEventDescriptor;
        /**
         * Creates and initializes a descriptor with no parameter or attribute values set.
         */
        static null():NSAppleEventDescriptor;
        /**
         * Creates and initializes a descriptor for an Apple event record whose data has yet to be set.
         */
        static record():NSAppleEventDescriptor;
        static currentProcess():NSAppleEventDescriptor;
        /**
         * Returns a descriptor obtained by coercing the receiver to the specified type.
         */
        coerce(descriptorType: DescType):NSAppleEventDescriptor | null;
        /**
         * Returns the descriptor at the specified (one-based) position in the receiving descriptor list.
         */
        atIndex(index: number):NSAppleEventDescriptor | null;
        /**
         * Inserts a descriptor at the specified (one-based) position in the receiving descriptor list, replacing the existing descriptor, if any, at that position.
         */
        insert(descriptor: NSAppleEventDescriptor, index: number):void;
        /**
         * Removes the descriptor at the specified (one-based) position in the receiving descriptor list.
         */
        remove(index: number):void;
        /**
         * Returns the receiver’s descriptor for the specified keyword.
         */
        forKeyword(keyword: AEKeyword):NSAppleEventDescriptor | null;
        /**
         * Returns the keyword for the descriptor at the specified (one-based) position in the receiver.
         */
        keywordForDescriptor(index: number):AEKeyword;
        /**
         * Adds a descriptor, identified by a keyword, to the receiver.
         */
        setDescriptor(descriptor: NSAppleEventDescriptor, keyword: AEKeyword):void;
        /**
         * Returns a descriptor for the receiver’s Apple event attribute identified by the specified keyword.
         */
        attributeDescriptor(keyword: AEKeyword):NSAppleEventDescriptor | null;
        /**
         * Returns a descriptor for the receiver’s Apple event parameter identified by the specified keyword.
         */
        paramDescriptor(keyword: AEKeyword):NSAppleEventDescriptor | null;
        /**
         * Removes the receiver’s parameter descriptor identified by the specified keyword.
         */
        removeParamDescriptor(keyword: AEKeyword):void;
        /**
         * Adds a descriptor to the receiver as an attribute identified by the specified keyword.
         */
        setAttribute(descriptor: NSAppleEventDescriptor, keyword: AEKeyword):void;
        /**
         * Adds a descriptor to the receiver as an Apple event parameter identified by the specified keyword.
         */
        setParam(descriptor: NSAppleEventDescriptor, keyword: AEKeyword):void;
        sendEvent(sendOptions: NSAppleEventDescriptor.SendOptions, timeoutInSeconds: TimeInterval):NSAppleEventDescriptor;
    }
    /**
     * the same name namespace of class NSAppleEventDescriptor
     */
    export namespace NSAppleEventDescriptor {
        export class SendOptions {
            static alwaysInteract: NSAppleEventDescriptor.SendOptions;
            static canInteract: NSAppleEventDescriptor.SendOptions;
            static canSwitchLayer: NSAppleEventDescriptor.SendOptions;
            static defaultOptions: NSAppleEventDescriptor.SendOptions;
            static dontAnnotate: NSAppleEventDescriptor.SendOptions;
            static dontExecute: NSAppleEventDescriptor.SendOptions;
            static dontRecord: NSAppleEventDescriptor.SendOptions;
            static neverInteract: NSAppleEventDescriptor.SendOptions;
            static noReply: NSAppleEventDescriptor.SendOptions;
            static queueReply: NSAppleEventDescriptor.SendOptions;
            static waitForReply: NSAppleEventDescriptor.SendOptions;
            constructor (rawValue: number);
        }
    }
    export class NSAppleEventManager extends NSObject {
        /**
         * Returns the descriptor for 
         *  if an Apple event is being handled on the current thread.
         */
        currentAppleEvent: NSAppleEventDescriptor | null;
        /**
         * Returns the corresponding reply event descriptor if an Apple event is being handled on the current thread.
         */
        currentReplyAppleEvent: NSAppleEventDescriptor | null;
        /**
         * Returns the single instance of 
         * , creating it first if it doesn’t exist.
         */
        static shared():NSAppleEventManager;
        /**
         * If an Apple event handler has been registered for the event specified by 
         *  and 
         * , removes it.
         */
        removeEventHandler(eventClass: AEEventClass, eventID: AEEventID):void;
        /**
         * Registers the Apple event handler specified by 
         *  for the event specified by 
         *  and 
         * .
         */
        setEventHandler(handler: any, handleEventSelector: Selector, eventClass: AEEventClass, eventID: AEEventID):void;
        /**
         * Causes the Apple event specified by 
         *  to be dispatched to the appropriate Apple event handler, if one has been registered by calling 
         * .
         */
        dispatchRawAppleEvent(theAppleEvent: UnsafePointer<AppleEvent>, theReply: UnsafeMutablePointer<AppleEvent>, handlerRefCon: SRefCon):OSErr;
        /**
         * Given a nonzero 
         *  returned by an invocation of 
         * , returns the descriptor for the event whose handling was suspended.
         */
        appleEvent(suspensionID: NSAppleEventManager.SuspensionID):NSAppleEventDescriptor;
        /**
         * Given a nonzero 
         *  returned by an invocation of 
         * , returns the corresponding reply event descriptor.
         */
        replyAppleEvent(suspensionID: NSAppleEventManager.SuspensionID):NSAppleEventDescriptor;
        /**
         * Given a nonzero 
         *  returned by an invocation of 
         * , signal that handling of the suspended event may now continue.
         */
        resume(suspensionID: NSAppleEventManager.SuspensionID):void;
        /**
         * Given a nonzero 
         *  returned by an invocation of 
         * , sets the values that will be returned by subsequent invocations of 
         *  and 
         *  to be the event whose handling was suspended and its corresponding reply event, respectively.
         */
        setCurrentAppleEventAndReplyEventWithSuspensionID(suspensionID: NSAppleEventManager.SuspensionID):void;
        /**
         * Suspends the handling of the current event and returns an ID that must be used to resume the handling of the event if an Apple event is being handled on the current thread.
         */
        suspendCurrentAppleEvent():NSAppleEventManager.SuspensionID | null;
    }
    export class NSScriptCommand extends NSObject {
        /**
         * If the receiver was constructed by Cocoa scripting's built-in Apple event handling, returns the Apple event descriptor from which it was constructed.
         */
        appleEvent: NSAppleEventDescriptor | null;
        /**
         * Returns the object or objects to which the command is to be sent (called both the “receivers” or “targets” of script commands).
         */
        evaluatedReceivers: any;
        /**
         * Sets the object specifier to 
         *  that, when evaluated, indicates the receiver or receivers of the command.
         */
        receiversSpecifier: NSScriptObjectSpecifier | null;
        /**
         * Sets the arguments of the command to 
         * .
         */
        arguments: Record<string, any> | null;
        /**
         * Returns a dictionary containing the arguments of the command, evaluated from object specifiers to objects if necessary. The keys in the dictionary are the argument names.
         */
        evaluatedArguments: Record<string, any> | null;
        /**
         * Sets the object that corresponds to the direct parameter of the Apple event from which the receiver derives.
         */
        directParameter: any;
        /**
         * Returns the command description for the command.
         */
        commandDescription: NSScriptCommandDescription;
        /**
         * Sets a descriptor for the expected type that will be put in the reply Apple event if the sender requested a reply, execution of the receiver completes, and an error number was set.
         */
        scriptErrorExpectedTypeDescriptor: NSAppleEventDescriptor | null;
        /**
         * Sets a script error number that is associated with the execution of the command and is returned in the reply Apple event, if a reply was requested by the sender.
         */
        scriptErrorNumber: number;
        /**
         * Sets a descriptor for an object that will be put in the reply Apple event if the sender requested a reply, execution of the receiver completes, and an error number was set.
         */
        scriptErrorOffendingObjectDescriptor: NSAppleEventDescriptor | null;
        /**
         * Sets a script error string that is associated with execution of the command.
         */
        scriptErrorString: string;
        isWellFormed: boolean;
        /**
         * Returns an a script command object initialized from the passed command description.
         */
        constructor (commandDef: NSScriptCommandDescription);
        constructor (inCoder: NSCoder);
        /**
         * If a command is being executed in the current thread by Cocoa scripting's built-in Apple event handling, return the command.
         */
        static current():NSScriptCommand | null;
        /**
         * Executes the command if it is valid and returns the result, if any.
         */
        execute():any;
        /**
         * Overridden by subclasses to provide a default implementation for the command represented by the receiver.
         */
        performDefaultImplementation():any;
        /**
         * Suspends the execution of the receiver.
         */
        suspendExecution():void;
        /**
         * If a successful, unmatched, invocation of 
         *  has been made, resume the execution of the command.
         */
        resumeExecution(result: any):void;
    }
    export class NSQuitCommand extends NSScriptCommand {
        /**
         * Returns a constant indicating how to deal with closing any modified documents.
         */
        saveOptions: NSSaveOptions;
    }
    export class NSSetCommand extends NSScriptCommand {
        /**
         * Returns a specifier that identifies the attribute or relationship that is to be set for the receiver of the 
         *  AppleScript command.
         */
        keySpecifier: NSScriptObjectSpecifier;
        /**
         * Sets the receiver’s object specifier.
         */
        setReceiversSpecifier(receiversRef: NSScriptObjectSpecifier | null):void;
    }
    export class NSMoveCommand extends NSScriptCommand {
        /**
         * Returns a specifier for the object or objects to be moved.
         */
        keySpecifier: NSScriptObjectSpecifier;
        /**
         * Sets the receiver’s object specifier.
         */
        setReceiversSpecifier(receiversRef: NSScriptObjectSpecifier | null):void;
    }
    export class NSCreateCommand extends NSScriptCommand {
        /**
         * Returns the class description for the class that is to be created.
         */
        createClassDescription: NSScriptClassDescription;
        /**
         * Returns a dictionary that contains the properties that were specified in the 
         *  Apple event command that has been converted to this 
         *  object.
         */
        resolvedKeyDictionary: Record<string, any>;
    }
    export class NSDeleteCommand extends NSScriptCommand {
        /**
         * Returns a specifier for the object or objects to be deleted.
         */
        keySpecifier: NSScriptObjectSpecifier;
        /**
         * Sets the receiver’s object specifier.
         */
        setReceiversSpecifier(receiversRef: NSScriptObjectSpecifier | null):void;
    }
    export class NSExistsCommand extends NSScriptCommand {}
    export class NSGetCommand extends NSScriptCommand {}
    export class NSCloneCommand extends NSScriptCommand {
        /**
         * Returns a specifier for the object or objects to be cloned.
         */
        keySpecifier: NSScriptObjectSpecifier;
        /**
         * Sets the receiver’s object specifier;.
         */
        setReceiversSpecifier(receiversRef: NSScriptObjectSpecifier | null):void;
    }
    export class NSCountCommand extends NSScriptCommand {}
    export class NSCloseCommand extends NSScriptCommand {
        /**
         * Returns a constant indicating how to deal with closing any modified documents.
         */
        saveOptions: NSSaveOptions;
    }
    export enum NSSaveOptions {
        yes = 0,
        no = 1,
        ask = 2,
    }
    export class NSScriptObjectSpecifier extends NSObject {
        /**
         * Returns the actual object represented by the nested series of object specifiers.
         */
        objectsByEvaluatingSpecifier: any;
        /**
         * Sets the class description of the receiver’s container specifier to a given specifier.
         */
        containerClassDescription: NSScriptClassDescription | null;
        /**
         * Sets whether the receiver’s container should be an object involved in a filter reference or the top-level object.
         */
        containerIsObjectBeingTested: boolean;
        /**
         * Sets whether the receiver’s container is to be the container for a range specifier or a top-level object.
         */
        containerIsRangeContainerObject: boolean;
        /**
         * Sets the container specifier of the receiver.
         */
        container: NSScriptObjectSpecifier | null;
        /**
         * Sets the receiver’s child reference.
         */
        child: NSScriptObjectSpecifier | null;
        /**
         * Sets the key of the receiver.
         */
        key: string;
        /**
         * Returns the class description of the objects specified by the receiver.
         */
        keyClassDescription: NSScriptClassDescription | null;
        /**
         * Returns the object specifier in which an evaluation error occurred.
         */
        evaluationError: NSScriptObjectSpecifier | null;
        /**
         * Sets the value of the evaluation error.
         */
        evaluationErrorNumber: number;
        /**
         * Returns an Apple event descriptor that represents the receiver.
         */
        descriptor: NSAppleEventDescriptor | null;
        /**
         * Returns a new object specifier for an Apple event descriptor.
         */
        constructor (descriptor: NSAppleEventDescriptor);
        /**
         * Returns an 
         *  object initialized with the given attributes.
         */
        constructor (classDesc: NSScriptClassDescription, container: NSScriptObjectSpecifier | null, property: string);
        /**
         * Returns an 
         *  object initialized with a given container specifier  and key.
         */
        constructor (container: NSScriptObjectSpecifier, property: string);
        constructor (inCoder: NSCoder);
        /**
         * This primitive method must be overridden by subclasses to return a pointer to an array of indices identifying objects in the key of a given container that are identified by the receiver of the message. 
         */
        indicesOfObjectsByEvaluating(container: any, count: UnsafeMutablePointer<Int>):UnsafeMutablePointer<Int> | null;
        /**
         * Returns the actual object or objects specified by the receiver as evaluated in the context of given container object. 
         */
        objectsByEvaluating(containers: any):any;
    }
    export class NSPropertySpecifier extends NSScriptObjectSpecifier {}
    export class NSPositionalSpecifier extends NSObject {
        /**
         * Returns the container in which the new or copied object or objects should be placed.
         */
        insertionContainer: any;
        /**
         * Returns an insertion index that indicates where the new or copied object or objects should be placed.
         */
        insertionIndex: number;
        /**
         * Returns the key that identifies the relationship into which the new or copied object or objects should be inserted.
         */
        insertionKey: string;
        /**
         * Returns a Boolean value that indicates whether evaluation has been successful and the object to be inserted should actually replace the keyed, indexed object in the insertion container.
         */
        insertionReplaces: boolean;
        /**
         * Returns the object specifier specified at initialization time.
         */
        objectSpecifier: NSScriptObjectSpecifier;
        /**
         * Returns the insertion position specified at initialization time.
         */
        position: NSPositionalSpecifier.InsertionPosition;
        /**
         * Initializes a positional specifier with a given position relative to another given specifier.
         */
        constructor (position: NSPositionalSpecifier.InsertionPosition, specifier: NSScriptObjectSpecifier);
        /**
         * Sets the class description for the object or objects to be inserted.
         */
        setInsertionClassDescription(classDescription: NSScriptClassDescription):void;
        /**
         * Causes the receiver to evaluate its position.
         */
        evaluate():void;
    }
    /**
     * the same name namespace of class NSPositionalSpecifier
     */
    export namespace NSPositionalSpecifier {
        export enum InsertionPosition {
            after = 0,
            before = 1,
            beginning = 2,
            end = 3,
            replace = 4,
        }
    }
    export class NSRandomSpecifier extends NSScriptObjectSpecifier {}
    export class NSRangeSpecifier extends NSScriptObjectSpecifier {
        /**
         * Sets the object specifier representing the last object of the range to a given object.
         */
        endSpecifier: NSScriptObjectSpecifier | null;
        /**
         * Returns the object specifier representing the first object of the range.
         */
        startSpecifier: NSScriptObjectSpecifier | null;
        /**
         * Returns a range specifier initialized with the given properties.
         */
        constructor (classDesc: NSScriptClassDescription, container: NSScriptObjectSpecifier | null, property: string, startSpec: NSScriptObjectSpecifier | null, endSpec: NSScriptObjectSpecifier | null);
        constructor (inCoder: NSCoder);
    }
    export class NSUniqueIDSpecifier extends NSScriptObjectSpecifier {
        /**
         * Returns the ID encapsulated by the receiver.
         */
        uniqueID: any;
        /**
         * Returns an 
         *  object, initialized with the given arguments.
         */
        constructor (classDesc: NSScriptClassDescription, container: NSScriptObjectSpecifier | null, property: string, uniqueID: any);
        constructor (inCoder: NSCoder);
    }
    export class NSWhoseSpecifier extends NSScriptObjectSpecifier {
        /**
         * Sets the end sub-element identifier for the specifier to the value of a given sub-element.
         */
        endSubelementIdentifier: NSWhoseSpecifier.SubelementIdentifier;
        /**
         * Sets the index position of the last sub-element within the range of objects being tested that pass the specifier’s test.
         */
        endSubelementIndex: number;
        /**
         * Returns the start sub-element identifier for the receiver.
         */
        startSubelementIdentifier: NSWhoseSpecifier.SubelementIdentifier;
        /**
         * Returns the index position of the first sub-element within the range of objects being tested that pass the receiver's test.
         */
        startSubelementIndex: number;
        /**
         * Returns the test object encapsulated by the receiver.
         */
        test: NSScriptWhoseTest;
        /**
         * Returns an 
         *  object initialized with the given attributes.
         */
        constructor (classDesc: NSScriptClassDescription, container: NSScriptObjectSpecifier | null, property: string, test: NSScriptWhoseTest);
        constructor (inCoder: NSCoder);
    }
    /**
     * the same name namespace of class NSWhoseSpecifier
     */
    export namespace NSWhoseSpecifier {
        export enum SubelementIdentifier {
            indexSubelement = 0,
            everySubelement = 1,
            middleSubelement = 2,
            randomSubelement = 3,
            noSubelement = 4,
        }
    }
    export class NSNameSpecifier extends NSScriptObjectSpecifier {
        /**
         * Sets the name encapsulated with the receiver for the specified object in the container.
         */
        name: string;
        /**
         * Invokes the super class’s 
         *  method and then sets the name instance variable to 
         * .
         */
        constructor (classDesc: NSScriptClassDescription, container: NSScriptObjectSpecifier | null, property: string, name: string);
        constructor (inCoder: NSCoder);
    }
    export class NSMiddleSpecifier extends NSScriptObjectSpecifier {}
    export class NSIndexSpecifier extends NSScriptObjectSpecifier {
        /**
         * Sets the value of the receiver’s 
         *  property.
         */
        index: number;
        /**
         * Initializes an allocated 
         *  object with a class description, container specifier, collection key, and object index.
         */
        constructor (classDesc: NSScriptClassDescription, container: NSScriptObjectSpecifier | null, property: string, index: number);
    }
    export class NSRelativeSpecifier extends NSScriptObjectSpecifier {
        /**
         * Sets the specifier for the base object.
         */
        baseSpecifier: NSScriptObjectSpecifier | null;
        /**
         * Sets the relative position encapsulated by the receiver.
         */
        relativePosition: NSRelativeSpecifier.RelativePosition;
        /**
         * Invokes the super class’s 
         *  method and initializes the relative position and base specifier to 
         *  and 
         * .
         */
        constructor (classDesc: NSScriptClassDescription, container: NSScriptObjectSpecifier | null, property: string, relPos: NSRelativeSpecifier.RelativePosition, baseSpecifier: NSScriptObjectSpecifier | null);
        constructor (inCoder: NSCoder);
    }
    /**
     * the same name namespace of class NSRelativeSpecifier
     */
    export namespace NSRelativeSpecifier {
        export enum RelativePosition {
            after = 0,
            before = 1,
        }
    }
    export class NSScriptSuiteRegistry extends NSObject {
        /**
         * Returns the names of the suite definitions currently loaded by the application.
         */
        suiteNames: string[];
        /**
         * Sets the single, shared instance of 
         *  to 
         * .
         */
        static setShared(registry: NSScriptSuiteRegistry):void;
        /**
         * Returns the single, shared instance of 
         * , creating it first if it doesn’t exist.
         */
        static shared():NSScriptSuiteRegistry;
        /**
         * Returns the name of the suite definition associated with the given four-character Apple event code, 
         * .
         */
        suite(appleEventCode: FourCharCode):string;
        /**
         * Returns the class descriptions contained in the suite identified by 
         * .
         */
        classDescriptions(suiteName: string):Record<string, NSScriptClassDescription> | null;
        /**
         * Returns the class description associated with the given four-character Apple event code, 
         * .
         */
        classDescription(appleEventCode: FourCharCode):NSScriptClassDescription | null;
        /**
         * Registers class description 
         *  for use by Cocoa’s built-in scripting support by storing it in a per-suite internal dictionary under the class name.
         */
        register(classDescription: NSScriptClassDescription):void;
        /**
         * Returns the command descriptions contained in the suite identified by 
         * .
         */
        commandDescriptions(suiteName: string):Record<string, NSScriptCommandDescription> | null;
        /**
         * Returns the command description identified by a suite’s four-character Apple event code of the class (
         * ) and the four-character Apple event code of the command (
         * ).
         */
        commandDescription(appleEventClassCode: FourCharCode, appleEventIDCode: FourCharCode):NSScriptCommandDescription | null;
        /**
         * Returns an 
         *  object that contains data in 
         *  resource format describing the scriptability information currently known to the application.
         */
        aeteResource(languageName: string):Data | null;
        /**
         * Returns the Apple event code associated with the suite named 
         * , such as 
         *  for the Core suite.
         */
        appleEventCode(suiteName: string):FourCharCode;
        /**
         * Returns the bundle containing the suite-definition property list (extension 
         * ) identified by 
         * .
         */
        bundle(suiteName: string):Bundle | null;
        /**
         * Loads the suite definition encapsulated in 
         * ; previously, this suite definition was parsed from a 
         *  property list contained in a framework or in 
         * .
         */
        loadSuite(suiteDeclaration: Record<AnyHashable, any>, bundle: Bundle):void;
        /**
         * Loads the suite definitions in bundle 
         * , invoking 
         *  for each suite found.
         */
        loadSuites(bundle: Bundle):void;
    }
    export class NSScriptClassDescription extends NSClassDescription {
        /**
         * Returns the class description instance for the superclass of the receiver’s class.
         */
        superclass: NSScriptClassDescription | null;
        /**
         * Returns the name of the class the receiver describes, as provided at initialization time.
         */
        className: string;
        /**
         * Returns the value of the 
         *  entry of the class dictionary from which the receiver was instantiated.
         */
        defaultSubcontainerAttributeKey: string;
        /**
         * Returns the name of the Objective-C class instantiated to implement the scripting class.
         */
        implementationClassName: string;
        /**
         * Returns the name of the receiver’s suite.
         */
        suiteName: string;
        /**
         * Returns the Apple event code associated with the receiver’s class.
         */
        appleEventCode: FourCharCode;
        /**
         * Initializes and returns a newly allocated instance of 
         * .
         */
        constructor (suiteName: string, className: string, classDeclaration: Record<AnyHashable, any> | null);
        /**
         * Returns the class description for the specified class or, if it is not scriptable, for the first superclass that is.
         */
        constructor (aClass: AnyClass);
        /**
         * Returns the class description instance for the class type of the specified attribute or relationship.
         */
        forKey(key: string):NSScriptClassDescription | null;
        /**
         * Returns a Boolean value indicating whether an insertion location must be specified when creating a new object in the specified to-many relationship of the receiver.
         */
        isLocationRequiredToCreate(toManyRelationshipKey: string):boolean;
        /**
         * Returns a Boolean value indicating whether a primary or secondary Apple event code in the receiver matches the passed code.
         */
        matchesAppleEventCode(appleEventCode: FourCharCode):boolean;
        /**
         * Returns a Boolean value indicating whether the described class has an ordered to-many relationship identified by the specified key.
         */
        hasOrderedToManyRelationship(key: string):boolean;
        /**
         * Returns a Boolean value indicating whether the described class has a property identified by the specified key.
         */
        hasProperty(key: string):boolean;
        /**
         * Returns a Boolean value indicating whether the described class has a readable property identified by the specified key.
         */
        hasReadableProperty(key: string):boolean;
        /**
         * Returns a Boolean value indicating whether the described class has a writable property identified by the specified key.
         */
        hasWritableProperty(key: string):boolean;
        /**
         * Given an Apple event code that identifies a property or element class, returns the key for the corresponding attribute, one-to-one relationship, or one-to-many relationship.
         */
        key(appleEventCode: FourCharCode):string;
        /**
         * Returns the name of the declared type of the attribute or relationship identified by the passed key.
         */
        type(key: string):string;
        /**
         * Returns the selector associated with the receiver for the specified command description.
         */
        selector(commandDescription: NSScriptCommandDescription):Selector | null;
        /**
         * Returns a Boolean value indicating whether the receiver or any superclass supports the specified command.
         */
        supportsCommand(commandDescription: NSScriptCommandDescription):boolean;
    }
    export class NSClassDescription extends NSObject {
        /**
         * Overridden by subclasses to return the names of attributes of instances of the described class.
         */
        attributeKeys: string[];
        /**
         * Overridden by subclasses to return the keys for the to-many relationship properties of instances of the described class.
         */
        toManyRelationshipKeys: string[];
        /**
         * Overridden by subclasses to return the keys for the to-one relationship properties of instances of the described class.
         */
        toOneRelationshipKeys: string[];
        /**
         * Returns the class description for a given class.
         */
        constructor (aClass: AnyClass);
        /**
         * Removes all 
         *  objects from the cache.
         */
        static invalidateClassDescriptionCache():void;
        /**
         * Registers an 
         *  object for a given class in the 
         *  cache.
         */
        static register(description: NSClassDescription, aClass: AnyClass):void;
        /**
         * Overridden by subclasses to return the name of the inverse relationship from a relationship specified by a given key. 
         */
        inverse(relationshipKey: string):string;
    }
    export class NSScriptCommandDescription extends NSObject {
        /**
         * Returns the four-character code for the Apple event class of the receiver’s command.
         */
        appleEventClassCode: FourCharCode;
        /**
         * Returns the four-character code for the Apple event ID of the receiver’s command.
         */
        appleEventCode: FourCharCode;
        /**
         * Returns the name of the class that will be instantiated to handle the command.
         */
        commandClassName: string;
        /**
         * Returns the name of the command.
         */
        commandName: string;
        /**
         * Returns the name of the suite that contains the command described by the receiver.
         */
        suiteName: string;
        /**
         * Returns the names (or keys) for all arguments of the receiver’s command.
         */
        argumentNames: string[];
        /**
         * Returns the Apple event code that identifies the command’s return type.
         */
        appleEventCodeForReturnType: FourCharCode;
        /**
         * Returns the return type of the command.
         */
        returnType: string;
        /**
         * Initializes and returns a newly allocated instance of 
         * .
         */
        constructor (suiteName: string, commandName: string, commandDeclaration: Record<AnyHashable, any> | null);
        constructor (inCoder: NSCoder);
        /**
         * Returns the Apple event code for the specified command argument of the receiver.
         */
        appleEventCodeForArgument(argumentName: string):FourCharCode;
        /**
         * Returns a Boolean value that indicates whether the command argument identified by the specified argument key is an optional argument.
         */
        isOptionalArgument(argumentName: string):boolean;
        /**
         * Returns the type of the command argument identified by the specified key. 
         */
        typeForArgument(argumentName: string):string;
        /**
         * Creates and returns an instance of the command object described by the receiver.
         */
        createCommandInstance():NSScriptCommand;
    }
    export class NSScriptWhoseTest extends NSObject {
        constructor ();
        constructor (inCoder: NSCoder);
        /**
         * Returns a Boolean value that indicates whether the test represented by the receiver evaluates to 
         * .
         */
        isTrue():boolean;
    }
    export class NSSpecifierTest extends NSScriptWhoseTest {
        /**
         * Returns a specifier test initialized to evaluate a test object against an object specified by an object specifier using a given comparison operation.
         */
        constructor (obj1: NSScriptObjectSpecifier | null, compOp: NSSpecifierTest.TestComparisonOperation, obj2: any);
        constructor (inCoder: NSCoder);
    }
    /**
     * the same name namespace of class NSSpecifierTest
     */
    export namespace NSSpecifierTest {
        export enum TestComparisonOperation {
            equal = 0,
            lessThanOrEqual = 1,
            lessThan = 2,
            greaterThanOrEqual = 3,
            greaterThan = 4,
            beginsWith = 5,
            endsWith = 6,
            contains = 7,
        }
    }
    export class NSLogicalTest extends NSScriptWhoseTest {
        /**
         * Returns an 
         *  object initialized to perform an 
         *  operation with the 
         *  objects in a given array.
         */
        constructor (subTests: NSSpecifierTest[]);
        /**
         * Returns an 
         *  object initialized to perform a 
         *  operation on the given 
         *  object.
         */
        constructor (subTest: NSScriptWhoseTest);
        /**
         * Returns an 
         *  object initialized to perform an 
         *  operation with the 
         *  objects in a given array.
         */
        constructor (subTests: NSSpecifierTest[]);
    }
    export class NSScriptCoercionHandler extends NSObject {
        /**
         * Returns the shared 
         *  for the application.
         */
        static shared():NSScriptCoercionHandler;
        /**
         * Returns an object of a given class representing a given value. 
         */
        coerceValue(value: any, toClass: AnyClass):any;
        /**
         * Registers a given object (typically a class) to handle coercions (conversions) from one given class to another.
         */
        registerCoercer(coercer: any, selector: Selector, fromClass: AnyClass, toClass: AnyClass):void;
    }
    export class NSScriptExecutionContext extends NSObject {
        /**
         * Sets the top-level object for an object-specifier evaluation.
         */
        topLevelObject: any;
        /**
         * Sets the top-level container object currently being tested in a “whose” qualifier to a given object.
         */
        objectBeingTested: any;
        /**
         * Sets the top-level container object for a range-specifier evaluation to a give object.
         */
        rangeContainerObject: any;
        /**
         * Returns the shared 
         *  instance.
         */
        static shared():NSScriptExecutionContext;
    }
    export class FileManager extends NSObject {
        /**
         * The shared file manager object for the process.
         */
        static default: FileManager;
        /**
         * The home directory for the current user.
         */
        homeDirectoryForCurrentUser: URL;
        /**
         * The temporary directory for the current user.
         */
        temporaryDirectory: URL;
        /**
         * An opaque token that represents the current user’s iCloud Drive Documents identity.
         */
        ubiquityIdentityToken: ((NSCoding & NSCopying & NSObjectProtocol));
        /**
         * The delegate of the file manager object.
         */
        delegate: FileManagerDelegate | null;
        /**
         * The path to the program’s current directory.
         */
        currentDirectoryPath: string;
        /**
         * Initializes a file manager object that is authorized to perform privileged file system operations.
         */
        constructor (authorization: NSWorkspace.Authorization);
        /**
         * Returns the home directory for the specified user.
         */
        homeDirectory(userName: string):URL | null;
        /**
         * Locates and optionally creates the specified common directory in a domain.
         */
        url(directory: FileManager.SearchPathDirectory, domain: FileManager.SearchPathDomainMask, url: URL | null, shouldCreate: boolean):URL;
        /**
         * Returns an array of URLs for the specified common directory in the requested domains.
         */
        urls(directory: FileManager.SearchPathDirectory, domainMask: FileManager.SearchPathDomainMask):URL[];
        /**
         * Returns the container directory associated with the specified security application group identifier.
         */
        containerURL(groupIdentifier: string):URL | null;
        /**
         * Performs a shallow search of the specified directory and returns URLs for the contained items. 
         */
        contentsOfDirectory(url: URL, keys: URLResourceKey[] | null, mask: FileManager.DirectoryEnumerationOptions):URL[];
        /**
         * Returns a directory enumerator object that can be used to perform a deep enumeration of the directory at the specified URL.
         */
        enumerator(url: URL, keys: URLResourceKey[] | null, mask: FileManager.DirectoryEnumerationOptions, handler: ()=>any | null):FileManager.DirectoryEnumerator | null;
        /**
         * Returns an array of URLs that identify the mounted volumes available on the device.
         */
        mountedVolumeURLs(propertyKeys: URLResourceKey[] | null, options: FileManager.VolumeEnumerationOptions):URL[] | null;
        /**
         * Performs a deep enumeration of the specified directory and returns the paths of all of the contained subdirectories. 
         */
        subpathsOfDirectory(path: string):string[];
        /**
         * Returns an array of strings identifying the paths for all items in the specified directory. 
         */
        subpaths(path: string):string[] | null;
        /**
         * Creates a directory with the given attributes at the specified URL.
         */
        createDirectory(url: URL, createIntermediates: boolean, attributes: Record<FileAttributeKey, any> | null):void;
        /**
         * Creates a file with the specified content and attributes at the given location.
         */
        createFile(path: string, data: Data | null, attr: Record<FileAttributeKey, any> | null):boolean;
        /**
         * Removes the file or directory at the specified URL.
         */
        removeItem(URL: URL):void;
        /**
         * Moves an item to the trash.
         */
        trashItem(url: URL, outResultingURL: AutoreleasingUnsafeMutablePointer<NSURL?> | null):void;
        /**
         * Replaces the contents of the item at the specified URL in a manner that ensures no data loss occurs.
         */
        replaceItemAt(originalItemURL: URL, newItemURL: URL, backupItemName: string, options: FileManager.ItemReplacementOptions):URL | null;
        /**
         * Replaces the contents of the item at the specified URL in a manner that ensures no data loss occurs.
         */
        replaceItem(originalItemURL: URL, newItemURL: URL, backupItemName: string, options: FileManager.ItemReplacementOptions, resultingURL: AutoreleasingUnsafeMutablePointer<NSURL?> | null):void;
        /**
         * Copies the file at the specified URL to a new location synchronously.
         */
        copyItem(srcURL: URL, dstURL: URL):void;
        /**
         * Moves the file or directory at the specified URL to a new location synchronously.
         */
        moveItem(srcURL: URL, dstURL: URL):void;
        /**
         * Returns a Boolean indicating whether the item is targeted for storage in iCloud.
         */
        isUbiquitousItem(url: URL):boolean;
        /**
         * Indicates whether the item at the specified URL should be stored in iCloud.
         */
        setUbiquitous(flag: boolean, url: URL, destinationURL: URL):void;
        /**
         * Starts downloading (if necessary) the specified item to the local system.
         */
        startDownloadingUbiquitousItem(url: URL):void;
        /**
         * Removes the local copy of the specified item that’s stored in iCloud.
         */
        evictUbiquitousItem(url: URL):void;
        /**
         * Returns the services provided by the File Provider extension that manages the item at the given URL.
         */
        getFileProviderServicesForItem(url: URL, completionHandler: ()=>any):void;
        /**
         * Creates a symbolic link at the specified URL that points to an item at the given URL. 
         */
        createSymbolicLink(url: URL, destURL: URL):void;
        /**
         * Creates a hard link between the items at the specified URLs.
         */
        linkItem(srcURL: URL, dstURL: URL):void;
        /**
         * Returns the path of the item pointed to by a symbolic link.
         */
        destinationOfSymbolicLink(path: string):string;
        /**
         * Returns a Boolean value that indicates whether a file or directory exists at a specified path.
         */
        fileExists(path: string):boolean;
        /**
         * Returns a Boolean value that indicates whether the invoking object appears able to read a specified file. 
         */
        isReadableFile(path: string):boolean;
        /**
         * Returns a Boolean value that indicates whether the invoking object appears able to write to a specified file.
         */
        isWritableFile(path: string):boolean;
        /**
         * Returns a Boolean value that indicates whether the operating system appears able to execute a specified file.
         */
        isExecutableFile(path: string):boolean;
        /**
         * Returns a Boolean value that indicates whether the invoking object appears able to delete a specified file.
         */
        isDeletableFile(path: string):boolean;
        /**
         * Returns an array of strings representing the user-visible components of a given path.
         */
        componentsToDisplay(path: string):string[] | null;
        /**
         * Returns the display name of the file or directory at a specified path.
         */
        displayName(path: string):string;
        /**
         * Returns the attributes of the item at a given path.
         */
        attributesOfItem(path: string):Record<FileAttributeKey, any>;
        /**
         * Returns a dictionary that describes the attributes of the mounted file system on which a given path resides.
         */
        attributesOfFileSystem(path: string):Record<FileAttributeKey, any>;
        /**
         * Sets the attributes of the specified file or directory.
         */
        setAttributes(attributes: Record<FileAttributeKey, any>, path: string):void;
        /**
         * Returns the contents of the file at the specified path.
         */
        contents(path: string):Data | null;
        /**
         * Returns a Boolean value that indicates whether the files or directories in specified paths have the same contents.
         */
        contentsEqual(path1: string, path2: string):boolean;
        /**
         * Determines the type of relationship that exists between a directory and an item.
         */
        getRelationship(outRelationship: UnsafeMutablePointer<FileManager.URLRelationship>, directoryURL: URL, otherURL: URL):void;
        /**
         * Returns a C-string representation of a given path that properly encodes Unicode strings for use by the file system.
         */
        fileSystemRepresentation(path: string):UnsafePointer<CChar>;
        /**
         * Returns an 
         *  object whose contents are derived from the specified C-string path.
         */
        string(str: UnsafePointer<CChar>, len: number):string;
        /**
         * Changes the path of the current working directory to the specified path. 
         */
        changeCurrentDirectoryPath(path: string):boolean;
        /**
         * Starts the process of unmounting the specified volume.
         */
        unmountVolume(url: URL, mask: FileManager.UnmountOptions, completionHandler: ()=>any):void;
        /**
         * Changes the attributes of a given file or directory. 
         */
        changeFileAttributes(attributes: Record<AnyHashable, any>, path: string):boolean;
        /**
         * Returns a dictionary that describes the POSIX attributes of the file specified at a given.
         */
        fileAttributes(path: string, yorn: boolean):Record<AnyHashable, any> | null;
        /**
         * Returns a dictionary that describes the attributes of the mounted file system on which a given path resides.
         */
        fileSystemAttributes(path: string):Record<AnyHashable, any> | null;
        /**
         * Returns the directories and files (including symbolic links) contained in a given directory.
         */
        directoryContents(path: string):any[] | null;
        /**
         * Returns the path of the directory or file that a symbolic link at a given path refers to.
         */
        pathContentOfSymbolicLink(path: string):string;
        /**
         * Replaces the contents of the item at the specified URL in a manner that ensures no data loss occurs.
         */
        replaceItemAtURL(originalItemURL: NSURL, newItemURL: NSURL, backupItemName: string, options: FileManager.ItemReplacementOptions):NSURL | null;
    }
    /**
     * the same name namespace of class FileManager
     */
    export namespace FileManager {
        export class DirectoryEnumerator extends NSEnumerator {
            /**
             * A dictionary with the attributes of the directory at which enumeration started.
             */
            directoryAttributes: Record<FileAttributeKey, any> | null;
            /**
             * A dictionary with the attributes of the most recently returned file or subdirectory (as referenced by the pathname).
             */
            fileAttributes: Record<FileAttributeKey, any> | null;
            /**
             * The number of levels deep the current object is in the directory hierarchy being enumerated.
             */
            level: number;
            isEnumeratingDirectoryPostOrder: boolean;
            /**
             * Causes the receiver to skip recursion into the most recently obtained subdirectory.
             */
            skipDescendents():void;
            /**
             * Causes the receiver to skip recursion into the most recently obtained subdirectory.
             */
            skipDescendants():void;
        }
        export class VolumeEnumerationOptions {
            /**
             * The enumeration skips hidden volumes.
             */
            static skipHiddenVolumes: FileManager.VolumeEnumerationOptions;
            /**
             * The enumeration produces file reference URLs rather than path-based URLs.
             */
            static produceFileReferenceURLs: FileManager.VolumeEnumerationOptions;
            /**
             * Create a volume enumeration options value.
             */
            constructor (rawValue: number);
        }
        export class ItemReplacementOptions {
            /**
             * Only metadata from the new item is used, and metadata from the original item isn’t preserved (default).
             */
            static usingNewMetadataOnly: FileManager.ItemReplacementOptions;
            /**
             * The backup item remains in place after a successful replacement.
             */
            static withoutDeletingBackupItem: FileManager.ItemReplacementOptions;
            /**
             * Creates a value for a file replacement operation.
             */
            constructor (rawValue: number);
        }
        export enum URLRelationship {
            contains = 0,
            same = 1,
            other = 2,
        }
        export class UnmountOptions {
            /**
             * Specifies that all partitions on an unmountable disk should be unmounted.
             */
            static allPartitionsAndEjectDisk: FileManager.UnmountOptions;
            /**
             * Specifies that no UI should accompany the unmount operation.
             */
            static withoutUI: FileManager.UnmountOptions;
            /**
             * Creates an unmount option set from the given raw value.
             */
            constructor (rawValue: number);
        }
        export class DirectoryEnumerationOptions {
            /**
             * An option to perform a shallow enumeration that doesn’t descend into directories.
             */
            static skipsSubdirectoryDescendants: FileManager.DirectoryEnumerationOptions;
            /**
             * An option to treat packages like files and not descend into their contents.
             */
            static skipsPackageDescendants: FileManager.DirectoryEnumerationOptions;
            /**
             * An option to skip hidden files.
             */
            static skipsHiddenFiles: FileManager.DirectoryEnumerationOptions;
            static includesDirectoriesPostOrder: FileManager.DirectoryEnumerationOptions;
            static producesRelativePathURLs: FileManager.DirectoryEnumerationOptions;
            /**
             * Creates a directory enumeration options value.
             */
            constructor (rawValue: number);
        }
        export enum SearchPathDirectory {
            applicationDirectory = 1,
            demoApplicationDirectory = 2,
            developerApplicationDirectory = 3,
            adminApplicationDirectory = 4,
            libraryDirectory = 5,
            developerDirectory = 6,
            userDirectory = 7,
            documentationDirectory = 8,
            documentDirectory = 9,
            coreServiceDirectory = 10,
            autosavedInformationDirectory = 11,
            desktopDirectory = 12,
            cachesDirectory = 13,
            applicationSupportDirectory = 14,
            downloadsDirectory = 15,
            inputMethodsDirectory = 16,
            moviesDirectory = 17,
            musicDirectory = 18,
            picturesDirectory = 19,
            printerDescriptionDirectory = 20,
            sharedPublicDirectory = 21,
            preferencePanesDirectory = 22,
            applicationScriptsDirectory = 23,
            itemReplacementDirectory = 99,
            allApplicationsDirectory = 100,
            allLibrariesDirectory = 101,
            trashDirectory = 102,
        }
        export class SearchPathDomainMask {
            /**
             * The user’s home directory—the place to install user’s personal items (
             * ).
             */
            static userDomainMask: FileManager.SearchPathDomainMask;
            /**
             * The place to install items available to everyone on this machine.
             */
            static localDomainMask: FileManager.SearchPathDomainMask;
            /**
             * The place to install items available on the network (
             * ).
             */
            static networkDomainMask: FileManager.SearchPathDomainMask;
            /**
             * A directory for system files provided by Apple (
             * ) .
             */
            static systemDomainMask: FileManager.SearchPathDomainMask;
            /**
             * All domains. 
             */
            static allDomainsMask: FileManager.SearchPathDomainMask;
            /**
             * Creates a search path domain mask.
             */
            constructor (rawValue: number);
        }
    }
    export class NSFileProviderService extends NSObject {
        /**
         * The File Provider service’s name.
         */
        name: NSFileProviderServiceName;
        /**
         * Asynchronously returns the service’s connection object.
         */
        getFileProviderConnection(completionHandler: ()=>any):void;
    }
    export class NSFileProviderServiceName {
        /**
         * Instantiates a new service name from the provided string.
         */
        constructor (rawValue: string);
        /**
         * Instantiates a new service name from the provided string.
         */
        constructor (rawValue: string);
    }
    export class FileAttributeKey {
        /**
         * The key in a file attribute dictionary whose value indicates whether the file is read-only.
         */
        static appendOnly: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates whether the file is busy.
         */
        static busy: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates the file's creation date.
         */
        static creationDate: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates the identifier for the device on which the file resides.
         */
        static deviceIdentifier: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates whether the file’s extension is hidden.
         */
        static extensionHidden: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates the file’s group ID.
         */
        static groupOwnerAccountID: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates the group name of the file’s owner.
         */
        static groupOwnerAccountName: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates the file’s HFS creator code.
         */
        static hfsCreatorCode: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates the file’s HFS type code.
         */
        static hfsTypeCode: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates whether the file is mutable. 
         */
        static immutable: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates the file’s last modified date.
         */
        static modificationDate: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates the file’s owner's account ID.
         */
        static ownerAccountID: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates the name of the file’s owner.
         */
        static ownerAccountName: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates the file’s Posix permissions.
         */
        static posixPermissions: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value identifies the protection level for this file.
         */
        static protectionKey: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates the file’s reference count.
         */
        static referenceCount: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates the file’s size in bytes.
         */
        static size: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates the file’s filesystem file number. 
         */
        static systemFileNumber: FileAttributeKey;
        /**
         * The key in a file system attribute dictionary whose value indicates the number of free nodes in the file system.
         */
        static systemFreeNodes: FileAttributeKey;
        /**
         * The key in a file system attribute dictionary whose value indicates the amount of free space on the file system.
         */
        static systemFreeSize: FileAttributeKey;
        /**
         * The key in a file system attribute dictionary whose value indicates the number of nodes in the file system.
         */
        static systemNodes: FileAttributeKey;
        /**
         * The key in a file system attribute dictionary whose value indicates the filesystem number of the file system.
         */
        static systemNumber: FileAttributeKey;
        /**
         * The key in a file system attribute dictionary whose value indicates the size of the file system.
         */
        static systemSize: FileAttributeKey;
        /**
         * The key in a file attribute dictionary whose value indicates the file’s type. 
         */
        static type: FileAttributeKey;
        /**
         * Creates a file attribute key from a string.
         */
        constructor (rawValue: string);
        /**
         * Creates a file attribute key from a raw value string.
         */
        constructor (rawValue: string);
    }
    export class FileAttributeType {
        /**
         * A block special file.
         */
        static typeBlockSpecial: FileAttributeType;
        /**
         * A character special file.
         */
        static typeCharacterSpecial: FileAttributeType;
        /**
         * A directory.
         */
        static typeDirectory: FileAttributeType;
        /**
         * A regular file.
         */
        static typeRegular: FileAttributeType;
        /**
         * A socket.
         */
        static typeSocket: FileAttributeType;
        /**
         * A symbolic link.
         */
        static typeSymbolicLink: FileAttributeType;
        /**
         * A file whose type is unknown.
         */
        static typeUnknown: FileAttributeType;
        /**
         * Creates a file attribute type value.
         */
        constructor (rawValue: string);
    }
    export class FileProtectionType {
        /**
         * The file is stored in an encrypted format on disk and cannot be read from or written to while the device is locked or booting.
         */
        static complete: FileProtectionType;
        /**
         * The file is stored in an encrypted format on disk after it is closed.
         */
        static completeUnlessOpen: FileProtectionType;
        /**
         * The file is stored in an encrypted format on disk and cannot be accessed until after the device has booted.
         */
        static completeUntilFirstUserAuthentication: FileProtectionType;
        /**
         * The file has no special protections associated with it.
         */
        static none: FileProtectionType;
        /**
         * Creates a file protection type value.
         */
        constructor (rawValue: string);
    }
    export interface FileManagerDelegate extends NSObjectProtocol {
        /**
         * Asks the delegate if the file manager should move the specified item to the new URL. 
         */
        fileManager(fileManager: FileManager, srcURL: URL, dstURL: URL):boolean;
    }
    export class FileHandle extends NSObject {
        /**
         * The file handle associated with the standard error file.
         */
        static standardError: FileHandle;
        /**
         * The file handle associated with the standard input file.
         */
        static standardInput: FileHandle;
        /**
         * The file handle associated with the standard output file.
         */
        static standardOutput: FileHandle;
        /**
         * The file handle associated with a null device.
         */
        static nullDevice: FileHandle;
        /**
         * Posted when the file handle reads the data currently available in a file or at a communications channel.
         */
        static readCompletionNotification: NSNotification.Name;
        /**
         * The POSIX file descriptor associated with the receiver.
         */
        fileDescriptor: Int32;
        /**
         * The file’s contents, as an asynchronous sequence of bytes.
         */
        bytes: FileHandle.AsyncBytes;
        /**
         * The data currently available in the receiver.
         */
        availableData: Data;
        /**
         * The block to use for reading the contents of the file handle asynchronously.
         */
        readabilityHandler: ()=>any | null;
        /**
         * The block to use for writing the contents of the file handle asynchronously.
         */
        writeabilityHandler: ()=>any | null;
        /**
         * The position of the file pointer within the file represented by the file handle.
         */
        offsetInFile: UInt64;
        /**
         * Creates and returns a file handle object associated with the specified file descriptor.
         */
        constructor (fd: Int32);
        /**
         * Creates and returns a file handle object associated with the specified file descriptor and deallocation policy.
         */
        constructor (fd: Int32, closeopt: boolean);
        /**
         * Returns a file handle initialized for reading the file, device, or named socket at the specified path.
         */
        constructor (path: string);
        /**
         * Returns a file handle initialized for reading the file, device, or named socket at the specified URL.
         */
        constructor (url: URL);
        /**
         * Returns a file handle initialized for writing to the file, device, or named socket at the specified path.
         */
        constructor (path: string);
        /**
         * Returns a file handle initialized for writing to the file, device, or named socket at the specified URL.
         */
        constructor (url: URL);
        /**
         * Returns a file handle initialized for reading and writing to the file, device, or named socket at the specified path.
         */
        constructor (path: string);
        /**
         * Returns a file handle initialized for reading and writing to the file, device, or named socket at the specified URL.
         */
        constructor (url: URL);
        /**
         * Returns a file handle initialized from data in an unarchiver.
         */
        constructor (coder: NSCoder);
        /**
         * Reads the available data synchronously up to the end of file or maximum number of bytes.
         */
        readToEnd():Data | null;
        /**
         * Reads data synchronously up to the specified number of bytes.
         */
        read(count: number):Data | null;
        /**
         * Accepts a socket connection (for stream-type sockets only) in the background and creates a file handle for the “near” (client) end of the communications channel.
         */
        acceptConnectionInBackgroundAndNotify():void;
        /**
         * Reads from the file or communications channel in the background and posts a notification when finished.
         */
        readInBackgroundAndNotify():void;
        /**
         * Reads to the end of file from the file or communications channel in the background and posts a notification when finished.
         */
        readToEndOfFileInBackgroundAndNotify():void;
        /**
         * Asynchronously checks to see if data is available.
         */
        waitForDataInBackgroundAndNotify():void;
        /**
         * Gets the position of the file pointer within the file.
         */
        offset():UInt64;
        /**
         * Places the file pointer at the end of the file referenced by the file handle and returns the new file offset.
         */
        seekToEnd():UInt64;
        /**
         * Moves the file pointer to the specified offset within the file.
         */
        seek(offset: UInt64):void;
        /**
         * Disallows further access to the represented file or communications channel and signals end of file on communications channels that permit writing.
         */
        close():void;
        /**
         * Causes all in-memory data and attributes of the file represented by the file handle to write to permanent storage.
         */
        synchronize():void;
        /**
         * Truncates or extends the file represented by the file handle to a specified offset within the file and puts the file pointer at that position.
         */
        truncate(offset: UInt64):void;
        /**
         * Reads the available data synchronously up to the end of file or maximum number of bytes.
         */
        readDataToEndOfFile():Data;
        /**
         * Reads data synchronously up to the specified number of bytes.
         */
        readData(length: number):Data;
        /**
         * Writes the specified data synchronously to the file handle.
         */
        write(data: Data):void;
        /**
         * Places the file pointer at the end of the file referenced by the file handle and returns the new file offset.
         */
        seekToEndOfFile():UInt64;
        /**
         * Disallows further access to the represented file or communications channel and signals end of file on communications channels that permit writing.
         */
        closeFile():void;
        /**
         * Causes all in-memory data and attributes of the file represented by the handle to write to permanent storage.
         */
        synchronizeFile():void;
        /**
         * Truncates or extends the file represented by the file handle to a specified offset within the file and puts the file pointer at that position.
         */
        truncateFile(offset: UInt64):void;
    }
    /**
     * the same name namespace of class FileHandle
     */
    export namespace FileHandle {
        export class AsyncBytes {
            /**
             * The file bytes as an asynchronous sequence of characters.
             */
            characters: AsyncCharacterSequence<FileHandle.AsyncBytes>;
            /**
             * The file bytes as an asynchronous sequence of Unicode scalar values.
             */
            unicodeScalars: AsyncUnicodeScalarSequence<FileHandle.AsyncBytes>;
            /**
             * The file bytes as an asynchronous sequence of lines of text.
             */
            lines: AsyncLineSequence<FileHandle.AsyncBytes>;
            /**
             * Returns a Boolean value that indicates whether the asynchronous sequence contains the given element.
             */
            contains(search: UInt8):boolean;
            /**
             * Returns a Boolean value that indicates whether all elements produced by the asynchronous sequence satisfy the given predicate.
             */
            allSatisfy(predicate: Bool):boolean;
            /**
             * Returns the first element of the sequence that satisfies the given predicate.
             */
            first(predicate: Bool):UInt8 | null;
            /**
             * Returns the minimum element in an asynchronous sequence of comparable elements.
             */
            min():UInt8 | null;
            /**
             * Returns the maximum element in an asynchronous sequence of comparable elements.
             */
            max():UInt8 | null;
            /**
             * Returns an asynchronous sequence, up to the specified maximum length, containing the initial elements of the base asynchronous sequence.
             */
            prefix(count: number):AsyncPrefixSequence<FileHandle.AsyncBytes>;
            /**
             * Omits elements from the base asynchronous sequence until a given closure returns false, after which it passes through all remaining elements.
             */
            drop(predicate: Bool):AsyncDropWhileSequence<FileHandle.AsyncBytes>;
            /**
             * Omits a specified number of elements from the base asynchronous sequence, then passes through all remaining elements.
             */
            dropFirst(count: number):AsyncDropFirstSequence<FileHandle.AsyncBytes>;
            /**
             * Creates an asynchronous sequence that contains, in order, the elements of the base sequence that satisfy the given predicate.
             */
            filter(isIncluded: Bool):AsyncFilterSequence<FileHandle.AsyncBytes>;
            /**
             * Creates the asynchronous iterator that produces elements of this asynchronous sequence.
             */
            makeAsyncIterator():FileHandle.AsyncBytes.Iterator;
        }
        /**
         * the same name namespace of class AsyncBytes
         */
        export namespace AsyncBytes {
            export class Iterator {
                next():UInt8 | null;
            }
        }
    }
    export class NSFileSecurity extends NSObject {
        constructor (coder: NSCoder);
    }
    export class NSFileVersion extends NSObject {
        /**
         * The URL identifying the location of the file associated with the file version object.
         */
        url: URL;
        /**
         * The string containing the user-presentable name of the file version.
         */
        localizedName: string;
        /**
         * The user-presentable name of the computer on which the revision was saved.
         */
        localizedNameOfSavingComputer: string;
        /**
         * The modification date of the version.
         */
        modificationDate: Date | null;
        /**
         * The identifier for this version of the file. 
         */
        persistentIdentifier: NSCoding;
        /**
         * A Boolean value that specifies whether the system can delete the associated file at some future time.
         */
        isDiscardable: boolean;
        /**
         * A Boolean value indicating whether the contents of the version are in conflict with the contents of another version.
         */
        isConflict: boolean;
        /**
         * A Boolean value that indicates the version object is not in conflict (
         * ) or is in conflict (
         * ).
         */
        isResolved: boolean;
        hasLocalContents: boolean;
        hasThumbnail: boolean;
        originatorNameComponents: PersonNameComponents | null;
        /**
         * Returns the most recent version object for the file at the specified URL.
         */
        static currentVersionOfItem(url: URL):NSFileVersion | null;
        /**
         * Returns all versions of the specified file except the current version. 
         */
        static otherVersionsOfItem(url: URL):NSFileVersion[] | null;
        /**
         * Returns the version of the file that has the specified persistent ID.
         */
        static version(url: URL, persistentIdentifier: any):NSFileVersion | null;
        /**
         * Creates and returns a temporary directory to use for saving the contents of the file. 
         */
        static temporaryDirectoryURLForNewVersionOfItem(url: URL):URL;
        /**
         * Creates a version of the file at the specified location. 
         */
        static addOfItem(url: URL, contentsURL: URL, options: NSFileVersion.AddingOptions):NSFileVersion;
        /**
         * Returns an array of version objects that are currently in conflict for the specified URL.
         */
        static unresolvedConflictVersionsOfItem(url: URL):NSFileVersion[] | null;
        /**
         * Removes all versions of a file, except the current one, from the version store.
         */
        static removeOtherVersionsOfItem(url: URL):void;
        static getNonlocalVersionsOfItem(url: URL, completionHandler: ()=>any):void;
        /**
         * Replace the contents of the specified file with the contents of the current version’s file. 
         */
        replaceItem(url: URL, options: NSFileVersion.ReplacingOptions):URL;
        /**
         * Remove this version object and its associated file from the version store. 
         */
        remove():void;
    }
    /**
     * the same name namespace of class NSFileVersion
     */
    export namespace NSFileVersion {
        export class AddingOptions {
            static byMoving: NSFileVersion.AddingOptions;
            constructor (rawValue: number);
        }
        export class ReplacingOptions {
            static byMoving: NSFileVersion.ReplacingOptions;
            constructor (rawValue: number);
        }
    }
    export class FileWrapper extends NSObject {
        /**
         * This property contains a boolean value that indicates whether the file wrapper object is a regular-file.
         */
        isRegularFile: boolean;
        /**
         * This property contains a boolean value indicating whether the file wrapper is a directory file wrapper.
         */
        isDirectory: boolean;
        /**
         * A boolean that indicates whether the file wrapper object is a symbolic-link file wrapper.
         */
        isSymbolicLink: boolean;
        /**
         * The file wrappers contained by a directory file wrapper.
         */
        fileWrappers: Record<string, FileWrapper> | null;
        /**
         * The URL referenced by the file wrapper object, which must be a symbolic-link file wrapper.
         */
        symbolicLinkDestinationURL: URL | null;
        /**
         * The contents of the file wrapper as an opaque data object.
         */
        serializedRepresentation: Data | null;
        /**
         * The filename of the file wrapper object
         */
        filename: string;
        /**
         * The preferred filename for the file wrapper object.
         */
        preferredFilename: string;
        /**
         * A dictionary of file attributes.
         */
        fileAttributes: Record<string, any>;
        /**
         * The contents of the file-system node associated with a regular-file file wrapper.
         */
        regularFileContents: Data | null;
        /**
         * The icon that represents the file wrapper.
         */
        icon: NSImage | null;
        /**
         * Initializes a file wrapper instance whose kind is determined by the type of file-system node located by the URL.
         */
        constructor (url: URL, options: FileWrapper.ReadingOptions);
        /**
         * Initializes a file wrapper instance whose kind is determined by the type of file-system node located by the path.
         */
        constructor (path: string);
        /**
         * Initializes the receiver as a directory file wrapper, with a given file-wrapper list.
         */
        constructor (childrenByPreferredName: Record<string, FileWrapper>);
        /**
         * Initializes the receiver as a regular-file file wrapper.
         */
        constructor (contents: Data);
        /**
         * Initializes the receiver as a symbolic-link file wrapper.
         */
        constructor (path: string);
        /**
         * Initializes the receiver as a symbolic-link file wrapper that links to a specified file.
         */
        constructor (url: URL);
        /**
         * Initializes the receiver as a regular-file file wrapper from given serialized data.
         */
        constructor (serializeRepresentation: Data);
        constructor (inCoder: NSCoder);
        /**
         * Adds a child file wrapper to the receiver, which must be a directory file wrapper.
         */
        addFileWrapper(child: FileWrapper):string;
        /**
         * Removes a child file wrapper from the receiver, which must be a directory file wrapper.
         */
        removeFileWrapper(child: FileWrapper):void;
        /**
         * Creates a file wrapper from a given file-system node and adds it to the receiver, which must be a directory file wrapper.
         */
        addFile(path: string):string;
        /**
         * Creates a regular-file file wrapper with the given contents and adds it to the receiver, which must be a directory file wrapper.
         */
        addRegularFile(data: Data, fileName: string):string;
        /**
         * Creates a symbolic-link file wrapper pointing to a given file-system node and adds it to the receiver, which must be a directory file wrapper.
         */
        addSymbolicLink(path: string, filename: string):string;
        /**
         * Returns the dictionary key used by a directory to identify a given file wrapper.
         */
        keyForChildFileWrapper(child: FileWrapper):string;
        /**
         * Provides the pathname referenced by the file wrapper object, which must be a symbolic-link file wrapper.
         */
        symbolicLinkDestination():string;
        /**
         * Indicates whether the file wrapper needs to be updated to match a given file-system node.
         */
        needsToBeUpdated(path: string):boolean;
        /**
         * Indicates whether the contents of a file wrapper matches a directory, regular file, or symbolic link on disk.
         */
        matchesContents(url: URL):boolean;
        /**
         * Updates the file wrapper to match a given file-system node.
         */
        update(path: string):boolean;
        /**
         * Recursively rereads the entire contents of a file wrapper from the specified location on disk.
         */
        read(url: URL, options: FileWrapper.ReadingOptions):void;
        /**
         * Writes a file wrapper’s contents to a given file-system node.
         */
        write(path: string, atomicFlag: boolean, updateFilenamesFlag: boolean):boolean;
    }
    /**
     * the same name namespace of class FileWrapper
     */
    export namespace FileWrapper {
        export class ReadingOptions {
            /**
             * The option to read files immediately after creating a file wrapper.
             */
            static immediate: FileWrapper.ReadingOptions;
            /**
             * Whether file mapping for regular file wrappers is disallowed.
             */
            static withoutMapping: FileWrapper.ReadingOptions;
            constructor (rawValue: number);
        }
        export class WritingOptions {
            /**
             * Whether writing is done atomically.
             */
            static atomic: FileWrapper.WritingOptions;
            /**
             * Whether descendant file wrappers’
             *  properties are set if the writing succeeds.
             */
            static withNameUpdating: FileWrapper.WritingOptions;
            constructor (rawValue: number);
        }
    }
    export interface NSFilePresenter extends NSObjectProtocol {
        /**
         * The URL of the presented file or directory.
         */
        presentedItemURL: URL | null;
        /**
         * The operation queue in which to execute presenter-related messages. 
         */
        presentedItemOperationQueue: OperationQueue;
        /**
         * The URL of a secondary item’s primary presented file or directory.
         */
        primaryPresentedItemURL: URL | null;
        /**
         * Notifies your object that another object or process wants to read the presented file or directory. 
         */
        relinquishPresentedItem(reader: ()=>any):void;
        /**
         * Tells your object to save any unsaved changes for the presented item.
         */
        savePresentedItemChanges(completionHandler: ()=>any):void;
        /**
         * Tells your object that its presented item is about to be deleted.
         */
        accommodatePresentedItemDeletion(completionHandler: ()=>any):void;
        /**
         * Tells your object that the presented item moved or was renamed.
         */
        presentedItemDidMove(newURL: URL):void;
        /**
         * Tells your object that the presented item’s contents or attributes changed.
         */
        presentedItemDidChange():void;
        /**
         * Tells the delegate that a new version of the file or file package was added. 
         */
        presentedItemDidGain(version: NSFileVersion):void;
        /**
         * Tells the delegate that a version of the file or file package was removed.
         */
        presentedItemDidLose(version: NSFileVersion):void;
        /**
         * Tells the delegate that some other entity resolved a version conflict for the presenter’s file or file package. 
         */
        presentedItemDidResolveConflict(version: NSFileVersion):void;
        /**
         * Tells the delegate that the item inside the presented directory gained a new version.
         */
        presentedSubitem(url: URL, version: NSFileVersion):void;
        /**
         * Tells the delegate that some entity wants to delete an item that is inside of a presented directory. 
         */
        accommodatePresentedSubitemDeletion(url: URL, completionHandler: ()=>any):void;
        /**
         * Tells the delegate that an item was added to the presented directory.
         */
        presentedSubitemDidAppear(url: URL):void;
        /**
         * Tells the delegate that the contents or attributes of the specified item changed.
         */
        presentedSubitemDidChange(url: URL):void;
        /**
         * A list of ubiquity attributes used to generate and send notifications whenever an attribute in the list changes.
         */
        observedPresentedItemUbiquityAttributes: Set<URLResourceKey>;
        /**
         * Tells your object that the file or file package's ubiquity attributes have changed.
         */
        presentedItemDidChangeUbiquityAttributes(attributes: Set<URLResourceKey>):void;
    }
    export class NSFileAccessIntent extends NSObject {
        /**
         * The current URL for the item managed by the file access intent instance. (read-only)
         */
        url: URL;
        /**
         * Returns a file access intent object for reading the given URL with the provided options.
         */
        static readingIntent(url: URL, options: NSFileCoordinator.ReadingOptions):Self;
        /**
         * Returns a file access intent object for writing to the given URL with the provided options.
         */
        static writingIntent(url: URL, options: NSFileCoordinator.WritingOptions):Self;
    }
    export class NSFileCoordinator extends NSObject {
        /**
         * Returns an array containing the currently registered file presenter objects. 
         */
        static filePresenters: NSFilePresenter[];
        /**
         * A string that uniquely identifies the file access that was performed by this file coordinator.
         */
        purposeIdentifier: string;
        /**
         * Initializes and returns a file coordinator object using the specified file presenter.
         */
        constructor (filePresenterOrNil: NSFilePresenter | null);
        /**
         * Registers the specified file presenter object so that it can receive notifications.
         */
        static addFilePresenter(filePresenter: NSFilePresenter):void;
        /**
         * Unregisters the specified file presenter object.
         */
        static removeFilePresenter(filePresenter: NSFilePresenter):void;
        /**
         * Performs a number of coordinated-read or -write operations asynchronously.
         */
        coordinate(intents: NSFileAccessIntent[], queue: OperationQueue, accessor: ()=>any):void;
        /**
         * Prepare to read or write from multiple files in a single batch operation.
         */
        prepare(readingURLs: URL[], readingOptions: NSFileCoordinator.ReadingOptions, writingURLs: URL[], writingOptions: NSFileCoordinator.WritingOptions, outError: NSErrorPointer, batchAccessor: ()=>any):void;
        /**
         * Announces that your app is moving a file to a new URL.
         */
        item(oldURL: URL, newURL: URL):void;
        /**
         * Cancels any active file coordination calls.
         */
        cancel():void;
    }
    /**
     * the same name namespace of class NSFileCoordinator
     */
    export namespace NSFileCoordinator {
        export class ReadingOptions {
            static withoutChanges: NSFileCoordinator.ReadingOptions;
            static resolvesSymbolicLink: NSFileCoordinator.ReadingOptions;
            /**
             * Specify this constant if you want to read an item’s metadata without triggering a download.
             */
            static immediatelyAvailableMetadataOnly: NSFileCoordinator.ReadingOptions;
            /**
             * Specify this content when reading an item for the purpose of uploading its contents. 
             */
            static forUploading: NSFileCoordinator.ReadingOptions;
            /**
             * Instantiates a reading option using an unsigned integer.
             */
            constructor (rawValue: number);
        }
        export class WritingOptions {
            static forDeleting: NSFileCoordinator.WritingOptions;
            static forMoving: NSFileCoordinator.WritingOptions;
            static forMerging: NSFileCoordinator.WritingOptions;
            static forReplacing: NSFileCoordinator.WritingOptions;
            /**
             * Select this option when writing to change the file’s metadata only and not its contents.
             */
            static contentIndependentMetadataOnly: NSFileCoordinator.WritingOptions;
            /**
             * Instantiates a writing option using an unsigned integer.
             */
            constructor (rawValue: number);
        }
    }
    export interface NSCoding {
        /**
         * Returns an object initialized from data in a given unarchiver.
         */
         (coder: NSCoder);
        /**
         * Encodes the receiver using a given archiver.
         */
        encode(coder: NSCoder):void;
    }
    export interface NSSecureCoding extends NSCoding {
        /**
         * A Boolean value that indicates whether or not the class supports secure coding.
         */
        static supportsSecureCoding: boolean;
    }
    export interface DecodableWithConfiguration {
        /**
         * Creates a new instance by retrieving the instance’s data from the specified decoder with help from the provided configuration.
         */
         (decoder: Decoder, configuration: Self.DecodingConfiguration);
    }
    export interface DecodingConfigurationProviding {
        /**
         * The configuration instance that assists in decoding another type.
         */
        static decodingConfiguration: Self.DecodingConfiguration;
    }
    export interface EncodableWithConfiguration {
        /**
         * Encodes the value into the specified encoder with help from the provided configuration.
         */
        encode(encoder: Encoder, configuration: Self.EncodingConfiguration):void;
    }
    export interface EncodingConfigurationProviding {
        /**
         * The configuration instance that assists in encoding another type.
         */
        static encodingConfiguration: Self.EncodingConfiguration;
    }
    export class JSONEncoder {
        /**
         * A value that determines the readability, size, and element order of the encoded JSON object.
         */
        outputFormatting: JSONEncoder.OutputFormatting;
        /**
         * A value that determines how to encode a  type’s coding keys as JSON keys.
         */
        keyEncodingStrategy: JSONEncoder.KeyEncodingStrategy;
        /**
         * A dictionary you use to customize the encoding process by providing contextual information.
         */
        userInfo: Record<CodingUserInfoKey, any>;
        /**
         * The strategy used when encoding dates as part of a JSON object.
         */
        dateEncodingStrategy: JSONEncoder.DateEncodingStrategy;
        /**
         * The strategy that an encoder uses to encode raw data.
         */
        dataEncodingStrategy: JSONEncoder.DataEncodingStrategy;
        /**
         * The strategy used by an encoder when it encounters exceptional floating-point values.
         */
        nonConformingFloatEncodingStrategy: JSONEncoder.NonConformingFloatEncodingStrategy;
        /**
         * Creates a new, reusable JSON encoder with the default formatting settings and encoding strategies.
         */
        constructor ();
    }
    /**
     * the same name namespace of class JSONEncoder
     */
    export namespace JSONEncoder {
        export class OutputFormatting {
            /**
             * The output formatting option that uses ample white space and indentation to make output easy to read.
             */
            static prettyPrinted: JSONEncoder.OutputFormatting;
            /**
             * The output formatting option that sorts keys in lexicographic order.
             */
            static sortedKeys: JSONEncoder.OutputFormatting;
            /**
             * The output formatting option specifies that the output doesn’t prefix slash characters with escape characters.
             */
            static withoutEscapingSlashes: JSONEncoder.OutputFormatting;
            /**
             * A raw value representing a set of output formatting options.
             */
            rawValue: number;
            /**
             * A Boolean value that indicates whether the set has no elements.
             */
            isEmpty: boolean;
            /**
             * Creates a new set of output formatting options from the given raw value.
             */
            constructor (rawValue: number);
            /**
             * Creates an empty option set.
             */
            constructor ();
            /**
             * Creates a set containing the elements of the given array literal.
             */
            constructor (...arrayLiteral: JSONEncoder.OutputFormatting);
            /**
             * Returns a Boolean value that indicates whether a given element is a member of the option set.
             */
            contains(member: JSONEncoder.OutputFormatting):boolean;
            /**
             * Adds the given element to the option set if it is not already a member.
             */
            insert(newMember: JSONEncoder.OutputFormatting):any;
            /**
             * Inserts the given element into the set.
             */
            update(newMember: JSONEncoder.OutputFormatting):JSONEncoder.OutputFormatting | null;
            /**
             * Removes the given element and all elements subsumed by it.
             */
            remove(member: JSONEncoder.OutputFormatting):JSONEncoder.OutputFormatting | null;
            /**
             * Returns a new option set of the elements contained in this set, in the given set, or in both.
             */
            union(other: JSONEncoder.OutputFormatting):JSONEncoder.OutputFormatting;
            /**
             * Inserts the elements of another set into this option set.
             */
            formUnion(other: JSONEncoder.OutputFormatting):void;
            /**
             * Returns a new option set with only the elements contained in both this set and the given set.
             */
            intersection(other: JSONEncoder.OutputFormatting):JSONEncoder.OutputFormatting;
            /**
             * Removes all elements of this option set that are not also present in the given set.
             */
            formIntersection(other: JSONEncoder.OutputFormatting):void;
            /**
             * Returns a new option set with the elements contained in this set or in the given set, but not in both.
             */
            symmetricDifference(other: JSONEncoder.OutputFormatting):JSONEncoder.OutputFormatting;
            /**
             * Replaces this set with a new set containing all elements contained in either this set or the given set, but not in both.
             */
            formSymmetricDifference(other: JSONEncoder.OutputFormatting):void;
            /**
             * Returns a new set containing the elements of this set that do not occur in the given set.
             */
            subtracting(other: JSONEncoder.OutputFormatting):JSONEncoder.OutputFormatting;
            /**
             * Removes the elements of the given set from this set.
             */
            subtract(other: JSONEncoder.OutputFormatting):void;
            /**
             * Returns a Boolean value that indicates whether the set is a subset of another set.
             */
            isSubset(other: JSONEncoder.OutputFormatting):boolean;
            /**
             * Returns a Boolean value that indicates whether this set is a strict subset of the given set.
             */
            isStrictSubset(other: JSONEncoder.OutputFormatting):boolean;
            /**
             * Returns a Boolean value that indicates whether the set is a superset of the given set.
             */
            isSuperset(other: JSONEncoder.OutputFormatting):boolean;
            /**
             * Returns a Boolean value that indicates whether this set is a strict superset of the given set.
             */
            isStrictSuperset(other: JSONEncoder.OutputFormatting):boolean;
            /**
             * Returns a Boolean value that indicates whether the set has no members in common with the given set.
             */
            isDisjoint(other: JSONEncoder.OutputFormatting):boolean;
        }
        export enum KeyEncodingStrategy {
            convertToSnakeCase,
            useDefaultKeys,
            custom,
        }
        export enum DateEncodingStrategy {
            deferredToDate,
            iso8601,
            formatted,
            custom,
            millisecondsSince1970,
            secondsSince1970,
        }
        export enum DataEncodingStrategy {
            base64,
            custom,
            deferredToData,
        }
        export enum NonConformingFloatEncodingStrategy {
            convertToString,
            throw,
        }
    }
    export class JSONDecoder {
        /**
         * A value that determines how to decode a type’s coding keys from JSON keys.
         */
        keyDecodingStrategy: JSONDecoder.KeyDecodingStrategy;
        /**
         * A dictionary you use to customize the decoding process by providing contextual information.
         */
        userInfo: Record<CodingUserInfoKey, any>;
        /**
         * Specifies that decoding supports the JSON5 syntax.
         */
        allowsJSON5: boolean;
        /**
         * Specifies that decoding assumes the top level of the JSON data is a dictionary, even if it doesn’t begin and end with braces.
         */
        assumesTopLevelDictionary: boolean;
        /**
         * The strategy used when decoding dates from part of a JSON object.
         */
        dateDecodingStrategy: JSONDecoder.DateDecodingStrategy;
        /**
         * The strategy that a decoder uses to decode raw data.
         */
        dataDecodingStrategy: JSONDecoder.DataDecodingStrategy;
        /**
         * The strategy used by a decoder when it encounters exceptional floating-point values.
         */
        nonConformingFloatDecodingStrategy: JSONDecoder.NonConformingFloatDecodingStrategy;
        /**
         * Creates a new, reusable JSON decoder with the default formatting settings and decoding strategies.
         */
        constructor ();
    }
    /**
     * the same name namespace of class JSONDecoder
     */
    export namespace JSONDecoder {
        export enum KeyDecodingStrategy {
            convertFromSnakeCase,
            useDefaultKeys,
            custom,
        }
        export enum DateDecodingStrategy {
            deferredToDate,
            iso8601,
            formatted,
            custom,
            millisecondsSince1970,
            secondsSince1970,
        }
        export enum DataDecodingStrategy {
            base64,
            custom,
            deferredToData,
        }
        export enum NonConformingFloatDecodingStrategy {
            convertFromString,
            throw,
        }
    }
    export class NSString extends NSObject {
        /**
         * Returns a zero-terminated list of the encodings string objects support in the application’s environment.
         */
        static availableStringEncodings: UnsafePointer<UInt>;
        /**
         * Returns the C-string encoding assumed for any method accepting a C string as an argument.
         */
        static defaultCStringEncoding: number;
        /**
         * The number of UTF-16 code units in the receiver.
         */
        length: number;
        /**
         * A null-terminated UTF8 representation of the string.
         */
        utf8String: UnsafePointer<CChar> | null;
        /**
         * An unsigned integer that can be used as a hash table address.
         */
        hash: number;
        /**
         * A lowercase representation of the string.
         */
        lowercased: string;
        /**
         * Returns a version of the string with all letters converted to lowercase, taking into account the current locale.
         */
        localizedLowercase: string;
        /**
         * An uppercase representation of the string.
         */
        uppercased: string;
        /**
         * Returns a version of the string with all letters converted to uppercase, taking into account the current locale.
         */
        localizedUppercase: string;
        /**
         * A capitalized representation of the string.
         */
        capitalized: string;
        /**
         * Returns a capitalized representation of the receiver using the current locale.
         */
        localizedCapitalized: string;
        /**
         * A string made by normalizing the string’s contents using the Unicode Normalization Form D.
         */
        decomposedStringWithCanonicalMapping: string;
        /**
         * A string made by normalizing the receiver’s contents using the Unicode Normalization Form KD.
         */
        decomposedStringWithCompatibilityMapping: string;
        /**
         * A string made by normalizing the string’s contents using the Unicode Normalization Form C.
         */
        precomposedStringWithCanonicalMapping: string;
        /**
         * A string made by normalizing the receiver’s contents using the Unicode Normalization Form KC.
         */
        precomposedStringWithCompatibilityMapping: string;
        /**
         * The floating-point value of the string as a 
         * .
         */
        doubleValue: number;
        /**
         * The floating-point value of the string as a 
         * .
         */
        floatValue: Float;
        /**
         * The integer value of the string.
         */
        intValue: Int32;
        /**
         * The 
         *  value of the string.
         */
        integerValue: number;
        /**
         * The 
         *  value of the string.
         */
        longLongValue: Int64;
        /**
         * The Boolean value of the string.
         */
        boolValue: boolean;
        /**
         * This 
         *  object.
         */
        description: string;
        /**
         * The fastest encoding to which the receiver may be converted without loss of information.
         */
        fastestEncoding: number;
        /**
         * The smallest encoding to which the receiver can be converted without loss of information.
         */
        smallestEncoding: number;
        /**
         * The file-system path components of the receiver.
         */
        pathComponents: string[];
        /**
         * A file system-specific representation of the receiver. 
         */
        fileSystemRepresentation: UnsafePointer<CChar>;
        /**
         * A Boolean value that indicates whether the receiver represents an absolute path.
         */
        isAbsolutePath: boolean;
        /**
         * The last path component of the receiver.
         */
        lastPathComponent: string;
        /**
         * The path extension, if any, of the string as interpreted as a path.
         */
        pathExtension: string;
        /**
         * A new string that replaces the current home directory portion of the current path with a tilde (
         * ) character.
         */
        abbreviatingWithTildeInPath: string;
        /**
         * A new string made by deleting the last path component from the receiver, along with any final path separator.
         */
        deletingLastPathComponent: string;
        /**
         * A new string made by deleting the extension (if any, and only the last) from the receiver.
         */
        deletingPathExtension: string;
        /**
         * A new string made by expanding the initial component of the receiver to its full path value.
         */
        expandingTildeInPath: string;
        /**
         * A new string made from the receiver by resolving all symbolic links and standardizing path. 
         */
        resolvingSymlinksInPath: string;
        /**
         * A new string made by removing extraneous path components from the receiver.
         */
        standardizingPath: string;
        /**
         * Returns a new string made from the receiver by replacing all percent encoded sequences with the matching UTF-8 characters.
         */
        removingPercentEncoding: string;
        customPlaygroundQuickLook: PlaygroundQuickLook;
        /**
         * Returns an initialized 
         *  object that contains no characters.
         */
        constructor ();
        /**
         * Returns an initialized 
         *  object containing a given number of bytes from a given buffer of bytes interpreted in a given encoding.
         */
        constructor (bytes: UnsafeRawPointer, len: number, encoding: number);
        /**
         * Returns an initialized 
         *  object that contains a given number of bytes from a given buffer of bytes interpreted in a given encoding, and optionally frees the buffer.
         */
        constructor (bytes: UnsafeMutableRawPointer, len: number, encoding: number, freeBuffer: boolean);
        /**
         * Returns an initialized 
         *  object that contains a given number of characters from a given C array of UTF-16 code units.
         */
        constructor (characters: UnsafePointer<unichar>, length: number);
        /**
         * Returns an initialized 
         *  object that contains a given number of characters from a given C array of UTF-16 code units.
         */
        constructor (characters: UnsafeMutablePointer<unichar>, length: number, freeBuffer: boolean);
        /**
         * Returns an 
         *  object initialized by copying the characters from another given string.
         */
        constructor (aString: string);
        /**
         * Returns an 
         *  object initialized using the characters in a given C array, interpreted according to a given encoding. 
         */
        constructor (nullTerminatedCString: UnsafePointer<CChar>, encoding: number);
        /**
         * Returns an 
         *  object initialized by copying the characters from a given C array of UTF8-encoded bytes.
         */
        constructor (nullTerminatedCString: UnsafePointer<CChar>);
        /**
         * Returns an 
         *  object initialized by using a given format string as a template into which the remaining argument values are substituted without any localization.
         */
        constructor (format: string, argList: CVaListPointer);
        /**
         * Returns an 
         *  object initialized by using a given format string as a template into which the remaining argument values are substituted according to given locale information. This method is meant to be called from within a variadic function, where the argument list will be available.
         */
        constructor (format: string, locale: any, argList: CVaListPointer);
        /**
         * Returns an 
         *  object initialized by converting given data into UTF-16 code units using a given encoding.
         */
        constructor (data: Data, encoding: number);
        /**
         * Returns an 
         *  object initialized by reading data from the file at a given path using a given encoding.
         */
        constructor (path: string, enc: number);
        /**
         * Returns an 
         *  object initialized by reading data from the file at a given path and returns by reference the encoding used to interpret the characters. 
         */
        constructor (path: string, enc: UnsafeMutablePointer<UInt> | null);
        /**
         * Returns an 
         *  object initialized by reading data from a given URL interpreted using a given encoding. 
         */
        constructor (url: URL, enc: number);
        /**
         * Returns an 
         *  object initialized by reading data from a given URL and returns by reference the encoding used to interpret the data. 
         */
        constructor (url: URL, enc: UnsafeMutablePointer<UInt> | null);
        /**
         * Initializes the receiver, a newly allocated 
         *  object, by converting the data in a given C-string from the default C-string encoding into the Unicode character encoding.
         */
        constructor (bytes: UnsafePointer<CChar>);
        /**
         * Initializes the receiver, a newly allocated 
         *  object, by converting the data in a given C-string from the default C-string encoding into the Unicode character encoding.
         */
        constructor (bytes: UnsafePointer<CChar>, length: number);
        /**
         * Initializes the receiver, a newly allocated 
         *  object, by converting the data in a given C-string from the default C-string encoding into the Unicode character encoding.
         */
        constructor (bytes: UnsafeMutablePointer<CChar>, length: number, freeBuffer: boolean);
        /**
         * Initializes the receiver, a newly allocated 
         *  object, by reading data from the file named by 
         * .
         */
        constructor (path: string);
        /**
         * Initializes the receiver, a newly allocated 
         *  object, by reading data from the location named by a given URL.
         */
        constructor (url: URL);
        constructor (bytes: UnsafeMutableRawPointer, len: number, encoding: number, deallocator: ()=>any | null);
        constructor (chars: UnsafeMutablePointer<unichar>, len: number, deallocator: ()=>any | null);
        constructor (coder: NSCoder);
        constructor (format: NSString, ...args: CVarArg);
        constructor (format: NSString, locale: Locale | null, ...args: CVarArg);
        /**
         * Create an instance initialized to 
         * .
         */
        constructor (value: StaticString);
        /**
         * Returns a localized string intended for display in a notification alert.
         */
        static localizedUserNotificationString(key: string, arguments: any[] | null):string;
        static localizedStringWithFormat(format: NSString, ...args: CVarArg):Self;
        /**
         * Returns the string encoding for the given data as detected by attempting to create a string according to the specified encoding options.
         */
        static stringEncoding(data: Data, opts: Record<StringEncodingDetectionOptionsKey, any> | null, _string: AutoreleasingUnsafeMutablePointer<NSString?> | null, usedLossyConversion: UnsafeMutablePointer<ObjCBool> | null):number;
        /**
         * Returns a human-readable string giving the name of a given encoding.
         */
        static localizedName(encoding: number):string;
        /**
         * Returns a string built from the strings in a given array by concatenating them with a path separator between each pair.
         */
        static path(components: string[]):string;
        /**
         * Creates a new string using a given C-string.
         */
        static string(bytes: UnsafePointer<CChar>):any;
        static deferredLocalizedIntentsString(format: string, ...args: CVarArg):NSString;
        /**
         * Returns the number of bytes required to store the receiver in a given encoding.
         */
        lengthOfBytes(enc: number):number;
        /**
         * Returns the maximum number of bytes needed to store the receiver in a given encoding.
         */
        maximumLengthOfBytes(enc: number):number;
        /**
         * Returns the character at a given UTF-16 code unit index.
         */
        character(index: number):unichar;
        /**
         * Copies characters from a given range in the receiver into a given buffer.
         */
        getCharacters(buffer: UnsafeMutablePointer<unichar>, range: NSRange):void;
        /**
         * Gets a given range of characters as bytes in a specified encoding.
         */
        getBytes(buffer: UnsafeMutableRawPointer | null, maxBufferCount: number, usedBufferCount: UnsafeMutablePointer<Int> | null, encoding: number, options: NSString.EncodingConversionOptions, range: NSRange, leftover: NSRangePointer | null):boolean;
        /**
         * Returns a representation of the string as a C string using a given encoding.
         */
        cString(encoding: number):UnsafePointer<CChar> | null;
        /**
         * Converts the string to a given encoding and stores it in a buffer.
         */
        getCString(buffer: UnsafeMutablePointer<CChar>, maxBufferCount: number, encoding: number):boolean;
        /**
         * Returns the result of invoking 
         *  with 
         *  as the only option.
         */
        caseInsensitiveCompare(_string: string):ComparisonResult;
        /**
         * Compares the string with a given string using a case-insensitive, localized, comparison. 
         */
        localizedCaseInsensitiveCompare(_string: string):ComparisonResult;
        /**
         * Returns the result of invoking 
         *  with no options and the receiver’s full extent as the range.
         */
        compare(_string: string):ComparisonResult;
        /**
         * Compares the string and a given string using a localized comparison.
         */
        localizedCompare(_string: string):ComparisonResult;
        /**
         * Compares strings as sorted by the Finder.
         */
        localizedStandardCompare(_string: string):ComparisonResult;
        /**
         * Returns a Boolean value that indicates whether a given string matches the beginning characters of the receiver.
         */
        hasPrefix(str: string):boolean;
        /**
         * Returns a Boolean value that indicates whether a given string matches the ending characters of the receiver. 
         */
        hasSuffix(str: string):boolean;
        /**
         * Returns a Boolean value that indicates whether a given string is equal to the receiver using a literal Unicode-based comparison.
         */
        isEqual(aString: string):boolean;
        appendingFormat(format: NSString, ...args: CVarArg):NSString;
        /**
         * Returns a new string made by appending a given string to the receiver.
         */
        appending(aString: string):string;
        /**
         * Returns a new string formed from the receiver by either removing characters from the end, or by appending as many occurrences as necessary of a given pad string.
         */
        padding(newLength: number, padString: string, padIndex: number):string;
        /**
         * Returns an array containing substrings from the receiver that have been divided by a given separator.
         */
        components(separator: string):string[];
        /**
         * Returns a new string made by removing from both ends of the receiver characters contained in a given character set.
         */
        trimmingCharacters(set: CharacterSet):string;
        /**
         * Returns a new string containing the characters of the receiver from the one at a given index to the end.
         */
        substring(from: number):string;
        /**
         * Creates a string suitable for comparison by removing the specified character distinctions from a string.
         */
        folding(options: NSString.CompareOptions, locale: Locale | null):string;
        /**
         * Returns a new string by applying a specified transform to the string.
         */
        applyingTransform(transform: StringTransform, reverse: boolean):string;
        /**
         * Returns a Boolean value indicating whether the string contains a given string by performing a case-sensitive, locale-unaware search.
         */
        contains(str: string):boolean;
        /**
         * Returns a Boolean value indicating whether the string contains a given string by performing a case-insensitive, locale-aware search.
         */
        localizedCaseInsensitiveContains(str: string):boolean;
        /**
         * Returns a Boolean value indicating whether the string contains a given string by performing a case and diacritic insensitive, locale-aware search.
         */
        localizedStandardContains(str: string):boolean;
        /**
         * Finds and returns the range in the string of the first character from a given character set.
         */
        rangeOfCharacter(searchSet: CharacterSet):NSRange;
        /**
         * Finds and returns the range of the first occurrence of a given string within the string.
         */
        range(searchString: string):NSRange;
        /**
         * Finds and returns the range of the first occurrence of a given string within the string by performing a case and diacritic insensitive, locale-aware search.
         */
        localizedStandardRange(str: string):NSRange;
        /**
         * Enumerates all the lines in the string.
         */
        enumerateLines(block: ()=>any):void;
        /**
         * Enumerates the substrings of the specified type in the specified range of the string.
         */
        enumerateSubstrings(range: NSRange, opts: NSString.EnumerationOptions, block: ()=>any):void;
        /**
         * Returns a new string in which all occurrences of a target string in the receiver are replaced by another given string.
         */
        replacingOccurrences(target: string, replacement: string):string;
        /**
         * Returns a new string in which the characters in a specified range of the receiver are replaced by a given string.
         */
        replacingCharacters(range: NSRange, replacement: string):string;
        /**
         * Returns a string containing characters the receiver and a given string have in common, starting from the beginning of each up to the first characters that aren’t equivalent.
         */
        commonPrefix(str: string, mask: NSString.CompareOptions):string;
        /**
         * Performs linguistic analysis on the specified string by enumerating the specific range of the string, providing the Block with the located tags.
         */
        enumerateLinguisticTags(range: NSRange, scheme: NSLinguisticTagScheme, options: NSLinguisticTagger.Options, orthography: NSOrthography | null, block: ()=>any):void;
        /**
         * Returns an array of linguistic tags for the specified range and requested tags within the receiving string.
         */
        linguisticTags(range: NSRange, scheme: NSLinguisticTagScheme, options: NSLinguisticTagger.Options, orthography: NSOrthography | null, tokenRanges: AutoreleasingUnsafeMutablePointer<NSArray?> | null):NSLinguisticTag[];
        /**
         * Returns by reference the beginning of the first line and the end of the last line touched by the given range.
         */
        getLineStart(startPtr: UnsafeMutablePointer<Int> | null, lineEndPtr: UnsafeMutablePointer<Int> | null, contentsEndPtr: UnsafeMutablePointer<Int> | null, range: NSRange):void;
        /**
         * Returns the range of characters representing the line or lines containing a given range.
         */
        lineRange(range: NSRange):NSRange;
        /**
         * Returns by reference the beginning of the first paragraph and the end of the last paragraph touched by the given range.
         */
        getParagraphStart(startPtr: UnsafeMutablePointer<Int> | null, parEndPtr: UnsafeMutablePointer<Int> | null, contentsEndPtr: UnsafeMutablePointer<Int> | null, range: NSRange):void;
        /**
         * Returns the range of characters representing the paragraph or paragraphs containing a given range.
         */
        paragraphRange(range: NSRange):NSRange;
        /**
         * Returns the range in the receiver of the composed character sequence located at a given index.
         */
        rangeOfComposedCharacterSequence(index: number):NSRange;
        /**
         * Returns the range in the string of the composed character sequences for a given range.
         */
        rangeOfComposedCharacterSequences(range: NSRange):NSRange;
        /**
         * Writes the contents of the receiver to a file at a given path using a given encoding. 
         */
        write(path: string, useAuxiliaryFile: boolean, enc: number):void;
        /**
         * Parses the receiver as a text representation of a property list, returning an 
         * , 
         * , 
         * , or 
         *  object, according to the topmost element.
         */
        propertyList():any;
        /**
         * Returns a dictionary object initialized with the keys and values found in the receiver.
         */
        propertyListFromStringsFileFormat():Record<AnyHashable, any> | null;
        /**
         * Draws the receiver with the font and other display characteristics of the given attributes, at the specified point in the current graphics context.
         */
        draw(point: CGPoint, attrs: Record<NSAttributedString.Key, any> | null):void;
        /**
         * Calculates and returns the bounding rect for the receiver drawn using the given options and display characteristics, within the specified rectangle in the current graphics context.
         */
        boundingRect(size: CGSize, options: NSStringDrawingOptions, attributes: Record<NSAttributedString.Key, any> | null, context: NSStringDrawingContext | null):CGRect;
        /**
         * Returns the bounding box size the receiver occupies when drawn with the given attributes.
         */
        size(attrs: Record<NSAttributedString.Key, any> | null):CGSize;
        /**
         * Returns a string variation suitable for the specified presentation width. 
         */
        variantFittingPresentationWidth(width: number):string;
        /**
         * Returns a Boolean value that indicates whether the receiver can be converted to a given encoding without loss of information. 
         */
        canBeConverted(encoding: number):boolean;
        /**
         * Returns an 
         *  object containing a representation of the receiver encoded using a given encoding.
         */
        data(encoding: number):Data | null;
        /**
         * Interprets the receiver as a path in the file system and attempts to perform filename completion, returning a numeric value that indicates whether a match was possible, and by reference the longest path that matches the receiver. 
         */
        completePath(outputName: AutoreleasingUnsafeMutablePointer<NSString?> | null, flag: boolean, outputArray: AutoreleasingUnsafeMutablePointer<NSArray?> | null, filterTypes: string[] | null):number;
        /**
         * Interprets the receiver as a system-independent path and fills a buffer with a C-string in a format and encoding suitable for use with file-system calls.
         */
        getFileSystemRepresentation(cname: UnsafeMutablePointer<CChar>, max: number):boolean;
        /**
         * Returns a new string made by appending to the receiver a given string.
         */
        appendingPathComponent(str: string):string;
        /**
         * Returns a new string made by appending to the receiver an extension separator followed by a given extension.
         */
        appendingPathExtension(str: string):string;
        /**
         * Returns an array of strings made by separately appending to the receiver each string in a given array.
         */
        strings(paths: string[]):string[];
        /**
         * Returns a new string made from the receiver by replacing all characters not in the specified set with percent-encoded characters.
         */
        addingPercentEncoding(allowedCharacters: CharacterSet):string;
        /**
         * Returns a representation of the receiver as a C string in the default C-string encoding, possibly losing information in converting to that encoding.
         */
        lossyCString():UnsafePointer<CChar> | null;
        /**
         * Returns the length in char-sized units of the receiver’s C-string representation in the default C-string encoding.
         */
        cStringLength():number;
        /**
         * Returns a representation of the receiver using a given encoding to determine the percent escapes necessary to convert the receiver into a legal URL string.
         */
        addingPercentEscapes(enc: number):string;
        /**
         * Returns a new string made by replacing in the receiver all percent escapes with the matching characters as determined by a given encoding.
         */
        replacingPercentEscapes(enc: number):string;
    }
    /**
     * the same name namespace of class NSString
     */
    export namespace NSString {
        export class CompareOptions {
            /**
             * A case-insensitive search.
             */
            static caseInsensitive: NSString.CompareOptions;
            /**
             * Exact character-by-character equivalence.
             */
            static literal: NSString.CompareOptions;
            /**
             * Search from end of source string.
             */
            static backwards: NSString.CompareOptions;
            /**
             * Search is limited to start (or end, if 
             * ) of source string.
             */
            static anchored: NSString.CompareOptions;
            /**
             * Numbers within strings are compared using numeric value, that is, 
             *  < 
             *  < 
             * . 
             */
            static numeric: NSString.CompareOptions;
            /**
             * Search ignores diacritic marks.
             */
            static diacriticInsensitive: NSString.CompareOptions;
            /**
             * Search ignores width differences in characters that have full-width and half-width forms, as occurs in East Asian character sets.
             */
            static widthInsensitive: NSString.CompareOptions;
            /**
             * Comparisons are forced to return either 
             *  or 
             *  if the strings are equivalent but not strictly equal.
             */
            static forcedOrdering: NSString.CompareOptions;
            /**
             * The search string is treated as an ICU-compatible regular expression. If set, no other options can apply except 
             *  and 
             * . You can use this option only with the 
             * … methods and 
             * .
             */
            static regularExpression: NSString.CompareOptions;
            constructor (rawValue: number);
        }
        export class EncodingConversionOptions {
            /**
             * Allows lossy conversion.
             */
            static allowLossy: NSString.EncodingConversionOptions;
            /**
             * Specifies an external representation (with a byte-order mark, if necessary, to indicate endianness).
             */
            static externalRepresentation: NSString.EncodingConversionOptions;
            constructor (rawValue: number);
        }
        export class EnumerationOptions {
            static byLines: NSString.EnumerationOptions;
            static byParagraphs: NSString.EnumerationOptions;
            static byComposedCharacterSequences: NSString.EnumerationOptions;
            static byWords: NSString.EnumerationOptions;
            static bySentences: NSString.EnumerationOptions;
            static reverse: NSString.EnumerationOptions;
            static substringNotRequired: NSString.EnumerationOptions;
            static localized: NSString.EnumerationOptions;
            static byCaretPositions: NSString.EnumerationOptions;
            static byDeletionClusters: NSString.EnumerationOptions;
            constructor (rawValue: number);
        }
    }
    export class JSONSerialization extends NSObject {
        /**
         * Returns a Foundation object from given JSON data.
         */
        static jsonObject(data: Data, opt: JSONSerialization.ReadingOptions):any;
        /**
         * Returns JSON data from a Foundation object.
         */
        static data(obj: any, opt: JSONSerialization.WritingOptions):Data;
        /**
         * Writes a given JSON object to a stream.
         */
        static writeJSONObject(obj: any, stream: OutputStream, opt: JSONSerialization.WritingOptions, error: NSErrorPointer):number;
        /**
         * Returns a Boolean value that indicates whether the serializer can convert a given object to JSON data.
         */
        static isValidJSONObject(obj: any):boolean;
    }
    /**
     * the same name namespace of class JSONSerialization
     */
    export namespace JSONSerialization {
        export class ReadingOptions {
            /**
             * Specifies that arrays and dictionaries in the returned object are mutable.
             */
            static mutableContainers: JSONSerialization.ReadingOptions;
            /**
             * Specifies that leaf strings in the JSON object graph are mutable.
             */
            static mutableLeaves: JSONSerialization.ReadingOptions;
            /**
             * Specifies that the parser allows top-level objects that aren’t arrays or dictionaries.
             */
            static fragmentsAllowed: JSONSerialization.ReadingOptions;
            /**
             * Specifies that reading serialized JSON data supports the JSON5 syntax.
             */
            static json5Allowed: JSONSerialization.ReadingOptions;
            /**
             * Specifies that the parser assumes the top level of the JSON data is a dictionary, even if it doesn’t begin and end with curly braces.
             */
            static topLevelDictionaryAssumed: JSONSerialization.ReadingOptions;
            /**
             * A deprecated option that specifies that the parser should allow top-level objects that aren’t arrays or dictionaries.
             */
            static allowFragments: JSONSerialization.ReadingOptions;
            /**
             * Creates a set of JSON formatting options from an integer that represents those options.
             */
            constructor (rawValue: number);
        }
        export class WritingOptions {
            /**
             * Specifies that the parser should allow top-level objects that aren’t arrays or dictionaries.
             */
            static fragmentsAllowed: JSONSerialization.WritingOptions;
            /**
             * Specifies that the output uses white space and indentation to make the resulting data more readable.
             */
            static prettyPrinted: JSONSerialization.WritingOptions;
            /**
             * Specifies that the output sorts keys in lexicographic order.
             */
            static sortedKeys: JSONSerialization.WritingOptions;
            /**
             * Specifies that the output doesn’t prefix slash characters with escape characters.
             */
            static withoutEscapingSlashes: JSONSerialization.WritingOptions;
            /**
             * Creates a set of JSON formatting options from an integer that represents those options.
             */
            constructor (rawValue: number);
        }
    }
    export class PropertyListEncoder {
        /**
         * A value that determines which property list format is used during encoding.
         */
        outputFormat: PropertyListSerialization.PropertyListFormat;
        /**
         * A dictionary you use to customize the encoding process by providing contextual information.
         */
        userInfo: Record<CodingUserInfoKey, any>;
        /**
         * Creates a new, reusable property list encoder with the default formatting settings.
         */
        constructor ();
    }
    export class PropertyListDecoder {
        /**
         * A dictionary you use to customize decoding by providing contextual information.
         */
        userInfo: Record<CodingUserInfoKey, any>;
        /**
         * Creates a new, reusable property list decoder.
         */
        constructor ();
    }
    export class PropertyListSerialization extends NSObject {
        /**
         * Returns an 
         *  object containing a given property list in a specified format.
         */
        static data(plist: any, format: PropertyListSerialization.PropertyListFormat, opt: PropertyListSerialization.WriteOptions):Data;
        /**
         * Writes a property list to the specified stream.
         */
        static writePropertyList(plist: any, stream: OutputStream, format: PropertyListSerialization.PropertyListFormat, opt: PropertyListSerialization.WriteOptions, error: NSErrorPointer):number;
        /**
         * Creates and returns a property list from the specified data.
         */
        static propertyList(data: Data, opt: PropertyListSerialization.ReadOptions, format: UnsafeMutablePointer<PropertyListSerialization.PropertyListFormat> | null):any;
        /**
         * This method is obsolete and will be deprecated soon.
         */
        static dataFromPropertyList(plist: any, format: PropertyListSerialization.PropertyListFormat, errorString: UnsafeMutablePointer<NSString?> | null):Data | null;
        /**
         * This method is deprecated. Use 
         *  instead.
         */
        static propertyListFromData(data: Data, opt: PropertyListSerialization.MutabilityOptions, format: UnsafeMutablePointer<PropertyListSerialization.PropertyListFormat> | null, errorString: UnsafeMutablePointer<NSString?> | null):any;
    }
    /**
     * the same name namespace of class PropertyListSerialization
     */
    export namespace PropertyListSerialization {
        export class MutabilityOptions {
            /**
             * Causes the returned property list to have mutable containers but immutable leaves.
             */
            static mutableContainers: PropertyListSerialization.MutabilityOptions;
            /**
             * Causes the returned property list to have mutable containers and leaves.
             */
            static mutableContainersAndLeaves: PropertyListSerialization.MutabilityOptions;
            constructor (rawValue: number);
        }
        export enum PropertyListFormat {
            openStep = 1,
            xml = 100,
            binary = 200,
        }
    }
    export class XMLDTD extends XMLNode {
        /**
         * Returns the receiver’s public identifier.
         */
        publicID: string;
        /**
         * Returns the receiver’s system identifier.
         */
        systemID: string;
        /**
         * Initializes and returns an 
         *  object created from the DTD declarations in a URL-referenced source.
         */
        constructor (url: URL, mask: XMLNode.Options);
        /**
         * Initializes and returns an 
         *  object created from the DTD declarations encapsulated in an 
         *  object 
         */
        constructor (data: Data, mask: XMLNode.Options);
        constructor ();
        /**
         * Returns a DTD node representing the predefined entity declaration with the specified name.
         */
        static predefinedEntityDeclaration(name: string):XMLDTDNode | null;
        /**
         * Adds a child node to the end of the list of existing children.
         */
        addChild(child: XMLNode):void;
        /**
         * Inserts a child node in the receiver’s list of children at a specific location in the list.
         */
        insertChild(child: XMLNode, index: number):void;
        /**
         * Inserts an array of child nodes at a specified location in the receiver’s list of children.
         */
        insertChildren(children: XMLNode[], index: number):void;
        /**
         * Removes the child node at a particular location in the receiver’s list of children.
         */
        removeChild(index: number):void;
        /**
         * Replaces a child at a particular index with another child.
         */
        replaceChild(index: number, node: XMLNode):void;
        /**
         * Removes all existing children of the receiver and replaces them with an array of new child nodes.
         */
        setChildren(children: XMLNode[] | null):void;
        /**
         * Returns the DTD node representing an element declaration for a specified element.
         */
        elementDeclaration(name: string):XMLDTDNode | null;
        /**
         * Returns the DTD node representing an attribute-list declaration for a given attribute and its element.
         */
        attributeDeclaration(name: string, elementName: string):XMLDTDNode | null;
        /**
         * Returns the DTD node representing the entity declaration for a specified entity.
         */
        entityDeclaration(name: string):XMLDTDNode | null;
        /**
         * Returns the DTD node representing the notation declaration identified by the specified notation name.
         */
        notationDeclaration(name: string):XMLDTDNode | null;
    }
    export class XMLDTDNode extends XMLNode {
        /**
         * Returns the receiver’s DTD kind.
         */
        dtdKind: XMLDTDNode.DTDKind;
        /**
         * Returns the name of the notation associated with the receiver.
         */
        notationName: string;
        /**
         * Returns the public identifier associated with the receiver.
         */
        publicID: string;
        /**
         * Returns the system identifier associated with the receiver.
         */
        systemID: string;
        isExternal: boolean;
        /**
         * Returns an 
         *  object initialized with the DTD declaration in a given string.
         */
        constructor (_string: string);
        constructor ();
        constructor (kind: XMLNode.Kind, options: XMLNode.Options);
    }
    /**
     * the same name namespace of class XMLDTDNode
     */
    export namespace XMLDTDNode {
        export enum DTDKind {
            cdataAttribute = 6,
            entitiesAttribute = 11,
            entityAttribute = 10,
            enumerationAttribute = 14,
            idAttribute = 7,
            idRefAttribute = 8,
            idRefsAttribute = 9,
            nmTokenAttribute = 12,
            nmTokensAttribute = 13,
            notationAttribute = 15,
            anyDeclaration = 18,
            elementDeclaration = 20,
            emptyDeclaration = 17,
            mixedDeclaration = 19,
            undefinedDeclaration = 16,
            general = 1,
            parameter = 4,
            parsed = 2,
            predefined = 5,
            unparsed = 3,
        }
    }
    export class XMLDocument extends XMLNode {
        /**
         * Sets the character encoding of the receiver to 
         * , 
         */
        characterEncoding: string;
        /**
         * Sets the kind of output content for the receiver.
         */
        documentContentKind: XMLDocument.ContentKind;
        /**
         * Returns an 
         *  object representing the internal DTD associated with the receiver.
         */
        dtd: XMLDTD | null;
        /**
         * Sets a Boolean value that specifies whether the receiver represents a standalone XML document.
         */
        isStandalone: boolean;
        /**
         * Returns the MIME type for the receiver.
         */
        mimeType: string;
        /**
         * Sets the version of the receiver’s XML.
         */
        version: string;
        /**
         * Returns the XML string representation of the receiver—that is, the entire document—encapsulated in a data object. 
         */
        xmlData: Data;
        /**
         * Initializes and returns an NSXMLDocument object created from the XML or HTML contents of a URL-referenced source
         */
        constructor (url: URL, mask: XMLNode.Options);
        /**
         * Initializes and returns an 
         *  object created from an 
         *  object.
         */
        constructor (data: Data, mask: XMLNode.Options);
        /**
         *  Returns an 
         *  object initialized with a single child, the root element.
         */
        constructor (element: XMLElement | null);
        /**
         * Initializes and returns an 
         *  object created from a string containing XML markup text.
         */
        constructor (_string: string, mask: XMLNode.Options);
        constructor ();
        /**
         * Overridden by subclasses to substitute a custom class for an NSXML class that the parser uses to create node instances.
         */
        static replacementClass(cls: AnyClass):AnyClass;
        /**
         * Returns the root element of the receiver.
         */
        rootElement():XMLElement | null;
        /**
         * Set the root element of the receiver. 
         */
        setRootElement(root: XMLElement):void;
        /**
         * Adds a child node after the last of the receiver’s existing children.
         */
        addChild(child: XMLNode):void;
        /**
         * Inserts a node object at specified position in the receiver’s array of children.
         */
        insertChild(child: XMLNode, index: number):void;
        /**
         * Inserts an array of children at a specified position in the receiver’s array of children.
         */
        insertChildren(children: XMLNode[], index: number):void;
        /**
         * Removes the child node of the receiver located at a specified position in its array of children.
         */
        removeChild(index: number):void;
        /**
         * Replaces the child node of the receiver located at a specified position in its array of children with another node.
         */
        replaceChild(index: number, node: XMLNode):void;
        /**
         * Sets the child nodes of the receiver.
         */
        setChildren(children: XMLNode[] | null):void;
        /**
         * Applies the XSLT pattern rules and templates (specified as a data object) to the receiver and returns a document object containing transformed XML or HTML markup.
         */
        object(xslt: Data, arguments: Record<string, string> | null):any;
        /**
         * Applies the XSLT pattern rules and templates located at a specified URL to the receiver and returns a document object containing transformed XML markup or an 
         *  object containing plain text, RTF text, and so on.
         */
        objectByApplyingXSLT(xsltURL: URL, argument: Record<string, string> | null):any;
        /**
         * Validates the document against the governing schema and returns whether the document conforms to the schema.
         */
        validate():void;
    }
    /**
     * the same name namespace of class XMLDocument
     */
    export namespace XMLDocument {
        export enum ContentKind {
            html = 2,
            text = 3,
            xhtml = 1,
            xml = 0,
        }
    }
    export class XMLNode extends NSObject {}
    /**
     * the same name namespace of class XMLNode
     */
    export namespace XMLNode {
        export class Options {
            /**
             * Formats HTML into valid XHTML during processing of the document. 
             */
            static documentTidyHTML: XMLNode.Options;
            /**
             * Changes malformed XML into valid XML during processing of the document. 
             */
            static documentTidyXML: XMLNode.Options;
            /**
             * Validates this document against its DTD (internal or external) or XML Schema. 
             */
            static documentValidate: XMLNode.Options;
            /**
             * Replaces all XInclude nodes in the document with the nodes referred to.
             */
            static documentXInclude: XMLNode.Options;
            /**
             * Includes a content type declaration for HTML or XHTML in the output of the document. 
             */
            static documentIncludeContentTypeDeclaration: XMLNode.Options;
        }
    }
    export class XMLElement extends XMLNode {
        /**
         * Sets all attributes of the receiver at once, replacing any existing attribute nodes.
         */
        attributes: XMLNode[] | null;
        /**
         * Sets all of the namespace nodes of the receiver at once, replacing any existing namespace nodes.
         */
        namespaces: XMLNode[] | null;
        /**
         * Returns an 
         *  object initialized with the specified name.
         */
        constructor (name: string);
        /**
         * Returns an 
         *  object initialized with a specified name and a single text-node child containing a specified value.
         */
        constructor (name: string, _string: string);
        /**
         * Returns an 
         *  object created from a specified string containing XML markup. 
         */
        constructor (_string: string);
        /**
         * Returns an 
         *  object initialized with the specified name and URI.
         */
        constructor (name: string, URI: string);
        constructor (kind: XMLNode.Kind, options: XMLNode.Options);
        /**
         * Returns the child element nodes (as 
         *  objects) of the receiver that have a specified name.
         */
        elements(name: string):XMLElement[];
        /**
         * Adds a child node at the end of the receiver’s current list of children.
         */
        addChild(child: XMLNode):void;
        /**
         * Inserts a new child node at a specified location in the receiver’s list of child nodes.
         */
        insertChild(child: XMLNode, index: number):void;
        /**
         * Inserts an array of child nodes at a specified location in the receiver’s list of children.
         */
        insertChildren(children: XMLNode[], index: number):void;
        /**
         * Removes the child node of the receiver identified by a given index.
         */
        removeChild(index: number):void;
        /**
         * Replaces a child node at a specified location with another child node.
         */
        replaceChild(index: number, node: XMLNode):void;
        /**
         * Sets all child nodes of the receiver at once, replacing any existing children.
         */
        setChildren(children: XMLNode[] | null):void;
        /**
         * Coalesces adjacent text nodes of the receiver that you have explicitly added, optionally including CDATA sections.
         */
        normalizeAdjacentTextNodesPreservingCDATA(preserve: boolean):void;
        /**
         * Adds an attribute node to the receiver.
         */
        addAttribute(attribute: XMLNode):void;
        /**
         * Returns the attribute node of the receiver with the specified name.
         */
        attribute(name: string):XMLNode | null;
        /**
         * Removes an attribute node identified by name.
         */
        removeAttribute(name: string):void;
        /**
         * Sets the attributes of the receiver based on the key-value pairs specified in the passed dictionary.
         */
        setAttributesWith(attributes: Record<string, string>):void;
        /**
         * Sets the attributes of the receiver based on the key-value pairs specified in the passed-in dictionary.
         */
        setAttributesAs(attributes: Record<AnyHashable, any>):void;
        /**
         * Adds a namespace node to the receiver.
         */
        addNamespace(aNamespace: XMLNode):void;
        /**
         * Returns the namespace node with a specified prefix.
         */
        namespace(name: string):XMLNode | null;
        /**
         * Removes a namespace node that is identified by a given prefix.
         */
        removeNamespace(name: string):void;
        /**
         * Returns the namespace node with the prefix matching the given qualified name.
         */
        resolveNamespace(name: string):XMLNode | null;
        /**
         * Returns the prefix associated with the specified URI.
         */
        resolvePrefix(namespaceURI: string):string;
    }
    export class XMLParser extends NSObject {
        /**
         * Indicates an error in XML parsing.
         */
        static errorDomain: string;
        /**
         * A delegate object that receives messages about the parsing process.
         */
        delegate: XMLParserDelegate | null;
        /**
         * A Boolean value that determines whether the parser reports the namespaces and qualified names of elements.
         */
        shouldProcessNamespaces: boolean;
        /**
         * A Boolean value that determines whether the parser reports the prefixes indicating the scope of namespace declarations.
         */
        shouldReportNamespacePrefixes: boolean;
        /**
         * A Boolean value that determines whether the parser reports declarations of external entities.
         */
        shouldResolveExternalEntities: boolean;
        /**
         * An 
         *  object from which you can obtain information about a parsing error.
         */
        parserError: Error | null;
        /**
         * The column number of the XML document being processed by the parser.
         */
        columnNumber: number;
        /**
         * The line number of the XML document being processed by the parser.
         */
        lineNumber: number;
        /**
         * The public identifier of the external entity referenced in the XML document.
         */
        publicID: string;
        /**
         * The system identifier of the external entity referenced in the XML document.
         */
        systemID: string;
        allowedExternalEntityURLs: Set<URL> | null;
        externalEntityResolvingPolicy: XMLParser.ExternalEntityResolvingPolicy;
        /**
         * Initializes a parser with the XML content referenced by the given URL.
         */
        constructor (url: URL);
        /**
         * Initializes a parser with the XML contents encapsulated in a given data object. 
         */
        constructor (data: Data);
        /**
         * Initializes a parser with the XML contents from the specified stream and parses it.
         */
        constructor (stream: InputStream);
        /**
         * Starts the event-driven parsing operation.
         */
        parse():boolean;
        /**
         * Stops the parser object.
         */
        abortParsing():void;
    }
    /**
     * the same name namespace of class XMLParser
     */
    export namespace XMLParser {
        export enum ExternalEntityResolvingPolicy {
            always = 3,
            never = 0,
            noNetwork = 1,
            sameOriginOnly = 2,
        }
        export enum ErrorCode {
            internalError = 1,
            outOfMemoryError = 2,
            documentStartError = 3,
            emptyDocumentError = 4,
            prematureDocumentEndError = 5,
            invalidHexCharacterRefError = 6,
            invalidDecimalCharacterRefError = 7,
            invalidCharacterRefError = 8,
            invalidCharacterError = 9,
            characterRefAtEOFError = 10,
            characterRefInPrologError = 11,
            characterRefInEpilogError = 12,
            characterRefInDTDError = 13,
            entityRefAtEOFError = 14,
            entityRefInPrologError = 15,
            entityRefInEpilogError = 16,
            entityRefInDTDError = 17,
            parsedEntityRefAtEOFError = 18,
            parsedEntityRefInPrologError = 19,
            parsedEntityRefInEpilogError = 20,
            parsedEntityRefInInternalSubsetError = 21,
            entityReferenceWithoutNameError = 22,
            entityReferenceMissingSemiError = 23,
            parsedEntityRefNoNameError = 24,
            parsedEntityRefMissingSemiError = 25,
            undeclaredEntityError = 26,
            unparsedEntityError = 28,
            entityIsExternalError = 29,
            entityIsParameterError = 30,
            unknownEncodingError = 31,
            encodingNotSupportedError = 32,
            stringNotStartedError = 33,
            stringNotClosedError = 34,
            namespaceDeclarationError = 35,
            entityNotStartedError = 36,
            entityNotFinishedError = 37,
            lessThanSymbolInAttributeError = 38,
            attributeNotStartedError = 39,
            attributeNotFinishedError = 40,
            attributeHasNoValueError = 41,
            attributeRedefinedError = 42,
            literalNotStartedError = 43,
            literalNotFinishedError = 44,
            commentNotFinishedError = 45,
            processingInstructionNotStartedError = 46,
            processingInstructionNotFinishedError = 47,
            notationNotStartedError = 48,
            notationNotFinishedError = 49,
            attributeListNotStartedError = 50,
            attributeListNotFinishedError = 51,
            mixedContentDeclNotStartedError = 52,
            mixedContentDeclNotFinishedError = 53,
            elementContentDeclNotStartedError = 54,
            elementContentDeclNotFinishedError = 55,
            xmlDeclNotStartedError = 56,
            xmlDeclNotFinishedError = 57,
            conditionalSectionNotStartedError = 58,
            conditionalSectionNotFinishedError = 59,
            externalSubsetNotFinishedError = 60,
            doctypeDeclNotFinishedError = 61,
            misplacedCDATAEndStringError = 62,
            cdataNotFinishedError = 63,
            misplacedXMLDeclarationError = 64,
            spaceRequiredError = 65,
            separatorRequiredError = 66,
            nmtokenRequiredError = 67,
            nameRequiredError = 68,
            pcdataRequiredError = 69,
            uriRequiredError = 70,
            publicIdentifierRequiredError = 71,
            ltRequiredError = 72,
            gtRequiredError = 73,
            ltSlashRequiredError = 74,
            equalExpectedError = 75,
            tagNameMismatchError = 76,
            unfinishedTagError = 77,
            standaloneValueError = 78,
            invalidEncodingNameError = 79,
            commentContainsDoubleHyphenError = 80,
            invalidEncodingError = 81,
            externalStandaloneEntityError = 82,
            invalidConditionalSectionError = 83,
            entityValueRequiredError = 84,
            notWellBalancedError = 85,
            extraContentError = 86,
            invalidCharacterInEntityError = 87,
            parsedEntityRefInInternalError = 88,
            entityRefLoopError = 89,
            entityBoundaryError = 90,
            invalidURIError = 91,
            uriFragmentError = 92,
            noDTDError = 94,
            delegateAbortedParseError = 512,
        }
    }
    export interface XMLParserDelegate extends NSObjectProtocol {
        /**
         * Sent by the parser object to the delegate when it begins parsing a document.
         */
        parserDidStartDocument(parser: XMLParser):void;
        /**
         * Sent by the parser object to the delegate when it has successfully completed parsing.
         */
        parserDidEndDocument(parser: XMLParser):void;
        /**
         * Sent by a parser object to its delegate when it encounters a start tag for a given element.
         */
        parser(parser: XMLParser, elementName: string, namespaceURI: string, qName: string, attributeDict: Record<string, string>):void;
    }
    export class NSKeyedArchiver extends NSCoder {
        /**
         * The encoded data for the archiver.
         */
        encodedData: Data;
        /**
         * The format in which the receiver encodes its data.
         */
        outputFormat: PropertyListSerialization.PropertyListFormat;
        /**
         * Indicates whether the archiver requires all archived classes to resist object substitution attacks.
         */
        requiresSecureCoding: boolean;
        /**
         * The archiver’s delegate.
         */
        delegate: NSKeyedArchiverDelegate | null;
        /**
         * Creates an archiver to encode data, and optionally disables secure coding.
         */
        constructor (requiresSecureCoding: boolean);
        /**
         * Initializes an archiver to encode data.
         */
        constructor ();
        /**
         * Initializes an archiver to encode data into a given a mutable-data object.
         */
        constructor (data: NSMutableData);
        /**
         * Encodes an object graph with the given root object into a data representation, optionally requiring secure coding.
         */
        static archivedData(object: any, requiresSecureCoding: boolean):Data;
        /**
         * Archives an object graph rooted at a given object to a file at a given path.
         */
        static archiveRootObject(rootObject: any, path: string):boolean;
        /**
         * Sets a global translation mapping to encode instances of a given class with the provided name, rather than their real name.
         */
        static setClassName(codedName: string, cls: AnyClass):void;
        /**
         * Returns the class name with which the archiver class encodes instances of a given class.
         */
        static className(cls: AnyClass):string;
        /**
         * Instructs the receiver to construct the final data stream.
         */
        finishEncoding():void;
        /**
         * Encodes a given Boolean value and associates it with a key.
         */
        encode(value: boolean, key: string):void;
        /**
         * Encodes a given number of bytes from a given C array of bytes and associates them with a key.
         */
        encodeBytes(bytes: UnsafePointer<UInt8> | null, length: number, key: string):void;
        /**
         * Encodes a reference to a given object and associates it with a key only if it has been unconditionally encoded elsewhere in the archive.
         */
        encodeConditionalObject(object: any, key: string):void;
    }
    export interface NSKeyedArchiverDelegate extends NSObjectProtocol {
        /**
         * Informs the delegate that a given object has been encoded.
         */
        archiver(archiver: NSKeyedArchiver, object: any):void;
        /**
         * Notifies the delegate that encoding has finished.
         */
        archiverDidFinish(archiver: NSKeyedArchiver):void;
        /**
         * Notifies the delegate that encoding is about to finish.
         */
        archiverWillFinish(archiver: NSKeyedArchiver):void;
    }
    export class NSKeyedUnarchiver extends NSCoder {
        /**
         * Indicates whether the receiver requires all unarchived classes to conform to 
         * .
         */
        requiresSecureCoding: boolean;
        /**
         * The action to take when this unarchiver fails to decode an entry.
         */
        decodingFailurePolicy: NSCoder.DecodingFailurePolicy;
        /**
         * The receiver’s delegate.
         */
        delegate: NSKeyedUnarchiverDelegate | null;
        /**
         * Initializes an archiver to decode data from the specified location.
         */
        constructor (data: Data);
        /**
         * Initializes an archiver to decode data.
         */
        constructor ();
        /**
         * Initializes an archiver to decode data from the specified location.
         */
        constructor (data: Data);
        /**
         * Decodes a previously-archived object graph, and returns the root object.
         */
        static unarchiveTopLevelObjectWithData(data: Data):any;
        /**
         * Decodes a previously-archived object graph, returning the root object as one of the specified classes.
         */
        static unarchivedObject(classes: Set<AnyHashable>, data: Data):any;
        /**
         * Decodes and returns the object graph previously encoded by 
         *  and stored in a given 
         *  object.
         */
        static unarchiveObject(data: Data):any;
        /**
         * Sets a global translation mapping to decode objects encoded with a given class name as instances of a given class instead.
         */
        static setClass(cls: AnyClass | null, codedName: string):void;
        /**
         * Returns the class from which this unarchiver instantiates an encoded object with a given class name.
         */
        static class(codedName: string):AnyClass | null;
        static unarchivedArrayOfObjects(classes: AnyClass[], data: Data):any[] | null;
        static unarchivedDictionary(keyClasses: AnyClass[], objectClasses: AnyClass[], data: Data):Record<AnyHashable, any> | null;
        /**
         * Returns a Boolean value that indicates whether the archive contains a value for a given key within the current decoding scope.
         */
        containsValue(key: string):boolean;
        /**
         * Decodes a Boolean value associated with a given key.
         */
        decodeBool(key: string):boolean;
        /**
         * Decodes a stream of bytes associated with a given key.
         */
        decodeBytes(key: string, lengthp: UnsafeMutablePointer<Int> | null):UnsafePointer<UInt8> | null;
        /**
         * Decodes a double-precision floating-point value associated with a given key.
         */
        decodeDouble(key: string):number;
        /**
         * Decodes a single-precision floating-point value associated with a given key.
         */
        decodeFloat(key: string):Float;
        /**
         * Decodes a 32-bit integer value associated with a given key.
         */
        decodeInt32(key: string):Int32;
        /**
         * Decodes a 64-bit integer value associated with a given key.
         */
        decodeInt64(key: string):Int64;
        /**
         * Decodes and returns an object associated with a given key.
         */
        decodeObject(key: string):any;
        /**
         * Tells the receiver that you are finished decoding objects.
         */
        finishDecoding():void;
    }
    export interface NSKeyedUnarchiverDelegate extends NSObjectProtocol {
        /**
         * Informs the delegate that the class with a given name is not available during decoding.
         */
        unarchiver(unarchiver: NSKeyedUnarchiver, name: string, classNames: string[]):AnyClass | null;
        /**
         * Notifies the delegate that decoding has finished.
         */
        unarchiverDidFinish(unarchiver: NSKeyedUnarchiver):void;
        /**
         * Notifies the delegate that decoding is about to finish.
         */
        unarchiverWillFinish(unarchiver: NSKeyedUnarchiver):void;
    }
    export class NSCoder extends NSObject {
        /**
         * A Boolean value that indicates whether the receiver supports keyed coding of objects.
         */
        allowsKeyedCoding: boolean;
        /**
         * The action the coder should take when decoding fails.
         */
        decodingFailurePolicy: NSCoder.DecodingFailurePolicy;
        /**
         * Indicates whether the archiver requires all archived classes to resist object substitution attacks.
         */
        requiresSecureCoding: boolean;
        /**
         * The set of coded classes allowed for secure coding.
         */
        allowedClasses: Set<AnyHashable> | null;
        /**
         * An error in the top-level encode.
         */
        error: Error | null;
        /**
         * The system version in effect for the archive.
         */
        systemVersion: UInt32;
        /**
         * Returns a Core Graphics affine transform structure corresponding to the data in a given string.
         */
        static cgAffineTransform(_string: string):CGAffineTransform;
        /**
         * Returns a Core Graphics point structure corresponding to the data in a given string.
         */
        static cgPoint(_string: string):CGPoint;
        /**
         * Returns a Core Graphics rectangle structure corresponding to the data in a given string.
         */
        static cgRect(_string: string):CGRect;
        /**
         * Returns a Core Graphics size structure corresponding to the data in a given string.
         */
        static cgSize(_string: string):CGSize;
        /**
         * Returns a Core Graphics vector corresponding to the data in a given string.
         */
        static cgVector(_string: string):CGVector;
        /**
         * Returns a directional edge insets structure based on data in the specified string.
         */
        static nsDirectionalEdgeInsets(_string: string):NSDirectionalEdgeInsets;
        /**
         * Returns a UIKit edge insets structure based on the data in the specified string.
         */
        static uiEdgeInsets(_string: string):UIEdgeInsets;
        /**
         * Returns a UIKit offset structure corresponding to the data in a given string.
         */
        static uiOffset(_string: string):UIOffset;
        /**
         * Returns a string formatted to contain the data from a rectangle.
         */
        static string(rect: CGRect):string;
        /**
         * Returns a Boolean value that indicates whether an encoded value is available for a string.
         */
        containsValue(key: string):boolean;
        /**
         * Encodes an array of the given Objective-C type, provided the number of items and a pointer.
         */
        encodeArray(type: UnsafePointer<CChar>, count: number, array: UnsafeRawPointer):void;
        /**
         * Encodes a Boolean value and associates it with the string 
         * .
         */
        encode(value: boolean, key: string):void;
        /**
         * An encoding method for subclasses to override such that it creates a copy, rather than a proxy, when decoded.
         */
        encodeBycopyObject(anObject: any):void;
        /**
         * An encoding method for subclasses to override such that it creates a proxy, rather than a copy, when decoded.
         */
        encodeByrefObject(anObject: any):void;
        /**
         * Encodes a buffer of data of an unspecified type.
         */
        encodeBytes(byteaddr: UnsafeRawPointer | null, length: number):void;
        /**
         * An encoding method for subclasses to override to conditionally encode an object, preserving common references to it.
         */
        encodeConditionalObject(object: any):void;
        /**
         * Encodes a C integer value and associates it with the string key.
         */
        encodeCInt(value: Int32, key: string):void;
        /**
         * Encodes a property list.
         */
        encodePropertyList(aPropertyList: any):void;
        /**
         * An encoding method for subclasses to override to encode an interconnected group of objects, starting with the provided root object.
         */
        encodeRootObject(rootObject: any):void;
        /**
         * Encodes a value of the given type at the given address.
         */
        encodeValue(type: UnsafePointer<CChar>, addr: UnsafeRawPointer):void;
        /**
         * Decode an object as one of several expected types, failing if the archived type doesn’t match any of the types.
         */
        decodeObject(classes: AnyClass[] | null, key: string):any;
        /**
         * Decodes a previously-encoded object.
         */
        decodeTopLevelObject():any;
        /**
         * Decodes an array of 
         *  items, whose Objective-C type is given by 
         * .
         */
        decodeArray(itemType: UnsafePointer<CChar>, count: number, array: UnsafeMutableRawPointer):void;
        /**
         * Decodes and returns a boolean value that was previously encoded with 
         *  and associated with the string 
         * .
         */
        decodeBool(key: string):boolean;
        /**
         * Decodes a buffer of data that was previously encoded with 
         *  and associated with the string 
         * .
         */
        decodeBytes(key: string, lengthp: UnsafeMutablePointer<Int> | null):UnsafePointer<UInt8> | null;
        /**
         * Decodes and returns an 
         *  object that was previously encoded with 
         * . Subclasses must override this method.
         */
        decodeData():Data | null;
        /**
         * Decodes and returns a double value that was previously encoded with either 
         *  or 
         *  and associated with the string 
         * .
         */
        decodeDouble(key: string):number;
        /**
         * Decodes and returns a float value that was previously encoded with 
         *  or 
         *  and associated with the string 
         * .
         */
        decodeFloat(key: string):Float;
        /**
         * Decodes and returns an int value that was previously encoded with 
         * , 
         * , 
         * , or 
         *  and associated with the string 
         * .
         */
        decodeCInt(key: string):Int32;
        /**
         * Decodes and returns an NSInteger value that was previously encoded with 
         * , 
         * , 
         * , or 
         *  and associated with the string 
         * .
         */
        decodeInteger(key: string):number;
        /**
         * Decodes and returns a 32-bit integer value that was previously encoded with 
         * , 
         * , 
         * , or 
         *  and associated with the string 
         * .
         */
        decodeInt32(key: string):Int32;
        /**
         * Decodes and returns a 64-bit integer value that was previously encoded with 
         * , 
         * , 
         * , or 
         *  and associated with the string 
         * .
         */
        decodeInt64(key: string):Int64;
        /**
         * Decodes and returns an NSPoint structure that was previously encoded with 
         * .
         */
        decodePoint():NSPoint;
        /**
         * Decodes a property list that was previously encoded with 
         * .
         */
        decodePropertyList():any;
        /**
         * Decodes and returns an NSRect structure that was previously encoded with 
         * .
         */
        decodeRect():NSRect;
        /**
         * Decodes and returns an NSSize structure that was previously encoded with 
         * .
         */
        decodeSize():NSSize;
        /**
         * Decodes a single value, whose Objective-C type is given by 
         * .
         */
        decodeValue(type: UnsafePointer<CChar>, data: UnsafeMutableRawPointer):void;
        /**
         * Decodes and returns the Core Graphics affine transform structure associated with the specified key in the coder’s archive. 
         */
        decodeCGAffineTransform(key: string):CGAffineTransform;
        /**
         * Decodes and returns the Core Graphics point structure associated with the specified key in the coder’s archive. 
         */
        decodeCGPoint(key: string):CGPoint;
        /**
         * Decodes and returns the Core Graphics rectangle structure associated with the specified key in the coder’s archive. 
         */
        decodeCGRect(key: string):CGRect;
        /**
         * Decodes and returns the Core Graphics size structure associated with the specified key in the coder’s archive. 
         */
        decodeCGSize(key: string):CGSize;
        /**
         * Decodes and returns the Core Graphics vector data associated with the specified key in the coder’s archive.
         */
        decodeCGVector(key: string):CGVector;
        /**
         * Decodes and returns the UIKit directional edge insets structure associated with the specified key in the coder’s archive.
         */
        decodeDirectionalEdgeInsets(key: string):NSDirectionalEdgeInsets;
        /**
         * Decodes and returns the UIKit edge insets structure associated with the specified key in the coder’s archive.
         */
        decodeUIEdgeInsets(key: string):UIEdgeInsets;
        /**
         * Decodes and returns the UIKit offset structure associated with the specified key in the coder’s archive. 
         */
        decodeUIOffset(key: string):UIOffset;
        /**
         * Returns the Core Media time structure associated with a given key.
         */
        decodeTime(key: string):CMTime;
        /**
         * Returns the Core Media time range structure associated with a given key.
         */
        decodeTimeRange(key: string):CMTimeRange;
        /**
         * Returns the Core Media time mapping structure associated with a given key.
         */
        decodeTimeMapping(key: string):CMTimeMapping;
        /**
         * Signals to this coder that the decode operation has failed.
         */
        failWithError(error: Error):void;
        /**
         * This method is present for historical reasons and is not used with keyed archivers.
         */
        version(className: string):number;
        decodeArrayOfObjects(classes: AnyClass[], key: string):any[] | null;
        decodeDictionary(keyClasses: AnyClass[], classes: AnyClass[], key: string):Record<AnyHashable, any> | null;
    }
    /**
     * the same name namespace of class NSCoder
     */
    export namespace NSCoder {
        export enum DecodingFailurePolicy {
            raiseException = 0,
            setErrorAndReturn = 1,
        }
    }
    export class NSSecureUnarchiveFromDataTransformer extends ValueTransformer {
        /**
         * A list of allowed classes the top-level object in an archive must conform to, for encoding and decoding.
         */
        static allowedTopLevelClasses: AnyClass[];
    }
    export class NSArchiver extends NSCoder {
        /**
         * The receiver's archive data.
         */
        archiverData: NSMutableData;
        /**
         * Returns an archiver, initialized to encode stream and version information into a given mutable data object.
         */
        constructor (mdata: NSMutableData);
        /**
         * Returns a data object containing the encoded form of the object graph whose root object is given.
         */
        static archivedData(rootObject: any):Data;
        /**
         * Creates a temporary instance of 
         *  and archives an object graph by encoding it into a data object and writing the resulting data object to a specified file.
         */
        static archiveRootObject(rootObject: any, path: string):boolean;
        /**
         * Archives a given object along with all the objects to which it is connected.
         */
        encodeRootObject(rootObject: any):void;
        /**
         * Conditionally archives a given object.
         */
        encodeConditionalObject(object: any):void;
        /**
         * Returns the name of the class used to archive instances of the class with a given true name.
         */
        classNameEncoded(trueName: string):string;
        /**
         * Encodes a substitute name for the class with a given true name.
         */
        encodeClassName(trueName: string, inArchiveName: string):void;
        /**
         * Causes the receiver to treat subsequent requests to encode a given object as though they were requests to encode another given object.
         */
        replace(object: any, newObject: any):void;
    }
    export class NSUnarchiver extends NSCoder {
        /**
         * A Boolean value that indicates whether the receiver has reached the end of the encoded data while decoding.
         */
        isAtEnd: boolean;
        /**
         * The system version number in effect when the archive was created.
         */
        systemVersion: UInt32;
        /**
         * Returns an 
         *  object initialized to read an archive from a given data object. 
         */
        constructor (data: Data);
        /**
         * Decodes and returns the object archived in a given 
         *  object.
         */
        static unarchiveObject(data: Data):any;
        /**
         * Returns the name of the class used when instantiating objects whose ostensible class, according to the archived data, is a given name.
         */
        static classNameDecoded(inArchiveName: string):string;
        /**
         * Instructs instances of 
         *  to use the class with a given name when instantiating objects whose ostensible class, according to the archived data, is another given name.
         */
        static decodeClassName(inArchiveName: string, trueName: string):void;
        /**
         * Causes the receiver to substitute one given object for another whenever the latter is extracted from the archive.
         */
        replace(object: any, newObject: any):void;
    }
    export class UserDefaults extends NSObject {
        /**
         * Returns the shared defaults object.
         */
        static standard: UserDefaults;
        /**
         * The domain consisting of defaults parsed from the application’s arguments. These are one or more pairs of the form
         *  included in the command-line invocation of the application. 
         */
        static argumentDomain: string;
        /**
         * The domain consisting of defaults meant to be seen by all applications.
         */
        static globalDomain: string;
        /**
         * The domain consisting of a set of temporary defaults whose values can be set by the application to ensure that searches will always be successful.
         */
        static registrationDomain: string;
        /**
         * Posted when user defaults are changed within the current process.
         */
        static didChangeNotification: NSNotification.Name;
        /**
         * Posted when more data is stored in user defaults than is allowed. 
         */
        static sizeLimitExceededNotification: NSNotification.Name;
        /**
         * Posted when ubiquitous defaults finish downloading data, either the first time a device is connected to an iCloud account or when a user switches their primary iCloud account.
         */
        static completedInitialCloudSyncNotification: NSNotification.Name;
        /**
         * Posted when the user changes the primary iCloud account.
         */
        static didChangeCloudAccountsNotification: NSNotification.Name;
        /**
         * Posted when a cloud default is set, but no iCloud user is logged in.
         */
        static noCloudAccountNotification: NSNotification.Name;
        /**
         * The current volatile domain names.
         */
        volatileDomainNames: string[];
        /**
         * Creates a user defaults object initialized with the defaults for the app and current user.
         */
        constructor ();
        /**
         * Creates a user defaults object initialized with the defaults for the specified database name.
         */
        constructor (suitename: string);
        /**
         * Creates a user defaults object initialized with the defaults for the specified user account.
         */
        constructor (username: string);
        /**
         * This method has no effect and shouldn't be used.
         */
        static resetStandardUserDefaults():void;
        /**
         * Returns the object associated with the specified key.
         */
        object(defaultName: string):any;
        /**
         * Returns the URL associated with the specified key.
         */
        url(defaultName: string):URL | null;
        /**
         * Returns the array associated with the specified key.
         */
        array(defaultName: string):any[] | null;
        /**
         * Returns the dictionary object associated with the specified key.
         */
        dictionary(defaultName: string):Record<string, any> | null;
        /**
         * Returns the string associated with the specified key.
         */
        string(defaultName: string):string;
        /**
         * Returns the array of strings associated with the specified key.
         */
        stringArray(defaultName: string):string[] | null;
        /**
         * Returns the data object associated with the specified key.
         */
        data(defaultName: string):Data | null;
        /**
         * Returns the Boolean value associated with the specified key.
         */
        bool(defaultName: string):boolean;
        /**
         * Returns the integer value associated with the specified key.
         */
        integer(defaultName: string):number;
        /**
         * Returns the float value associated with the specified key.
         */
        float(defaultName: string):Float;
        /**
         * Returns the double value associated with the specified key.
         */
        double(defaultName: string):number;
        /**
         * Returns a dictionary that contains a union of all key-value pairs in the domains in the search list.
         */
        dictionaryRepresentation():Record<string, any>;
        /**
         * Sets the value of the specified default key.
         */
        set(value: any, defaultName: string):void;
        /**
         * Removes the value of the specified default key.
         */
        removeObject(defaultName: string):void;
        /**
         * Inserts the specified domain name into the receiver’s search list.
         */
        addSuite(suiteName: string):void;
        /**
         * Removes the specified domain name from the receiver’s search list.
         */
        removeSuite(suiteName: string):void;
        /**
         * Adds the contents of the specified dictionary to the registration domain.
         */
        register(registrationDictionary: Record<string, any>):void;
        /**
         * Returns a dictionary representation of the defaults for the specified domain.
         */
        persistentDomain(domainName: string):Record<string, any> | null;
        /**
         * Sets a dictionary for the specified persistent domain.  
         */
        setPersistentDomain(domain: Record<string, any>, domainName: string):void;
        /**
         * Removes the contents of the specified persistent domain from the user’s defaults.
         */
        removePersistentDomain(domainName: string):void;
        /**
         * Returns an array of the current persistent domain names.
         */
        persistentDomainNames():any[];
        /**
         * Returns the dictionary for the specified volatile domain.  
         */
        volatileDomain(domainName: string):Record<string, any>;
        /**
         * Sets the dictionary for the specified volatile domain.  
         */
        setVolatileDomain(domain: Record<string, any>, domainName: string):void;
        /**
         * Removes the specified volatile domain from the user’s defaults.
         */
        removeVolatileDomain(domainName: string):void;
        /**
         * Returns a Boolean value indicating whether the specified key is managed by an administrator.
         */
        objectIsForced(key: string):boolean;
        /**
         * Waits for any pending asynchronous updates to the defaults database and returns; this method is unnecessary and shouldn't be used.
         */
        synchronize():boolean;
    }
    export class NSUbiquitousKeyValueStore extends NSObject {
        /**
         * Returns the shared iCloud key-value store object.
         */
        static default: NSUbiquitousKeyValueStore;
        /**
         * Posted when the value of one or more keys in the local key-value store changed due to incoming data pushed from iCloud.
         */
        static didChangeExternallyNotification: NSNotification.Name;
        /**
         * A dictionary containing all of the key-value pairs in the key-value store.
         */
        dictionaryRepresentation: Record<string, any>;
        /**
         * Returns the array associated with the specified key.
         */
        array(aKey: string):any[] | null;
        /**
         * Returns the Boolean value associated with the specified key.
         */
        bool(aKey: string):boolean;
        /**
         * Returns the data object associated with the specified key.
         */
        data(aKey: string):Data | null;
        /**
         * Returns the dictionary object associated with the specified key.
         */
        dictionary(aKey: string):Record<string, any> | null;
        /**
         * Returns the double value associated with the specified key.
         */
        double(aKey: string):number;
        /**
         * Returns the 
         *  value associated with the specified key.
         */
        longLong(aKey: string):Int64;
        /**
         * Returns the object associated with the specified key.
         */
        object(aKey: string):any;
        /**
         * Returns the string associated with the specified key.
         */
        string(aKey: string):string;
        /**
         * Sets an array object for the specified key in the key-value store. 
         */
        set(anArray: any[] | null, aKey: string):void;
        /**
         * Explicitly synchronizes in-memory keys and values with those stored on disk.
         */
        synchronize():boolean;
        /**
         * Removes the value associated with the specified key from the key-value store.
         */
        removeObject(aKey: string):void;
    }
    export class NSMetadataQuery extends NSObject {
        /**
         * An array containing the search scopes.
         */
        searchScopes: any[];
        /**
         * The predicate used to filter query results.
         */
        predicate: NSPredicate | null;
        /**
         * An array of sort descriptor objects.
         */
        sortDescriptors: NSSortDescriptor[];
        /**
         * An array of attributes whose values are gathered by the query.
         */
        valueListAttributes: string[];
        /**
         * An array of grouping attributes. (read-only)
         */
        groupingAttributes: string[] | null;
        /**
         * The interval at which notification of updated results occurs.
         */
        notificationBatchingInterval: TimeInterval;
        /**
         * The query’s delegate.
         */
        delegate: NSMetadataQueryDelegate | null;
        /**
         * An array of objects that define the query’s scope.
         */
        searchItems: any[] | null;
        /**
         * A Boolean value that indicates whether the query has started. (read-only)
         */
        isStarted: boolean;
        /**
         * A Boolean value that indicates whether the receiver is in the initial gathering phase of the query. (read-only)
         */
        isGathering: boolean;
        /**
         * A Boolean value that indicates whether the query has stopped.
         */
        isStopped: boolean;
        /**
         * An array containing the query’s results.
         */
        results: any[];
        /**
         * The number of results returned by the query. (read-only)
         */
        resultCount: number;
        /**
         * An array containing hierarchical groups of query results. (read-only)
         */
        groupedResults: NSMetadataQueryResultGroup[];
        /**
         * A dictionary containing the value lists generated by the query.
         */
        valueLists: Record<string, NSMetadataQueryAttributeValueTuple[]>;
        /**
         * The queue on which query result notifications are posted.
         */
        operationQueue: OperationQueue | null;
        /**
         * Attempts to start the query.
         */
        start():boolean;
        /**
         * Stops the receiver’s current query from gathering any further results.
         */
        stop():void;
        /**
         * Returns the query result at a specific index.
         */
        result(idx: number):any;
        /**
         * Returns the index of a query result object in the receiver’s results array.
         */
        index(result: any):number;
        /**
         * Enumerates the current set of results using the given block.
         */
        enumerateResults(block: ()=>any):void;
        /**
         * Returns the value for the attribute name 
         *  at the index in the results specified by 
         * .
         */
        value(attrName: string, idx: number):any;
        /**
         * Enables updates to the query results.
         */
        enableUpdates():void;
        /**
         * Disables updates to the query results.
         */
        disableUpdates():void;
    }
    export class NSMetadataQueryResultGroup extends NSObject {
        /**
         * The result group’s attribute name.
         */
        attribute: string;
        /**
         * The result group’s value.
         */
        value: any;
        /**
         * An array containing the result group’s result objects.
         */
        results: any[];
        /**
         * The number of results returned by the result group.
         */
        resultCount: number;
        /**
         * An array containing the result group’s subgroups.
         */
        subgroups: NSMetadataQueryResultGroup[] | null;
        /**
         * Returns the query result at a specific index.
         */
        result(idx: number):any;
    }
    export class NSMetadataQueryAttributeValueTuple extends NSObject {
        /**
         * The attribute name for the tuple’s value.
         */
        attribute: string;
        /**
         * The number of instances of the value for the tuple’s attribute.
         */
        count: number;
        /**
         * The value of the tuple’s attribute.
         */
        value: any;
    }
    export class StringTransform {
        /**
         * A constant containing the transliteration of a string from any script to Latin script.
         */
        static toLatin: StringTransform;
        /**
         * A constant containing the transliteration of a string from Latin script to Arabic script.
         */
        static latinToArabic: StringTransform;
        /**
         * A constant containing the transliteration of a string from Latin script to Cyrillic script.
         */
        static latinToCyrillic: StringTransform;
        /**
         * A constant containing the transliteration of a string from Latin script to Greek script. 
         */
        static latinToGreek: StringTransform;
        /**
         * A constant containing the transliteration of a string from Latin script to Hangul script.
         */
        static latinToHangul: StringTransform;
        /**
         * A constant containing the transliteration of a string from Latin script to Hebrew script.
         */
        static latinToHebrew: StringTransform;
        /**
         * A constant containing the transliteration of a string from Latin script to Hiragana script.
         */
        static latinToHiragana: StringTransform;
        /**
         * A constant containing the transliteration of a string from Latin script to Katakana script.
         */
        static latinToKatakana: StringTransform;
        /**
         * A constant containing the transliteration of a string from Latin script to Thai script.
         */
        static latinToThai: StringTransform;
        /**
         * A constant containing the transliteration of a string from Hiragana script to Katakana script.
         */
        static hiraganaToKatakana: StringTransform;
        /**
         * A constant containing the transliteration of a string from Han script to Latin.
         */
        static mandarinToLatin: StringTransform;
        /**
         * A constant containing the transformation of a string by removing diacritics.
         */
        static stripDiacritics: StringTransform;
        /**
         * A constant containing the transformation of a string by removing combining marks.
         */
        static stripCombiningMarks: StringTransform;
        /**
         * A constant containing the transformation of a string from full-width CJK characters to half-width forms.
         */
        static fullwidthToHalfwidth: StringTransform;
        /**
         * An identifier for a transform that converts characters to Unicode names.
         */
        static toUnicodeName: StringTransform;
        /**
         * A constant containing the transformation of a string from characters to XML hexadecimal escape codes.
         */
        static toXMLHex: StringTransform;
        constructor (rawValue: string);
        constructor (rawValue: string);
    }
    export interface NSMetadataQueryDelegate extends NSObjectProtocol {
        /**
         * Returns a different object for a given query result object.
         */
        metadataQuery(query: NSMetadataQuery, result: NSMetadataItem):any;
    }
    export class NSMetadataItem extends NSObject {
        /**
         * An array containing the attribute keys for the metadata item’s values.
         */
        attributes: string[];
        /**
         * Initializes a metadata item with a given URL.
         */
        constructor (url: URL);
        /**
         * Returns the receiver’s metadata attribute name specified by a given key.
         */
        value(key: string):any;
        /**
         * Returns a dictionary containing the key-value pairs for the attribute names specified by a given array of keys.
         */
        values(keys: string[]):Record<string, any> | null;
    }
    export class URLSession extends NSObject {
        /**
         * The shared singleton session object.
         */
        static shared: URLSession;
        /**
         * A copy of the configuration object for this session.
         */
        configuration: URLSessionConfiguration;
        /**
         * The delegate assigned when this object was created.
         */
        delegate: URLSessionDelegate | null;
        /**
         * The operation queue provided when this object was created.
         */
        delegateQueue: OperationQueue;
        /**
         * An app-defined descriptive label for the session.
         */
        sessionDescription: string;
        /**
         * Creates a session with the specified session configuration.
         */
        constructor (configuration: URLSessionConfiguration);
        /**
         * Creates a session with the specified session configuration, delegate, and operation queue.
         */
        constructor (configuration: URLSessionConfiguration, delegate: URLSessionDelegate | null, queue: OperationQueue | null);
        constructor ();
        static new():Self;
        /**
         * Retrieves the contents of a URL based on the specified URL request and delivers an asynchronous sequence of bytes.
         */
        bytes(request: URLRequest, delegate: URLSessionTaskDelegate | null):any;
        /**
         * Downloads the contents of a URL based on the specified URL request and delivers the data asynchronously.
         */
        data(request: URLRequest, delegate: URLSessionTaskDelegate | null):any;
        /**
         * Retrieves the contents of a URL based on the specified URL request and delivers the URL of the saved file asynchronously.
         */
        download(request: URLRequest, delegate: URLSessionTaskDelegate | null):any;
        /**
         * Uploads data to a URL based on the specified URL request and delivers the result asynchronously.
         */
        upload(request: URLRequest, bodyData: Data, delegate: URLSessionTaskDelegate | null):any;
        /**
         * Creates a task that retrieves the contents of the specified URL.
         */
        dataTask(url: URL):URLSessionDataTask;
        /**
         * Creates a download task that retrieves the contents of the specified URL and saves the results to a file.
         */
        downloadTask(url: URL):URLSessionDownloadTask;
        /**
         * Creates a task that performs an HTTP request for the specified URL request object and uploads the provided data.
         */
        uploadTask(request: URLRequest, bodyData: Data):URLSessionUploadTask;
        /**
         * Creates a task that establishes a bidirectional TCP/IP connection to a specified hostname and port.
         */
        streamTask(hostname: string, port: number):URLSessionStreamTask;
        /**
         * Creates a WebSocket task for the provided URL.
         */
        webSocketTask(url: URL):URLSessionWebSocketTask;
        /**
         * Invalidates the session, allowing any outstanding tasks to finish.
         */
        finishTasksAndInvalidate():void;
        /**
         * Flushes cookies and credentials to disk, clears transient caches, and ensures that future requests occur on a new TCP connection.
         */
        flush(completionHandler: ()=>any):void;
        /**
         * Asynchronously calls a completion callback with all data, upload, and download tasks in a session.
         */
        getTasksWithCompletionHandler(completionHandler: ()=>any):void;
        /**
         * Asynchronously calls a completion callback with all tasks in a session
         */
        getAllTasks(completionHandler: ()=>any):void;
        /**
         * Cancels all outstanding tasks and then invalidates the session.
         */
        invalidateAndCancel():void;
        /**
         * Empties all cookies, caches and credential stores, removes disk files, flushes in-progress downloads to disk, and ensures that future requests occur on a new socket.
         */
        reset(completionHandler: ()=>any):void;
        /**
         * Returns a publisher that wraps a URL session data task for a given URL request.
         */
        dataTaskPublisher(request: URLRequest):URLSession.DataTaskPublisher;
    }
    /**
     * the same name namespace of class URLSession
     */
    export namespace URLSession {
        export enum AuthChallengeDisposition {
            useCredential = 0,
            performDefaultHandling = 1,
            cancelAuthenticationChallenge = 2,
            rejectProtectionSpace = 3,
        }
        export enum DelayedRequestDisposition {
            cancel = 2,
            continueLoading = 0,
            useNewRequest = 1,
        }
        export class AsyncBytes {
            /**
             * The bytes loaded by the session, as an asychronous sequence of characters.
             */
            characters: AsyncCharacterSequence<URLSession.AsyncBytes>;
            /**
             * The bytes loaded by the session, as an asychronous sequence of Unicode scalar values.
             */
            unicodeScalars: AsyncUnicodeScalarSequence<URLSession.AsyncBytes>;
            /**
             * The bytes loaded by the session, as an asychronous sequence of lines of text.
             */
            lines: AsyncLineSequence<URLSession.AsyncBytes>;
            /**
             * The URL session task that performs the data transfer.
             */
            task: URLSessionDataTask;
            /**
             * Returns a Boolean value that indicates whether the asynchronous sequence contains the given element.
             */
            contains(search: UInt8):boolean;
            /**
             * Returns a Boolean value that indicates whether all elements produced by the asynchronous sequence satisfy the given predicate.
             */
            allSatisfy(predicate: Bool):boolean;
            /**
             * Returns the first element of the sequence that satisfies the given predicate.
             */
            first(predicate: Bool):UInt8 | null;
            /**
             * Returns the minimum element in an asynchronous sequence of comparable elements.
             */
            min():UInt8 | null;
            /**
             * Returns the maximum element in an asynchronous sequence of comparable elements.
             */
            max():UInt8 | null;
            /**
             * Returns an asynchronous sequence, up to the specified maximum length, containing the initial elements of the base asynchronous sequence.
             */
            prefix(count: number):AsyncPrefixSequence<URLSession.AsyncBytes>;
            /**
             * Omits a specified number of elements from the base asynchronous sequence, then passes through all remaining elements.
             */
            dropFirst(count: number):AsyncDropFirstSequence<URLSession.AsyncBytes>;
            /**
             * Omits elements from the base asynchronous sequence until a given closure returns false, after which it passes through all remaining elements.
             */
            drop(predicate: Bool):AsyncDropWhileSequence<URLSession.AsyncBytes>;
            /**
             * Creates an asynchronous sequence that contains, in order, the elements of the base sequence that satisfy the given predicate.
             */
            filter(isIncluded: Bool):AsyncFilterSequence<URLSession.AsyncBytes>;
            makeAsyncIterator():URLSession.AsyncBytes.Iterator;
        }
        /**
         * the same name namespace of class AsyncBytes
         */
        export namespace AsyncBytes {
            export class Iterator {
                next():UInt8 | null;
            }
        }
        export enum ResponseDisposition {
            cancel = 0,
            allow = 1,
            becomeDownload = 2,
            becomeStream = 3,
        }
        export class DataTaskPublisher {
            /**
             * The URL request performed by the data task associated with this publisher.
             */
            request: URLRequest;
            /**
             * The URL session that performs the data task associated with this publisher.
             */
            session: URLSession;
            values: AsyncThrowingPublisher<URLSession.DataTaskPublisher>;
            /**
             * Creates a data task publisher from the provided URL request and URL session.
             */
            constructor (request: URLRequest, session: URLSession);
            allSatisfy(predicate: ()=>any):Publishers.AllSatisfy<URLSession.DataTaskPublisher>;
            append(...elements: any):any;
            assertNoFailure(prefix: string, file: StaticString, line: number):Publishers.AssertNoFailure<URLSession.DataTaskPublisher>;
            breakpoint(receiveSubscription: ()=>any | null, receiveOutput: ()=>any | null, receiveCompletion: ()=>any | null):Publishers.Breakpoint<URLSession.DataTaskPublisher>;
            breakpointOnError():Publishers.Breakpoint<URLSession.DataTaskPublisher>;
            buffer(size: number, prefetch: Publishers.PrefetchStrategy, whenFull: Publishers.BufferingStrategy<URLError>):Publishers.Buffer<URLSession.DataTaskPublisher>;
            collect():Publishers.Collect<URLSession.DataTaskPublisher>;
            contains(predicate: ()=>any):Publishers.ContainsWhere<URLSession.DataTaskPublisher>;
            count():Publishers.Count<URLSession.DataTaskPublisher>;
            drop(predicate: ()=>any):Publishers.DropWhile<URLSession.DataTaskPublisher>;
            dropFirst(count: number):Publishers.Drop<URLSession.DataTaskPublisher>;
            eraseToAnyPublisher():any;
            filter(isIncluded: ()=>any):Publishers.Filter<URLSession.DataTaskPublisher>;
            first():Publishers.First<URLSession.DataTaskPublisher>;
            handleEvents(receiveSubscription: ()=>any | null, receiveOutput: ()=>any | null, receiveCompletion: ()=>any | null, receiveCancel: ()=>any | null, receiveRequest: ()=>any | null):Publishers.HandleEvents<URLSession.DataTaskPublisher>;
            ignoreOutput():Publishers.IgnoreOutput<URLSession.DataTaskPublisher>;
            last():Publishers.Last<URLSession.DataTaskPublisher>;
            max(areInIncreasingOrder: ()=>any):Publishers.Comparison<URLSession.DataTaskPublisher>;
            merge(other: URLSession.DataTaskPublisher):Publishers.MergeMany<URLSession.DataTaskPublisher>;
            min(areInIncreasingOrder: ()=>any):Publishers.Comparison<URLSession.DataTaskPublisher>;
            output(index: number):Publishers.Output<URLSession.DataTaskPublisher>;
            prefix(maxLength: number):Publishers.Output<URLSession.DataTaskPublisher>;
            prepend(...elements: any):any;
            print(prefix: string, stream: TextOutputStream | null):Publishers.Print<URLSession.DataTaskPublisher>;
            removeDuplicates(predicate: ()=>any):Publishers.RemoveDuplicates<URLSession.DataTaskPublisher>;
            replaceEmpty(output: any):Publishers.ReplaceEmpty<URLSession.DataTaskPublisher>;
            replaceError(output: any):Publishers.ReplaceError<URLSession.DataTaskPublisher>;
            retry(retries: number):Publishers.Retry<URLSession.DataTaskPublisher>;
            share():Publishers.Share<URLSession.DataTaskPublisher>;
            sink(receiveCompletion: ()=>any, receiveValue: ()=>any):AnyCancellable;
            tryAllSatisfy(predicate: Bool):Publishers.TryAllSatisfy<URLSession.DataTaskPublisher>;
            tryContains(predicate: Bool):Publishers.TryContainsWhere<URLSession.DataTaskPublisher>;
            tryDrop(predicate: Bool):Publishers.TryDropWhile<URLSession.DataTaskPublisher>;
            tryFilter(isIncluded: Bool):Publishers.TryFilter<URLSession.DataTaskPublisher>;
            tryFirst(predicate: Bool):Publishers.TryFirstWhere<URLSession.DataTaskPublisher>;
            tryLast(predicate: Bool):Publishers.TryLastWhere<URLSession.DataTaskPublisher>;
            tryMax(areInIncreasingOrder: Bool):Publishers.TryComparison<URLSession.DataTaskPublisher>;
            tryMin(areInIncreasingOrder: Bool):Publishers.TryComparison<URLSession.DataTaskPublisher>;
            tryPrefix(predicate: Bool):Publishers.TryPrefixWhile<URLSession.DataTaskPublisher>;
            tryRemoveDuplicates(predicate: Bool):Publishers.TryRemoveDuplicates<URLSession.DataTaskPublisher>;
        }
    }
    export class URLSessionConfiguration extends NSObject {
        /**
         * A default session configuration object.
         */
        static default: URLSessionConfiguration;
        /**
         * A session configuration that uses no persistent storage for caches, cookies, or credentials.
         */
        static ephemeral: URLSessionConfiguration;
        /**
         * The background session identifier of the configuration object.
         */
        identifier: string;
        /**
         * A dictionary of additional headers to send with requests.
         */
        httpAdditionalHeaders: Record<AnyHashable, any> | null;
        /**
         * The type of network service for all tasks within sessions based on this configuration.
         */
        networkServiceType: NSURLRequest.NetworkServiceType;
        /**
         * A Boolean value that determines whether connections should be made over a cellular network.
         */
        allowsCellularAccess: boolean;
        /**
         * The timeout interval to use when waiting for additional data.
         */
        timeoutIntervalForRequest: TimeInterval;
        /**
         * The maximum amount of time that a resource request should be allowed to take.
         */
        timeoutIntervalForResource: TimeInterval;
        /**
         * The identifier for the shared container into which files in background URL sessions should be downloaded.
         */
        sharedContainerIdentifier: string;
        /**
         * A Boolean value that indicates whether the session should wait for connectivity to become available, or fail immediately.
         */
        waitsForConnectivity: boolean;
        /**
         * A policy constant that determines when cookies should be accepted.
         */
        httpCookieAcceptPolicy: HTTPCookie.AcceptPolicy;
        /**
         * A Boolean value that determines whether requests should contain cookies from the cookie store.
         */
        httpShouldSetCookies: boolean;
        /**
         * The cookie store for storing cookies within this session.
         */
        httpCookieStorage: HTTPCookieStorage | null;
        /**
         * The minimum TLS protocol version that the client should accept when making connections in this session.
         */
        tlsMinimumSupportedProtocolVersion: tls_protocol_version_t;
        /**
         * The maximum TLS protocol version that the client should request when making connections in this session.
         */
        tlsMaximumSupportedProtocolVersion: tls_protocol_version_t;
        /**
         * A credential store that provides credentials for authentication.
         */
        urlCredentialStorage: URLCredentialStorage | null;
        /**
         * The minimum TLS protocol to accept during protocol negotiation.
         */
        tlsMinimumSupportedProtocol: SSLProtocol;
        /**
         * The maximum TLS protocol version that the client should request when making connections in this session.
         */
        tlsMaximumSupportedProtocol: SSLProtocol;
        /**
         * The URL cache for providing cached responses to requests within the session.
         */
        urlCache: URLCache | null;
        /**
         * A predefined constant that determines when to return a response from the cache.
         */
        requestCachePolicy: NSURLRequest.CachePolicy;
        /**
         * A Boolean value that indicates whether the app should be resumed or launched in the background when transfers finish.
         */
        sessionSendsLaunchEvents: boolean;
        /**
         * A Boolean value that determines whether background tasks can be scheduled at the discretion of the system for optimal performance.
         */
        isDiscretionary: boolean;
        /**
         * A Boolean value that indicates whether TCP connections should be kept open when the app moves to the background.
         */
        shouldUseExtendedBackgroundIdleMode: boolean;
        /**
         * An array of extra protocol subclasses that handle requests in a session.
         */
        protocolClasses: AnyClass[] | null;
        /**
         * A service type that specifies the Multipath TCP connection policy for transmitting data over Wi-Fi and cellular interfaces.
         */
        multipathServiceType: URLSessionConfiguration.MultipathServiceType;
        /**
         * The maximum number of simultaneous connections to make to a given host.
         */
        httpMaximumConnectionsPerHost: number;
        /**
         * A Boolean value that determines whether the session should use HTTP pipelining.
         */
        httpShouldUsePipelining: boolean;
        /**
         * A dictionary containing information about the proxy to use within this session.
         */
        connectionProxyDictionary: Record<AnyHashable, any> | null;
        /**
         * A Boolean value that indicates whether connections may use the network when the user has specified Low Data Mode.
         */
        allowsConstrainedNetworkAccess: boolean;
        /**
         * A Boolean value that indicates whether connections may use a network interface that the system considers expensive.
         */
        allowsExpensiveNetworkAccess: boolean;
        requiresDNSSECValidation: boolean;
        /**
         * Creates an empty session configuration.
         */
        constructor ();
        /**
         * Creates a session configuration object that allows HTTP and HTTPS uploads or downloads to be performed in the background.
         */
        static background(identifier: string):URLSessionConfiguration;
        /**
         * Creates an empty session configuration.
         */
        static new():Self;
        /**
         * Returns a session configuration object that allows HTTP and HTTPS uploads or downloads to be performed in the background.
         */
        static backgroundSessionConfiguration(identifier: string):URLSessionConfiguration;
    }
    /**
     * the same name namespace of class URLSessionConfiguration
     */
    export namespace URLSessionConfiguration {
        export enum MultipathServiceType {
            none = 0,
            handover = 1,
            interactive = 2,
            aggregate = 3,
        }
    }
    export class HTTPCookieStorage extends NSObject {
        /**
         * The shared cookie storage instance.
         */
        static shared: HTTPCookieStorage;
        /**
         * The cookie storage’s cookie accept policy.
         */
        cookieAcceptPolicy: HTTPCookie.AcceptPolicy;
        /**
         * The cookie storage’s cookies.
         */
        cookies: HTTPCookie[] | null;
        /**
         * Returns the cookie storage instance for the container associated with the specified app group identifier.
         */
        static sharedCookieStorage(identifier: string):HTTPCookieStorage;
        /**
         * Removes cookies that were stored after a given date.
         */
        removeCookies(date: Date):void;
        /**
         * Deletes the specified cookie from the cookie storage.
         */
        deleteCookie(cookie: HTTPCookie):void;
        /**
         * Stores a specified cookie in the cookie storage if the cookie accept policy permits.
         */
        setCookie(cookie: HTTPCookie):void;
        /**
         * Adds an array of cookies to the cookie storage if the storage’s cookie acceptance policy permits.
         */
        setCookies(cookies: HTTPCookie[], URL: URL | null, mainDocumentURL: URL | null):void;
        /**
         * Stores an array of cookies in the cookie storage, on behalf of the provided task, if the cookie accept policy permits.
         */
        storeCookies(cookies: HTTPCookie[], task: URLSessionTask):void;
        /**
         * Fetches cookies relevant to the specified task and passes them to the completion handler.
         */
        getCookiesFor(task: URLSessionTask, completionHandler: ()=>any):void;
        /**
         * Returns all of the cookie storage’s cookies, sorted according to a given set of sort descriptors.
         */
        sortedCookies(sortOrder: NSSortDescriptor[]):HTTPCookie[];
    }
    export class HTTPCookie extends NSObject {}
    /**
     * the same name namespace of class HTTPCookie
     */
    export namespace HTTPCookie {
        export enum AcceptPolicy {
            always = 0,
            never = 1,
            onlyFromMainDocumentDomain = 2,
        }
    }
    export class HTTPCookieStringPolicy {
        /**
         * A policy that prohibits a cross-site request from including the cookie.
         */
        static sameSiteStrict: HTTPCookieStringPolicy;
        /**
         * A policy that allows certain cross-site requests to include the cookie.
         */
        static sameSiteLax: HTTPCookieStringPolicy;
        /**
         * Creates an HTTP cookie string policy from the given raw string.
         */
        constructor (rawValue: string);
    }
    export class HTTPCookiePropertyKey {
        /**
         * An 
         *  object containing the comment for the cookie.
         */
        static comment: HTTPCookiePropertyKey;
        /**
         * An 
         *  object or 
         *  object containing the comment URL for the cookie.
         */
        static commentURL: HTTPCookiePropertyKey;
        /**
         * An 
         *  object stating whether the cookie should be discarded at the end of the session. 
         */
        static discard: HTTPCookiePropertyKey;
        /**
         * An 
         *  object containing the domain for the cookie. 
         */
        static domain: HTTPCookiePropertyKey;
        /**
         * An 
         *  object or 
         *  object specifying the expiration date for the cookie. 
         */
        static expires: HTTPCookiePropertyKey;
        /**
         * An 
         *  object containing an integer value stating how long in seconds the cookie should be kept, at most.
         */
        static maximumAge: HTTPCookiePropertyKey;
        /**
         * An 
         *  object containing the name of the cookie (required).
         */
        static name: HTTPCookiePropertyKey;
        /**
         * An NSURL or 
         *  object containing the URL that set this cookie. 
         */
        static originURL: HTTPCookiePropertyKey;
        /**
         * An 
         *  object containing the path for the cookie.
         */
        static path: HTTPCookiePropertyKey;
        /**
         * An 
         *  object containing comma-separated integer values specifying the ports for the cookie.
         */
        static port: HTTPCookiePropertyKey;
        /**
         * A string indicating the same-site policy for the cookie.
         */
        static sameSitePolicy: HTTPCookiePropertyKey;
        /**
         * An 
         *  object indicating that the cookie should be transmitted only over secure channels.
         */
        static secure: HTTPCookiePropertyKey;
        /**
         * An 
         *  object containing the value of the cookie. 
         */
        static value: HTTPCookiePropertyKey;
        /**
         * An 
         *  object that specifies the version of the cookie.
         */
        static version: HTTPCookiePropertyKey;
        /**
         * Creates an HTTP cookie property key using the given string.
         */
        constructor (rawValue: string);
        /**
         * Creates an HTTP cookie property key using the given string.
         */
        constructor (rawValue: string);
    }
    export class URLProtocol extends NSObject {
        /**
         * The protocol’s cached response.
         */
        cachedResponse: CachedURLResponse | null;
        /**
         * The object the protocol uses to communicate with the URL loading system.
         */
        client: URLProtocolClient | null;
        /**
         * The protocol’s request.
         */
        request: URLRequest;
        /**
         * The protocol’s task.
         */
        task: URLSessionTask | null;
        /**
         * Creates a URL protocol instance to handle the request.
         */
        constructor (request: URLRequest, cachedResponse: CachedURLResponse | null, client: URLProtocolClient | null);
        /**
         * Creates a URL protocol instance to handle the task.
         */
        constructor (task: URLSessionTask, cachedResponse: CachedURLResponse | null, client: URLProtocolClient | null);
        /**
         * Attempts to register a subclass of 
         * , making it visible to the URL loading system.
         */
        static registerClass(protocolClass: AnyClass):boolean;
        /**
         * Unregisters the specified subclass of 
         * .
         */
        static unregisterClass(protocolClass: AnyClass):void;
        /**
         * Determines whether the protocol subclass can handle the specified request.
         */
        static canInit(request: URLRequest):boolean;
        /**
         * Fetches the property associated with the specified key in the specified request.
         */
        static property(key: string, request: URLRequest):any;
        /**
         * Sets the property associated with the specified key in the specified request.
         */
        static setProperty(value: any, key: string, request: NSMutableURLRequest):void;
        /**
         * Removes the property associated with the specified key in the specified request.
         */
        static removeProperty(key: string, request: NSMutableURLRequest):void;
        /**
         * Returns a canonical version of the specified request.
         */
        static canonicalRequest(request: URLRequest):URLRequest;
        /**
         * A Boolean value indicating whether two requests are equivalent for cache purposes.
         */
        static requestIsCacheEquivalent(a: URLRequest, b: URLRequest):boolean;
        /**
         * Starts protocol-specific loading of the request.
         */
        startLoading():void;
        /**
         * Stops protocol-specific loading of the request.
         */
        stopLoading():void;
    }
    export interface URLProtocolClient extends NSObjectProtocol {
        /**
         * Tells the client that the protocol implementation has created a response object for the request.
         */
        urlProtocol(protocol: URLProtocol, response: URLResponse, policy: URLCache.StoragePolicy):void;
        /**
         * Tells the client that the protocol implementation has finished loading.
         */
        urlProtocolDidFinishLoading(protocol: URLProtocol):void;
    }
    export interface URLSessionDelegate extends NSObjectProtocol,Sendable {
        /**
         * Tells the URL session that the session has been invalidated.
         */
        urlSession(session: URLSession, error: Error | null):void;
        /**
         * Tells the delegate that all messages enqueued for a session have been delivered.
         */
        urlSessionDidFinishEvents(session: URLSession):void;
    }
    export interface URLSessionTaskDelegate extends URLSessionDelegate {
        /**
         * Tells the delegate that the task finished transferring data.
         */
        urlSession(session: URLSession, task: URLSessionTask, error: Error | null):void;
    }
    export class URLSessionTaskMetrics extends NSObject {
        /**
         * An array of metrics for each individual request-response transaction made during the execution of the task.
         */
        transactionMetrics: URLSessionTaskTransactionMetrics[];
        /**
         * The time interval between when a task is instantiated and when the task is completed.
         */
        taskInterval: DateInterval;
        /**
         * The number of redirects that occurred during the execution of the task.
         */
        redirectCount: number;
        /**
         * Creates a task metrics instance.
         */
        constructor ();
        static new():Self;
    }
    /**
     * the same name namespace of class URLSessionTaskMetrics
     */
    export namespace URLSessionTaskMetrics {
        export enum ResourceFetchType {
            unknown = 0,
            networkLoad = 1,
            serverPush = 2,
            localCache = 3,
        }
        export enum DomainResolutionProtocol {
            https = 4,
            tcp = 2,
            tls = 3,
            udp = 1,
            unknown = 0,
        }
    }
    export class URLSessionTaskTransactionMetrics extends NSObject {
        /**
         * The transaction request.
         */
        request: URLRequest;
        /**
         * The transaction response.
         */
        response: URLResponse | null;
        /**
         * The time when the task started fetching the resource, from the server or locally. 
         */
        fetchStartDate: Date | null;
        /**
         * The time immediately before the task started the name lookup for the resource.
         */
        domainLookupStartDate: Date | null;
        /**
         * The time after the name lookup was completed.
         */
        domainLookupEndDate: Date | null;
        /**
         * The time immediately before the task started establishing a TCP connection to the server.
         */
        connectStartDate: Date | null;
        /**
         * The time immediately before the task started the TLS security handshake to secure the current connection.
         */
        secureConnectionStartDate: Date | null;
        /**
         * The time immediately after the security handshake completed.
         */
        secureConnectionEndDate: Date | null;
        /**
         * The time immediately after the task finished establishing the connection to the server.
         */
        connectEndDate: Date | null;
        /**
         * The time immediately before the task started requesting the resource, regardless of whether it is retrieved from the server or local resources.
         */
        requestStartDate: Date | null;
        /**
         * The time immediately after the task finished requesting the resource, regardless of whether it was retrieved from the server or local resources.
         */
        requestEndDate: Date | null;
        /**
         * The time immediately after the task received the first byte of the response from the server or from local resources.
         */
        responseStartDate: Date | null;
        /**
         * The time immediately after the task received the last byte of the resource.
         */
        responseEndDate: Date | null;
        /**
         * The size of the upload body data, file, or stream, in bytes.
         */
        countOfRequestBodyBytesBeforeEncoding: Int64;
        /**
         * The number of bytes transferred for the request body.
         */
        countOfRequestBodyBytesSent: Int64;
        /**
         * The number of bytes transferred for the request header.
         */
        countOfRequestHeaderBytesSent: Int64;
        /**
         * The size of data delivered to your delegate or completion handler.
         */
        countOfResponseBodyBytesAfterDecoding: Int64;
        /**
         * The number of bytes transferred for the response body.
         */
        countOfResponseBodyBytesReceived: Int64;
        /**
         * The number of bytes transferred for the response header.
         */
        countOfResponseHeaderBytesReceived: Int64;
        /**
         * The network protocol used to fetch the resource.
         */
        networkProtocolName: string;
        /**
         * The IP address string of the remote interface for the connection.
         */
        remoteAddress: string;
        /**
         * The port number of the remote interface for the connection.
         */
        remotePort: number;
        /**
         * The IP address string of the local interface for the connection.
         */
        localAddress: string;
        /**
         * The port number of the local interface for the connection.
         */
        localPort: number;
        /**
         * The TLS cipher suite the task negotiated with the endpoint for the connection.
         */
        negotiatedTLSCipherSuite: tls_ciphersuite_t | null;
        /**
         * The TLS protocol version the task negotiated with the endpoint for the connection.
         */
        negotiatedTLSProtocolVersion: tls_protocol_version_t | null;
        /**
         * A Boolean value that indicates whether the connection operates over a cellular interface.
         */
        isCellular: boolean;
        /**
         * A Boolean value that indicates whether the connection operates over an expensive interface.
         */
        isExpensive: boolean;
        /**
         * A Boolean value that indicates whether the connection operates over an interface marked as constrained.
         */
        isConstrained: boolean;
        /**
         * A Boolean value that indicastes whether the task used a proxy connection to fetch the resource.
         */
        isProxyConnection: boolean;
        /**
         * A Boolean value that indicates whether the task used a persistent connection to fetch the resource.
         */
        isReusedConnection: boolean;
        /**
         * A Boolean value that indicates whether the connection uses a successfully negotiated multipath protocol.
         */
        isMultipath: boolean;
        /**
         * A value that indicates whether the resource was loaded, pushed, or retrieved from the local cache.
         */
        resourceFetchType: URLSessionTaskMetrics.ResourceFetchType;
        domainResolutionProtocol: URLSessionTaskMetrics.DomainResolutionProtocol;
        /**
         * Creates a transaction metrics instance.
         */
        constructor ();
        /**
         * Creates a new transaction metrics instance.
         */
        static new():Self;
    }
    export class StringEncodingDetectionOptionsKey {
        static allowLossyKey: StringEncodingDetectionOptionsKey;
        static disallowedEncodingsKey: StringEncodingDetectionOptionsKey;
        static fromWindowsKey: StringEncodingDetectionOptionsKey;
        static likelyLanguageKey: StringEncodingDetectionOptionsKey;
        static lossySubstitutionKey: StringEncodingDetectionOptionsKey;
        static suggestedEncodingsKey: StringEncodingDetectionOptionsKey;
        static useOnlySuggestedEncodingsKey: StringEncodingDetectionOptionsKey;
        constructor (rawValue: string);
    }
    export class URLSessionDataTask extends URLSessionTask {
        constructor ();
        static new():Self;
    }
    export interface URLSessionDataDelegate extends URLSessionTaskDelegate {
        /**
         * Tells the delegate that the data task received the initial reply (headers) from the server.
         */
        urlSession(session: URLSession, dataTask: URLSessionDataTask, response: URLResponse, completionHandler: ()=>any):void;
    }
    export class URLSessionDownloadTask extends URLSessionTask {
        /**
         * Initializes a download task.
         */
        constructor ();
        /**
         * Creates and initializes a download task.
         */
        static new():Self;
        /**
         * Cancels a download and calls a callback with resume data for later use.
         */
        cancel(completionHandler: ()=>any):void;
    }
    export interface URLSessionDownloadDelegate extends URLSessionTaskDelegate {
        /**
         * Tells the delegate that a download task has finished downloading.
         */
        urlSession(session: URLSession, downloadTask: URLSessionDownloadTask, location: URL):void;
    }
    export class URLSessionUploadTask extends URLSessionDataTask {
        constructor ();
        static new():Self;
    }
    export class URLSessionStreamTask extends URLSessionTask {
        constructor ();
        static new():Self;
        /**
         * Asynchronously reads a number of bytes from the stream, and calls a handler upon completion.
         */
        readData(minBytes: number, maxBytes: number, timeout: TimeInterval, completionHandler: ()=>any):void;
        /**
         * Asynchronously writes the specified data to the stream, and calls a handler upon completion.
         */
        write(data: Data, timeout: TimeInterval, completionHandler: ()=>any):void;
        /**
         * Completes any already enqueued reads and writes, and then invokes the 
         *  delegate message.
         */
        captureStreams():void;
        /**
         * Completes any enqueued reads and writes, and then closes the read side of the underlying socket.
         */
        closeRead():void;
        /**
         * Completes any enqueued reads and writes, and then closes the write side of the underlying socket.
         */
        closeWrite():void;
        /**
         * Completes any enqueued reads and writes, and establishes a secure connection.
         */
        startSecureConnection():void;
        /**
         * Completes any enqueued reads and writes, and closes the secure connection.
         */
        stopSecureConnection():void;
    }
    export interface URLSessionStreamDelegate extends URLSessionTaskDelegate {
        /**
         * Tells the delegate that a better route to the host has been detected for the stream.
         */
        urlSession(session: URLSession, streamTask: URLSessionStreamTask):void;
    }
    export class URLSessionWebSocketTask extends URLSessionTask {
        /**
         * The maximum number of bytes to buffer before the receive call fails with an error.
         */
        maximumMessageSize: number;
        /**
         * A code that indicates the reason a connection closed.
         */
        closeCode: URLSessionWebSocketTask.CloseCode;
        /**
         * A block of data that provides further information about why a connection closed.
         */
        closeReason: Data | null;
        /**
         * Sends a WebSocket message, receiving the result in a completion handler.
         */
        send(message: URLSessionWebSocketTask.Message, completionHandler: ()=>any):void;
        /**
         * Reads a WebSocket message once all the frames of the message are available.
         */
        receive(completionHandler: ()=>any):void;
        /**
         * Sends a ping frame from the client side, with a closure to receive the pong from the server endpoint.
         */
        sendPing(pongReceiveHandler: ()=>any):void;
        /**
         * Sends a close frame with the given close code and optional close reason.
         */
        cancel(closeCode: URLSessionWebSocketTask.CloseCode, reason: Data | null):void;
    }
    /**
     * the same name namespace of class URLSessionWebSocketTask
     */
    export namespace URLSessionWebSocketTask {
        export enum Message {
            data,
            string,
        }
        export enum CloseCode {
            abnormalClosure = 1006,
            goingAway = 1001,
            internalServerError = 1011,
            invalid = 0,
            invalidFramePayloadData = 1007,
            mandatoryExtensionMissing = 1010,
            messageTooBig = 1009,
            noStatusReceived = 1005,
            normalClosure = 1000,
            policyViolation = 1008,
            protocolError = 1002,
            tlsHandshakeFailure = 1015,
            unsupportedData = 1003,
        }
    }
    export interface URLSessionWebSocketDelegate extends URLSessionTaskDelegate {
        /**
         * Tells the delegate that the WebSocket task successfully negotiated the handshake with the endpoint, indicating the negotiated protocol.
         */
        urlSession(session: URLSession, webSocketTask: URLSessionWebSocketTask, protocol: string):void;
    }
    export class URLSessionTask extends NSObject {
        /**
         * The default URL session task priority, used implicitly for any task you have not prioritized.
         */
        static defaultPriority: Float;
        /**
         * A low URL session task priority, with a floating point value above the minimum of 
         *  and below the default value.
         */
        static lowPriority: Float;
        /**
         * A high URL session task priority, with a floating point value above the default value and below the maximum of 
         * .
         */
        static highPriority: Float;
        /**
         * The current state of the task—active, suspended, in the process of being canceled, or completed.
         */
        state: URLSessionTask.State;
        /**
         * The relative priority at which you’d like a host to handle the task, specified as a floating point value between 
         *  (lowest priority) and 
         *  (highest priority).
         */
        priority: Float;
        /**
         * A representation of the overall task progress.
         */
        progress: Progress;
        /**
         * The number of bytes that the task expects to receive in the response body.
         */
        countOfBytesExpectedToReceive: Int64;
        /**
         * The number of bytes that the task has received from the server in the response body.
         */
        countOfBytesReceived: Int64;
        /**
         * The number of bytes that the task expects to send in the request body.
         */
        countOfBytesExpectedToSend: Int64;
        /**
         * The number of bytes that the task has sent to the server in the request body.
         */
        countOfBytesSent: Int64;
        /**
         * The URL request object currently being handled by the task.
         */
        currentRequest: URLRequest | null;
        /**
         * The original request object passed when the task was created.
         */
        originalRequest: URLRequest | null;
        /**
         * The server’s response to the currently active request.
         */
        response: URLResponse | null;
        /**
         * An app-provided string value for the current task.
         */
        taskDescription: string;
        /**
         * An identifier uniquely identifying the task within a given session.
         */
        taskIdentifier: number;
        /**
         * An error object that indicates why the task failed.
         */
        error: Error | null;
        /**
         * A Boolean value that determines whether to deliver a partial response body in increments.
         */
        prefersIncrementalDelivery: boolean;
        /**
         * A delegate specific to the task.
         */
        delegate: URLSessionTaskDelegate | null;
        /**
         * A best-guess upper bound on the number of bytes the client expects to receive.
         */
        countOfBytesClientExpectsToReceive: Int64;
        /**
         * A best-guess upper bound on the number of bytes the client expects to send.
         */
        countOfBytesClientExpectsToSend: Int64;
        /**
         * The earliest date at which the network load should begin.
         */
        earliestBeginDate: Date | null;
        /**
         * Initializes an empty URL sesson task.
         */
        constructor ();
        /**
         * Creates a new URL session task.
         */
        static new():Self;
        /**
         * Cancels the task.
         */
        cancel():void;
        /**
         * Resumes the task, if it is suspended.
         */
        resume():void;
        /**
         * Temporarily suspends a task.
         */
        suspend():void;
    }
    /**
     * the same name namespace of class URLSessionTask
     */
    export namespace URLSessionTask {
        export enum State {
            running = 0,
            suspended = 1,
            canceling = 2,
            completed = 3,
        }
    }
    export class URLRequest {
        /**
         * The request’s cache policy.
         */
        cachePolicy: URLRequest.CachePolicy;
        /**
         * The HTTP request method.
         */
        httpMethod: string;
        /**
         * The URL of the request.
         */
        url: URL | null;
        /**
         * The data sent as the message body of a request, such as for an HTTP POST request.
         */
        httpBody: Data | null;
        /**
         * The stream used to deliver the HTTP body.
         */
        httpBodyStream: InputStream | null;
        /**
         * The main document URL associated with this request.
         */
        mainDocumentURL: URL | null;
        /**
         * A dictionary containing all of the HTTP header fields for a request.
         */
        allHTTPHeaderFields: Record<string, string> | null;
        /**
         * The timeout interval of the request.
         */
        timeoutInterval: TimeInterval;
        /**
         * A Boolean value indicating whether cookies will be sent with and set for this request.
         */
        httpShouldHandleCookies: boolean;
        /**
         * A Boolean value indicating whether the request should transmit before the previous response is received.
         */
        httpShouldUsePipelining: boolean;
        /**
         * A Boolean value indicating whether the request is allowed to use the built-in cellular radios to satisfy the request.
         */
        allowsCellularAccess: boolean;
        /**
         * A Boolean value that indicates whether the request may use the network when the user has specified Low Data Mode.
         */
        allowsConstrainedNetworkAccess: boolean;
        /**
         * A Boolean value that indicates whether connections may use a network interface that the system considers expensive.
         */
        allowsExpensiveNetworkAccess: boolean;
        /**
         * The service type associated with this request.
         */
        networkServiceType: URLRequest.NetworkServiceType;
        /**
         * The entity that initiates the network request.
         */
        attribution: URLRequest.Attribution;
        /**
         * A textual description of the request.
         */
        description: string;
        /**
         * A textual description of the request suitable for debugging.
         */
        debugDescription: string;
        /**
         * A mirror that reflects the request.
         */
        customMirror: Mirror;
        /**
         * The computed hash value for the request.
         */
        hashValue: number;
        assumesHTTP3Capable: boolean;
        requiresDNSSECValidation: boolean;
        /**
         * Creates and initializes a URL request with the given URL, cache policy, and timeout interval.
         */
        constructor (url: URL, cachePolicy: URLRequest.CachePolicy, timeoutInterval: TimeInterval);
        /**
         * Adds a value to the header field.
         */
        addValue(value: string, field: string):void;
        /**
         * Sets a value for the header field.
         */
        setValue(value: string, field: string):void;
        /**
         * Retrieves a header value.
         */
        value(field: string):string;
        hash(hasher: Hasher):void;
    }
    export class NSURLRequest extends NSObject {}
    /**
     * the same name namespace of class NSURLRequest
     */
    export namespace NSURLRequest {
        export enum CachePolicy {
            useProtocolCachePolicy = 0,
            reloadIgnoringLocalCacheData = 1,
            reloadIgnoringLocalAndRemoteCacheData = 4,
            returnCacheDataElseLoad = 2,
            returnCacheDataDontLoad = 3,
            reloadRevalidatingCacheData = 5,
        }
        export enum NetworkServiceType {
            default = 0,
            voip = 1,
            video = 2,
            background = 3,
            voice = 4,
            callSignaling = 11,
            responsiveData = 6,
            avStreaming = 8,
            responsiveAV = 9,
        }
    }
    export class NSMutableURLRequest extends NSURLRequest {
        /**
         * The request’s cache policy.
         */
        cachePolicy: NSURLRequest.CachePolicy;
        /**
         * The HTTP request method.
         */
        httpMethod: string;
        /**
         * The URL being requested.
         */
        url: URL | null;
        /**
         * The request body.
         */
        httpBody: Data | null;
        /**
         * The request body as an input stream.
         */
        httpBodyStream: InputStream | null;
        /**
         * The main document URL.
         */
        mainDocumentURL: URL | null;
        /**
         * A dictionary containing all of the HTTP header fields for a request.
         */
        allHTTPHeaderFields: Record<string, string> | null;
        /**
         * The request’s timeout interval, in seconds.
         */
        timeoutInterval: TimeInterval;
        /**
         * A Boolean value that indicates whether the request should use the default cookie handling for the request.
         */
        httpShouldHandleCookies: boolean;
        /**
         * A Boolean value that indicates whether the request can continue transmitting data before receiving a response from an earlier transmission.
         */
        httpShouldUsePipelining: boolean;
        /**
         * A Boolean value that indicates whether a connection can use the device’s cellular network (if present).
         */
        allowsCellularAccess: boolean;
        /**
         * A Boolean value that indicates whether connections may use the network when the user has specified Low Data Mode.
         */
        allowsConstrainedNetworkAccess: boolean;
        /**
         * A Boolean value that indicates whether connections may use a network interface that the system considers expensive.
         */
        allowsExpensiveNetworkAccess: boolean;
        /**
         * The network service type of the connection.
         */
        networkServiceType: NSURLRequest.NetworkServiceType;
        /**
         * The entity that initiates the network request.
         */
        attribution: NSURLRequest.Attribution;
        assumesHTTP3Capable: boolean;
        requiresDNSSECValidation: boolean;
        /**
         * Adds a value to the header field.
         */
        addValue(value: string, field: string):void;
        /**
         * Sets a value for the header field.
         */
        setValue(value: string, field: string):void;
        /**
         * Binds a URL request to the network interface associated with the hotspot helper command instance.
         */
        bind(command: NEHotspotHelperCommand):void;
    }
    export class NSMutableString extends NSString {
        /**
         * Returns an 
         *  object initialized with initial storage for a given number of characters,
         */
        constructor (capacity: number);
        /**
         * Adds to the end of the receiver the characters of a given string.
         */
        append(aString: string):void;
        /**
         * Transliterates the receiver by applying a specified ICU string transform.
         */
        applyTransform(transform: StringTransform, reverse: boolean, range: NSRange, resultingRange: NSRangePointer | null):boolean;
        /**
         * Removes from the receiver the characters in a given range.
         */
        deleteCharacters(range: NSRange):void;
        /**
         * Inserts into the receiver the characters of a given string at a given location.
         */
        insert(aString: string, loc: number):void;
        /**
         * Replaces the characters from 
         *  with those in 
         * .
         */
        replaceCharacters(range: NSRange, aString: string):void;
        /**
         * Replaces all occurrences of a given string in a given range with another given string, returning the number of replacements.
         */
        replaceOccurrences(target: string, replacement: string, options: NSString.CompareOptions, searchRange: NSRange):number;
        /**
         * Replaces the characters of the receiver with those in a given string.
         */
        setString(aString: string):void;
        appendFormat(format: NSString, ...args: CVarArg):void;
    }
    export class URLResponse extends NSObject {
        /**
         * The expected length of the response’s content.
         */
        expectedContentLength: Int64;
        /**
         * A suggested filename for the response data.
         */
        suggestedFilename: string;
        /**
         * The MIME type of the response.
         */
        mimeType: string;
        /**
         * The name of the text encoding provided by the response’s originating source.
         */
        textEncodingName: string;
        /**
         * The URL for the response.
         */
        url: URL | null;
        /**
         * Creates an initialized 
         *  object with the URL, MIME type, length, and text encoding set to given values.
         */
        constructor (URL: URL, MIMEType: string, length: number, name: string);
    }
    export class HTTPURLResponse extends URLResponse {
        /**
         * All HTTP header fields of the response.
         */
        allHeaderFields: Record<AnyHashable, any>;
        /**
         * The response’s HTTP status code.
         */
        statusCode: number;
        /**
         * Initializes an HTTP URL response object with a status code, protocol version, and response headers.
         */
        constructor (url: URL, statusCode: number, HTTPVersion: string, headerFields: Record<string, string> | null);
        /**
         * Returns a localized string corresponding to a specified HTTP status code.
         */
        static localizedString(statusCode: number):string;
        /**
         * Returns the value that corresponds to the given header field.
         */
        value(field: string):string;
    }
    export class CachedURLResponse extends NSObject {
        /**
         * The cached response’s data.
         */
        data: Data;
        /**
         * The URL response object associated with the instance.
         */
        response: URLResponse;
        /**
         * The cached response’s storage policy.
         */
        storagePolicy: URLCache.StoragePolicy;
        /**
         * The cached response’s user info dictionary.
         */
        userInfo: Record<AnyHashable, any> | null;
        /**
         * Creates a cached URL response instance.
         */
        constructor (response: URLResponse, data: Data);
        /**
         * Creates a cached URL response instance with a given server response, data, user-info dictionary, and storage policy.
         */
        constructor (response: URLResponse, data: Data, userInfo: Record<AnyHashable, any> | null, storagePolicy: URLCache.StoragePolicy);
    }
    export class URLCache extends NSObject {}
    /**
     * the same name namespace of class URLCache
     */
    export namespace URLCache {
        export enum StoragePolicy {
            allowed = 0,
            allowedInMemoryOnly = 1,
            notAllowed = 2,
        }
    }
    export class URLAuthenticationChallenge extends NSObject {
        /**
         * The receiver’s protection space.
         */
        protectionSpace: URLProtectionSpace;
        /**
         * The URL response object representing the last authentication failure.
         */
        failureResponse: URLResponse | null;
        /**
         * The receiver’s count of failed authentication attempts.
         */
        previousFailureCount: number;
        /**
         * The proposed credential for this challenge.
         */
        proposedCredential: URLCredential | null;
        /**
         * The error object representing the last authentication failure.
         */
        error: Error | null;
        /**
         * The sender of the challenge.
         */
        sender: URLAuthenticationChallengeSender | null;
        /**
         * Creates an authentication challenge from an existing challenge instance.
         */
        constructor (challenge: URLAuthenticationChallenge, sender: URLAuthenticationChallengeSender);
        /**
         * Initializes an authentication challenge from parameters you provide.
         */
        constructor (space: URLProtectionSpace, credential: URLCredential | null, previousFailureCount: number, response: URLResponse | null, error: Error | null, sender: URLAuthenticationChallengeSender);
    }
    export class URLCredential extends NSObject {
        /**
         * The credential’s user name.
         */
        user: string;
        /**
         * The intermediate certificates of the credential, if it is a client certificate credential.
         */
        certificates: any[];
        /**
         * A Boolean value that indicates whether the credential has a password.
         */
        hasPassword: boolean;
        /**
         * The credential’s password.
         */
        password: string;
        /**
         * The identity of this credential if it is a client certificate credential.
         */
        identity: SecIdentity | null;
        /**
         * The credential’s persistence setting.
         */
        persistence: URLCredential.Persistence;
        /**
         * Creates a URL credential instance for resolving a client certificate authentication challenge.
         */
        constructor (identity: SecIdentity, certArray: any[] | null, persistence: URLCredential.Persistence);
        /**
         * Creates a URL credential instance for server trust authentication, initialized with a accepted trust.
         */
        constructor (trust: SecTrust);
        /**
         * Creates a URL credential instance initialized with a given user name and password, using a given persistence setting.
         */
        constructor (user: string, password: string, persistence: URLCredential.Persistence);
    }
    /**
     * the same name namespace of class URLCredential
     */
    export namespace URLCredential {
        export enum Persistence {
            none = 0,
            forSession = 1,
            permanent = 2,
            synchronizable = 3,
        }
    }
    export class URLCredentialStorage extends NSObject {
        /**
         * The shared URL credential storage instance.
         */
        static shared: URLCredentialStorage;
        /**
         * The credentials for all available protection spaces.
         */
        allCredentials: Record<[String, URLCredential]>;
        /**
         * Returns the default credential for the specified protection space.
         */
        defaultCredential(space: URLProtectionSpace):URLCredential | null;
        /**
         * Gets the default credential for the specified protection space, which is being accessed by the given task, and passes it to the provided completion handler.
         */
        getDefaultCredential(space: URLProtectionSpace, task: URLSessionTask, completionHandler: ()=>any):void;
        /**
         * Sets the default credential for a specified protection space.
         */
        setDefaultCredential(credential: URLCredential, space: URLProtectionSpace):void;
        /**
         * Removes the specified credential from the credential storage for the specified protection space.
         */
        remove(credential: URLCredential, space: URLProtectionSpace):void;
        /**
         * Adds a credential to the credential storage for the specified protection space. 
         */
        set(credential: URLCredential, space: URLProtectionSpace):void;
        /**
         * Returns a dictionary containing the credentials for the specified protection space.
         */
        credentials(space: URLProtectionSpace):Record<string, URLCredential> | null;
        /**
         * Gets a dictionary containing the credentials for the specified protection space, on behalf of the given task, and passes the dictionary to the provided completion handler.
         */
        getCredentials(protectionSpace: URLProtectionSpace, task: URLSessionTask, completionHandler: ()=>any):void;
    }
    export class URLProtectionSpace extends NSObject {
        /**
         * The authentication method used by the receiver.
         */
        authenticationMethod: string;
        /**
         * The acceptable certificate-issuing authorities for client certificate authentication.
         */
        distinguishedNames: Data[] | null;
        /**
         * The receiver’s host.
         */
        host: string;
        /**
         * The receiver’s port.
         */
        port: number;
        /**
         * The receiver’s protocol.
         */
        protocol: string;
        /**
         * The receiver's proxy type.
         */
        proxyType: string;
        /**
         * The receiver’s authentication realm
         */
        realm: string;
        /**
         * A Boolean value that indicates whether the credentials for the protection space can be sent securely.
         */
        receivesCredentialSecurely: boolean;
        /**
         * A representation of the server’s SSL transaction state.
         */
        serverTrust: SecTrust | null;
        /**
         * Creates a protection space object from the given host, port, protocol, realm, and authentication method.
         */
        constructor (host: string, port: number, protocol: string, realm: string, authenticationMethod: string);
        /**
         * Creates a protection space object representing a proxy server. 
         */
        constructor (host: string, port: number, type: string, realm: string, authenticationMethod: string);
        isProxy():boolean;
    }
    export class URLError {
        /**
         * The domain of the URL error.
         */
        static errorDomain: string;
        /**
         * App Transport Security disallowed a connection because there is no secure network connection.
         */
        static appTransportSecurityRequiresSecureConnection: URLError.Code;
        /**
         * An app or app extension attempted to connect to a background session that is already connected to a process.
         */
        static backgroundSessionInUseByAnotherProcess: URLError.Code;
        /**
         * The shared container identifier of the URL session configuration is needed but hasn’t been set.
         */
        static backgroundSessionRequiresSharedContainer: URLError.Code;
        /**
         * The app is suspended or exits while a background data task is processing.
         */
        static backgroundSessionWasDisconnected: URLError.Code;
        /**
         * The URL Loading System received bad data from the server.
         */
        static badServerResponse: URLError.Code;
        /**
         * A malformed URL prevented a URL request from being initiated.
         */
        static badURL: URLError.Code;
        /**
         * A connection was attempted while a phone call is active on a network that doesn’t support simultaneous phone and data communication, such as EDGE or GPRS.
         */
        static callIsActive: URLError.Code;
        /**
         * An asynchronous load has been canceled.
         */
        static cancelled: URLError.Code;
        /**
         * A download task couldn’t close the downloaded file on disk.
         */
        static cannotCloseFile: URLError.Code;
        /**
         * An attempt to connect to a host failed.
         */
        static cannotConnectToHost: URLError.Code;
        /**
         * A download task couldn’t create the downloaded file on disk because of an I/O failure.
         */
        static cannotCreateFile: URLError.Code;
        /**
         * Content data received during a connection request had an unknown content encoding.
         */
        static cannotDecodeContentData: URLError.Code;
        /**
         * Content data received during a connection request couldn’t be decoded for a known content encoding.
         */
        static cannotDecodeRawData: URLError.Code;
        /**
         * The host name for a URL couldn’t be resolved.
         */
        static cannotFindHost: URLError.Code;
        /**
         * A request to load an item only from the cache could not be satisfied.
         */
        static cannotLoadFromNetwork: URLError.Code;
        /**
         * A download task was unable to move a downloaded file on disk.
         */
        static cannotMoveFile: URLError.Code;
        /**
         * A download task was unable to open the downloaded file on disk.
         */
        static cannotOpenFile: URLError.Code;
        /**
         * A task couldn’t parse a response.
         */
        static cannotParseResponse: URLError.Code;
        /**
         * A download task was unable to remove a downloaded file from disk.
         */
        static cannotRemoveFile: URLError.Code;
        /**
         * A download task was unable to write to the downloaded file on disk.
         */
        static cannotWriteToFile: URLError.Code;
        /**
         * A server certificate was rejected.
         */
        static clientCertificateRejected: URLError.Code;
        /**
         * A client certificate was required to authenticate an SSL connection during a request.
         */
        static clientCertificateRequired: URLError.Code;
        /**
         * The length of the resource data exceeds the maximum allowed.
         */
        static dataLengthExceedsMaximum: URLError.Code;
        /**
         * The cellular network disallowed a connection.
         */
        static dataNotAllowed: URLError.Code;
        /**
         * The host address couldn’t be found via DNS lookup.
         */
        static dnsLookupFailed: URLError.Code;
        /**
         * A download task failed to decode an encoded file during the download.
         */
        static downloadDecodingFailedMidStream: URLError.Code;
        /**
         * A download task failed to decode an encoded file after downloading.
         */
        static downloadDecodingFailedToComplete: URLError.Code;
        /**
         * The specified file doesn’t exist.
         */
        static fileDoesNotExist: URLError.Code;
        /**
         * A request for an FTP file resulted in the server responding that the file is not a plain file, but a directory.
         */
        static fileIsDirectory: URLError.Code;
        /**
         * A redirect loop has been detected or the threshold for number of allowable redirects has been exceeded (currently 16).
         */
        static httpTooManyRedirects: URLError.Code;
        /**
         * The attempted connection required activating a data context while roaming, but international roaming is disabled.
         */
        static internationalRoamingOff: URLError.Code;
        /**
         * A client or server connection was severed in the middle of an in-progress load.
         */
        static networkConnectionLost: URLError.Code;
        /**
         * A resource couldn’t be read because of insufficient permissions.
         */
        static noPermissionsToReadFile: URLError.Code;
        /**
         * A network resource was requested, but an internet connection hasn’t been established and can’t be established automatically.
         */
        static notConnectedToInternet: URLError.Code;
        /**
         * A redirect was specified by way of server response code, but the server didn’t accompany this code with a redirect URL.
         */
        static redirectToNonExistentLocation: URLError.Code;
        /**
         * A body stream is needed but the client didn’t provide one.  
         */
        static requestBodyStreamExhausted: URLError.Code;
        /**
         * A requested resource couldn’t be retrieved.
         */
        static resourceUnavailable: URLError.Code;
        /**
         * A server certificate is expired, or is not yet valid.
         */
        static serverCertificateHasBadDate: URLError.Code;
        /**
         * A server certificate wasn’t signed by any root server.
         */
        static serverCertificateHasUnknownRoot: URLError.Code;
        /**
         * A server certificate isn’t valid yet.
         */
        static serverCertificateNotYetValid: URLError.Code;
        /**
         * A server certificate was signed by a root server that isn’t trusted.
         */
        static serverCertificateUntrusted: URLError.Code;
        /**
         * An attempt to establish a secure connection failed for reasons that can’t be expressed more specifically.
         */
        static secureConnectionFailed: URLError.Code;
        /**
         * An asynchronous operation timed out.
         */
        static timedOut: URLError.Code;
        /**
         * The URL Loading System encountered an error that it can’t interpret.  
         */
        static unknown: URLError.Code;
        /**
         * A properly formed URL couldn’t be handled by the framework.
         */
        static unsupportedURL: URLError.Code;
        /**
         * Authentication is required to access a resource.
         */
        static userAuthenticationRequired: URLError.Code;
        /**
         * An asynchronous request for authentication has been canceled by the user.
         */
        static userCancelledAuthentication: URLError.Code;
        /**
         * A server reported that a URL has a non-zero content length, but terminated the network connection gracefully without sending any data.
         */
        static zeroByteResource: URLError.Code;
        /**
         * The error code within the domain.
         */
        errorCode: number;
        /**
         * The error’s user-info dictionary.
         */
        errorUserInfo: Record<string, any>;
        /**
         * A string containing the localized description of the error.
         */
        localizedDescription: string;
        /**
         * The URL which caused a load to fail.
         */
        failingURL: URL | null;
        /**
         * The state of a failed SSL handshake.
         */
        failureURLPeerTrust: SecTrust | null;
        /**
         * The string for the URL which caused a load to fail.
         */
        failureURLString: string;
        /**
         * An opaque data object used to resume a failed download task.
         */
        downloadTaskResumeData: Data | null;
        /**
         * The reason for canceling a background task.
         */
        backgroundTaskCancelledReason: URLError.BackgroundTaskCancelledReason | null;
        /**
         * The reason the network was unavailable for a task.
         */
        networkUnavailableReason: URLError.NetworkUnavailableReason | null;
        /**
         * The hash value.
         */
        hashValue: number;
        code: Code;
        userInfo: Record<string, any>;
        constructor (code: Code, userInfo: Record<string, any>);
        hash(hasher: Hasher):void;
    }
    /**
     * the same name namespace of class URLError
     */
    export namespace URLError {
        export enum BackgroundTaskCancelledReason {
            backgroundUpdatesDisabled,
            insufficientSystemResources,
            userForceQuitApplication,
        }
        export enum NetworkUnavailableReason {
            cellular,
            constrained,
            expensive,
        }
        export class Code {
            /**
             * App Transport Security disallowed a connection because there is no secure network connection.
             */
            static appTransportSecurityRequiresSecureConnection: URLError.Code;
            /**
             * An app or app extension attempted to connect to a background session that is already connected to a process.
             */
            static backgroundSessionInUseByAnotherProcess: URLError.Code;
            /**
             * The shared container identifier of the URL session configuration is needed but has not been set.
             */
            static backgroundSessionRequiresSharedContainer: URLError.Code;
            /**
             * The app is suspended or exits while a background data task is processing.
             */
            static backgroundSessionWasDisconnected: URLError.Code;
            /**
             * The URL Loading system received bad data from the server.
             */
            static badServerResponse: URLError.Code;
            /**
             * A malformed URL prevented a URL request from being initiated.
             */
            static badURL: URLError.Code;
            /**
             * A connection was attempted while a phone call is active on a network that does not support simultaneous phone and data communication (EDGE or GPRS).
             */
            static callIsActive: URLError.Code;
            /**
             * An asynchronous load has been canceled.
             */
            static cancelled: URLError.Code;
            /**
             * A download task couldn’t close the downloaded file on disk.
             */
            static cannotCloseFile: URLError.Code;
            /**
             * An attempt to connect to a host failed.
             */
            static cannotConnectToHost: URLError.Code;
            /**
             * A download task couldn’t create the downloaded file on disk because of an I/O failure.
             */
            static cannotCreateFile: URLError.Code;
            /**
             * Content data received during a connection request had an unknown content encoding.
             */
            static cannotDecodeContentData: URLError.Code;
            /**
             * Content data received during a connection request could not be decoded for a known content encoding.
             */
            static cannotDecodeRawData: URLError.Code;
            /**
             * The host name for a URL could not be resolved.
             */
            static cannotFindHost: URLError.Code;
            /**
             * A request to load an item only from the cache could not be satisfied.
             */
            static cannotLoadFromNetwork: URLError.Code;
            /**
             * A download task was unable to move a downloaded file on disk.
             */
            static cannotMoveFile: URLError.Code;
            /**
             * A download task was unable to open the downloaded file on disk.
             */
            static cannotOpenFile: URLError.Code;
            /**
             * A task could not parse a response.
             */
            static cannotParseResponse: URLError.Code;
            /**
             * A download task was unable to remove a downloaded file from disk.
             */
            static cannotRemoveFile: URLError.Code;
            /**
             * A download task was unable to write to the downloaded file on disk.
             */
            static cannotWriteToFile: URLError.Code;
            /**
             * A server certificate was rejected.
             */
            static clientCertificateRejected: URLError.Code;
            /**
             * A client certificate was required to authenticate an SSL connection during a request.
             */
            static clientCertificateRequired: URLError.Code;
            /**
             * The length of the resource data exceeds the maximum allowed.
             */
            static dataLengthExceedsMaximum: URLError.Code;
            /**
             * The cellular network disallowed a connection.
             */
            static dataNotAllowed: URLError.Code;
            /**
             * The host address could not be found via DNS lookup.
             */
            static dnsLookupFailed: URLError.Code;
            /**
             * A download task failed to decode an encoded file during the download.
             */
            static downloadDecodingFailedMidStream: URLError.Code;
            /**
             * A download task failed to decode an encoded file after downloading.
             */
            static downloadDecodingFailedToComplete: URLError.Code;
            /**
             * A file does not exist.
             */
            static fileDoesNotExist: URLError.Code;
            /**
             * A request for an FTP file resulted in the server responding that the file is not a plain file, but a directory.
             */
            static fileIsDirectory: URLError.Code;
            /**
             * A redirect loop has been detected or the threshold for number of allowable redirects has been exceeded (currently 16).
             */
            static httpTooManyRedirects: URLError.Code;
            /**
             * The attempted connection required activating a data context while roaming, but international roaming is disabled.
             */
            static internationalRoamingOff: URLError.Code;
            /**
             * A client or server connection was severed in the middle of an in-progress load.
             */
            static networkConnectionLost: URLError.Code;
            /**
             * A resource couldn’t be read because of insufficient permissions.
             */
            static noPermissionsToReadFile: URLError.Code;
            /**
             * A network resource was requested, but an internet connection has not been established and cannot be established automatically.
             */
            static notConnectedToInternet: URLError.Code;
            /**
             * A redirect was specified by way of server response code, but the server did not accompany this code with a redirect URL.
             */
            static redirectToNonExistentLocation: URLError.Code;
            /**
             * A body stream is needed but the client did not provide one.  
             */
            static requestBodyStreamExhausted: URLError.Code;
            /**
             * A requested resource couldn’t be retrieved.
             */
            static resourceUnavailable: URLError.Code;
            /**
             * An attempt to establish a secure connection failed for reasons that can’t be expressed more specifically.
             */
            static secureConnectionFailed: URLError.Code;
            /**
             * A server certificate had a date which indicates it has expired, or is not yet valid.
             */
            static serverCertificateHasBadDate: URLError.Code;
            /**
             * A server certificate was not signed by any root server.
             */
            static serverCertificateHasUnknownRoot: URLError.Code;
            /**
             * A server certificate is not yet valid.
             */
            static serverCertificateNotYetValid: URLError.Code;
            /**
             * A server certificate was signed by a root server that isn’t trusted.
             */
            static serverCertificateUntrusted: URLError.Code;
            /**
             * An asynchronous operation timed out.
             */
            static timedOut: URLError.Code;
            /**
             * The URL Loading System encountered an error that it can’t interpret.  
             */
            static unknown: URLError.Code;
            /**
             * A properly formed URL couldn’t be handled by the framework.
             */
            static unsupportedURL: URLError.Code;
            /**
             * Authentication is required to access a resource.
             */
            static userAuthenticationRequired: URLError.Code;
            /**
             * An asynchronous request for authentication has been canceled by the user.
             */
            static userCancelledAuthentication: URLError.Code;
            /**
             * A server reported that a URL has a non-zero content length, but terminated the network connection gracefully without sending any data.
             */
            static zeroByteResource: URLError.Code;
            rawValue: number;
            hashValue: number;
            constructor (rawValue: number);
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
    }
    export class NSURLConnection extends NSObject {
        /**
         * A deep copy of the original connection request.
         */
        originalRequest: URLRequest;
        /**
         * The current connection request.
         */
        currentRequest: URLRequest;
        /**
         * The Newsstand download object associated with the URL connection.
         */
        newsstandAssetDownload: NKAssetDownload | null;
        /**
         * Returns an initialized URL connection and begins to load the data for the URL request.
         */
        constructor (request: URLRequest, delegate: any);
        /**
         * Returns an initialized URL connection and begins to load the data for the URL request, if specified.
         */
        constructor (request: URLRequest, delegate: any, startImmediately: boolean);
        /**
         * Returns whether a request can be handled based on a preflight evaluation.
         */
        static canHandle(request: URLRequest):boolean;
        /**
         * Performs a synchronous load of the specified URL request.
         */
        static sendSynchronousRequest(request: URLRequest, response: AutoreleasingUnsafeMutablePointer<URLResponse?> | null):Data;
        /**
         * Loads the data for a URL request and executes a handler block on an operation queue when the request completes or fails.
         */
        static sendAsynchronousRequest(request: URLRequest, queue: OperationQueue, handler: ()=>any):void;
        /**
         * Causes the connection to begin loading data, if it has not already.
         */
        start():void;
        /**
         * Cancels an asynchronous load of a request.
         */
        cancel():void;
        /**
         * Determines the run loop and mode that the connection uses to call methods on its delegate.
         */
        schedule(aRunLoop: RunLoop, mode: RunLoop.Mode):void;
        /**
         * Determines the operation queue that is used to call methods on the connection’s delegate.
         */
        setDelegateQueue(queue: OperationQueue | null):void;
        /**
         * Causes the connection to stop calling delegate methods in the specified run loop and mode.
         */
        unschedule(aRunLoop: RunLoop, mode: RunLoop.Mode):void;
    }
    export interface NSURLConnectionDelegate extends NSObjectProtocol {
        /**
         * Tells the delegate that the connection will send a request for an authentication challenge.
         */
        connection(connection: NSURLConnection, challenge: URLAuthenticationChallenge):void;
        /**
         * Sent to determine whether the URL loader should use the credential storage for authenticating the connection.
         */
        connectionShouldUseCredentialStorage(connection: NSURLConnection):boolean;
    }
    export interface NSURLConnectionDataDelegate extends NSURLConnectionDelegate {
        /**
         * Sent when the connection has received sufficient data to construct the URL response for its request.
         */
        connection(connection: NSURLConnection, response: URLResponse):void;
        /**
         * Sent when a connection has finished loading successfully.
         */
        connectionDidFinishLoading(connection: NSURLConnection):void;
    }
    export interface NSURLConnectionDownloadDelegate extends NSURLConnectionDelegate {
        /**
         * Sent to the delegate to deliver progress information for a download of a URL asset to a destination file.
         */
        connection(connection: NSURLConnection, bytesWritten: Int64, totalBytesWritten: Int64, expectedTotalBytes: Int64):void;
        /**
         * Sent to the delegate when an URL connection resumes downloading a URL asset that was earlier suspended. 
         */
        connectionDidResumeDownloading(connection: NSURLConnection, totalBytesWritten: Int64, expectedTotalBytes: Int64):void;
        /**
         * Sent to the delegate when the URL connection has successfully downloaded the URL asset to a destination file.
         */
        connectionDidFinishDownloading(connection: NSURLConnection, destinationURL: URL):void;
    }
    export class NSURLDownload extends NSObject {
        /**
         * Returns the resume data for a download that is not yet complete.
         */
        resumeData: Data | null;
        /**
         * Returns whether the receiver deletes partially downloaded files when a download stops prematurely.
         */
        deletesFileUponFailure: boolean;
        /**
         * Returns the request that initiated the receiver’s download.
         */
        request: URLRequest;
        /**
         * Returns an initialized URL download for a URL request and begins to download the data for the request.
         */
        constructor (request: URLRequest, delegate: NSURLDownloadDelegate | null);
        /**
         * Returns an initialized NSURLDownload object that will resume downloading the specified data to the specified file and begins the download.
         */
        constructor (resumeData: Data, delegate: NSURLDownloadDelegate | null, path: string);
        /**
         * Returns whether a URL download object can resume a download that was decoded with the specified MIME type.
         */
        static canResumeDownloadDecoded(MIMEType: string):boolean;
        /**
         * Sets the destination path of the downloaded file.
         */
        setDestination(path: string, allowOverwrite: boolean):void;
        /**
         * Cancels the receiver’s download and deletes the downloaded file.
         */
        cancel():void;
    }
    export interface NSURLDownloadDelegate extends NSObjectProtocol {
        /**
         * Sent to determine whether the delegate is able to respond to a protection space’s form of authentication.
         */
        download(connection: NSURLDownload, protectionSpace: URLProtectionSpace):boolean;
        /**
         * Sent to determine whether the URL loader should consult the credential storage to authenticate the download.
         */
        downloadShouldUseCredentialStorage(download: NSURLDownload):boolean;
        /**
         * Sent immediately after a download object begins a download.
         */
        downloadDidBegin(download: NSURLDownload):void;
        /**
         * Sent when a download object has completed downloading successfully and has written its results to disk.
         */
        downloadDidFinish(download: NSURLDownload):void;
    }
    export class NSURLHandle extends NSObject {}
    /**
     * the same name namespace of class NSURLHandle
     */
    export namespace NSURLHandle {
        export enum Status {
            notLoaded = 0,
            loadSucceeded = 1,
            loadInProgress = 2,
            loadFailed = 3,
        }
    }
    export interface URLAuthenticationChallengeSender extends NSObjectProtocol {
        /**
         * Cancels a given authentication challenge.
         */
        cancel(challenge: URLAuthenticationChallenge):void;
        /**
         * Attempt to continue downloading a request without providing a credential for a given challenge.
         */
        continueWithoutCredential(challenge: URLAuthenticationChallenge):void;
        /**
         * Attempt to use a given credential for a given authentication challenge.
         */
        use(credential: URLCredential, challenge: URLAuthenticationChallenge):void;
        /**
         * Causes the system-provided default behavior to be used.
         */
        performDefaultHandling(challenge: URLAuthenticationChallenge):void;
        /**
         * Rejects the currently supplied protection space.
         */
        rejectProtectionSpaceAndContinue(challenge: URLAuthenticationChallenge):void;
    }
    export class NetService extends NSObject {
        /**
         * This key identifies the error that occurred during the most recent operation. 
         */
        static errorCode: string;
        /**
         * This key identifies the originator of the error, which is either the 
         *  object or the mach network layer. For most errors, you should not need the value provided by this key.
         */
        static errorDomain: string;
        /**
         * A read-only array containing 
         *  objects, each of which contains a socket address for the service.
         */
        addresses: Data[] | null;
        /**
         * A string containing the domain for this service.
         */
        domain: string;
        /**
         * Specifies whether to also publish, resolve, or monitor this service over peer-to-peer Bluetooth and Wi-Fi, if available. 
         *  by default.
         */
        includesPeerToPeer: boolean;
        /**
         * A string containing the name of this service.
         */
        name: string;
        /**
         * The type of the published service.
         */
        type: string;
        /**
         * The delegate for the receiver.
         */
        delegate: NetServiceDelegate | null;
        /**
         * The port on which the service is listening for connections.
         */
        port: number;
        /**
         * A string containing the DNS hostname for this service.
         */
        hostName: string;
        /**
         * Returns the receiver, initialized as a network service of a given type and sets the initial host information.
         */
        constructor (domain: string, type: string, name: string);
        /**
         * Initializes the receiver for publishing a network service of type 
         *  at the socket location specified by 
         * , 
         * , and 
         * .
         */
        constructor (domain: string, type: string, name: string, port: Int32);
        /**
         * Returns an 
         *  object representing a TXT record formed from a given dictionary.
         */
        static data(txtDictionary: Record<string, Data>):Data;
        /**
         * Returns a dictionary representing a TXT record given as an 
         *  object.
         */
        static dictionary(txtData: Data):Record<string, Data>;
        /**
         * Creates a pair of input and output streams for the receiver and returns a Boolean value that indicates whether they were retrieved successfully.
         */
        getInputStream(inputStream: UnsafeMutablePointer<InputStream?> | null, outputStream: UnsafeMutablePointer<OutputStream?> | null):boolean;
        /**
         * Returns the TXT record for the receiver.
         */
        txtRecordData():Data | null;
        /**
         * Sets the TXT record for the receiver, and returns a Boolean value that indicates whether the operation was successful.
         */
        setTXTRecord(recordData: Data | null):boolean;
        /**
         * Adds the service to the specified run loop.
         */
        schedule(aRunLoop: RunLoop, mode: RunLoop.Mode):void;
        /**
         * Removes the service from the given run loop for a given mode.
         */
        remove(aRunLoop: RunLoop, mode: RunLoop.Mode):void;
        /**
         * Attempts to advertise the receiver’s on the network.
         */
        publish():void;
        /**
         * Starts a resolve process for the service.
         */
        resolve():void;
        /**
         * Starts the monitoring of TXT-record updates for the receiver.
         */
        startMonitoring():void;
        /**
         * Halts a currently running attempt to publish or resolve a service.
         */
        stop():void;
        /**
         * Stops the monitoring of TXT-record updates for the receiver.
         */
        stopMonitoring():void;
    }
    /**
     * the same name namespace of class NetService
     */
    export namespace NetService {
        export enum ErrorCode {
            unknownError = -72000,
            collisionError = -72001,
            notFoundError = -72002,
            activityInProgress = -72003,
            badArgumentError = -72004,
            cancelledError = -72005,
            invalidError = -72006,
            timeoutError = -72007,
            missingRequiredConfigurationError = -72008,
        }
        export class Options {
            /**
             * Specifies that the network service should not rename itself in the event of a name collision.
             */
            static noAutoRename: NetService.Options;
            static listenForConnections: NetService.Options;
            constructor (rawValue: number);
        }
    }
    export interface NetServiceDelegate extends NSObjectProtocol {
        /**
         * Notifies the delegate that the network is ready to publish the service.
         */
        netServiceWillPublish(sender: NetService):void;
        /**
         * Notifies the delegate that a service could not be published.
         */
        netService(sender: NetService, errorDict: Record<string, NSNumber>):void;
        /**
         * Notifies the delegate that a service was successfully published.
         */
        netServiceDidPublish(sender: NetService):void;
        /**
         * Notifies the delegate that the network is ready to resolve the service.
         */
        netServiceWillResolve(sender: NetService):void;
        /**
         * Informs the delegate that the address for a given service was resolved.
         */
        netServiceDidResolveAddress(sender: NetService):void;
        /**
         * Informs the delegate that a 
         *  or 
         *  request was stopped.
         */
        netServiceDidStop(sender: NetService):void;
    }
    export class NetServiceBrowser extends NSObject {
        /**
         * The delegate object for this instance.
         */
        delegate: NetServiceBrowserDelegate | null;
        /**
         * Whether to browse over peer-to-peer Bluetooth and Wi-Fi, if available. 
         * , by default.
         */
        includesPeerToPeer: boolean;
        /**
         * Initializes an allocated 
         *  object.
         */
        constructor ();
        /**
         * Initiates a search for domains visible to the host. This method returns immediately.
         */
        searchForBrowsableDomains():void;
        /**
         * Initiates a search for domains in which the host may register services.
         */
        searchForRegistrationDomains():void;
        /**
         * Starts a search for services of a particular type within a specific domain.
         */
        searchForServices(type: string, domainString: string):void;
        /**
         * Halts a currently running search or resolution.
         */
        stop():void;
        /**
         * Adds the receiver to the specified run loop.
         */
        schedule(aRunLoop: RunLoop, mode: RunLoop.Mode):void;
        /**
         * Removes the receiver from the specified run loop.
         */
        remove(aRunLoop: RunLoop, mode: RunLoop.Mode):void;
    }
    export interface NetServiceBrowserDelegate extends NSObjectProtocol {
        /**
         * Tells the delegate the sender found a domain.
         */
        netServiceBrowser(browser: NetServiceBrowser, domainString: string, moreComing: boolean):void;
        /**
         * Tells the delegate that a search is commencing.
         */
        netServiceBrowserWillSearch(browser: NetServiceBrowser):void;
        /**
         * Tells the delegate that a search was stopped.
         */
        netServiceBrowserDidStopSearch(browser: NetServiceBrowser):void;
    }
    export interface NSXPCProxyCreating {
        /**
         * Returns a proxy object with no error handling block.
         */
        remoteObjectProxy():any;
        /**
         * Returns a proxy object that invokes the error handling block if an error occurs on the connection.
         */
        remoteObjectProxyWithErrorHandler(handler: ()=>any):any;
        synchronousRemoteObjectProxyWithErrorHandler(handler: ()=>any):any;
    }
    export class NSXPCConnection extends NSObject {
        /**
         * An interruption handler that is called if the remote process exits or crashes.
         */
        interruptionHandler: ()=>any | null;
        /**
         * An invalidation handler that is called if the connection can not be formed or the connection has terminated and may not be re-established.
         */
        invalidationHandler: ()=>any | null;
        /**
         * The name of the XPC service that this connection was configured to connect to.
         */
        serviceName: string;
        /**
         * If the connection was created with an 
         *  object, returns the endpoint object used.
         */
        endpoint: NSXPCListenerEndpoint;
        /**
         * The 
         *  object that describes the protocol for the exported object on this connection.
         */
        exportedInterface: NSXPCInterface | null;
        /**
         * An exported object for the connection.
         */
        exportedObject: any;
        /**
         * Defines the 
         *  object that describes the protocol for the object represented by the 
         * . 
         */
        remoteObjectInterface: NSXPCInterface | null;
        /**
         * Returns a proxy for the remote object (that is, the 
         *  from the other side of this connection).
         */
        remoteObjectProxy: any;
        /**
         * The BSM audit session identifier for the connecting process.
         */
        auditSessionIdentifier: au_asid_t;
        /**
         * The process ID (PID) of the connecting process.
         */
        processIdentifier: pid_t;
        /**
         * The effective group ID (EGID) of the connecting process.
         */
        effectiveGroupIdentifier: gid_t;
        /**
         * The effective user ID (EUID) of the connecting process.
         */
        effectiveUserIdentifier: uid_t;
        /**
         * Initializes an 
         *  object to connect to an 
         *  object in another process, identified by an 
         *  object.
         */
        constructor (endpoint: NSXPCListenerEndpoint);
        /**
         * Initializes an 
         *  object to connect to a LaunchAgent or LaunchDaemon with a name advertised in a 
         * .
         */
        constructor (name: string, options: NSXPCConnection.Options);
        /**
         * Initializes an 
         *  object to connect to an 
         *  object in an XPC service, identified by a service name.
         */
        constructor (serviceName: string);
        /**
         * Returns the current connection, in the context of a call to a method on your exported object.
         */
        static current():NSXPCConnection | null;
        /**
         * Activates the connection.
         */
        activate():void;
        /**
         * Starts or resumes handling of messages on a connection.
         */
        resume():void;
        /**
         * Invalidates the connection.
         */
        invalidate():void;
        /**
         * Suspends the connection.
         */
        suspend():void;
        /**
         * Add a barrier block to execute on the connection.
         */
        scheduleSendBarrierBlock(block: ()=>any):void;
        /**
         * Returns a proxy for the remote object (that is, the object exported from the other side of this connection) with the specified error handler.
         */
        remoteObjectProxyWithErrorHandler(handler: ()=>any):any;
        synchronousRemoteObjectProxyWithErrorHandler(handler: ()=>any):any;
        /**
         * Sets the code signing requirement for this connection.
         */
        setCodeSigningRequirement(requirement: string):void;
    }
    /**
     * the same name namespace of class NSXPCConnection
     */
    export namespace NSXPCConnection {
        export class Options {
            static privileged: NSXPCConnection.Options;
            constructor (rawValue: number);
        }
    }
    export class NSXPCInterface extends NSObject {
        /**
         * The Objective-C protocol that this interface is based on.
         */
        protocol: Protocol;
        /**
         * Returns an NSXPCInterface instance for a given protocol.
         */
        constructor (protocol: Protocol);
        /**
         * Returns the current list of allowed classes that can appear within the specified collection object argument to the specified method.
         */
        classes(sel: Selector, arg: number, ofReply: boolean):Set<AnyHashable>;
        /**
         * Returns the interface previously set for the specified selector and parameter.
         */
        forSelector(sel: Selector, arg: number, ofReply: boolean):NSXPCInterface | null;
        /**
         * Sets the classes that can appear within the (numerically) specified collection object argument to the specified method.
         */
        setClasses(classes: Set<AnyHashable>, sel: Selector, arg: number, ofReply: boolean):void;
        /**
         * Configures a specific parameter of a method to be sent as a proxy object instead of copied.
         */
        setInterface(ifc: NSXPCInterface, sel: Selector, arg: number, ofReply: boolean):void;
        setXPCType(type: xpc_type_t, sel: Selector, arg: number, ofReply: boolean):void;
        xpcType(sel: Selector, arg: number, ofReply: boolean):xpc_type_t | null;
    }
    export class NSXPCCoder extends NSCoder {
        /**
         * The connection currently performing encoding or decoding.
         */
        connection: NSXPCConnection | null;
        /**
         * An optional user information object associated with the coder.
         */
        userInfo: NSObjectProtocol | null;
        /**
         * Encodes an object to send over an XPC connection.
         */
        encodeXPCObject(xpcObject: xpc_object_t, key: string):void;
        /**
         * Decodes an object and validates that its type matches the type a service provides over XPC.
         */
        decodeXPCObject(type: xpc_type_t, key: string):xpc_object_t | null;
    }
    export class NSXPCListener extends NSObject {
        /**
         * The delegate for the listener.
         */
        delegate: NSXPCListenerDelegate | null;
        /**
         * Returns an endpoint object that may be sent over an existing connection.
         */
        endpoint: NSXPCListenerEndpoint;
        /**
         * Initializes a listener in a LaunchAgent or LaunchDaemon which has a name advertised in a 
         *  file.
         */
        constructor (name: string);
        /**
         * Returns the singleton listener used to listen for incoming connections in an XPC service.
         */
        static service():NSXPCListener;
        /**
         * Returns a new anonymous listener connection.
         */
        static anonymous():NSXPCListener;
        /**
         * Activates the listener.
         */
        activate():void;
        /**
         * Starts processing of incoming requests.
         */
        resume():void;
        /**
         * Invalidates the listener.
         */
        invalidate():void;
        /**
         * Suspends the listener.
         */
        suspend():void;
        /**
         * Sets the code signing requirement for connections to this listener.
         */
        setConnectionCodeSigningRequirement(requirement: string):void;
    }
    export interface NSXPCListenerDelegate extends NSObjectProtocol {
        /**
         * Accepts or rejects a new connection to the listener.
         */
        listener(listener: NSXPCListener, newConnection: NSXPCConnection):boolean;
    }
    export class NSXPCListenerEndpoint extends NSObject {}
    export class NSAttributedString extends NSObject {}
    /**
     * the same name namespace of class NSAttributedString
     */
    export namespace NSAttributedString {
        export class Key {
            /**
             * A key that indicates whether to speak punctuation. 
             */
            static accessibilitySpeechPunctuation: NSAttributedString.Key;
            /**
             * A key that indicates the language to use when speaking a string. 
             */
            static accessibilitySpeechLanguage: NSAttributedString.Key;
            /**
             * A key that indicates the pitch to apply to spoken content. 
             */
            static accessibilitySpeechPitch: NSAttributedString.Key;
            /**
             * A key that indicates whether to queue an announcement behind existing speech or to interrupt it.
             */
            static accessibilitySpeechQueueAnnouncement: NSAttributedString.Key;
            /**
             * A key that indicates the pronunciation of a specific word or phrase, such as a proper name.
             */
            static accessibilitySpeechIPANotation: NSAttributedString.Key;
            /**
             * A key for specifying custom attributes to apply to the text.
             */
            static accessibilityTextCustom: NSAttributedString.Key;
            /**
             * A key for specifying the heading level of the text.
             */
            static accessibilityTextHeadingLevel: NSAttributedString.Key;
        }
    }
    export class NSKeyValueOperator {
        /**
         * The 
         *  array operator.
         */
        static averageKeyValueOperator: NSKeyValueOperator;
        /**
         * The 
         *  array operator.
         */
        static countKeyValueOperator: NSKeyValueOperator;
        /**
         * The 
         *  array operator.
         */
        static distinctUnionOfArraysKeyValueOperator: NSKeyValueOperator;
        /**
         * The 
         *  array operator.
         */
        static distinctUnionOfObjectsKeyValueOperator: NSKeyValueOperator;
        /**
         * The 
         *  array operator.
         */
        static distinctUnionOfSetsKeyValueOperator: NSKeyValueOperator;
        /**
         * The 
         *  array operator.
         */
        static maximumKeyValueOperator: NSKeyValueOperator;
        /**
         * The 
         *  array operator.
         */
        static minimumKeyValueOperator: NSKeyValueOperator;
        /**
         * The 
         *  array operator.
         */
        static sumKeyValueOperator: NSKeyValueOperator;
        /**
         * The 
         *  array operator.
         */
        static unionOfArraysKeyValueOperator: NSKeyValueOperator;
        /**
         * The 
         *  array operator.
         */
        static unionOfObjectsKeyValueOperator: NSKeyValueOperator;
        /**
         * The 
         *  array operator.
         */
        static unionOfSetsKeyValueOperator: NSKeyValueOperator;
        constructor (rawValue: string);
    }
    export interface NSCopying {
        /**
         * Returns a new instance that’s a copy of the receiver.
         */
        copy(zone: NSZone | null):any;
    }
    export interface NSMutableCopying {
        /**
         * Returns a new instance that’s a mutable copy of the receiver.
         */
        mutableCopy(zone: NSZone | null):any;
    }
    export class NSNumber extends NSValue {
        /**
         * The number object's value expressed as a Boolean value.
         */
        boolValue: boolean;
        /**
         * The number object's value expressed as a 
         * .
         */
        int8Value: CChar;
        /**
         * The number object's value expressed as an 
         *  structure.
         */
        decimalValue: Decimal;
        /**
         * The number object's value expressed as a 
         * , converted as necessary.
         */
        doubleValue: number;
        /**
         * The number object's value expressed as a 
         * , converted as necessary.
         */
        floatValue: Float;
        /**
         * The number object's value expressed as an 
         * , converted as necessary.
         */
        int32Value: Int32;
        /**
         * The number object's value expressed as an 
         *  object, converted as necessary.
         */
        intValue: number;
        /**
         * The number object’s value expressed as a 
         * , converted as necessary.
         */
        int64Value: Int64;
        /**
         * The number object's value expressed as a 
         * , converted as necessary.
         */
        int16Value: Int16;
        /**
         * The number object's value expressed as an unsigned 
         * , converted as necessary.
         */
        uint8Value: UInt8;
        /**
         * The number object's value expressed as an 
         *  object, converted as necessary.
         */
        uintValue: number;
        /**
         * The number object's value expressed as an unsigned 
         * , converted as necessary.
         */
        uint32Value: UInt32;
        /**
         * The number object’s value expressed as an unsigned 
         * , converted as necessary.
         */
        uint64Value: UInt64;
        /**
         * The number object's value expressed as an unsigned 
         * , converted as necessary.
         */
        uint16Value: UInt16;
        /**
         * The number object's value expressed as a human-readable string.
         */
        stringValue: string;
        /**
         * Returns an 
         *  object initialized to contain a given value, treated as a 
         * .
         */
        constructor (value: boolean);
        /**
         * Create an instance initialized to 
         * .
         */
        constructor (value: boolean);
        constructor (coder: NSCoder);
        /**
         * Create an instance initialized to 
         * .
         */
        constructor (value: number);
        /**
         * Create an instance initialized to 
         * .
         */
        constructor (value: number);
        /**
         * Returns a string that represents the contents of the number object for a given locale.
         */
        description(locale: any):string;
        /**
         * Returns an 
         *  value that indicates whether the number object’s value is greater than, equal to, or less than a given number.
         */
        compare(otherNumber: NSNumber):ComparisonResult;
        /**
         * Returns a Boolean value that indicates whether the number object’s value and a given number are equal. 
         */
        isEqual(_number: NSNumber):boolean;
    }
    export class NSValue extends NSObject {
        /**
         * A C string containing the Objective-C type of the data contained in the value object.
         */
        objCType: UnsafePointer<CChar>;
        /**
         * Returns the value as an untyped pointer.
         */
        pointerValue: UnsafeMutableRawPointer | null;
        /**
         * The value as a non-retained pointer to an object.
         */
        nonretainedObjectValue: any;
        /**
         * The Foundation range structure representation of the value.
         */
        rangeValue: NSRange;
        /**
         * The Foundation point structure representation of the value.
         */
        pointValue: NSPoint;
        /**
         * The Foundation size structure representation of the value.
         */
        sizeValue: NSSize;
        /**
         * The Foundation rectangle structure representation of the value.
         */
        rectValue: NSRect;
        /**
         * Returns the CoreGraphics point structure representation of the value.
         */
        cgPointValue: CGPoint;
        /**
         * Returns the CoreGraphics vector structure representation of the value.
         */
        cgVectorValue: CGVector;
        /**
         * Returns the CoreGraphics size structure representation of the value.
         */
        cgSizeValue: CGSize;
        /**
         * Returns the CoreGraphics rectangle structure representation of the value.
         */
        cgRectValue: CGRect;
        /**
         * Returns the CoreGraphics affine transform representation of the value.
         */
        cgAffineTransformValue: CGAffineTransform;
        /**
         * Returns the UIKit edge insets structure representation of the value.
         */
        uiEdgeInsetsValue: UIEdgeInsets;
        /**
         * Returns the UIKit offset structure representation of the value.
         */
        uiOffsetValue: UIOffset;
        /**
         * The CoreAnimation transform structure representation of the value.
         */
        caTransform3DValue: CATransform3D;
        /**
         * The CoreMedia time structure representation of the value.
         */
        timeValue: CMTime;
        /**
         * The CoreMedia time range structure representation of the value.
         */
        timeRangeValue: CMTimeRange;
        /**
         * The CoreMedia time mapping structure representation of the value.
         */
        timeMappingValue: CMTimeMapping;
        /**
         * The CoreLocation geographic coordinate structure representation of the value.
         */
        mkCoordinateValue: CLLocationCoordinate2D;
        /**
         * The MapKit coordinate span structure representation of the value.
         */
        mkCoordinateSpanValue: MKCoordinateSpan;
        /**
         * The three-element Scene Kit vector representation of the value.
         */
        scnVector3Value: SCNVector3;
        /**
         * The four-element Scene Kit vector representation of the value.
         */
        scnVector4Value: SCNVector4;
        /**
         * The Scene Kit 4 x 4 matrix representation of the value.
         */
        scnMatrix4Value: SCNMatrix4;
        directionalEdgeInsetsValue: NSDirectionalEdgeInsets;
        edgeInsetsValue: NSEdgeInsets;
        videoDimensionsValue: CMVideoDimensions;
        /**
         * Initializes a value object to contain the specified value, interpreted with the specified Objective-C type.
         */
        constructor (value: UnsafeRawPointer, type: UnsafePointer<CChar>);
        /**
         * Creates a value object containing the specified value, interpreted with the specified Objective-C type.
         */
        constructor (value: UnsafeRawPointer, type: UnsafePointer<CChar>);
        /**
         * Creates a value object containing the specified pointer.
         */
        constructor (pointer: UnsafeRawPointer | null);
        /**
         * Creates a value object containing the specified object.
         */
        constructor (anObject: any);
        /**
         * Creates a new value object containing the specified Foundation range structure.
         */
        constructor (range: NSRange);
        /**
         * Creates a new value object containing the specified Foundation point structure.
         */
        constructor (point: NSPoint);
        /**
         * Creates a new value object containing the specified Foundation size structure.
         */
        constructor (size: NSSize);
        /**
         * Creates a new value object containing the specified Foundation rectangle structure.
         */
        constructor (rect: NSRect);
        /**
         * Creates a new value object containing the specified CoreGraphics point structure.
         */
        constructor (point: CGPoint);
        /**
         * Creates a new value object containing the specified CoreGraphics vector structure.
         */
        constructor (vector: CGVector);
        /**
         * Creates a new value object containing the specified CoreGraphics size structure.
         */
        constructor (size: CGSize);
        /**
         * Creates a new value object containing the specified CoreGraphics rectangle structure.
         */
        constructor (rect: CGRect);
        /**
         * Creates a new value object containing the specified CoreGraphics affine transform structure.
         */
        constructor (transform: CGAffineTransform);
        /**
         * Creates a new value object containing the specified UIKit edge insets structure.
         */
        constructor (insets: UIEdgeInsets);
        /**
         * Creates a new value object containing the specified UIKit offset structure.
         */
        constructor (insets: UIOffset);
        /**
         * Creates a new value object containing the specified CoreAnimation transform structure.
         */
        constructor (t: CATransform3D);
        /**
         * Creates a new value object containing the specified CoreMedia time structure.
         */
        constructor (time: CMTime);
        /**
         * Creates a new value object containing the specified CoreMedia time range structure.
         */
        constructor (timeRange: CMTimeRange);
        /**
         * Creates a new value object containing the specified CoreMedia time mapping structure.
         */
        constructor (timeMapping: CMTimeMapping);
        /**
         * Creates a new value object containing the specified CoreLocation geographic coordinate structure.
         */
        constructor (coordinate: CLLocationCoordinate2D);
        /**
         * Creates a new value object containing the specified MapKit coordinate span structure.
         */
        constructor (span: MKCoordinateSpan);
        /**
         * Creates a value object that contains the specified three-element SceneKit vector.
         */
        constructor (v: SCNVector3);
        /**
         * Creates a value object that contains the specified four-element SceneKit vector.
         */
        constructor (v: SCNVector4);
        /**
         * Creates a value object that contains the specified SceneKit 4 x 4 matrix.
         */
        constructor (v: SCNMatrix4);
        constructor (coder: NSCoder);
        constructor (insets: NSDirectionalEdgeInsets);
        constructor (insets: NSEdgeInsets);
        constructor (dimensions: CMVideoDimensions);
        /**
         * Copies the value into the specified buffer.
         */
        getValue(value: UnsafeMutableRawPointer):void;
        /**
         * Returns a Boolean value that indicates whether the value object and another value object are equal.
         */
        isEqual(value: NSValue):boolean;
    }
    export class ValueTransformer extends NSObject {
        /**
         * Returns the value transformer identified by a given identifier.
         */
        constructor (name: NSValueTransformerName);
        /**
         * Registers the provided value transformer with a given identifier.
         */
        static setValueTransformer(transformer: ValueTransformer | null, name: NSValueTransformerName):void;
        /**
         * Returns an array of all the registered value transformers.
         */
        static valueTransformerNames():NSValueTransformerName[];
        /**
         * Returns a Boolean value that indicates whether the receiver can reverse a transformation.
         */
        static allowsReverseTransformation():boolean;
        /**
         * Returns the class of the value returned by the receiver for a forward transformation.
         */
        static transformedValueClass():AnyClass;
        /**
         * Returns the result of transforming a given value. 
         */
        transformedValue(value: any):any;
        /**
         * Returns the result of the reverse transformation of a given value. 
         */
        reverseTransformedValue(value: any):any;
    }
    export class NSValueTransformerName {
        /**
         * This value transformer returns 
         *  if the value is 
         * . 
         */
        static isNilTransformerName: NSValueTransformerName;
        /**
         * This value transformer returns 
         *  if the value is non-
         * .
         */
        static isNotNilTransformerName: NSValueTransformerName;
        /**
         * The name of the value transformer that attempts to unarchive data stored inside a keyed archive in an object you provide.
         */
        static keyedUnarchiveFromDataTransformerName: NSValueTransformerName;
        /**
         * This value transformer negates a boolean value, transforming 
         *  to 
         *  and 
         *  to 
         * .
         */
        static negateBooleanTransformerName: NSValueTransformerName;
        /**
         * The name of the value transformer that attempts to unarchive data from an object you provide.
         */
        static unarchiveFromDataTransformerName: NSValueTransformerName;
        /**
         * The name of the value transformer that creates then returns an object by attempting to unarchive the data to a class that supports secure coding.
         */
        static secureUnarchiveFromDataTransformerName: NSValueTransformerName;
        constructor (rawValue: string);
        constructor (rawValue: string);
    }
    export interface ReferenceConvertible extends CustomDebugStringConvertible,CustomStringConvertible,Hashable {}
    export class NSProxy {
        /**
         * A string containing the real class name and the id of the receiver as a hexadecimal number.
         */
        description: string;
        /**
         * A string containing the real class name and the id of the receiver as a hexadecimal number.
         */
        debugDescription: string;
        /**
         * Returns a new instance of the receiving class 
         */
        static alloc():Self;
        /**
         * Returns a Boolean value that indicates whether the receiving class responds to a given selector.
         */
        static responds(aSelector: Selector):boolean;
        /**
         * Returns 
         *  (the class object).
         */
        static class():AnyClass;
        /**
         * The garbage collector invokes this method on the receiver before disposing of the memory it uses.
         */
        finalize():void;
        /**
         * Passes a given invocation to the real object the proxy represents.
         */
        forwardInvocation(invocation: NSInvocation):void;
        /**
         * Deallocates the memory occupied by the receiver.
         */
        dealloc():void;
    }
    export interface NSMachPortDelegate extends PortDelegate {
        /**
         * Process an incoming Mach message.
         */
        handleMachMessage(msg: UnsafeMutableRawPointer):void;
    }
    export class NSMachPort extends Port {
        /**
         * The Mach port used by the receiver, represented as an integer.
         */
        machPort: UInt32;
        /**
         * Initializes a newly allocated 
         *  object with a given Mach port.
         */
        constructor (machPort: UInt32);
        /**
         * Initializes a newly allocated 
         *  object with a given Mach port and the specified options. 
         */
        constructor (machPort: UInt32, f: NSMachPort.Options);
        /**
         * Creates and returns a port object configured with the given Mach port.
         */
        static port(machPort: UInt32):Port;
        /**
         * Removes the receiver from the run loop mode 
         *  of 
         * .
         */
        remove(runLoop: RunLoop, mode: RunLoop.Mode):void;
        /**
         * Schedules the receiver into the run loop mode 
         *  of 
         * .
         */
        schedule(runLoop: RunLoop, mode: RunLoop.Mode):void;
        /**
         * Returns the receiver’s delegate.
         */
        delegate():NSMachPortDelegate | null;
        /**
         * Sets the receiver’s delegate to a given object.
         */
        setDelegate(anObject: NSMachPortDelegate | null):void;
    }
    /**
     * the same name namespace of class NSMachPort
     */
    export namespace NSMachPort {
        export class Options {
            /**
             * Remove a receive right when the 
             *  object is invalidated or destroyed.
             */
            static deallocateReceiveRight: NSMachPort.Options;
            /**
             * Deallocate a send right when the 
             *  object is invalidated or destroyed.
             */
            static deallocateSendRight: NSMachPort.Options;
            constructor (rawValue: number);
        }
    }
    export class MessagePort extends Port {}
    export interface PortDelegate extends NSObjectProtocol {
        /**
         * Processes a given incoming message on the port.
         */
        handle(message: NSPortMessage):void;
    }
    export class PortMessage extends NSObject {
        /**
         * Returns the data components of the receiver.
         */
        components: any[] | null;
        /**
         * For an outgoing message, returns the port on which replies to the receiver will arrive. For an incoming message, returns the port the receiver did arrive on.
         */
        receivePort: Port | null;
        /**
         * For an outgoing message, returns the port the receiver will send itself through. For an incoming message, returns the port replies to the receiver should be sent through.
         */
        sendPort: Port | null;
        /**
         * Returns the identifier for the receiver.
         */
        msgid: UInt32;
        /**
         * Initializes a newly allocated 
         *  object to send given data on a given port and to receiver replies on another given port.
         */
        constructor (sendPort: Port | null, replyPort: Port | null, components: any[] | null);
        /**
         * Attempts to send the message before 
         * , returning 
         *  if successful or 
         *  if the operation times out.
         */
        send(date: Date):boolean;
    }
    export class NSProtocolChecker extends NSProxy {
        /**
         * Returns the protocol object the receiver uses.
         */
        protocol: Protocol;
        /**
         * Returns the target of the receiver.
         */
        target: NSObject | null;
        /**
         * Initializes a newly allocated 
         *  instance that will forward any messages in 
         *  to 
         * , the protocol checker’s target.
         */
        constructor (anObject: NSObject, aProtocol: Protocol);
    }
    export class RunLoop extends NSObject {
        /**
         * Returns the run loop for the current thread.
         */
        static current: RunLoop;
        /**
         * Returns the run loop of the main thread.
         */
        static main: RunLoop;
        /**
         * The receiver's current input mode.
         */
        currentMode: RunLoop.Mode | null;
        /**
         * The minimum tolerance the run loop scheduler allows.
         */
        minimumTolerance: RunLoop.SchedulerTimeType.Stride;
        /**
         * The run loop scheduler’s definition of the current moment in time.
         */
        now: RunLoop.SchedulerTimeType;
        /**
         * Performs one pass through the run loop in the specified mode and returns the date at which the next timer is scheduled to fire.
         */
        limitDate(mode: RunLoop.Mode):Date | null;
        /**
         * Returns the receiver's underlying 
         *  object.
         */
        getCFRunLoop():CFRunLoop;
        /**
         * Registers a given timer with a given input mode.
         */
        add(timer: Timer, mode: RunLoop.Mode):void;
        /**
         * Removes a port from the specified input mode of the run loop.
         */
        remove(aPort: Port, mode: RunLoop.Mode):void;
        /**
         * Puts the receiver into a permanent loop, during which time it processes data from all attached input sources.
         */
        run():void;
        /**
         * Runs the loop once or until the specified date, accepting input only for the specified mode.
         */
        acceptInput(mode: RunLoop.Mode, limitDate: Date):void;
        /**
         * Schedules a block that the run loop invokes.
         */
        perform(block: ()=>any):void;
        /**
         * Cancels the sending of a previously scheduled message.
         */
        cancelPerform(aSelector: Selector, target: any, arg: any):void;
        /**
         * Cancels all outstanding ordered performs scheduled with a given target.
         */
        cancelPerformSelectors(target: any):void;
        /**
         * Performs the action at some time after the specified date, using the scheduler’s minimum tolerance.
         */
        schedule(options: RunLoop.SchedulerOptions | null, action: ()=>any):void;
    }
    /**
     * the same name namespace of class RunLoop
     */
    export namespace RunLoop {
        export class Mode {
            /**
             * A pseudo-mode that includes one or more other run loop modes.
             */
            static common: RunLoop.Mode;
            /**
             * The mode set to handle input sources other than connection objects.
             */
            static default: RunLoop.Mode;
            /**
             * The mode set when tracking events modally, such as a mouse-dragging loop.
             */
            static eventTracking: RunLoop.Mode;
            /**
             * The mode set when waiting for input from a modal panel, such as a save or open panel.
             */
            static modalPanel: RunLoop.Mode;
            /**
             * The mode set while tracking in controls takes place.
             */
            static tracking: RunLoop.Mode;
            /**
             * Creates a run loop mode using the specified string value.
             */
            constructor (rawValue: string);
            /**
             * Creates a run loop mode using the specified raw string value.
             */
            constructor (rawValue: string);
        }
        export class SchedulerTimeType {
            /**
             * The date this type represents.
             */
            date: Date;
            /**
             * The computed hash value of the run loop scheduler time.
             */
            hashValue: number;
            /**
             * Initializes a run loop scheduler time with the given date.
             */
            constructor (date: Date);
            /**
             * Creates a run loop scheduler time with data from an unarchiver.
             */
            constructor (decoder: Decoder);
            /**
             * Returns a run loop scheduler time calculated by advancing this instance’s time by the given interval.
             */
            advanced(n: RunLoop.SchedulerTimeType.Stride):RunLoop.SchedulerTimeType;
            /**
             * Returns the distance to another run loop scheduler time.
             */
            distance(other: RunLoop.SchedulerTimeType):RunLoop.SchedulerTimeType.Stride;
            /**
             * Hashes the essential components of the run loop schedule time by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
            /**
             * Encodes the run loop schedule time into the given encoder.
             */
            encode(encoder: Encoder):void;
        }
        /**
         * the same name namespace of class SchedulerTimeType
         */
        export namespace SchedulerTimeType {
            export class Stride {
                /**
                 * The zero value.
                 */
                static zero: RunLoop.SchedulerTimeType.Stride;
                /**
                 * The value of this time interval in seconds.
                 */
                magnitude: TimeInterval;
                /**
                 * The value of this time interval in seconds.
                 */
                timeInterval: TimeInterval;
                /**
                 * Creates a run loop scheduler time interval from the given time interval.
                 */
                constructor (timeInterval: TimeInterval);
                /**
                 * Creates a run loop scheduler time interval from a floating-point seconds value.
                 */
                constructor (value: TimeInterval);
                /**
                 * Creates a run loop scheduler time interval from an integer seconds value.
                 */
                constructor (value: TimeInterval);
                /**
                 * Creates a run loop scheduler time interval with data from an unarchiver.
                 */
                constructor (decoder: Decoder);
                /**
                 * Converts the specified number of microseconds into an instance of this scheduler time type.
                 */
                static microseconds(us: number):RunLoop.SchedulerTimeType.Stride;
                /**
                 * Converts the specified number of milliseconds into an instance of this scheduler time type.
                 */
                static milliseconds(ms: number):RunLoop.SchedulerTimeType.Stride;
                /**
                 * Converts the specified number of nanoseconds into an instance of this scheduler time type.
                 */
                static nanoseconds(ns: number):RunLoop.SchedulerTimeType.Stride;
                /**
                 * Converts the specified number of seconds, as a floating-point value, into an instance of this scheduler time type.
                 */
                static seconds(s: number):RunLoop.SchedulerTimeType.Stride;
                /**
                 * Replaces this value with its additive inverse.
                 */
                negate():void;
                /**
                 * Encodes the run loop schedule time interval into the given encoder.
                 */
                encode(encoder: Encoder):void;
            }
        }
        export class SchedulerOptions {}
    }
    export class Thread extends NSObject {
        /**
         * Returns a Boolean value that indicates whether the current thread is the main thread.
         */
        static isMainThread: boolean;
        /**
         * Returns the thread object representing the current thread of execution.
         */
        static current: Thread;
        /**
         * Returns an array containing the call stack return addresses.
         */
        static callStackReturnAddresses: NSNumber[];
        /**
         * Returns an array containing the call stack symbols.
         */
        static callStackSymbols: string[];
        /**
         * A Boolean value that indicates whether the receiver is executing.
         */
        isExecuting: boolean;
        /**
         * A Boolean value that indicates whether the receiver has finished execution.
         */
        isFinished: boolean;
        /**
         * A Boolean value that indicates whether the receiver is cancelled.
         */
        isCancelled: boolean;
        /**
         * The thread object's dictionary.
         */
        threadDictionary: NSMutableDictionary;
        /**
         * The name of the receiver.
         */
        name: string;
        /**
         * The stack size of the receiver, in bytes.
         */
        stackSize: number;
        qualityOfService: QualityOfService;
        /**
         * Returns an initialized 
         *  object.
         */
        constructor ();
        /**
         * Returns an 
         *  object initialized with the given arguments.
         */
        constructor (target: any, selector: Selector, argument: any);
        constructor (block: ()=>any);
        /**
         * Detaches a new thread and uses the specified selector as the thread entry point.
         */
        static detachNewThreadSelector(selector: Selector, target: any, argument: any):void;
        /**
         * Blocks the current thread until the time specified.
         */
        static sleep(date: Date):void;
        /**
         * Terminates the current thread.
         */
        static exit():void;
        /**
         * Returns whether the application is multithreaded.
         */
        static isMultiThreaded():boolean;
        /**
         * Returns the current thread’s priority.
         */
        static threadPriority():number;
        /**
         * Sets the current thread’s priority.
         */
        static setThreadPriority(p: number):boolean;
        static detachNewThread(block: ()=>any):void;
        /**
         * Starts the receiver.
         */
        start():void;
        /**
         * The main entry point routine for the thread. 
         */
        main():void;
        /**
         * Changes the cancelled state of the receiver to indicate that it should exit.
         */
        cancel():void;
    }
    export interface NSLocking {
        /**
         * Attempts to acquire a lock, blocking a thread’s execution until the lock can be acquired.
         */
        lock():void;
        /**
         * Relinquishes a previously acquired lock.
         */
        unlock():void;
    }
    export class NSLock extends NSObject {
        /**
         * The name associated with the receiver.
         */
        name: string;
        /**
         * Attempts to acquire a lock before a given time and returns a Boolean value indicating whether the attempt was successful.
         */
        lock(limit: Date):boolean;
        /**
         * Attempts to acquire a lock and immediately returns a Boolean value that indicates whether the attempt was successful.
         */
        try():boolean;
    }
    export class NSRecursiveLock extends NSObject {
        /**
         * The name associated with the receiver. 
         */
        name: string;
        /**
         * Attempts to acquire a lock before a given date.
         */
        lock(limit: Date):boolean;
        /**
         * Attempts to acquire a lock, and immediately returns a Boolean value that indicates whether the attempt was successful.
         */
        try():boolean;
    }
    export class NSDistributedLock extends NSObject {
        /**
         * Returns the time the receiver was acquired by any of the 
         *  objects using the same path.
         */
        lockDate: Date;
        /**
         * Initializes an 
         *  object to use as the lock the file-system entry specified by a given path.
         */
        constructor (path: string);
        /**
         * Attempts to acquire the receiver and immediately returns a Boolean value that indicates whether the attempt was successful.
         */
        try():boolean;
        /**
         * Forces the lock to be relinquished.
         */
        break():void;
        /**
         * Relinquishes the receiver.
         */
        unlock():void;
    }
    export class NSConditionLock extends NSObject {
        /**
         * The condition associated with the receiver.
         */
        condition: number;
        /**
         * The name associated with the receiver.
         */
        name: string;
        /**
         * Initializes a newly allocated 
         *  object and sets its condition.
         */
        constructor (condition: number);
        /**
         * Attempts to acquire a lock before a specified moment in time.
         */
        lock(limit: Date):boolean;
        /**
         * Attempts to acquire a lock without regard to the receiver’s condition.
         */
        try():boolean;
        /**
         * Attempts to acquire a lock if the receiver’s condition is equal to the specified condition.
         */
        tryLock(condition: number):boolean;
        /**
         * Relinquishes the lock and sets the receiver’s condition.
         */
        unlock(condition: number):void;
    }
    export class NSCondition extends NSObject {
        /**
         * The name of the condition.
         */
        name: string;
        /**
         * Blocks the current thread until the condition is signaled.
         */
        wait():void;
        /**
         * Signals the condition, waking up one thread waiting on it.
         */
        signal():void;
        /**
         * Signals the condition, waking up all threads waiting on it.
         */
        broadcast():void;
    }
    export class Process extends NSObject {
        /**
         * Posted when the task has stopped execution.
         */
        static didTerminateNotification: NSNotification.Name;
        /**
         * The receiver’s process identifier.
         */
        processIdentifier: Int32;
        /**
         * A status that indicates whether the receiver is still running.
         */
        isRunning: boolean;
        /**
         * The exit status the receiver’s executable returns.
         */
        terminationStatus: Int32;
        /**
         * The reason the system terminated the task.
         */
        terminationReason: Process.TerminationReason;
        /**
         * The command arguments that the system uses to launch the executable.
         */
        arguments: string[] | null;
        /**
         * The current directory for the receiver.
         */
        currentDirectoryURL: URL | null;
        /**
         * The environment for the receiver.
         */
        environment: Record<string, string> | null;
        /**
         * The receiver’s executable.
         */
        executableURL: URL | null;
        /**
         * The default quality of service level the system applies to operations the task executes.
         */
        qualityOfService: QualityOfService;
        /**
         * The standard error for the receiver.
         */
        standardError: any;
        /**
         * The standard input for the receiver.
         */
        standardInput: any;
        /**
         * The standard output for the receiver.
         */
        standardOutput: any;
        /**
         * A completion block the system invokes when the task completes.
         */
        terminationHandler: ()=>any | null;
        /**
         * Sets the current directory for the receiver.
         */
        currentDirectoryPath: string;
        /**
         * Sets the receiver’s executable.
         */
        launchPath: string;
        /**
         * Returns an initialized process object with the environment of the current process.
         */
        constructor ();
        /**
         * Creates and runs a task with a specified executable and arguments.
         */
        static run(url: URL, arguments: string[], terminationHandler: ()=>any | null):Process;
        /**
         * Creates and launches a task with a specified executable and arguments.
         */
        static launchedProcess(path: string, arguments: string[]):Process;
        /**
         * Sends an interrupt signal to the receiver and all of its subtasks.
         */
        interrupt():void;
        /**
         * Resumes execution of a suspended task.
         */
        resume():boolean;
        /**
         * Suspends execution of the receiver task.
         */
        suspend():boolean;
        /**
         * Sends a terminate signal to the receiver and all of its subtasks.
         */
        terminate():void;
        /**
         * Blocks the process until the receiver is finished.
         */
        waitUntilExit():void;
        /**
         * Launches the task represented by the receiver.
         */
        launch():void;
    }
    /**
     * the same name namespace of class Process
     */
    export namespace Process {
        export enum TerminationReason {
            exit = 1,
            uncaughtSignal = 2,
        }
    }
    export class NSUserScriptTask extends NSObject {
        /**
         * The URL of the script file.
         */
        scriptURL: URL;
        /**
         * Return a user script task instance given a URL for a script file.
         */
        constructor (url: URL);
        /**
         * Executes the script with no input and ignoring any result.
         */
        execute(handler: NSUserScriptTask.CompletionHandler | null):void;
    }
    export class NSUserAppleScriptTask extends NSUserScriptTask {
        /**
         * Execute the AppleScript script by sending it the specified Apple event.
         */
        execute(event: NSAppleEventDescriptor | null, handler: NSUserAppleScriptTask.CompletionHandler | null):void;
    }
    export class NSUserAutomatorTask extends NSUserScriptTask {
        /**
         * The variables required by the Automator workflow.
         */
        variables: Record<string, any> | null;
        /**
         * Execute the Automator workflow by providing it as securely coded input.
         */
        execute(input: NSSecureCoding | null, handler: NSUserAutomatorTask.CompletionHandler | null):void;
    }
    export class NSUserUnixTask extends NSUserScriptTask {
        /**
         * The standard error stream.
         */
        standardError: FileHandle | null;
        /**
         * The standard input stream.
         */
        standardInput: FileHandle | null;
        /**
         * The standard output stream.
         */
        standardOutput: FileHandle | null;
        /**
         * Execute the unix script with the specified arguments.
         */
        execute(arguments: string[] | null, handler: NSUserUnixTask.CompletionHandler | null):void;
    }
    export class Stream extends NSObject {
        /**
         * Sets the receiver’s delegate.
         */
        delegate: StreamDelegate | null;
        /**
         * Returns the receiver’s status.
         */
        streamStatus: Stream.Status;
        /**
         * Returns an 
         *  object representing the stream error. 
         */
        streamError: Error | null;
        /**
         * Creates and returns by reference an 
         *  object and 
         *  object for a socket connection with a given host on a given port.
         */
        static getStreamsTo(host: Host, port: number, inputStream: AutoreleasingUnsafeMutablePointer<InputStream?> | null, outputStream: AutoreleasingUnsafeMutablePointer<OutputStream?> | null):void;
        /**
         * Creates and returns by reference a bound pair of input and output streams.
         */
        static getBoundStreams(bufferSize: number, inputStream: AutoreleasingUnsafeMutablePointer<InputStream?> | null, outputStream: AutoreleasingUnsafeMutablePointer<OutputStream?> | null):void;
        static getStreamsToHost(hostname: string, port: number, inputStream: AutoreleasingUnsafeMutablePointer<InputStream?> | null, outputStream: AutoreleasingUnsafeMutablePointer<OutputStream?> | null):void;
        /**
         * Returns the receiver’s property for a given key.
         */
        property(key: Stream.PropertyKey):any;
        /**
         * Attempts to set the value of a given property of the receiver and returns a Boolean value that indicates whether the value is accepted by the receiver.
         */
        setProperty(property: any, key: Stream.PropertyKey):boolean;
        /**
         * Opens the receiving stream.
         */
        open():void;
        /**
         * Closes the receiver.
         */
        close():void;
        /**
         * Schedules the receiver on a given run loop in a given mode.
         */
        schedule(aRunLoop: RunLoop, mode: RunLoop.Mode):void;
        /**
         * Removes the receiver from a given run loop running in a given mode.
         */
        remove(aRunLoop: RunLoop, mode: RunLoop.Mode):void;
    }
    /**
     * the same name namespace of class Stream
     */
    export namespace Stream {
        export enum Status {
            atEnd = 5,
            closed = 6,
            error = 7,
            notOpen = 0,
            open = 2,
            opening = 1,
            reading = 3,
            writing = 4,
        }
        export class Event {
            /**
             * The open has completed successfully.
             */
            static openCompleted: Stream.Event;
            /**
             * The stream has bytes to be read. 
             */
            static hasBytesAvailable: Stream.Event;
            /**
             * The stream can accept bytes for writing.
             */
            static hasSpaceAvailable: Stream.Event;
            /**
             * An error has occurred on the stream.
             */
            static errorOccurred: Stream.Event;
            /**
             * The end of the stream has been reached.
             */
            static endEncountered: Stream.Event;
            constructor (rawValue: number);
        }
        export class PropertyKey {
            /**
             * Value is an 
             *  instance containing the data written to a memory stream. 
             */
            static dataWrittenToMemoryStreamKey: Stream.PropertyKey;
            /**
             * Value is an 
             *  object containing the current absolute offset of the stream.
             */
            static fileCurrentOffsetKey: Stream.PropertyKey;
            /**
             * The type of service for the stream. Providing the service type allows the system to properly handle certain attributes of the stream, including routing and suspension behavior. Most streams do not need to set this property. See 
             *  for a list of possible values.
             */
            static networkServiceType: Stream.PropertyKey;
            static socketSecurityLevelKey: Stream.PropertyKey;
            /**
             * Value is an 
             *  object containing SOCKS proxy configuration information. 
             */
            static socksProxyConfigurationKey: Stream.PropertyKey;
            constructor (rawValue: string);
            constructor (rawValue: string);
        }
    }
    export class StreamNetworkServiceTypeValue {
        /**
         * Specifies that the stream is providing a background service.
         */
        static background: StreamNetworkServiceTypeValue;
        /**
         * Specifies that the stream is providing video service.
         */
        static video: StreamNetworkServiceTypeValue;
        /**
         * Specifies that the stream is providing voice service.
         */
        static voice: StreamNetworkServiceTypeValue;
        /**
         * Specifies that the stream is providing VoIP service.
         */
        static voIP: StreamNetworkServiceTypeValue;
        static callSignaling: StreamNetworkServiceTypeValue;
        constructor (rawValue: string);
    }
    export class StreamSOCKSProxyConfiguration {
        /**
         * Value is an 
         *  object that represents the SOCKS proxy host.
         */
        static hostKey: StreamSOCKSProxyConfiguration;
        /**
         * Value is an 
         *  object containing the user’s password.
         */
        static passwordKey: StreamSOCKSProxyConfiguration;
        /**
         * Value is an 
         *  object containing an integer that represents the port on which the proxy listens.
         */
        static portKey: StreamSOCKSProxyConfiguration;
        /**
         * Value is an 
         *  object containing the user’s name.
         */
        static userKey: StreamSOCKSProxyConfiguration;
        /**
         * Value is either 
         *  or 
         * . 
         */
        static versionKey: StreamSOCKSProxyConfiguration;
        constructor (rawValue: string);
    }
    export class StreamSOCKSProxyVersion {
        /**
         * Possible value for 
         * .
         */
        static version4: StreamSOCKSProxyVersion;
        /**
         * Possible value for 
         * .
         */
        static version5: StreamSOCKSProxyVersion;
        constructor (rawValue: string);
    }
    export class StreamSocketSecurityLevel {
        /**
         * Specifies that the highest level security protocol that can be negotiated be set as the security protocol for a socket stream.
         */
        static negotiatedSSL: StreamSocketSecurityLevel;
        /**
         * Specifies that no security level be set for a socket stream.
         */
        static none: StreamSocketSecurityLevel;
        /**
         * Specifies that SSL version 2 be set as the security protocol for a socket stream.
         */
        static ssLv2: StreamSocketSecurityLevel;
        /**
         * Specifies that SSL version 3 be set as the security protocol for a socket stream.
         */
        static ssLv3: StreamSocketSecurityLevel;
        /**
         * Specifies that TLS version 1 be set as the security protocol for a socket stream.
         */
        static tlSv1: StreamSocketSecurityLevel;
        constructor (rawValue: string);
    }
    export class InputStream extends Stream {
        /**
         * A Boolean value that indicates whether the receiver has bytes available to read.
         */
        hasBytesAvailable: boolean;
        /**
         * Initializes and returns an 
         *  object for reading from a given 
         *  object.
         */
        constructor (data: Data);
        /**
         * Initializes and returns an 
         *  object that reads data from the file at a given path.
         */
        constructor (path: string);
        /**
         * Initializes and returns an 
         *  object that reads data from the file at a given URL.
         */
        constructor (url: URL);
        /**
         * Reads up to a given number of bytes into a given buffer.
         */
        read(buffer: UnsafeMutablePointer<UInt8>, len: number):number;
        /**
         * Returns by reference a pointer to a read buffer and, by reference, the number of bytes available, and returns a Boolean value that indicates whether the buffer is available.
         */
        getBuffer(buffer: UnsafeMutablePointer<UnsafeMutablePointer<UInt8>?>, len: UnsafeMutablePointer<Int>):boolean;
    }
    export class OutputStream extends Stream {
        /**
         * A boolean value that indicates whether the receiver can be written to.
         */
        hasSpaceAvailable: boolean;
        /**
         * Returns an initialized output stream that will write to memory.
         */
        constructor (toMemory: any);
        /**
         * Returns an initialized output stream that can write to a provided buffer.
         */
        constructor (buffer: UnsafeMutablePointer<UInt8>, capacity: number);
        /**
         * Returns an initialized output stream for writing to a specified file.
         */
        constructor (path: string, shouldAppend: boolean);
        /**
         * Returns an initialized output stream for writing to a specified URL.
         */
        constructor (url: URL, shouldAppend: boolean);
        /**
         * Creates and returns an initialized output stream that will write stream data to memory.
         */
        static toMemory():Self;
        /**
         * Writes the contents of a provided data buffer to the receiver.
         */
        write(buffer: UnsafePointer<UInt8>, len: number):number;
    }
    export interface StreamDelegate extends NSObjectProtocol {
        /**
         * The delegate receives this message when a given event has occurred on a given stream.
         */
        stream(aStream: Stream, eventCode: Stream.Event):void;
    }
    export class Pipe extends NSObject {
        /**
         * The receiver's read file handle.
         */
        fileHandleForReading: FileHandle;
        /**
         * The receiver's write file handle.
         */
        fileHandleForWriting: FileHandle;
    }
    export class Host extends NSObject {
        /**
         * Returns one of the network addresses of the receiver.
         */
        address: string;
        /**
         * Returns all the network addresses of the receiver.
         */
        addresses: string[];
        /**
         * Returns one of the hostnames of the receiver.
         */
        name: string;
        /**
         * Returns the name used as by default when publishing 
         * .
         */
        localizedName: string;
        /**
         * Returns all the hostnames of the receiver.
         */
        names: string[];
        /**
         * Returns the 
         *  with the Internet address 
         * .
         */
        constructor (address: string);
        /**
         * Returns a host with a specific name.
         */
        constructor (name: string);
        /**
         * Returns an 
         *  object representing the host the process is running on.
         */
        static current():Self;
        /**
         * Indicates whether the receiver represents the same host as another 
         *  object.
         */
        isEqual(aHost: Host):boolean;
    }
    export class Port extends NSObject {
        /**
         * Posted from the 
         *  method, which is invoked when the 
         *  is deallocated or when it notices that its communication channel has been damaged. The notification object is the 
         *  object that has become invalid. This notification does not contain a 
         *  dictionary.
         */
        static didBecomeInvalidNotification: NSNotification.Name;
        /**
         * A Boolean value that indicates whether the receiver is valid.
         */
        isValid: boolean;
        /**
         * The number of bytes of space reserved by the receiver for sending data.
         */
        reservedSpaceLength: number;
        /**
         * Marks the receiver as invalid and posts an 
         *  to the default notification center.
         */
        invalidate():void;
        /**
         * Sets the receiver’s delegate to a given object.
         */
        setDelegate(anObject: PortDelegate | null):void;
        /**
         * Returns the receiver’s delegate.
         */
        delegate():PortDelegate | null;
        /**
         * This method is provided for subclasses that have custom types of 
         * .
         */
        send(limitDate: Date, components: NSMutableArray | null, receivePort: Port | null, headerSpaceReserved: number):boolean;
        /**
         * This method should be implemented by a subclass to stop monitoring of a port when removed from a give run loop in a given input mode. 
         */
        remove(runLoop: RunLoop, mode: RunLoop.Mode):void;
        /**
         * This method should be implemented by a subclass to set up monitoring of a port when added to a given run loop in a given input mode.
         */
        schedule(runLoop: RunLoop, mode: RunLoop.Mode):void;
    }
    export class SocketPort extends Port {
        /**
         * The receiver’s socket address structure stored inside an 
         *  object.
         */
        address: Data;
        /**
         * The protocol that the receiver uses for communication.
         */
        protocol: Int32;
        /**
         * The protocol family that the receiver uses for communication.
         */
        protocolFamily: Int32;
        /**
         * The receiver’s native socket identifier on the platform.
         */
        socket: SocketNativeHandle;
        /**
         * The receiver’s socket type.
         */
        socketType: Int32;
        /**
         * Initializes the receiver as a local TCP/IP socket of type 
         * .
         */
        constructor ();
        /**
         * Initializes the receiver as a local TCP/IP socket of type 
         * , listening on a specified port number.
         */
        constructor (port: UInt16);
        /**
         * Initializes the receiver as a local socket with the provided arguments.
         */
        constructor (family: Int32, type: Int32, protocol: Int32, address: Data);
        /**
         * Initializes the receiver with a previously created local socket.
         */
        constructor (family: Int32, type: Int32, protocol: Int32, sock: SocketNativeHandle);
        /**
         * Initializes the receiver as a TCP/IP socket of type 
         *  that can connect to a remote host on a specified port.
         */
        constructor (port: UInt16, hostName: string);
        /**
         * Initializes the receiver as a remote socket with the provided arguments.
         */
        constructor (family: Int32, type: Int32, protocol: Int32, address: Data);
    }
    export class NSSwappedDouble {
        v: UInt64;
        constructor ();
        constructor (v: UInt64);
    }
    export class NSSwappedFloat {
        v: UInt32;
        constructor ();
        constructor (v: UInt32);
    }
    export class _NSRange {}
    export enum NSGrammaticalGender {
        feminine = 1,
        masculine = 2,
        neuter = 3,
        notSet = 0,
    }
    export enum NSGrammaticalNumber {
        zero = 2,
        singular = 1,
        plural = 3,
        pluralTwo = 4,
        pluralFew = 5,
        pluralMany = 6,
        notSet = 0,
    }
    export enum NSGrammaticalPartOfSpeech {
        determiner = 1,
        pronoun = 2,
        letter = 3,
        adverb = 4,
        particle = 5,
        adjective = 6,
        adposition = 7,
        verb = 8,
        noun = 9,
        conjunction = 10,
        numeral = 11,
        interjection = 12,
        preposition = 13,
        abbreviation = 14,
        notSet = 0,
    }
    export class AttributedString {
        static defaultResolverSpecification: ResolverSpecification;
        static transferRepresentation: TransferRepresentation;
        /**
         * The position of the first character in a nonempty attributed string.
         */
        startIndex: AttributedString.Index;
        /**
         * The string’s past-the-end position — the position one greater than the last valid subscript argument.
         */
        endIndex: AttributedString.Index;
        /**
         * The characters of the attributed string, as a view into the underlying string.
         */
        characters: AttributedString.CharacterView;
        /**
         * The Unicode scalars of the attributed string, as a view into the underlying string.
         */
        unicodeScalars: AttributedString.UnicodeScalarView;
        /**
         * The attributed runs of the attributed string, as a view into the underlying string.
         */
        runs: AttributedString.Runs;
        /**
         * The hash value.
         */
        hashValue: number;
        /**
         * A textual representation of the instance.
         */
        description: string;
        /**
         * Creates an empty attributed string.
         */
        constructor ();
        /**
         * Creates an attributed string from an attributed substring.
         */
        constructor (substring: AttributedSubstring);
        /**
         * Creates an attributed string from a string and an attribute container.
         */
        constructor (_string: string, attributes: AttributeContainer);
        /**
         * Creates an attributed string from the specified string literal, with no attributes.
         */
        constructor (value: string);
        /**
         * Creates an instance initialized to the given value.
         */
        constructor (value: string);
        /**
         * Creates an instance initialized to the given value.
         */
        constructor (value: string);
        /**
         * Creates an attributed string by looking up a localized string from the app’s bundle.
         */
        constructor (key: String.LocalizationValue, options: AttributedString.FormattingOptions, table: string, bundle: Bundle | null, locale: Locale | null, comment: StaticString | null);
        /**
         * Creates a localized attributed string from a localized string resource.
         */
        constructor (resource: LocalizedStringResource);
        /**
         * Creates an attributed string by looking up a localized string from the app’s bundle, using a default value if necessary.
         */
        constructor (key: StaticString, defaultValue: String.LocalizationValue, options: AttributedString.FormattingOptions, table: string, bundle: Bundle | null, locale: Locale | null, comment: StaticString | null);
        /**
         * Creates an attributed string from a Markdown-formatted string using the provided options.
         */
        constructor (markdown: string, options: AttributedString.MarkdownParsingOptions, baseURL: URL | null);
        /**
         * Creates an attributed string from the contents of a specified URL that contains Markdown-formatted data, using the provided options.
         */
        constructor (url: URL, options: AttributedString.MarkdownParsingOptions, baseURL: URL | null);
        /**
         * Creates a new instance by decoding from the given decoder.
         */
        constructor (decoder: Decoder);
        /**
         * Creates a new instance by decoding from the specified decoder, with help from the provided configuration.
         */
        constructor (decoder: Decoder, configuration: AttributeScopeCodableConfiguration);
        constructor (resource: LocalizedStringResource, options: AttributedString.LocalizationOptions);
        /**
         * Sets the attributed string’s attributes to those in a specified attribute container.
         */
        setAttributes(attributes: AttributeContainer):void;
        /**
         * Returns an attributed string by setting the attributed string’s attributes to those in a specified attribute container.
         */
        settingAttributes(attributes: AttributeContainer):AttributedString;
        /**
         * Merges the attributed string’s attributes with those in a specified attribute container.
         */
        mergeAttributes(attributes: AttributeContainer, mergePolicy: AttributedString.AttributeMergePolicy):void;
        /**
         * Returns an attributed string by merging the attributed string’s attributes with those in a specified attribute container.
         */
        mergingAttributes(attributes: AttributeContainer, mergePolicy: AttributeMergePolicy):AttributedString;
        /**
         * Replaces occurrences of attributes in one attribute container with those in another attribute container.
         */
        replaceAttributes(attributes: AttributeContainer, others: AttributeContainer):void;
        /**
         * Returns an attributed string by replacing occurrences of attributes in one attribute container with those in another attribute container.
         */
        replacingAttributes(attributes: AttributeContainer, others: AttributeContainer):AttributedString;
        /**
         * Returns the position of the character offset a given distance, measured in characters, from a given string index.
         */
        index(i: Index, distance: number):Index;
        /**
         * Applies automatic grammar agreement inflection rules to the attributed string and returns the result.
         */
        inflected():AttributedString;
        /**
         * Hashes the essential components of this value by feeding them into the given hasher.
         */
        hash(hasher: Hasher):void;
        /**
         * Encodes the value into the given encoder.
         */
        encode(encoder: Encoder):void;
    }
    /**
     * the same name namespace of class AttributedString
     */
    export namespace AttributedString {
        export enum AttributeMergePolicy {
            keepCurrent,
            keepNew,
        }
        export class FormattingOptions {
            /**
             * An option to add an attribute that marks replacements in localized strings.
             */
            static applyReplacementIndexAttribute: AttributedString.FormattingOptions;
            /**
             * The raw value of the option set.
             */
            rawValue: number;
            /**
             * A Boolean value that indicates whether the set has no elements.
             */
            isEmpty: boolean;
            /**
             * Creates an empty option set.
             */
            constructor ();
            /**
             * Creates a set containing the elements of the given array literal.
             */
            constructor (...arrayLiteral: AttributedString.FormattingOptions);
            /**
             * Creates a formatting options instance from the given raw value.
             */
            constructor (rawValue: number);
            /**
             * Returns a Boolean value that indicates whether a given element is a member of the option set.
             */
            contains(member: AttributedString.FormattingOptions):boolean;
            /**
             * Removes all elements of this option set that are not also present in the given set.
             */
            formIntersection(other: AttributedString.FormattingOptions):void;
            /**
             * Replaces this set with a new set containing all elements contained in either this set or the given set, but not in both.
             */
            formSymmetricDifference(other: AttributedString.FormattingOptions):void;
            /**
             * Inserts the elements of another set into this option set.
             */
            formUnion(other: AttributedString.FormattingOptions):void;
            /**
             * Adds the given element to the option set if it is not already a member.
             */
            insert(newMember: AttributedString.FormattingOptions):any;
            /**
             * Returns a new option set with only the elements contained in both this set and the given set.
             */
            intersection(other: AttributedString.FormattingOptions):AttributedString.FormattingOptions;
            /**
             * Returns a Boolean value that indicates whether the set has no members in common with the given set.
             */
            isDisjoint(other: AttributedString.FormattingOptions):boolean;
            /**
             * Returns a Boolean value that indicates whether this set is a strict subset of the given set.
             */
            isStrictSubset(other: AttributedString.FormattingOptions):boolean;
            /**
             * Returns a Boolean value that indicates whether this set is a strict superset of the given set.
             */
            isStrictSuperset(other: AttributedString.FormattingOptions):boolean;
            /**
             * Returns a Boolean value that indicates whether the set is a subset of another set.
             */
            isSubset(other: AttributedString.FormattingOptions):boolean;
            /**
             * Returns a Boolean value that indicates whether the set is a superset of the given set.
             */
            isSuperset(other: AttributedString.FormattingOptions):boolean;
            /**
             * Removes the given element and all elements subsumed by it.
             */
            remove(member: AttributedString.FormattingOptions):AttributedString.FormattingOptions | null;
            /**
             * Removes the elements of the given set from this set.
             */
            subtract(other: AttributedString.FormattingOptions):void;
            /**
             * Returns a new set containing the elements of this set that do not occur in the given set.
             */
            subtracting(other: AttributedString.FormattingOptions):AttributedString.FormattingOptions;
            /**
             * Returns a new option set with the elements contained in this set or in the given set, but not in both.
             */
            symmetricDifference(other: AttributedString.FormattingOptions):AttributedString.FormattingOptions;
            /**
             * Returns a new option set of the elements contained in this set, in the given set, or in both.
             */
            union(other: AttributedString.FormattingOptions):AttributedString.FormattingOptions;
            /**
             * Inserts the given element into the set.
             */
            update(newMember: AttributedString.FormattingOptions):AttributedString.FormattingOptions | null;
        }
        export class MarkdownParsingOptions {
            /**
             * A Boolean value that indicates whether parsing allows extensions to Markdown that specify extended attributes.
             */
            allowsExtendedAttributes: boolean;
            /**
             * A Boolean value that indicates whether parsing applies attributes that indicate the position of attributed text in the original Markdown string.
             */
            appliesSourcePositionAttributes: boolean;
            /**
             * The policy for handling a parsing failure.
             */
            failurePolicy: AttributedString.MarkdownParsingOptions.FailurePolicy;
            /**
             * The syntax for interpreting a Markdown string.
             */
            interpretedSyntax: AttributedString.MarkdownParsingOptions.InterpretedSyntax;
            /**
             * The language code for this document.
             */
            languageCode: string;
            /**
             * Creates a Markdown parsing options instance with the specified values.
             */
            constructor (allowsExtendedAttributes: boolean, interpretedSyntax: AttributedString.MarkdownParsingOptions.InterpretedSyntax, failurePolicy: AttributedString.MarkdownParsingOptions.FailurePolicy, languageCode: string);
            /**
             * Creates a Markdown parsing options instance with the specified values, optionally marking the source position of attributed text.
             */
            constructor (allowsExtendedAttributes: boolean, interpretedSyntax: AttributedString.MarkdownParsingOptions.InterpretedSyntax, failurePolicy: AttributedString.MarkdownParsingOptions.FailurePolicy, languageCode: string, appliesSourcePositionAttributes: boolean);
        }
        /**
         * the same name namespace of class MarkdownParsingOptions
         */
        export namespace MarkdownParsingOptions {
            export enum FailurePolicy {
                returnPartiallyParsedIfPossible,
                throwError,
            }
            export enum InterpretedSyntax {
                full,
                inlineOnly,
                inlineOnlyPreservingWhitespace,
            }
        }
        export class MarkdownSourcePosition {
            /**
             * The line where the text begins in the Markdown source.
             */
            startLine: number;
            /**
             * The column where the text begins in the Markdown source.
             */
            startColumn: number;
            /**
             * The line where the text ends in the Markdown source.
             */
            endLine: number;
            /**
             * The column where the text ends in the Markdown source.
             */
            endColumn: number;
            /**
             * The hash value.
             */
            hashValue: number;
            /**
             * Creates a Markdown source position instance from its start and end line and column.
             */
            constructor (startLine: number, startColumn: number, endLine: number, endColumn: number);
            /**
             * Creates a new instance by decoding from the given decoder.
             */
            constructor (decoder: Decoder);
            /**
             * Encodes this value into the given encoder.
             */
            encode(encoder: Encoder):void;
            /**
             * Hashes the essential components of this value by feeding them into the given hasher.
             */
            hash(hasher: Hasher):void;
        }
        export class Index {}
        export class CharacterView {
            /**
             * The number of elements in the collection.
             */
            count: number;
            endIndex: AttributedString.Index;
            /**
             * The first element of the collection.
             */
            first: Character | null;
            /**
             * The indices that are valid for subscripting the collection, in ascending order.
             */
            indices: DefaultIndices<AttributedString.CharacterView>;
            /**
             * A Boolean value indicating whether the collection is empty.
             */
            isEmpty: boolean;
            /**
             * The last element of the collection.
             */
            last: Character | null;
            /**
             * A sequence containing the same elements as this sequence, but on which some operations, such as 
             *  and 
             * , are implemented lazily.
             */
            lazy: LazySequence<AttributedString.CharacterView>;
            publisher: any;
            startIndex: AttributedString.Index;
            /**
             * A value less than or equal to the number of elements in the collection.
             */
            underestimatedCount: number;
            constructor ();
            /**
             * Creates a new collection containing the specified number of a single, repeated value.
             */
            constructor (repeatedValue: Character, count: number);
            /**
             * Returns a Boolean value indicating whether every element of a sequence satisfies a given predicate.
             */
            allSatisfy(predicate: Bool):boolean;
            /**
             * Adds an element to the end of the collection.
             */
            append(newElement: Character):void;
            /**
             * Applies the given difference to this collection.
             */
            applying(difference: CollectionDifference<Character>):AttributedString.CharacterView | null;
            /**
             * Returns a Boolean value indicating whether the sequence contains the given element.
             */
            contains(element: Character):boolean;
            distance(start: AttributedString.Index, end: AttributedString.Index):number;
            /**
             * Returns a subsequence by skipping elements while 
             *  returns 
             *  and returning the remaining elements.
             */
            drop(predicate: Bool):Slice<AttributedString.CharacterView>;
            /**
             * Returns a subsequence containing all but the given number of initial elements.
             */
            dropFirst(k: number):Slice<AttributedString.CharacterView>;
            /**
             * Returns a subsequence containing all but the specified number of final elements.
             */
            dropLast(k: number):Slice<AttributedString.CharacterView>;
            /**
             * Returns a sequence of pairs (
             * , 
             * ), where 
             *  represents a consecutive integer starting at zero and 
             *  represents an element of the sequence.
             */
            enumerated():EnumeratedSequence<AttributedString.CharacterView>;
            /**
             * Returns a new collection of the same type containing, in order, the elements of the original collection that satisfy the given predicate.
             */
            filter(isIncluded: Bool):AttributedString.CharacterView;
            /**
             * Returns the first index where the specified value appears in the collection.
             */
            firstIndex(element: Character):AttributedString.Index | null;
            /**
             * Calls the given closure on each element in the sequence in the same order as a 
             * -
             *  loop.
             */
            forEach(body: Void):void;
            /**
             * Offsets the given index by the specified distance.
             */
            formIndex(i: AttributedString.Index, distance: number):void;
            index(i: AttributedString.Index, distance: number):AttributedString.Index;
            /**
             * Inserts a new element into the collection at the specified position.
             */
            insert(newElement: Character, i: AttributedString.Index):void;
            /**
             * Returns the last index where the specified value appears in the collection.
             */
            lastIndex(element: Character):AttributedString.Index | null;
            /**
             * Returns an iterator over the elements of the collection.
             */
            makeIterator():IndexingIterator<AttributedString.CharacterView>;
            /**
             * Returns the maximum element in the sequence.
             */
            max():Character | null;
            /**
             * Returns the minimum element in the sequence.
             */
            min():Character | null;
            /**
             * Removes and returns the last element of the collection.
             */
            popLast():Character | null;
            /**
             * Returns a subsequence, up to the specified maximum length, containing the initial elements of the collection.
             */
            prefix(maxLength: number):Slice<AttributedString.CharacterView>;
            /**
             * Returns a random element of the collection.
             */
            randomElement():Character | null;
            /**
             * Removes and returns the element at the specified position.
             */
            remove(position: AttributedString.Index):Character;
            /**
             * Removes all elements from the collection.
             */
            removeAll(keepCapacity: boolean):void;
            /**
             * Removes and returns the first element of the collection.
             */
            removeFirst():Character;
            /**
             * Removes and returns the last element of the collection.
             */
            removeLast():Character;
            /**
             * Removes the elements in the specified subrange from the collection.
             */
            removeSubrange(bounds: Range<AttributedString.Index>):void;
            /**
             * Prepares the collection to store the specified number of elements, when doing so is appropriate for the underlying type.
             */
            reserveCapacity(n: number):void;
            /**
             * Returns a view presenting the elements of the collection in reverse order.
             */
            reversed():ReversedCollection<AttributedString.CharacterView>;
            /**
             * Returns the elements of the sequence, shuffled.
             */
            shuffled():Character[];
            /**
             * Returns the elements of the sequence, sorted.
             */
            sorted():Character[];
            /**
             * Returns the longest possible subsequences of the collection, in order, that don’t contain elements satisfying the given predicate.
             */
            split(maxSplits: number, omittingEmptySubsequences: boolean, isSeparator: Bool):Slice<AttributedString.CharacterView>[];
            /**
             * Returns a subsequence, up to the given maximum length, containing the final elements of the collection.
             */
            suffix(maxLength: number):Slice<AttributedString.CharacterView>;
            trimPrefix(predicate: Bool):void;
            trimmingPrefix(predicate: Bool):Slice<AttributedString.CharacterView>;
        }
        export class UnicodeScalarView {
            /**
             * The number of elements in the collection.
             */
            count: number;
            endIndex: AttributedString.Index;
            /**
             * The first element of the collection.
             */
            first: UnicodeScalar | null;
            /**
             * The indices that are valid for subscripting the collection, in ascending order.
             */
            indices: DefaultIndices<AttributedString.UnicodeScalarView>;
            /**
             * A Boolean value indicating whether the collection is empty.
             */
            isEmpty: boolean;
            /**
             * The last element of the collection.
             */
            last: UnicodeScalar | null;
            /**
             * A sequence containing the same elements as this sequence, but on which some operations, such as 
             *  and 
             * , are implemented lazily.
             */
            lazy: LazySequence<AttributedString.UnicodeScalarView>;
            publisher: any;
            startIndex: AttributedString.Index;
            /**
             * A value less than or equal to the number of elements in the collection.
             */
            underestimatedCount: number;
            constructor ();
            /**
             * Creates a new collection containing the specified number of a single, repeated value.
             */
            constructor (repeatedValue: UnicodeScalar, count: number);
            /**
             * Returns a Boolean value indicating whether every element of a sequence satisfies a given predicate.
             */
            allSatisfy(predicate: Bool):boolean;
            /**
             * Adds an element to the end of the collection.
             */
            append(newElement: UnicodeScalar):void;
            /**
             * Applies the given difference to this collection.
             */
            applying(difference: CollectionDifference<UnicodeScalar>):AttributedString.UnicodeScalarView | null;
            /**
             * Returns a Boolean value indicating whether the sequence contains the given element.
             */
            contains(element: UnicodeScalar):boolean;
            distance(start: AttributedString.Index, end: AttributedString.Index):number;
            /**
             * Returns a subsequence by skipping elements while 
             *  returns 
             *  and returning the remaining elements.
             */
            drop(predicate: Bool):Slice<AttributedString.UnicodeScalarView>;
            /**
             * Returns a subsequence containing all but the given number of initial elements.
             */
            dropFirst(k: number):Slice<AttributedString.UnicodeScalarView>;
            /**
             * Returns a subsequence containing all but the specified number of final elements.
             */
            dropLast(k: number):Slice<AttributedString.UnicodeScalarView>;
            /**
             * Returns a sequence of pairs (
             * , 
             * ), where 
             *  represents a consecutive integer starting at zero and 
             *  represents an element of the sequence.
             */
            enumerated():EnumeratedSequence<AttributedString.UnicodeScalarView>;
            /**
             * Returns a new collection of the same type containing, in order, the elements of the original collection that satisfy the given predicate.
             */
            filter(isIncluded: Bool):AttributedString.UnicodeScalarView;
            /**
             * Returns the first index where the specified value appears in the collection.
             */
            firstIndex(element: UnicodeScalar):AttributedString.Index | null;
            /**
             * Calls the given closure on each element in the sequence in the same order as a 
             * -
             *  loop.
             */
            forEach(body: Void):void;
            /**
             * Offsets the given index by the specified distance.
             */
            formIndex(i: AttributedString.Index, distance: number):void;
            index(i: AttributedString.Index, distance: number):AttributedString.Index;
            /**
             * Inserts a new element into the collection at the specified position.
             */
            insert(newElement: UnicodeScalar, i: AttributedString.Index):void;
            /**
             * Returns the last index where the specified value appears in the collection.
             */
            lastIndex(element: UnicodeScalar):AttributedString.Index | null;
            /**
             * Returns an iterator over the elements of the collection.
             */
            makeIterator():IndexingIterator<AttributedString.UnicodeScalarView>;
            /**
             * Returns the maximum element in the sequence.
             */
            max():UnicodeScalar | null;
            /**
             * Returns the minimum element in the sequence.
             */
            min():UnicodeScalar | null;
            /**
             * Removes and returns the last element of the collection.
             */
            popLast():UnicodeScalar | null;
            /**
             * Returns a subsequence, up to the specified maximum length, containing the initial elements of the collection.
             */
            prefix(maxLength: number):Slice<AttributedString.UnicodeScalarView>;
            /**
             * Returns a random element of the collection.
             */
            randomElement():UnicodeScalar | null;
            /**
             * Removes and returns the element at the specified position.
             */
            remove(position: AttributedString.Index):UnicodeScalar;
            /**
             * Removes all elements from the collection.
             */
            removeAll(keepCapacity: boolean):void;
            /**
             * Removes and returns the first element of the collection.
             */
            removeFirst():UnicodeScalar;
            /**
             * Removes and returns the last element of the collection.
             */
            removeLast():UnicodeScalar;
            /**
             * Removes the elements in the specified subrange from the collection.
             */
            removeSubrange(bounds: Range<AttributedString.Index>):void;
            /**
             * Prepares the collection to store the specified number of elements, when doing so is appropriate for the underlying type.
             */
            reserveCapacity(n: number):void;
            /**
             * Returns a view presenting the elements of the collection in reverse order.
             */
            reversed():ReversedCollection<AttributedString.UnicodeScalarView>;
            /**
             * Returns the elements of the sequence, shuffled.
             */
            shuffled():UnicodeScalar[];
            /**
             * Returns the elements of the sequence, sorted.
             */
            sorted():UnicodeScalar[];
            /**
             * Returns the longest possible subsequences of the collection, in order, that don’t contain elements satisfying the given predicate.
             */
            split(maxSplits: number, omittingEmptySubsequences: boolean, isSeparator: Bool):Slice<AttributedString.UnicodeScalarView>[];
            /**
             * Returns a subsequence, up to the given maximum length, containing the final elements of the collection.
             */
            suffix(maxLength: number):Slice<AttributedString.UnicodeScalarView>;
            trimPrefix(predicate: Bool):void;
            trimmingPrefix(predicate: Bool):Slice<AttributedString.UnicodeScalarView>;
        }
        export class Runs {
            /**
             * The number of elements in the collection.
             */
            count: number;
            description: string;
            endIndex: AttributedString.Runs.Index;
            /**
             * The first element of the collection.
             */
            first: Run | null;
            /**
             * The indices that are valid for subscripting the collection, in ascending order.
             */
            indices: DefaultIndices<AttributedString.Runs>;
            /**
             * A Boolean value indicating whether the collection is empty.
             */
            isEmpty: boolean;
            /**
             * The last element of the collection.
             */
            last: Run | null;
            /**
             * A sequence containing the same elements as this sequence, but on which some operations, such as 
             *  and 
             * , are implemented lazily.
             */
            lazy: LazySequence<AttributedString.Runs>;
            publisher: any;
            startIndex: AttributedString.Runs.Index;
            /**
             * A value less than or equal to the number of elements in the collection.
             */
            underestimatedCount: number;
            /**
             * Returns a Boolean value indicating whether every element of a sequence satisfies a given predicate.
             */
            allSatisfy(predicate: Bool):boolean;
            /**
             * Returns a Boolean value indicating whether the sequence contains the given element.
             */
            contains(element: Run):boolean;
            distance(start: Index, end: Index):number;
            /**
             * Returns a subsequence by skipping elements while 
             *  returns 
             *  and returning the remaining elements.
             */
            drop(predicate: Bool):Slice<AttributedString.Runs>;
            /**
             * Returns a subsequence containing all but the given number of initial elements.
             */
            dropFirst(k: number):Slice<AttributedString.Runs>;
            /**
             * Returns a subsequence containing all but the specified number of final elements.
             */
            dropLast(k: number):Slice<AttributedString.Runs>;
            /**
             * Returns a sequence of pairs (
             * , 
             * ), where 
             *  represents a consecutive integer starting at zero and 
             *  represents an element of the sequence.
             */
            enumerated():EnumeratedSequence<AttributedString.Runs>;
            /**
             * Returns an array containing, in order, the elements of the sequence that satisfy the given predicate.
             */
            filter(isIncluded: Bool):Run[];
            /**
             * Returns the first index where the specified value appears in the collection.
             */
            firstIndex(element: Run):Index | null;
            /**
             * Calls the given closure on each element in the sequence in the same order as a 
             * -
             *  loop.
             */
            forEach(body: Void):void;
            /**
             * Offsets the given index by the specified distance.
             */
            formIndex(i: Index, distance: number):void;
            index(i: Index, distance: number):Index;
            /**
             * Returns the last index where the specified value appears in the collection.
             */
            lastIndex(element: Run):Index | null;
            /**
             * Returns an iterator over the elements of the collection.
             */
            makeIterator():IndexingIterator<AttributedString.Runs>;
            /**
             * Returns the maximum element in the sequence, using the given predicate as the comparison between elements.
             */
            max(areInIncreasingOrder: Bool):Run | null;
            /**
             * Returns the minimum element in the sequence, using the given predicate as the comparison between elements.
             */
            min(areInIncreasingOrder: Bool):Run | null;
            /**
             * Returns a subsequence, up to the specified maximum length, containing the initial elements of the collection.
             */
            prefix(maxLength: number):Slice<AttributedString.Runs>;
            /**
             * Returns a random element of the collection.
             */
            randomElement():Run | null;
            /**
             * Returns a view presenting the elements of the collection in reverse order.
             */
            reversed():ReversedCollection<AttributedString.Runs>;
            /**
             * Returns the elements of the sequence, shuffled.
             */
            shuffled():Run[];
            /**
             * Returns the elements of the sequence, sorted using the given predicate as the comparison between elements.
             */
            sorted(areInIncreasingOrder: Bool):Run[];
            /**
             * Returns the longest possible subsequences of the collection, in order, that don’t contain elements satisfying the given predicate.
             */
            split(maxSplits: number, omittingEmptySubsequences: boolean, isSeparator: Bool):Slice<AttributedString.Runs>[];
            /**
             * Returns a subsequence, up to the given maximum length, containing the final elements of the collection.
             */
            suffix(maxLength: number):Slice<AttributedString.Runs>;
            trimmingPrefix(predicate: Bool):Slice<AttributedString.Runs>;
        }
        /**
         * the same name namespace of class Runs
         */
        export namespace Runs {
            export class Index {
                advanced(n: number):AttributedString.Runs.Index;
                distance(other: AttributedString.Runs.Index):number;
            }
            export class Run {
                attributes: AttributeContainer;
                description: string;
                range: Range<AttributedString.Index>;
            }
        }
        export class InterpolationOptions {
            static insertAttributesWithoutMerging: AttributedString.InterpolationOptions;
            /**
             * A Boolean value that indicates whether the set has no elements.
             */
            isEmpty: boolean;
            rawValue: number;
            /**
             * Creates an empty option set.
             */
            constructor ();
            /**
             * Creates a set containing the elements of the given array literal.
             */
            constructor (...arrayLiteral: AttributedString.InterpolationOptions);
            constructor (rawValue: number);
            /**
             * Returns a Boolean value that indicates whether a given element is a member of the option set.
             */
            contains(member: AttributedString.InterpolationOptions):boolean;
            /**
             * Removes all elements of this option set that are not also present in the given set.
             */
            formIntersection(other: AttributedString.InterpolationOptions):void;
            /**
             * Replaces this set with a new set containing all elements contained in either this set or the given set, but not in both.
             */
            formSymmetricDifference(other: AttributedString.InterpolationOptions):void;
            /**
             * Inserts the elements of another set into this option set.
             */
            formUnion(other: AttributedString.InterpolationOptions):void;
            /**
             * Adds the given element to the option set if it is not already a member.
             */
            insert(newMember: AttributedString.InterpolationOptions):any;
            /**
             * Returns a new option set with only the elements contained in both this set and the given set.
             */
            intersection(other: AttributedString.InterpolationOptions):AttributedString.InterpolationOptions;
            /**
             * Returns a Boolean value that indicates whether the set has no members in common with the given set.
             */
            isDisjoint(other: AttributedString.InterpolationOptions):boolean;
            /**
             * Returns a Boolean value that indicates whether this set is a strict subset of the given set.
             */
            isStrictSubset(other: AttributedString.InterpolationOptions):boolean;
            /**
             * Returns a Boolean value that indicates whether this set is a strict superset of the given set.
             */
            isStrictSuperset(other: AttributedString.InterpolationOptions):boolean;
            /**
             * Returns a Boolean value that indicates whether the set is a subset of another set.
             */
            isSubset(other: AttributedString.InterpolationOptions):boolean;
            /**
             * Returns a Boolean value that indicates whether the set is a superset of the given set.
             */
            isSuperset(other: AttributedString.InterpolationOptions):boolean;
            /**
             * Removes the given element and all elements subsumed by it.
             */
            remove(member: AttributedString.InterpolationOptions):AttributedString.InterpolationOptions | null;
            /**
             * Removes the elements of the given set from this set.
             */
            subtract(other: AttributedString.InterpolationOptions):void;
  