"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.bundleArkTS = exports.toArkTS = exports.bundleSwift = exports.toSwift = exports.bundleKotlin = exports.toKotlin = exports.parse = void 0;
const path_1 = require("path");
const utils_1 = require("./utils");
const bindingsOverride = process.env['UTS_BINARY_PATH'];
const bindings = !!bindingsOverride
    ? require((0, path_1.resolve)(bindingsOverride))
    : require('./binding').default;
function resolveOptions(options) {
    const { input, output } = options;
    if (!input?.root) {
        return;
    }
    if (!input?.filename) {
        return;
    }
    if (!output?.outDir) {
        return;
    }
    if (output.sourceMap === true) {
        output.sourceMap = output.outDir;
    }
    else if (output.sourceMap === false ||
        typeof output.sourceMap === 'undefined') {
        output.sourceMap = '';
    }
    if (!output.imports) {
        // TODO
        output.imports = [];
    }
    input.root = (0, utils_1.normalizePath)(input.root);
    input.filename = (0, utils_1.normalizePath)(input.filename);
    output.outDir = (0, utils_1.normalizePath)(output.outDir);
    output.sourceMap = (0, utils_1.normalizePath)(output.sourceMap);
    output.logFilename = !!output.logFilename;
    output.noColor = !!output.noColor;
    if (output.extname && output.extname[0] === '.') {
        output.extname = output.extname.slice(1);
    }
    return options;
}
async function parse(source, options = {}) {
    options.noColor = !!options.noColor;
    return bindings
        .parse(source, toBuffer(options))
        .then((res) => JSON.parse(res));
}
exports.parse = parse;
async function toKotlin(options) {
    const kotlinOptions = resolveOptions(options);
    if (!kotlinOptions) {
        return Promise.resolve({});
    }
    return bindings
        .toKotlin(toBuffer(kotlinOptions))
        .then((res) => JSON.parse(res));
}
exports.toKotlin = toKotlin;
async function bundleKotlin(options) {
    const bundleOptions = resolveOptions(options);
    if (!bundleOptions) {
        return Promise.resolve({});
    }
    return bindings
        .bundleKotlin(toBuffer(bundleOptions))
        .then((res) => JSON.parse(res))
        .catch((error) => {
        return { error };
    });
}
exports.bundleKotlin = bundleKotlin;
async function toSwift(options) {
    const swiftOptions = resolveOptions(options);
    if (!swiftOptions) {
        return Promise.resolve({});
    }
    return bindings
        .toSwift(toBuffer(swiftOptions))
        .then((res) => JSON.parse(res))
        .catch((error) => {
        return { error };
    });
}
exports.toSwift = toSwift;
async function bundleSwift(options) {
    const bundleOptions = resolveOptions(options);
    if (!bundleOptions) {
        return Promise.resolve({});
    }
    return bindings
        .bundleSwift(toBuffer(bundleOptions))
        .then((res) => JSON.parse(res));
}
exports.bundleSwift = bundleSwift;
async function toArkTS(options) {
    const arkTSOptions = resolveOptions(options);
    if (!arkTSOptions) {
        return Promise.resolve({});
    }
    return bindings
        .toSwift(toBuffer(arkTSOptions))
        .then((res) => JSON.parse(res))
        .catch((error) => {
        return { error };
    });
}
exports.toArkTS = toArkTS;
async function bundleArkTS(options) {
    const bundleOptions = resolveOptions(options);
    if (!bundleOptions) {
        return Promise.resolve({});
    }
    return bindings
        .bundleArkTS(toBuffer(bundleOptions))
        .then((res) => JSON.parse(res));
}
exports.bundleArkTS = bundleArkTS;
function toBuffer(t) {
    // 'Buffer' only refers to a type, but is being used as a value here
    return Buffer.from(JSON.stringify(t));
}
