/*
 * Copyright (C) 2024 Huawei Device Co., Ltd.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * @file State management API file
 * @kit ArkUI
 */
/**
 * Function that returns default creator.
 *
 * @typedef { function } StorageDefaultCreator<T>
 * @returns { T } default creator
 * @syscap SystemCapability.ArkUI.ArkUI.Full
 * @crossplatform
 * @atomicservice
 * @since 12
 */
export declare type StorageDefaultCreator<T> = () => T;
/**
 * Define class constructor with arbitrary parameters.
 * @interface TypeConstructorWithArgs<T>
 * @syscap SystemCapability.ArkUI.ArkUI.Full
 * @crossplatform
 * @atomicservice
 * @since 12
 */
export interface TypeConstructorWithArgs<T> {
    /**
     * @param { any } args the arguments of the constructor.
     * @returns { T } return class instance.
     * @syscap SystemCapability.ArkUI.ArkUI.Full
     * @crossplatform
     * @atomicservice
     * @since 12
     */
    new (...args: any): T;
}
/**
 * AppStorageV2 is for UI state of app-wide access, has same life cycle as the app,
 * and saves database content only in memory.
 *
 * @syscap SystemCapability.ArkUI.ArkUI.Full
 * @crossplatform
 * @atomicservice
 * @since 12
 */
export declare class AppStorageV2 {
    /**
     * If the value for the given key is already available, return the value.
     * If not, add the key with value generated by calling defaultFunc and return it to caller.
     *
     * @param { TypeConstructorWithArgs<T> } type class type of the stored value.
     * @param { string | StorageDefaultCreator<T> } [keyOrDefaultCreator] alias name of the key, or the function generating the default value.
     * @param { StorageDefaultCreator<T> } [defaultCreator] the function generating the default value
     * @returns { T | undefined } the value of the existed key or the default value
     * @syscap SystemCapability.ArkUI.ArkUI.Full
     * @crossplatform
     * @atomicservice
     * @since 12
     */
    static connect<T extends object>(type: TypeConstructorWithArgs<T>, keyOrDefaultCreator?: string | StorageDefaultCreator<T>, defaultCreator?: StorageDefaultCreator<T>): T | undefined;
    /**
     * Removes data with the given key or given class type.
     *
     * @param { string | TypeConstructorWithArgs<T> } keyOrType key or class type removing
     * @syscap SystemCapability.ArkUI.ArkUI.Full
     * @crossplatform
     * @atomicservice
     * @since 12
     */
    static remove<T>(keyOrType: string | TypeConstructorWithArgs<T>): void;
    /**
     * Return the array of all keys.
     *
     * @returns { Array<string> } the array of all keys
     * @syscap SystemCapability.ArkUI.ArkUI.Full
     * @crossplatform
     * @atomicservice
     * @since 12
     */
    static keys(): Array<string>;
}
/**
 * Function that returns reason type when error.
 *
 * @typedef { function } PersistenceErrorCallback
 * @param { string } key persisted key when error
 * @param { 'quota' | 'serialization' | 'unknown' } reason reason type when error
 * @param { string } message more message when error
 * @syscap SystemCapability.ArkUI.ArkUI.Full
 * @crossplatform
 * @atomicservice
 * @since 12
 */
export declare type PersistenceErrorCallback = (key: string, reason: 'quota' | 'serialization' | 'unknown', message: string) => void;
/**
 * PersistenceV2 is for UI state of app-wide access, available on app re-start,
 * and saves database content in disk.
 *
 * @extends AppStorageV2
 * @syscap SystemCapability.ArkUI.ArkUI.Full
 * @crossplatform
 * @atomicservice
 * @since 12
 */
export declare class PersistenceV2 extends AppStorageV2 {
    /**
     * Used to manually persist data changes to disks.
     *
     * @param { string | TypeConstructorWithArgs<T> } keyOrType key or class type need to persist.
     * @syscap SystemCapability.ArkUI.ArkUI.Full
     * @crossplatform
     * @atomicservice
     * @since 12
     */
    static save<T>(keyOrType: string | TypeConstructorWithArgs<T>): void;
    /**
     * Be called when persisting has encountered an error.
     *
     * @param { PersistenceErrorCallback | undefined } callback called when error
     * @syscap SystemCapability.ArkUI.ArkUI.Full
     * @crossplatform
     * @atomicservice
     * @since 12
     */
    static notifyOnError(callback: PersistenceErrorCallback | undefined): void;
}
/**
 * Define TypeConstructor type.
 *
 * @interface TypeConstructor<T>
 * @syscap SystemCapability.ArkUI.ArkUI.Full
 * @crossplatform
 * @atomicservice
 * @since 12
 */
export interface TypeConstructor<T> {
    /**
     * @returns { T }
     * @syscap SystemCapability.ArkUI.ArkUI.Full
     * @crossplatform
     * @atomicservice
     * @since 12
     */
    new (): T;
}
/**
 * Function that returns PropertyDecorator.
 *
 * @typedef { function } TypeDecorator
 * @param { TypeConstructor<T> } type type info
 * @returns { PropertyDecorator } Type decorator
 * @syscap SystemCapability.ArkUI.ArkUI.Full
 * @crossplatform
 * @atomicservice
 * @since 12
 */
export declare type TypeDecorator = <T>(type: TypeConstructor<T>) => PropertyDecorator;
/**
 * Define Type PropertyDecorator, adds type information to an object.
 *
 * @syscap SystemCapability.ArkUI.ArkUI.Full
 * @crossplatform
 * @atomicservice
 * @since 12
 */
export declare const Type: TypeDecorator;
