"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveTscUniModuleIndexFileName = exports.resolveUVueOutputPluginDir = exports.resolveOutputPluginDir = exports.syncUniModuleFilesByCompiler = exports.compileUniModuleWithTsc = exports.createUniXSwiftCompilerOnce = exports.createUniXSwiftCompiler = exports.createUniXKotlinCompilerOnce = exports.createUniXKotlinCompiler = exports.createUniXArkTSCompilerOnce = exports.createUniXArkTSCompiler = void 0;
const path_1 = __importDefault(require("path"));
const fs_extra_1 = __importDefault(require("fs-extra"));
const fast_glob_1 = __importDefault(require("fast-glob"));
const compiler_1 = require("./tsc/compiler");
const shared_1 = require("./shared");
const utils_1 = require("./utils");
const shared_2 = require("@vue/shared");
function createUniXTargetLanguageCompiler(platform, language) {
    const tscInputDir = path_1.default.join(process.env.UNI_APP_X_TSC_DIR, platform);
    return (0, compiler_1.createUniXCompiler)(process.env.NODE_ENV === 'development' ? 'development' : 'production', language, {
        inputDir: tscInputDir,
        cacheDir: path_1.default.resolve(process.env.UNI_APP_X_TSC_CACHE_DIR, platform),
        outputDir: path_1.default.join(process.env.UNI_APP_X_UVUE_DIR, platform),
        paths: resolveUniXCompilerUniModulesPaths(platform, process.env.UNI_INPUT_DIR, tscInputDir),
        normalizeFileName: normalizeNodeModules,
    });
}
function createUniXArkTSCompiler() {
    return createUniXTargetLanguageCompiler('app-harmony', 'ArkTS');
}
exports.createUniXArkTSCompiler = createUniXArkTSCompiler;
exports.createUniXArkTSCompilerOnce = (0, shared_1.once)(createUniXArkTSCompiler);
function createUniXKotlinCompiler() {
    return createUniXTargetLanguageCompiler('app-android', 'Kotlin');
}
exports.createUniXKotlinCompiler = createUniXKotlinCompiler;
exports.createUniXKotlinCompilerOnce = (0, shared_1.once)(createUniXKotlinCompiler);
function createUniXSwiftCompiler() {
    return createUniXTargetLanguageCompiler('app-ios', 'Swift');
}
exports.createUniXSwiftCompiler = createUniXSwiftCompiler;
exports.createUniXSwiftCompilerOnce = (0, shared_1.once)(createUniXSwiftCompiler);
function resolveUniXCompilerUniModulesPaths(platform, inputDir, tscInputDir) {
    const paths = {};
    const uniModulesDir = path_1.default.resolve(inputDir, 'uni_modules');
    if (fs_extra_1.default.existsSync(uniModulesDir)) {
        fs_extra_1.default.readdirSync(uniModulesDir).forEach((dir) => {
            const pluginPath = `@/uni_modules/${dir}`;
            const pluginDir = path_1.default.resolve(uniModulesDir, dir);
            const utssdkDir = path_1.default.resolve(pluginDir, 'utssdk');
            const tscUtsSdkDir = path_1.default.resolve(tscInputDir, 'uni_modules', dir, 'utssdk');
            // utssdk 插件
            if (fs_extra_1.default.existsSync(utssdkDir)) {
                // 加密插件
                if (fs_extra_1.default.existsSync(path_1.default.resolve(pluginDir, 'encrypt'))) {
                    if (fs_extra_1.default.existsSync(path_1.default.resolve(utssdkDir, 'interface.uts'))) {
                        paths[pluginPath] = [path_1.default.resolve(tscUtsSdkDir, 'interface.uts.ts')];
                    }
                }
                else {
                    // 非加密插件
                    // utssdk/app-android/index.uts
                    if (fs_extra_1.default.existsSync(path_1.default.resolve(utssdkDir, platform, 'index.uts'))) {
                        paths[pluginPath] = [
                            path_1.default.resolve(tscUtsSdkDir, platform, 'index.uts.ts'),
                        ];
                        // utssdk/index.uts
                    }
                    else if (fs_extra_1.default.existsSync(path_1.default.resolve(utssdkDir, 'index.uts'))) {
                        paths[pluginPath] = [path_1.default.resolve(tscUtsSdkDir, 'index.uts.ts')];
                    }
                }
            }
        });
    }
    return paths;
}
const NODE_MODULES_REGEX = /(\.\.\/)?node_modules/g;
function normalizeNodeModules(str) {
    str = (0, shared_1.normalizePath)(str).replace(NODE_MODULES_REGEX, 'node-modules');
    // HBuilderX 内置模块路径转换
    str = str.replace(/.*\/plugins\/uniapp-cli-vite\/node[-_]modules/, 'node-modules');
    if (!(0, shared_1.isInHBuilderX)()) {
        // 内部测试
        if (str.includes('uni-app-next/packages/')) {
            str = str.replace(/.*\/uni-app-next\/packages\//, 'node-modules/@dcloudio/');
        }
    }
    if (process.env.UNI_PLATFORM === 'mp-alipay') {
        str = str.replace('node-modules/@', 'node-modules/npm-scope-');
    }
    return str;
}
async function compileUniModuleWithTsc(platform, pluginDir, uniXCompiler, { rootFiles, preprocessor, }) {
    // 初始化编译器
    await uniXCompiler.init();
    // 同步资源
    await syncUniModuleFilesByCompiler(platform, uniXCompiler, pluginDir, preprocessor);
    // 添加入口
    const indexFileName = resolveTscUniModuleIndexFileName(platform, pluginDir);
    if (indexFileName) {
        await uniXCompiler.addRootFile(indexFileName);
    }
    const userRootFiles = typeof rootFiles === 'function' ? rootFiles(platform) : rootFiles;
    if (userRootFiles && userRootFiles.length) {
        await uniXCompiler.addRootFiles(userRootFiles);
    }
    await uniXCompiler.close();
}
exports.compileUniModuleWithTsc = compileUniModuleWithTsc;
async function syncUniModuleFilesByCompiler(platform, compiler, pluginDir, preprocessor) {
    const start = Date.now();
    const inputDir = process.env.UNI_INPUT_DIR;
    const outputPluginDir = resolveOutputPluginDir(platform, inputDir, pluginDir);
    const uvueOutputPluginDir = resolveUVueOutputPluginDir(platform, inputDir, pluginDir);
    // 目前每次编译，都全量比对同步uni_modules目录下的文件，不然还要 watch dir
    const files = await syncUniModulesFiles(platform, pluginDir, outputPluginDir, true, preprocessor);
    const staticFiles = await syncUniModuleStaticFiles(pluginDir, uvueOutputPluginDir, preprocessor);
    if (staticFiles.length) {
        files.push(...staticFiles);
    }
    // copy vue files
    const vueFiles = await syncUniModuleVueFiles(pluginDir, uvueOutputPluginDir, preprocessor);
    if (vueFiles.length) {
        // 如果有组件，那将 uts 文件 copy 到 .uvue 目录下，避免 tsc 不 emit 相关的 uts 文件
        // 如果 tsc emit 了，那就会再次覆盖
        await syncUniModulesFiles(platform, pluginDir, uvueOutputPluginDir, false, preprocessor);
        compiler.debug(`${path_1.default.basename(pluginDir)} sync vue files(${vueFiles.length})`);
        files.push(...vueFiles);
    }
    // 如果是 customElements，且没有utssdk入口文件，需要自动生成一个
    const customElementsDir = path_1.default.resolve(pluginDir, 'customElements');
    if (fs_extra_1.default.existsSync(customElementsDir)) {
        const customElements = resolveCustomElements(customElementsDir);
        if (customElements.length) {
            const pluginId = path_1.default.basename(pluginDir);
            const customElementsCodes = customElements.map((name) => {
                if (platform === 'app-harmony') {
                    // 鸿蒙不需要，在入口 index.generated.ets 中 define 了
                    return `export * from '@/uni_modules/${pluginId}/customElements/${name}/${name}.uts'`;
                }
                else {
                    // 自动生成 customElements.define 代码，rust 那里会根据 define 来生成注册代码
                    const codes = [];
                    const source = `@/uni_modules/${pluginId}/customElements/${name}/${name}.uts`;
                    const className = (0, shared_2.capitalize)((0, shared_2.camelize)(name));
                    const elementClassName = `${className}Element`;
                    codes.push(`import { ${elementClassName} } from '${source}'`);
                    codes.push(`customElements.define('${name.replace('uni-', '')}', ${elementClassName})`);
                    codes.push(`export * from '${source}'`);
                    return codes.join('\n');
                }
            });
            const indexFileName = resolveTscUniModuleIndexFileName(platform, pluginDir);
            if (!indexFileName) {
                const indexFileName = path_1.default.resolve(resolveOutputPluginDir(platform, process.env.UNI_INPUT_DIR, pluginDir), `utssdk/${platform}/index.uts.ts`);
                fs_extra_1.default.outputFileSync(indexFileName, customElementsCodes.join('\n'));
            }
            else {
                const indexFileContent = fs_extra_1.default.readFileSync(indexFileName, 'utf-8');
                customElementsCodes.forEach((code) => {
                    if (!indexFileContent.includes(code)) {
                        fs_extra_1.default.appendFileSync(indexFileName, '\n' + code);
                    }
                });
            }
        }
    }
    compiler.debug(`${path_1.default.basename(pluginDir)} sync files(${files.length})`, Date.now() - start);
}
exports.syncUniModuleFilesByCompiler = syncUniModuleFilesByCompiler;
const isDir = (path) => {
    const stat = fs_extra_1.default.lstatSync(path);
    if (stat.isDirectory()) {
        return true;
    }
    else if (stat.isSymbolicLink()) {
        return fs_extra_1.default.lstatSync(fs_extra_1.default.realpathSync(path)).isDirectory();
    }
    return false;
};
function resolveCustomElements(customElementsDir) {
    const pluginDir = path_1.default.resolve(customElementsDir, '..');
    if (!(0, utils_1.isCustomElementsSupported)(pluginDir)) {
        return [];
    }
    const customElements = [];
    fs_extra_1.default.readdirSync(customElementsDir).forEach((name) => {
        const folder = path_1.default.resolve(customElementsDir, name);
        if (!isDir(folder)) {
            return;
        }
        const files = fs_extra_1.default.readdirSync(folder);
        // 读取文件夹文件列表，比对文件名（fs.existsSync在大小写不敏感的系统会匹配不准确）
        // customElements 的文件名是 uts 后缀
        const ext = '.uts';
        if (files.includes(name + ext)) {
            customElements.push(name);
        }
    });
    return customElements;
}
function resolveUniModuleGlobs() {
    const extname = `.{uts,ts,json}`;
    const globs = [
        `*.uts`,
        `customElements/**/*${extname}`,
        // test-uts/common/**/*
        `common/**/*${extname}`,
        `utssdk/**/*${extname}`,
        // 不copy components目录了，不单独编译，启用vite走完整流程编译
        // `components/**/*`,
    ];
    return globs;
}
function resolveUniModuleIgnoreGlobs(platform) {
    const globs = [
        `utssdk/app-android/config.json`,
        `utssdk/app-ios/config.json`,
        `utssdk/web/**/*`,
        `utssdk/mp-*/**/*`,
    ];
    if (platform === 'app-android' || platform === 'app-ios') {
        globs.push(`utssdk/app-harmony/**/*`);
    }
    if (platform === 'app-harmony') {
        globs.push(`utssdk/app-android/**/*`);
        globs.push(`utssdk/app-ios/**/*`);
    }
    return globs;
}
function resolveUniModuleVueGlobs() {
    const extname = `.{vue,uvue}`;
    const globs = [
        `utssdk/app-android/**/*${extname}`,
        `utssdk/app-ios/**/*${extname}`,
    ];
    return globs;
}
async function syncUniModuleStaticFiles(pluginDir, outputPluginDir, preprocessor) {
    return (0, fast_glob_1.default)(`static/**/*`, {
        cwd: pluginDir,
        absolute: false,
    }).then((files) => {
        return Promise.all(files.map((fileName) => syncUniModulesFile(fileName, pluginDir, outputPluginDir, false, preprocessor).then(() => fileName)));
    });
}
async function syncUniModuleVueFiles(pluginDir, outputPluginDir, preprocessor) {
    return (0, fast_glob_1.default)(resolveUniModuleVueGlobs(), {
        cwd: pluginDir,
        absolute: false,
    }).then((files) => {
        return Promise.all(files.map((fileName) => syncUniModulesFile(fileName, pluginDir, outputPluginDir, false, preprocessor).then(() => fileName)));
    });
}
async function syncUniModulesFiles(platform, pluginDir, outputPluginDir, rename, preprocessor) {
    return (0, fast_glob_1.default)(resolveUniModuleGlobs(), {
        cwd: pluginDir,
        absolute: false,
        ignore: resolveUniModuleIgnoreGlobs(platform),
    }).then((files) => {
        return Promise.all(files.map((fileName) => syncUniModulesFile(fileName, pluginDir, outputPluginDir, rename, preprocessor).then(() => fileName)));
    });
}
async function syncUniModulesFile(relativeFileName, pluginDir, outputPluginDir, rename, preprocessor) {
    const src = path_1.default.resolve(pluginDir, relativeFileName);
    if (rename) {
        const extname = path_1.default.extname(relativeFileName);
        if (['.uts', '.json', '.vue', '.uvue'].includes(extname)) {
            // test.uts => test.uts.ts
            // test.json => test.json.ts
            return writeFile(src, path_1.default.resolve(outputPluginDir, relativeFileName + '.ts'), preprocessor);
        }
    }
    return copyFile(src, path_1.default.resolve(outputPluginDir, relativeFileName));
}
const utsModuleFileCaches = new Map();
async function writeFile(src, dest, preprocessor) {
    const stat = await fs_extra_1.default.stat(src);
    const key = src + ',' + dest;
    if (utsModuleFileCaches.get(key) === stat.mtimeMs) {
        return;
    }
    utsModuleFileCaches.set(key, stat.mtimeMs);
    return preprocessor(fs_extra_1.default.readFileSync(src, 'utf-8'), src).then((content) => fs_extra_1.default.outputFile(dest, content));
}
async function copyFile(src, dest) {
    const stat = await fs_extra_1.default.stat(src);
    const key = src + ',' + dest;
    if (utsModuleFileCaches.get(key) === stat.mtimeMs) {
        return;
    }
    utsModuleFileCaches.set(key, stat.mtimeMs);
    return fs_extra_1.default.copy(src, dest, { overwrite: true });
}
function resolveOutputPluginDir(platform, inputDir, pluginDir) {
    return path_1.default.join(process.env.UNI_APP_X_TSC_DIR, platform, path_1.default.relative(inputDir, pluginDir));
}
exports.resolveOutputPluginDir = resolveOutputPluginDir;
function resolveUVueOutputPluginDir(platform, inputDir, pluginDir) {
    return path_1.default.join(process.env.UNI_APP_X_UVUE_DIR, platform, path_1.default.relative(inputDir, pluginDir));
}
exports.resolveUVueOutputPluginDir = resolveUVueOutputPluginDir;
function resolveTscUniModuleIndexFileName(platform, pluginDir) {
    pluginDir = resolveOutputPluginDir(platform, process.env.UNI_INPUT_DIR, pluginDir);
    let indexFileName = path_1.default.resolve(pluginDir, `utssdk/${platform}/index.uts.ts`);
    if (fs_extra_1.default.existsSync(indexFileName)) {
        return indexFileName;
    }
    indexFileName = path_1.default.resolve(pluginDir, 'utssdk/index.uts.ts');
    if (fs_extra_1.default.existsSync(indexFileName)) {
        return indexFileName;
    }
}
exports.resolveTscUniModuleIndexFileName = resolveTscUniModuleIndexFileName;
