"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.originalPositionForSync = exports.originalPositionFor = exports.generatedPositionFor = exports.resolveUtsPluginSourceMapFile = exports.resolveUTSPluginSourceMapFile = exports.resolveUTSSourceMapFile = void 0;
const fs_1 = require("fs");
const path_1 = require("path");
const shared_1 = require("@vue/shared");
const source_map_1 = require("source-map");
const source_map_js_1 = require("source-map-js");
const kotlin_1 = require("./kotlin");
const utils_1 = require("./utils");
const uvue_1 = require("./uvue");
const shared_2 = require("./shared");
const kotlin_2 = require("./stacktrace/kotlin");
const EXTNAME = {
    kotlin: '.kt',
    swift: '.swift',
    arkts: '.ets',
};
const PLATFORM_DIR = {
    kotlin: 'app-android',
    swift: 'app-ios',
    arkts: 'app-harmony',
};
const uniModulesUTSPackagePrefix = 'uts.sdk.modules.';
const initUniModulesUTSKotlinSourceMapFiles = (0, shared_2.once)((inputDir, outputDir) => {
    const target = 'kotlin';
    const sourceMapFiles = new Map();
    const uniModulesDir = (0, path_1.resolve)(inputDir, 'uni_modules');
    if ((0, fs_1.existsSync)(uniModulesDir)) {
        (0, fs_1.readdirSync)(uniModulesDir).forEach((dir) => {
            if ((0, fs_1.existsSync)((0, path_1.join)(uniModulesDir, dir, 'utssdk'))) {
                sourceMapFiles.set((0, utils_1.parseKotlinPackageWithPluginId)(dir, true), (0, path_1.join)(outputDir, '../.sourcemap/app', 'uni_modules', dir, 'utssdk', PLATFORM_DIR[target], `index${EXTNAME[target]}.map`));
            }
        });
    }
    return sourceMapFiles;
});
function resolveUTSSourceMapFile(target, // | 'swift',
filename, inputDir, outputDir, cacheDir) {
    if (target !== 'kotlin') {
        throw `only support kotlin, but got ${target}`;
    }
    if (cacheDir) {
        process.env.UNI_APP_X_CACHE_DIR = cacheDir;
    }
    if (!process.env.UNI_APP_X_CACHE_DIR) {
        throw 'UNI_APP_X_CACHE_DIR is not set';
    }
    inputDir = normalizePath(inputDir);
    outputDir = normalizePath(outputDir);
    const kotlinRootOutDir = (0, kotlin_1.kotlinDir)(outputDir);
    const kotlinSrcOutDir = (0, uvue_1.kotlinSrcDir)(kotlinRootOutDir);
    if (target === 'kotlin') {
        // uts插件
        if (filename.startsWith(uniModulesUTSPackagePrefix)) {
            for (const [key, value] of initUniModulesUTSKotlinSourceMapFiles(inputDir, outputDir)) {
                if (filename.startsWith(key)) {
                    return value;
                }
            }
            throw `${filename} sourcemap not found`;
            // uni-app x 主项目
        }
        else if (filename.startsWith('uni.')) {
            (0, kotlin_2.updateUTSKotlinSourceMapManifestCache)(process.env.UNI_APP_X_CACHE_DIR);
            filename = (0, path_1.resolve)(kotlinSrcOutDir, (0, kotlin_2.parseFilenameByClassName)(filename.replace(filename.split('.', 2).join('.') + '.', '')));
        }
    }
    filename = normalizePath(filename);
    if (filename.includes('/utssdk/')) {
        // 大概率是插件
        return resolveUTSPluginSourceMapFile(target, filename, inputDir, outputDir);
    }
    const fileExtname = (0, path_1.extname)(filename);
    // 如果是源文件，比如uvue,vue,uts等，则需要转换为kt文件路径
    const isSrcFile = fileExtname !== EXTNAME[target] &&
        filename.startsWith(inputDir) &&
        !filename.startsWith(outputDir);
    if (isSrcFile) {
        filename = normalizePath((0, path_1.resolve)(kotlinSrcOutDir, (0, path_1.relative)(inputDir, filename).replace(fileExtname, EXTNAME[target])));
    }
    // 文件是 kt 或 swift
    if ((0, path_1.extname)(filename) === EXTNAME[target]) {
        const relativeFileName = (0, path_1.relative)(kotlinSrcOutDir, filename);
        const sourceMapFile = resolveSourceMapFile(relativeFileName, (0, utils_1.resolveUniAppXSourceMapPath)((0, kotlin_1.kotlinDir)(outputDir)));
        if (sourceMapFile) {
            return sourceMapFile;
        }
    }
    if (isSrcFile) {
        const sourceMapFile = resolveSourceMapFile('index.kt', (0, utils_1.resolveUniAppXSourceMapPath)((0, kotlin_1.kotlinDir)(outputDir)));
        if (sourceMapFile) {
            return sourceMapFile;
        }
    }
    throw `${filename} sourcemap not found`;
    function resolveSourceMapFile(relativeFileName, sourceMapDir) {
        const sourceMapFile = (0, path_1.resolve)(sourceMapDir, relativeFileName + '.map');
        if ((0, fs_1.existsSync)(sourceMapFile)) {
            return sourceMapFile;
        }
    }
}
exports.resolveUTSSourceMapFile = resolveUTSSourceMapFile;
function resolveUTSPluginSourceMapFile(target, filename, inputDir, outputDir) {
    inputDir = normalizePath(inputDir);
    outputDir = normalizePath(outputDir);
    filename = normalizePath(filename);
    const pluginDir = resolvePluginDir(target, inputDir, outputDir, filename);
    if (!pluginDir) {
        throw `plugin dir not found`;
    }
    const is_uni_modules = (0, path_1.basename)((0, path_1.dirname)(pluginDir)) === 'uni_modules';
    if (is_uni_modules && (target === 'kotlin' || target === 'swift')) {
        // 传入的可能是混编文件，该文件路径可能是源码工程路径，也可能是copy后路径
        const extname = EXTNAME[target];
        if (extname && filename.endsWith(extname)) {
            // 重要：必须先判断输出目录，再判断源码目录
            // 如果是copy后路径，则可能是混编文件
            if (filename.startsWith(outputDir)) {
                // 不是index.kt、index.swift、index.ets。那就是混编文件
                const outputUTSSDKDir = (0, path_1.join)(outputDir, (0, path_1.relative)(inputDir, pluginDir), 'utssdk', PLATFORM_DIR[target]);
                if (
                // 开发时目录
                filename !==
                    normalizePath((0, path_1.join)(outputUTSSDKDir, 'index' + extname)) &&
                    // 发行时目录
                    filename !==
                        normalizePath((0, path_1.join)(outputUTSSDKDir, 'src', 'index' + extname))) {
                    return normalizePath((0, path_1.relative)(outputDir, filename)) + '.fake.map';
                }
            }
            else if (filename.startsWith(inputDir)) {
                // 如果是源码工程路径，则肯定是混编文件
                return normalizePath((0, path_1.relative)(inputDir, filename)) + '.fake.map';
            }
        }
    }
    const sourceMapFile = target === 'arkts'
        ? (0, path_1.join)(outputDir, '../.sourcemap/app-harmony', (0, path_1.relative)(inputDir, pluginDir), is_uni_modules ? 'utssdk' : '', PLATFORM_DIR[target], `index${EXTNAME[target]}.map`)
        : (0, path_1.join)((0, path_1.join)(outputDir, '../.sourcemap/app'), (0, path_1.relative)(inputDir, pluginDir), is_uni_modules ? 'utssdk' : '', PLATFORM_DIR[target], `index${EXTNAME[target]}.map`);
    if (!(0, fs_1.existsSync)(sourceMapFile)) {
        throw `${sourceMapFile} not found`;
    }
    return sourceMapFile;
}
exports.resolveUTSPluginSourceMapFile = resolveUTSPluginSourceMapFile;
// 兼容旧版本
exports.resolveUtsPluginSourceMapFile = resolveUTSPluginSourceMapFile;
function resolvePluginDir(target, inputDir, outputDir, filename) {
    if (target === 'arkts') {
        const parts = normalizePath(filename).split('/uni_modules/');
        if (parts.length > 1) {
            return (0, path_1.join)(inputDir, 'uni_modules', parts[parts.length - 1].split('/')[0]);
        }
    }
    // 目标文件是编译后 kt 或 swift 或 ets
    if (filename.startsWith(outputDir)) {
        const relativePath = (0, path_1.relative)(outputDir, filename);
        const hasSrc = normalizePath(relativePath).includes('/src/');
        // uni_modules/test-uts
        if (relativePath.startsWith('uni_modules')) {
            return (0, path_1.join)(inputDir, (0, path_1.join)(relativePath, hasSrc ? '../../../..' : '../../..'));
        }
        // utssdk/test-uts
        return (0, path_1.join)(inputDir, (0, path_1.join)(relativePath, hasSrc ? '../../..' : '../..'));
    }
    else if (filename.startsWith(inputDir)) {
        let parent = (0, path_1.dirname)(filename);
        const utssdkDir = normalizePath((0, path_1.join)(inputDir, 'utssdk'));
        const uniModulesDir = normalizePath((0, path_1.join)(inputDir, 'uni_modules'));
        while (parent) {
            const dir = (0, path_1.dirname)(parent);
            if (parent === dir) {
                // windows 上边会剩下一个盘符
                return;
            }
            if (dir === utssdkDir || dir === uniModulesDir) {
                return parent;
            }
            parent = dir;
        }
        throw `${filename} is not a uts plugin file`;
    }
    else {
        throw `${filename} is not in ${inputDir} or ${outputDir}`;
    }
}
const consumers = {};
/**
 * 解析源码文件，目前 uts 的 sourcemap 存储的都是相对目录
 * @param consumer
 * @param filename
 * @returns
 */
function resolveSource(consumer, filename) {
    filename = normalizePath(filename);
    return (consumer.sources.find((source) => filename.endsWith(source)) || filename);
}
/**
 * 根据源码文件名、行号、列号，返回生成后文件、行号、列号（根据 uts 文件返回 kt|swift 文件）
 * @param originalPosition
 * @returns
 */
async function generatedPositionFor({ sourceMapFile, filename, line, column, outputDir, platform, }) {
    if (sourceMapFile.endsWith('.fake.map')) {
        const relativeSource = sourceMapFile.replace('.fake.map', '');
        return Promise.resolve({
            source: relativeSource,
            relativeSource,
            line,
            column,
            lastColumn: column,
        });
    }
    return resolveSourceMapConsumer(sourceMapFile).then((consumer) => {
        const res = consumer.generatedPositionFor({
            source: resolveSource(consumer, filename),
            line,
            column,
            bias: column === 0 ? 2 /* BIAS.LEAST_UPPER_BOUND */ : 1 /* BIAS.GREATEST_LOWER_BOUND */,
        });
        let source = null;
        let relativeSource = null;
        if (outputDir) {
            // 根据 sourceMapFile 和 outputDir，计算出生成后的文件路径
            const normalizedSourceMapFile = normalizePath(sourceMapFile);
            if (!platform) {
                // 默认 app，目前硬编码识别
                platform = normalizedSourceMapFile.includes('/.sourcemap/app-harmony/')
                    ? 'app-harmony'
                    : 'app';
            }
            const sourceMapRootDirs = [
                {
                    sourceMapRootDir: normalizePath((0, shared_2.resolveSourceMapPath)(outputDir, platform === 'app-android' || platform === 'app-ios'
                        ? 'app'
                        : platform)),
                    outputDir: outputDir,
                },
            ];
            // 理论上以下逻辑需要 app-android 下生效，但目前可能没传platform，所以默认也生效
            const kotlinOutDir = (0, kotlin_1.kotlinDir)(outputDir);
            if (kotlinOutDir) {
                sourceMapRootDirs.push({
                    sourceMapRootDir: normalizePath((0, utils_1.resolveUniAppXSourceMapPath)(kotlinOutDir)),
                    outputDir: (0, path_1.join)(kotlinOutDir, 'src'),
                });
            }
            for (const { sourceMapRootDir, outputDir } of sourceMapRootDirs) {
                if (normalizedSourceMapFile.startsWith(sourceMapRootDir)) {
                    relativeSource = normalizePath((0, path_1.relative)(sourceMapRootDir, sourceMapFile).replace('.map', ''));
                    source = (0, path_1.join)(outputDir, relativeSource);
                    break;
                }
            }
        }
        return Object.assign(res, { source, relativeSource });
    });
}
exports.generatedPositionFor = generatedPositionFor;
/**
 * 根据生成后的文件名、行号、列号，返回源码文件、行号、列号（根据 kt|swift 文件返回 uts 文件）
 * @param generatedPosition
 * @returns
 */
async function originalPositionFor(generatedPosition) {
    if (generatedPosition.sourceMapFile.endsWith('.fake.map')) {
        const relativeSource = generatedPosition.sourceMapFile.replace('.fake.map', '');
        return Promise.resolve({
            source: relativeSource,
            relativeSource,
            line: generatedPosition.line,
            column: generatedPosition.column,
            name: null,
        });
    }
    return resolveSourceMapConsumer(generatedPosition.sourceMapFile).then((consumer) => {
        const res = consumer.originalPositionFor({
            line: generatedPosition.line,
            column: generatedPosition.column,
            bias: generatedPosition.column === 0
                ? 2 /* BIAS.LEAST_UPPER_BOUND */
                : 1 /* BIAS.GREATEST_LOWER_BOUND */,
        });
        if (generatedPosition.withSourceContent &&
            res.source &&
            (0, shared_1.hasOwn)(res, 'line') &&
            (0, shared_1.hasOwn)(res, 'column')) {
            return Object.assign(res, {
                sourceContent: consumer.sourceContentFor(res.source, true),
            });
        }
        if (res.source && generatedPosition.inputDir) {
            res.source = (0, path_1.join)(generatedPosition.inputDir, res.source);
        }
        return res;
    });
}
exports.originalPositionFor = originalPositionFor;
/**
 * 根据生成后的文件名、行号、列号，返回源码文件、行号、列号（根据 kt|swift 文件返回 uts 文件）
 * 同步API
 * @param generatedPosition
 * @returns
 */
function originalPositionForSync(generatedPosition) {
    if (generatedPosition.sourceMapFile.endsWith('.fake.map')) {
        const relativeSource = generatedPosition.sourceMapFile.replace('.fake.map', '');
        return {
            source: relativeSource,
            line: generatedPosition.line,
            column: generatedPosition.column,
        };
    }
    const consumer = resolveSourceMapConsumerSync(generatedPosition.sourceMapFile);
    const res = consumer.originalPositionFor({
        line: generatedPosition.line,
        column: generatedPosition.column,
        bias: generatedPosition.column === 0
            ? 2 /* BIAS.LEAST_UPPER_BOUND */
            : 1 /* BIAS.GREATEST_LOWER_BOUND */,
    });
    if (generatedPosition.withSourceContent &&
        res.source &&
        (0, shared_1.hasOwn)(res, 'line') &&
        (0, shared_1.hasOwn)(res, 'column')) {
        return Object.assign(res, {
            sourceContent: consumer.sourceContentFor(res.source, true) ?? '',
            sourceRoot: consumer.sourceRoot,
        });
    }
    if (res.source && generatedPosition.inputDir) {
        res.source = (0, path_1.join)(generatedPosition.inputDir, res.source);
    }
    return res;
}
exports.originalPositionForSync = originalPositionForSync;
async function resolveSourceMapConsumer(sourceMapFile) {
    const stats = (0, fs_1.statSync)(sourceMapFile);
    if (!stats.isFile()) {
        throw `${sourceMapFile} is not a file`;
    }
    const cache = consumers[sourceMapFile];
    if (!cache || cache.time !== stats.mtimeMs) {
        consumers[sourceMapFile] = {
            time: stats.mtimeMs,
            consumer: await new source_map_1.SourceMapConsumer((0, fs_1.readFileSync)(sourceMapFile, 'utf8')),
        };
    }
    return consumers[sourceMapFile].consumer;
}
function resolveSourceMapConsumerSync(sourceMapFile) {
    const stats = (0, fs_1.statSync)(sourceMapFile);
    if (!stats.isFile()) {
        throw `${sourceMapFile} is not a file`;
    }
    const cache = consumers[sourceMapFile];
    if (!cache || cache.time !== stats.mtimeMs) {
        consumers[sourceMapFile] = {
            time: stats.mtimeMs,
            consumer: new source_map_js_1.SourceMapConsumer(JSON.parse((0, fs_1.readFileSync)(sourceMapFile, 'utf8'))),
        };
    }
    return consumers[sourceMapFile].consumer;
}
function normalizePath(path) {
    return path.replace(/\\/g, '/');
}
